/*
 * This filter converts Sun rasterfiles into Impress file for printing on an
 * Imagen laser printer.  Converted by Walter Underwood, Ford Aerospace, Palo
 * Alto (wunder@ford-wdl1). 
 *
 * This filter used to convert MacPaint files into Impress files for printing on
 * an Imagen laser printer.  Based on MACimp for Tops-20 by Ed Pattermann,
 * Stanford. 
 *
 * Usage:  screendump | sunimp | ipr 
 *
 * History:      
 * --------
 * 25 July 1984  Winkler created. [macimp] 
 *
 * 27 July 1984  Burgess (sumex-aim): modified write_imp_init() code to reflect
 * appropriate owner and spooldate 
 *
 * 17 Oct. 1984  W. Underwood converted to read Sun 2 bitmaps as used by the
 * "screendump" and "screenload" programs.  This is suspected to be a "memory
 * pixrect" format. 
 *
 * 12 Dec. 1984  W. Underwood changed throw_away_sun_header() to
 * read_sun_header() to read and understand the rasterfile header. 
 *
 * 13 Oct. 1986  V. Jacobson changed to work on Sun-3/O.S. 3.x.  Changed
 * to handle rasterlines that are not a multiple of 32 bits.  Changed
 * to handle Sun 3/160 (color) framebuffer.
 */
static char *RCSident = "$Header: sunimp.c,v 1.9 86/10/13 04:35:21 van Exp $";

#include <stdio.h>
#include <rasterfile.h>

#define     MAG_POWER           1	/* can be 0, 1, or 2 */
#define     DOTS_PER_INCH       300	/* For Imagen 8/300 */

#define     OPAQUE              3
#define     OPERATION_TYPE      OPAQUE

#define     LOBYTE(number) ((char)(number & 0x00FF))
#define     HIBYTE(number16) ((char)((number16 >> 8) & 0x00FF))

#define     PATCH_SIZE          32
#define     PATCH_BYTES         4

typedef char patch[PATCH_SIZE][PATCH_BYTES];

int h_patches;			/* Number of Impress patches across */
int v_patches;			/* Number of Impress patches down */
int pad_bytes;			/* Bytes to fill last row of patches */
int ras_lines;			/* number of lines in sun raster */
int ras_bytes;			/* number of bytes per line */
int horiz_pad;			/* number of pad bytes per line */
unsigned char *remap;		/* table to map color/g.s. to 0/1 */

main ()
{
	read_sun_header ();
	write_imp_init ();
	process_data ();
	write_imp_end ();
	fflush (stdout);
}

process_data ()
{
	register int i, j, k;
	register int byte, pix;

	if (remap) {
		/* handle color/grey scale raster file */
		for (i = 0; i < ras_lines; i++) {
			for (j = 0; j < ras_bytes; j++) {
				pix = 0;
				for (k = 0; k < 8; k++) {
					if ((byte = getchar ()) == EOF)
						fatal ("Premature EOF in rasterfile");
					pix <<= 1;
					pix |= remap[byte];
				}
				add_byte ((char) pix);
			}
			for (j = 0; j < horiz_pad; j++)
				add_byte ((char) 0);
		}
	} else {
		/* handle monochrome raster file */
		for (i = 0; i < ras_lines; i++) {
			for (j = 0; j < ras_bytes; j++) {
				if ((byte = getchar ()) == EOF)
					fatal ("Premature EOF in rasterfile");
				add_byte ((char) byte);
			}
			for (j = 0; j < horiz_pad; j++)
				add_byte ((char) 0);
		}
	}
	for (i = 0; i < pad_bytes; i++)
		add_byte ((char) 0);
}

add_byte (one_byte)
	char one_byte;
{
	extern int h_patches;
	extern char *calloc ();

	static patch *patches = NULL;
	static short patch_num = 0;
	static short patch_row = 0;
	static short patch_col = 0;

	if (patches == NULL)	/* First time only */
		if (! (patches = (patch *) calloc (h_patches, sizeof (patch))))
			fatal ("Cannot alloc space for conversion array.\n");

	patches[patch_num][patch_row][patch_col] = one_byte;

	if (++patch_col >= PATCH_BYTES) {
		patch_col = 0;
		if (++patch_num >= h_patches) {
			patch_num = 0;
			if (++patch_row >= PATCH_SIZE) {
				patch_row = 0;
				write_patches (patches);
			}
		}
	}
}

write_patches (patches)
	patch *patches;
{
	register int patchnum, patchrow, patchcol;

	for (patchnum = 0; patchnum < h_patches; patchnum++)
		for (patchrow = 0; patchrow < PATCH_SIZE; patchrow++)
			for (patchcol = 0; patchcol < PATCH_BYTES; patchcol++)
				putchar (patches[patchnum][patchrow][patchcol]);
}

read_sun_header ()
{
	extern char *malloc ();
	register int i, j;
	struct rasterfile rf;

	rf.ras_magic = getw (stdin);
	if (feof (stdin) != 0)
		exit (0);
	rf.ras_width = getw (stdin);
	rf.ras_height = getw (stdin);
	rf.ras_depth = getw (stdin);
	rf.ras_length = getw (stdin);
	rf.ras_type = getw (stdin);
	rf.ras_maptype = getw (stdin);
	rf.ras_maplength = getw (stdin);

	/* Start checking the header */

	if (rf.ras_magic != RAS_MAGIC)
		error ("Warning: bad magic number in rasterfile (%d)\n",
		       rf.ras_magic);

	if (rf.ras_depth != 1 && rf.ras_depth != 8)
		fatal ("Sorry, can't handle %d bits per pixel.\n",
		       rf.ras_depth);

	if (rf.ras_type > 1)
		fatal ("Sorry, can only handle type 0 or 1.  ras_type= %d\n",
		       rf.ras_type);

	/* If there is a colormap, set up to re-map it to 0/1. */
	if (rf.ras_maplength) {
		register unsigned char *r, *g, *b;

		remap = (unsigned char *)malloc (256);
		j = rf.ras_maplength / 3;
		r = (unsigned char *)malloc (j);
		g = (unsigned char *)malloc (j);
		b = (unsigned char *)malloc (j);
		fread (r, 1, j, stdin);
		fread (g, 1, j, stdin);
		fread (b, 1, j, stdin);
		for (i = 0; i < 256; i++)
			remap[i] = ((r[i]&g[i]&b[i]) != 255);

		free (r); free (g); free (b);
	}
	ras_lines = rf.ras_height;
	ras_bytes = rf.ras_width / 8;
	horiz_pad = ras_bytes % PATCH_BYTES;
	h_patches = (rf.ras_width + (PATCH_SIZE - 1)) / PATCH_SIZE;
	v_patches = (rf.ras_height + (PATCH_SIZE - 1)) / PATCH_SIZE;

	pad_bytes = h_patches * PATCH_BYTES * 
		    (v_patches * PATCH_SIZE - rf.ras_height);

}

write_imp_init ()
{
	char *getenv ();
	char *ctime (), cimptime[26];
	long time (), imptime;

	imptime = time (0);	/* get time in internal form */
	strcpy (cimptime, ctime (&imptime));	/* put time in string form */
	cimptime[24] = '\0';	/* nullify \n in time string */

	printf ("@Document(Language ImPress, Name \"Sun 3 Rasterfile\"");
	printf (", Jobheader On, Owner \"%s\"", getenv ("USER"));
	printf (", Spooldate \"%s\")", cimptime);
	/* SET_ABS_H */
	putchar ((char) 135);
	putchar (HIBYTE ((short) DOTS_PER_INCH / 4));
	putchar (LOBYTE ((short) DOTS_PER_INCH / 4));

	/* SET_ABS_V */
	putchar ((char) 137);
	putchar (HIBYTE ((short) DOTS_PER_INCH));
	putchar (LOBYTE ((short) DOTS_PER_INCH));

	/* SET_MAGNIFICATION */
	putchar ((char) 236);
	putchar ((char) MAG_POWER);

	/* BITMAP */
	putchar ((char) 235);
	putchar ((char) OPERATION_TYPE);
	putchar ((char) h_patches);
	putchar ((char) v_patches);
}

write_imp_end ()
{
	/* ENDPAGE */
	putchar ((char) 219);

	/* EOF */
	putchar ((char) 255);
}

/*
 * error - type a message on error stream 
 */

/* VARARGS1 */
error (fmt, a1, a2, a3, a4, a5)
	char *fmt;
{
	fprintf (stderr, "sunimp: ");
	fprintf (stderr, fmt, a1, a2, a3, a4, a5);
	fprintf (stderr, "\n");
}


/*
 * fatal - type an error message and abort 
 */

/* VARARGS1 */
fatal (fmt, a1, a2, a3, a4, a5)
	char *fmt;
{
	error (fmt, a1, a2, a3, a4, a5);
	exit (1);
}
