/* File: /u1/oystr/HRPC/HRpcRTS/udpTransp.c  Date: 27-Feb-1986  */

/*
 * $Header$
 * INTERFACE:	UDP transport routines
 *
 * FUNCTION:	Supply primitive UDP transport functions
 *
 * IMPORTS:	Standard
 *
 * EXPORTS:	UdpMaxBufferSize, UdpOpenLink,  UdpCloseLink,
 *		UdpSendPacket,    UdpRecvPacket, UdpInitSend,
 *		UdpFinishSend,    UdpInitRecv,	UdpFinishRecv
 *
 * DESIGN:	Sure.
 *
 * $Log$
 *
 * Revision 1.1  87/08/20  ??:??:?? mss
 * Changes for lightweight process support.
 *
 * 27-Feb-1986:	Initial implementation, Jan Sanislo
 */

#include <HRPC/basicRpc.h>
#include <HRPC/CIncludes/Binding_defs.h>
#include <HRPC/hrpcErrCodes.h>
#include <HRPC/LWP/kEvents.h>
#include <HRPC/LWP/LWPdefs.h>
#include <HRPC/LWP/LWPtypes.h>
#include "../Transports/connDefs.h"
#include <netinet/in.h>
/* #include <sys/time.h> */
#include <errno.h>

#define UDPMAXBUFFERSIZE 8096

extern int AllocateBuffer();
extern int DeallocateBuffer();
extern int errno;

int UdpMaxBufferSize( fBptr )
    HRPCBinding *fBptr;
{
    return( UDPMAXBUFFERSIZE );
}

/*
 * Create a UDP socket from which to
 * send messages.  The local port number is
 * not saved. Close only via associated socket file descriptor.
 */

int UdpOpenLink( fBptr )
    HRPCBinding *fBptr;
{
    struct sockaddr_in myAddr;
    struct sockaddr_in *destAddr;
    register TransControl *tcptr;
    register ConnDescr *connptr;
    int spktyp = (int) fBptr->speaking;
    STMconnDescr *stmconnptr = &STMconnStates[spktyp];
    HRPCBinding *STMfBptr;
    extern void setConnReadProc();
    extern int HRPC_AddToREADFDS();
    extern int HRPC_SetSocketAsync();
    extern int HRPC_LWPcreateSTM();
    lwpPROCESS STMpid;
    int sigsheldprevious;
    
    /* check that addr has been filled in */
    destAddr = (struct sockaddr_in *) &(fBptr->transDescr.netAddr);
    if ( (destAddr->sin_family != AF_INET) ||
	 (destAddr->sin_addr.s_addr == 0)  ||
	 (destAddr->sin_port <= 0) )
	fatalerr("UdpOpenLink: netAddr not initialized.\n");

    tcptr = &(fBptr->transDescr);

    /*
     * check state.
     */
    if ( tcptr->sockfd > 0 ) {
	/* already open so return */
        /* tcptr->packetArrived = 0; */
	return;
    }
    
    /*
     * Create connection -
     * make socket and create SpeakTypeMgr
     */
    tcptr->sockfd = socket(AF_INET,SOCK_DGRAM,IPPROTO_UDP);
    if ( tcptr->sockfd < 0 )
	fatalperr("UdpOpenLink: create socket.\n");
    if ( HRPC_lwpType != SingleThread ) {
	HRPC_SetSocketAsync(tcptr->sockfd,1);
	if ( stmconnptr->connCount == 0 ) {
	    HRPC_LWPcreateSTM(fBptr, &STMfBptr, &STMpid);
	}
    }

    /* actually connect to it */
    myAddr.sin_addr.s_addr = INADDR_ANY;
    myAddr.sin_port = 0;
    
    if ( bind( tcptr->sockfd, &myAddr, sizeof(struct sockaddr_in) ) < 0 )
	fatalperr("UdpOpenLink: bind.\n");
    connptr = &connStates[tcptr->sockfd];
    fBptr->connState = CONN_IDLE;
    tcptr->packetArrived = 0;
    DISABLE_SignalHandler
    connptr->masterfd = connptr->connfd = tcptr->sockfd;
    connptr->connbptr = fBptr;
    setConnReadProc(tcptr->sockfd,fBptr);
    connptr->connCount = 1;
    ENABLE_SignalHandler
    connptr->selmask = (1 << tcptr->sockfd);
    if ( HRPC_lwpType != SingleThread ) {
	if ( stmconnptr->connCount == 0 ) {
	    stmconnptr->STMpid = STMpid;
		/* sanity checks */
	    if ( stmconnptr->BindingQ != (struct BindingQelement *) 0 ) {
		printf("UdpOpenLink: warning non-empty BindingQ\n");
	        stmconnptr->BindingQ = (struct BindingQelement *) 0;
	    }
	    DISABLE_SignalHandler
	    if ( stmconnptr->BufferQ != (struct BufferQelement *) 0 ) {
		printf("UdpOpenLink: warning non-empty BufferQ\n");
	        stmconnptr->BufferQ = (struct BufferQelement *) 0;
	    }
	    if ( stmconnptr->TaskQ != (struct PacketQelement *) 0 ) {
		printf("UdpOpenLink: warning non-empty TaskQ\n");
	        stmconnptr->TaskQ = (struct PacketQelement *) 0;
	    }
	    stmconnptr->BufferQsize = 0;
	    stmconnptr->STMbptr = STMfBptr;
	    ENABLE_SignalHandler
	}
	DISABLE_SignalHandler
	HRPC_AddToREADFDS(tcptr->sockfd);
	(stmconnptr->connCount)++;
	ENABLE_SignalHandler
    } else {
	DISABLE_SignalHandler
	if ( stmconnptr->connCount == 0 ) {
		/* sanity check */
	    if ( stmconnptr->BufferQ != (struct BufferQelement *) 0 ) {
                printf("UdpOpenLink: warning non-empty BufferQ\n");
	        stmconnptr->BufferQ = (struct BufferQelement *) 0;
	    }
	    stmconnptr->BufferQsize = 0;
            stmconnptr->STMbptr = fBptr;
	}
	(stmconnptr->connCount)++;
        ENABLE_SignalHandler
    }
}

int UdpCloseLink( fBptr )
    HRPCBinding *fBptr;
{
    register struct sockaddr_in *destAddr;
    register TransControl *tcptr;
    register ConnDescr *connptr;
    int		ConnStateFD, sigsheldprevious;
    extern struct BindingQelement *CleanUpBindingQ();
    
    /* Make sure it is actually open */
    destAddr = (struct sockaddr_in *) (&fBptr->transDescr.netAddr);
    tcptr = &fBptr->transDescr;
    if ( (destAddr->sin_port == 0) || (tcptr->sockfd == 0) ) {
	/* fatalerr("UdpCloseLink: link not open.\n"); */
	return;
    }

    connptr = &connStates[tcptr->sockfd];
    DISABLE_SignalHandler
    --(connptr->connCount);
    ENABLE_SignalHandler
    if ( connptr->connCount > 0 ) {
		/* connection still in use */
	DISABLE_SignalHandler
        --(STMconnStates[(int)fBptr->speaking].connCount);
	ENABLE_SignalHandler
	CleanUpBindingQ(fBptr,1);
	destAddr->sin_port = destAddr->sin_addr.s_addr = 0;
	fBptr->connState = CONN_CLOSED;
        tcptr->packetArrived = 0;
	tcptr->sockfd = 0;
	DeallocateBuffer( fBptr );
        return;
    }
    ConnStateFD = tcptr->sockfd;
    
    DISABLE_SignalHandler
    if ( close( tcptr->sockfd ) < 0 )
	fatalperr("UdpCloseLink: cannot close link.\n");
    if ( HRPC_lwpType != SingleThread ) HRPC_RmFromREADFDS(ConnStateFD);
    ENABLE_SignalHandler
    destAddr->sin_port = destAddr->sin_addr.s_addr = 0;
    HRPC_LWPcloseSocket(fBptr, ConnStateFD);
}

/*
 * Actually send a packet to whatever is on the other
 * end of netAddr.
 *		>>>> WARNING <<<<<
 * No retries are done here.  The retry strategy is
 * actually a part of the RPC protocol which defines
 * transaction IDs, message sequence numbers, etc.
 * Arguably this should be part of the transport,
 * but we can't put it here since e.g. the CMU stuff
 * would be different.
 */

int UdpSendPacket( fBptr )
    HRPCBinding *fBptr;
{
    int cnt, nbytes;
    ConnDescr *cdptr;
    struct sockaddr_in *toAddr;
    
    nbytes = (int) (fBptr->maxBufSize - fBptr->curBufSize);
    cdptr = &connStates[fBptr->transDescr.sockfd];
    if ( fBptr->connState == CONN_CLCALL ) {
	toAddr = (struct sockaddr_in *) &fBptr->transDescr.netAddr;
    }
    else {
	toAddr = (struct sockaddr_in *) &(fBptr->transDescr.replyTo);
    }
    
    cnt = sendto(fBptr->transDescr.sockfd,
		 (char *) fBptr->currentBuffer,
		 nbytes,
		 0, toAddr,
		 sizeof( struct sockaddr_in ) );
    if ( cnt != nbytes )
	fatalperr("UdpSendPacket: can't send full packet.\n");
}
    
/*
 * (try to) Receive a packet.  The timeout period is
 * wired to the SUN default.  This timeout should be
 * a configurable parameter - again based on what the
 * higher level RPC protocol expects.  However, see
 * the comment above.
 *
 * Return values:
 *	 1: packet received
 *	 0: time out
 *	-1: other error (actually blows us away for now)
 *
 * Unwarranted assumptions:
 *   The memory buffer in the binding is "free".
 */

int UdpRecvPacket( fBptr )
    HRPCBinding *fBptr;
{
    struct timeval tmo, *tmoptr;
    int readmask, readables;
    int selectStatus, readStatus, fromlen;
    struct sockaddr_in from;
    struct sockaddr_in *peer;
    struct sockaddr_in *fromptr;
    ConnDescr *cdptr;
    struct BindingQIdTableEntry idTable[BindingQIdTableMax];
    extern int  AddToBindingQ();

    readmask = 1 << fBptr->transDescr.sockfd;
    cdptr = &connStates[fBptr->transDescr.sockfd];
    if ( fBptr->connState == CONN_SRVCALL ) {
	tmoptr = (struct timeval *) 0;
    }
    else {
	tmoptr = &tmo;
	tmo.tv_sec  = 30;
        tmo.tv_usec = 0;
    }
    
loop:
    readables = readmask;
    if ( HRPC_lwpType != SingleThread ) {
	/* construct BindingQelement & add to connStates[fd].BindingQ */
	    /* first, set IdTable */
	if ( fBptr->connState != CONN_WCALL ) {
	    idTable[0].IdValue = fBptr->rpcDescr.replyId;
	    idTable[0].getPacketIdValue = fBptr->rpcDescr.GetReplyId;
	    AddToBindingQ(fBptr,idTable,1,0);
	} else
	    AddToBindingQ(fBptr,idTable,0,0);
    }
    (*HRPC_lwpDescr.SelectAndReadPacket)(fBptr, tmoptr, &readables,
						&selectStatus, &readStatus);
    if ( selectStatus < 0 )
	    /* should not happen in env with multiple threads */
	if ( errno == EINTR ) goto loop;
	else
	    fatalperr("UdpRecvPacket: select error.\n");

    if ( selectStatus == 0 )
	return( HRPC_REPLYTMO ) /* timed out */;

    if ( readables != readmask )
	fatalerr("UdpRecvPacket: select mask mismatch.\n");

	/* check status of read */
    if ( readStatus <= 0 ) {
	fatalperr("UdpRecvPacket: read error: %d.\n",readStatus);
    }

    return( 0 );
}

/*
 * Low-level buffer allocation/deallocation routines.
 */

memory UdpBufAlloc()
{
    memory newmem;

    newmem = (memory) malloc(UDPMAXBUFFERSIZE);
    if ( newmem == (memory) 0 ) {
	fatalerr("UdpBufAlloc: no memory.\n");
    }
    
    return( newmem );
}

int UdpBufDealloc( buffer )
    memory buffer;
{
    if ( (int) buffer > 0 )
	free( buffer );
    else
	fatalerr("UdpBufDealloc: bad buffer.\n");
}

/*
 * These are necessary for two reasons:
 *   1) some protocols (e.g. SPP) require
 *      explicit manipulation of transport-level packet headers
 *	(e.g. EOM bit).
 *   2) transport level operations are sensitive to whether we
 *	are receiving a request (e.g. must save network return
 *	address in the case of UDP) or a reply to a request.
 *	Likewise for sending request or reply.
 * These routines are called exclusively by RPC protocol routines.
 */

/*ARGSUSED*/
int UdpInitSend( fBptr, fDirec )
    HRPCBinding *fBptr;
    int fDirec;
{
    fBptr->connState = fDirec;
}

/*ARGSUSED*/
int UdpFinishSend( fBptr, fDirec )
    HRPCBinding *fBptr;
    int fDirec;
{
    fBptr->connState = fDirec;

    UdpSendPacket( fBptr );
    fBptr->connState = CONN_IDLE;
}

/*ARGSUSED*/
int UdpInitRecv( fBptr, fDirec )
    HRPCBinding *fBptr;
    int fDirec;
{
    fBptr->connState = fDirec;
    AllocateBuffer( fBptr );
    return( UdpRecvPacket( fBptr ) );
}

/*ARGSUSED*/
int UdpFinishRecv( fBptr, fDirec )
    HRPCBinding *fBptr;
    int fDirec;
{
    fBptr->connState = CONN_IDLE;
}
