/* File: /u1/oystr/HRPC/HRpcRTS/rpcSun.c  Date:  1-Mar-1986  */

/*
 * $Header$
 * INTERFACE:	Implements fake SUN RPC protocol
 *
 * FUNCTION:	See above
 *
 * IMPORTS:	basicRpc.h + SUN files
 *		+ rpcSunHdrs.c
 *		    SunCallHdr, SunReplyHdr, SunAuthInfo
 *
 * EXPORTS:	Standard RPC emulation routines
 *
 * DESIGN:	Why not.  It is probably worth writing down our
 *		terminology wrt the SUN record marking standard
 *		A message consists of a single record.  A record
 *		consists multiple segments.  Each segment begins
 *		with a four-byte header containing the number of
 *		bytes in the segment.  The last segment of the
 *		message has the high order bit set.  A buffer is
 *		a chunk of data read off the TCP connection.  A
 *		buffer is fragmented if it does not contain a full
 *		segment.  A buffer is multi-segmented if it contains
 *		parts of more than one segment.  Note that a buffer can
 *		be both fragmented and multi-segmented.
 *
 * DEFECTS:	Major changes required to support tcp transport.
 *		Which presents an interesting question - how to
 *		integrate transport dependencies into a different
 *		level of protocol.
 *
 *		Unwarranted assumptions:  It is possible that, for
 *		TCP connections, we may wind up with a fragment of
 *		the segment header in the buffer (e.g. first two bytes).
 *		We can thus get into a two level recursion of the form
 *		otw detects data needed -> rpc next packet -> otw
 *		segment header detects data needed -> rpc next packet.
 *		The assumption is that the last 'next packet' will actually
 *		do a read and that there will be enough data there to
 *		satisfy (minimally) the segment header otw.  By extension
 *		we generally assume that if we run out of data, the
 *		next read will produce enough to satisfy all outstanding
 *		otw requests.
 * $Log$
 * Revision 1.1  87/01/??  ??:??:?? mss
 * Changes for lightweight process support.
 *
 *  1-Mar-1986:	Initial implementation, Jan Sanislo
 */

/*
 * SUN specific stuff.
 */
#include <sys/types.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <rpc/auth.h>
#include <rpc/rpc_msg.h>
#include <sys/time.h>
#include <HRPC/basicRpc.h>
#include <HRPC/hrpcErrCodes.h>
#include <HRPC/bufferOps.h>
#include <netinet/in.h>
#include "../Transports/connDefs.h"
#include "../RpcProtos/rpcSun.h"


struct opaque_auth sunAuthNULL /* Must be 0 */;

/* "forward" defs. */
extern SunPacketOutgoing();
extern SunPacketIncoming();
extern SunAnswerPacket();
extern SunReplyPacket();
extern int AllocateBuffer();

SunInitOutgoing( fBptr, fProcedureNum, fVersNum )
    HRPCBinding *fBptr;
    int		 fProcedureNum;
    int		 fVersNum;
{
    struct   rpc_msg      *callHdr;
    register OtwControl   *otwptr = &(fBptr->otwDescr);
    register TransControl *tcptr  = &(fBptr->transDescr);
    struct   timeval	   now;
    SunCallInfo		  *scptr;
    int			   dummy = 0;
 
    /*
     * Open the connection.
     *		>>>>> WARNING <<<<<
     * Watch out for this call to OpenLink given DTC's
     * BS about "availability". Actually, this is a tough
     * call.
     */
    (*tcptr->OpenLink)( fBptr );
    (*tcptr->InitSend)( fBptr, CONN_CLCALL );
    
    /*
     * Set direction, get a buffer.
     */
    fBptr->otwOp = OTW_ENCODE;
    AllocateBuffer( fBptr );

    /*
     * See if we have a call header we can reuse.
     */
    if ( fBptr->rpcDescr.callState == (memory) NULL ) {
	scptr = (SunCallInfo *) calloc( 1, sizeof( SunCallInfo ) );
	if ( scptr == (SunCallInfo *) NULL ) {
	    fatalerr("SunInitOutgoing: can't alloc call info.\n");
	}
	fBptr->rpcDescr.callState = (memory) scptr;
    }
    else {
	scptr = (SunCallInfo *) fBptr->rpcDescr.callState;
    }
    callHdr = &(scptr->callHdr);

    if ( fBptr->transType != UDPTRANSP ) {
	/* Using a stream protocol, save buffer position for segment mark */
	GetBufPos( fBptr, &(scptr->countPos) );
	fBptr->curBufMark += sizeof(LongCardinal);
	fBptr->curBufSize -= sizeof(LongCardinal);
    }
    
    /*
     * Generate a header and save it where we need it.
     */
    (void) gettimeofday( &now, (struct timezone *) 0);
    callHdr->rm_xid = getpid() ^ (int) random() ^ now.tv_sec ^ now.tv_usec;
    callHdr->rm_direction = CALL;
    callHdr->rm_call.cb_rpcvers = RPC_MSG_VERSION;
    callHdr->rm_call.cb_prog = (int) fBptr->bndProgNum;
    callHdr->rm_call.cb_vers = fVersNum;
    callHdr->rm_call.cb_proc = fProcedureNum;

    /* Always use null authen. -- stand by for big hassle later */
    /* 		>>>>> WARNING <<<<<
     * We know we are doing an encode at this point.
     */
    callHdr->rm_call.cb_cred = sunAuthNULL;
    callHdr->rm_call.cb_verf = sunAuthNULL;

    fBptr->rpcDescr.PutPacket = SunPacketOutgoing;
    fBptr->rpcDescr.replyId = callHdr->rm_xid;
    SunCallHdr( fBptr, callHdr );

    /* Header encoded in output buffer, done here */
    return;
}

SunPacketOutgoing( fBptr )
    HRPCBinding *fBptr;
{
    BufPos oldPos;
    int    byteCount;
    SunCallInfo *scptr;
    
    if ( fBptr->transType != UDPTRANSP ) {
	/* Stuff segment byte count and send it off */
	scptr = (SunCallInfo *) fBptr->rpcDescr.callState;
	SetBufPos( fBptr, &(scptr->countPos), &oldPos );
	byteCount = oldPos.bufMark - scptr->countPos.bufMark
		      - sizeof(LongCardinal);
	(*fBptr->otwDescr.LongCardinal)(fBptr, &byteCount);
	SetBufPos( fBptr, &oldPos, 0);
    }
    else {
	/* should not happen if using UDP */
	fatalerr("SunPacketOutgoing: should not happen.\n");
    }
    (*fBptr->transDescr.SendPacket)(fBptr);
    if ( fBptr->transType != UDPTRANSP ) {
	/* record buffer position for next packet and save space */
	GetBufPos( fBptr, &(scptr->countPos) );
	fBptr->curBufMark += sizeof(LongCardinal);
	fBptr->curBufSize -= sizeof(LongCardinal);
    }
}

SunFinishOutgoing( fBptr )
    register HRPCBinding *fBptr;
{
    BufPos oldPos;
    int    byteCount;
    SunCallInfo *scptr;
    
    if ( fBptr->transType != UDPTRANSP ) {
	/* Stuff segment byte count and send it off */
	scptr = (SunCallInfo *) fBptr->rpcDescr.callState;
	SetBufPos( fBptr, &(scptr->countPos), &oldPos );
	byteCount = oldPos.bufMark - scptr->countPos.bufMark
		       - sizeof(LongCardinal);
	/* last segment, set indicator */
	byteCount |= LASTSEGTAG;
	(*fBptr->otwDescr.LongCardinal)(fBptr, &byteCount);
	SetBufPos( fBptr, &oldPos, 0);
    }

    (*fBptr->transDescr.FinishSend)(fBptr, CONN_CLCALL);
}


/*
 * Following is very nasty due to necessity of implementing
 * retransmit policy at RPC protocol level.  To quote DTC:
 * "This is not right!! Why is it doing this?!?!".
 *		>>>>> WARNING <<<<<
 * Major changes required for tcp adaptation.
 * Also have to redo send/receive buffers in presence of
 * multiple outstanding messages.
 */

#define MAXRECVTRIES 4
extern HRPCErrRec *SunReplyHdr();

SunInitAnswer( fBptr )
    HRPCBinding *fBptr;
{
    int		    rcvTries, rcvStat;
    struct rpc_msg *origCall;
    struct rpc_msg  reply;
    SunCallInfo    *scptr;
    struct PacketQelement sentPacketInfo, recvdPacketInfo;
    extern int	    AddToBufferQ();
    int		    saveBufSize;
    HRPCErrRec	   *errptr;

    /*
     * Hang around waiting for a reply.
     *		>>>>> WARNING <<<<<
     * If using TCP should check NEXTSEGINBUF flag -
     * no read if more data avail.  Can only happen
     * in LWP environment with multiplexed calls, which
     * means that we need more routing info in packet anyway.
     * Plus major problems, since connection is set up to be
     * one way only (either sending or receiving).
     */
	/* save sent packet in case we need to resend */
    sentPacketInfo.BufSize = fBptr->curBufSize;
    sentPacketInfo.BufMark = fBptr->curBufMark;
    sentPacketInfo.CurrentBuffer = fBptr->currentBuffer;
    fBptr->currentBuffer = 0;	/* (indirectly) force InitRecv to allocate 
					* a new buffer */
retry_loop:
    rcvStat = (*fBptr->transDescr.InitRecv)(fBptr, CONN_WREPLY);
	/* save packet allocated (indirectly) by InitRecv */
    recvdPacketInfo.BufSize = fBptr->curBufSize;
    recvdPacketInfo.BufMark = fBptr->curBufMark;
    recvdPacketInfo.CurrentBuffer = fBptr->currentBuffer;
    rcvTries = 1;

    while ( (rcvStat == HRPC_REPLYTMO) && (rcvTries < MAXRECVTRIES) ) {
	    /* Resend the original packet if not TCP */
	if ( fBptr->transType == UDPTRANSP ) {
		/* I will be sorry for doing this...*/
		/* grab packet to resend */
	    fBptr->curBufSize = sentPacketInfo.BufSize;
	    fBptr->curBufMark = sentPacketInfo.BufMark;
	    fBptr->currentBuffer = sentPacketInfo.CurrentBuffer;
		/* resend packet */
	    (*fBptr->transDescr.InitSend)(fBptr, CONN_CLCALL);
	    (*fBptr->transDescr.SendPacket)( fBptr );
		/* reinstate packet allocated (indirectly) by InitRecv */
	    fBptr->curBufSize = recvdPacketInfo.BufSize;
	    fBptr->curBufMark = recvdPacketInfo.BufMark;
	    fBptr->currentBuffer = recvdPacketInfo.CurrentBuffer;
	}
	rcvStat = (*fBptr->transDescr.InitRecv)(fBptr, CONN_WREPLY);
	rcvTries++;
    }

    /*
     * Check for timeout.
     */
    if ( (rcvTries >= MAXRECVTRIES) || rcvStat ) {
        struct sockaddr_in *sinp;

	sinp = (struct sockaddr_in *) &fBptr->transDescr.netAddr;
	errmsg("SunInitAnswer: no reply in %d retries, rcvStat 0x%x\n",
		rcvTries,rcvStat);
	fatalerr("   Program# %d @ IP addr %08x.\n",
		 fBptr->ifdPtr->progNum, sinp->sin_addr.s_addr);
    }

    /*
     * Decode the reply.
     */
    fBptr->otwOp = OTW_DECODE;
    scptr = (SunCallInfo *) fBptr->rpcDescr.callState;
    fBptr->rpcDescr.GetPacket = SunAnswerPacket;
    
    if ( fBptr->transType != UDPTRANSP ) {
	(*fBptr->otwDescr.LongCardinal)(fBptr, &scptr->segSize);
	scptr->bufSize = fBptr->curBufSize;
	scptr->segSize &= ~ LASTSEGTAG;
	SetFragFlags( scptr, fBptr );
    }
    
    errptr = SunReplyHdr( fBptr, &reply );
    if ( errptr ) {
	longjmp( fBptr->unwindTo, errptr );
    }

    /*
     * Make sure it is the one we are looking for.
     *		>>>>> WARNING <<<<<
     * If using TCP, then must scarf up remaining segments
     * (read to end of record).
     */
    if ( reply.rm_direction != REPLY ) {
	errmsg("SunInitAnswer: packet not a reply: %d.\n",reply.rm_direction);
	goto retry_loop;
    }
    origCall = (struct rpc_msg *) &(scptr->callHdr);
    if ( origCall->rm_xid != reply.rm_xid ) {
	errmsg("SunInitAnswer: request/reply mismatch: %d != %d.\n",
		origCall->rm_xid, reply.rm_xid);
	goto retry_loop;
    }

	/* don't need sent packet anymore, so release it's buffer */
    AddToBufferQ(fBptr->speaking,sentPacketInfo.CurrentBuffer);
}

SunAnswerPacket( fBptr )
    HRPCBinding *fBptr;
{
    SunCallInfo *scptr;
    
    if ( fBptr->transType == UDPTRANSP ) {
	fatalerr("SunAnswerPacket: can't happen with UDP.\n");
    }
    scptr = (SunCallInfo *) fBptr->rpcDescr.callState;

    /*
     * Check to see if next segment in buffer.
     */
    if ( scptr->fragFlags & NEXTSEGINBUF ) {
	/* (re)set bytes remaining */
	fBptr->curBufSize = scptr->bufSize - scptr->fragSize;
	scptr->fragFlags &= ~ NEXTSEGINBUF;
        /* In case of fragmented count field */
	scptr->segSize = scptr->fragSize = 4;
	(*fBptr->otwDescr.LongCardinal)(fBptr,&scptr->segSize);
	scptr->segSize &= ~LASTSEGTAG;
    }
    else {
	/*
	 * Must read more data.
	 * First fake segment size down to bytes expected - bytes consumed
	 */
	if ( scptr->fragFlags & FRAGINBUF ) {
	    scptr->segSize -= scptr->fragSize;
	   (*fBptr->transDescr.RecvPacket)( fBptr );
	}
	else {
            /*
             * it is time to read a whole new segment into the buffer.
             * the last segment was neither fragmented, nor accompanied
             * by another segment in curbuf (NEXTSEGINBUF not there)
             */
            (*fBptr->transDescr.RecvPacket)(fBptr);
            /*
             * since this a whole new segment, we have to recompute
             * its size and (of course) readjust fBptr->curBufSTUFF
             * to reflect our knowledge of the new segment
             *  (stolen fom SunInitAnswer)
             */
            (*fBptr->otwDescr.LongCardinal)(fBptr, &scptr->segSize);
            scptr->bufSize = fBptr->curBufSize;
            scptr->segSize &= ~ LASTSEGTAG;   
	}
    }
    /*
     * Sometimes we can get out of synch. The usual cases are
     * 1) segment header is *last* four bytes in the buffer,
     * and 2) when we do a read, we only get 4 bytes of a segment
     * header, leaving an empty data buffer.  Therefore, make sure
     * we leave here with the buffer containing some data to be
     * consumed.
     */
    if ( fBptr->curBufSize == 0 ) {
	(*fBptr->transDescr.RecvPacket)(fBptr);
	scptr->bufSize = fBptr->curBufSize;
    }
    SetFragFlags( scptr, fBptr );
}

SunFinishAnswer( fBptr )
    register HRPCBinding *fBptr;
{
    /*
     * Just free up the per call info for now
     *		>>>>> WARNING <<<<
     * This finishes one call - may still be something
     * in buffer related to next call.  Watch out for
     * this.
     */
    (void) free( fBptr->rpcDescr.callState );
    fBptr->rpcDescr.callState = (memory) NULL;
    (*fBptr->transDescr.FinishRecv)(fBptr, CONN_CLREPLY);
}

SunCloseRpc( fBptr )
    register HRPCBinding *fBptr;
{
    if ( fBptr->rpcDescr.callState != (memory) NULL ) {
	free( fBptr->rpcDescr.callState );
	fBptr->rpcDescr.callState = (memory) NULL;
    }
}

/*
 * SunGetReplyId obtains and returns (via replyId) the replyId portion of the
 * packet header, which is stored in the buffer associated with the binding
 * argument.  The binding argument should be the binding of an STM.  The code
 * is primarily stolen from SunInitAnswer.
 */
SunGetReplyId(fBptr,replyId)
    HRPCBinding	*fBptr;
    u_long	*replyId;
{
	struct rpc_msg	replyHdr;
	SunCallInfo	*scptr;

		/* See if we have a call header we can reuse. */
	if ( fBptr->rpcDescr.callState == (memory) NULL ) {
	    scptr = (SunCallInfo *) calloc( 1, sizeof( SunCallInfo ) );
	    if ( scptr == (SunCallInfo *) NULL ) {
		fatalerr("SunGetReplyId: can't alloc call info.\n");
	    }
	    fBptr->rpcDescr.callState = (memory) scptr;
	} else {
	    scptr = (SunCallInfo *) fBptr->rpcDescr.callState;
	}

		/* Decode the reply. */
	fBptr->otwOp = OTW_DECODE;
	if ( fBptr->transType != UDPTRANSP ) {
	    (*fBptr->otwDescr.LongCardinal)(fBptr, &scptr->segSize);
	    scptr->bufSize = fBptr->curBufSize;
	    scptr->segSize &= ~ LASTSEGTAG;
	    SetFragFlags( scptr, fBptr );
	}
	SunReplyHdr( fBptr, &replyHdr );
	*replyId = replyHdr.rm_xid;
}
