/* File: /u1/oystr/HRpcRTS/hrpcErr.c  Date: 24-Oct-1986  */

/*
 * $Header$
 * INTERFACE:	To error routines.
 *
 * FUNCTION:	(Very) Minimal support for error handling.
 *
 * IMPORTS:	Error codes.
 *
 * EXPORTS:	NewHRPCErrRec (only used internally by runtime)
 *		HRPCperror (can be used by application level code)
 *		HRPCAbortOnErr (produce core dump on error)
 *
 * DESIGN:	Very ad hoc.  Gotchas include use of LWPs and
 *		the "polymorphic" nature of some errors.  This
 *		stuff is almost guaranteed to choke on the REPORTS-type
 *		error handling, but at least will provide a clue.
 *
 * DEFECTS: 		>>>> WARNING <<<<
 *		Import and Export codes are defined in
 *		HRPC/Interfaces/Binding.cr.  These codes are
 *		technically defined by a remote interface.
 * $Log$
 * 24-Oct-1986:	Initial implementation, Jan Sanislo
 */

#include <stdio.h>
#include <HRPC/hrpcErrCodes.h>

/* "forward" */
extern void HRPCperror();

static int abortOnErr = 0;

int HRPCAbortOnErr( fDoIt )
    int fDoIt;
{
    int old;

    old = abortOnErr;
    abortOnErr = fDoIt;
    return( old );
}

HRPCErrRec *NewHRPCErrRec( fCode, fSize )
    HRPCErrVal fCode;
    int fSize;
{
    HRPCErrRec *eptr;

    if ( abortOnErr ) {
	HRPCErrRec tmperr;

	tmperr.hrpcErr = fCode;
	tmperr.errDetails = (char *) 0;
	HRPCperror( &tmperr );
	abort();
    }

    eptr = (HRPCErrRec *) calloc(1, sizeof(HRPCErrRec) );
    if ( eptr == (HRPCErrRec *) 0 ) {
	fatalerr("Can't alloc error record.\n");
	/*NOTREACHED*/
    }
    eptr->hrpcErr = fCode;
    if ( fSize > 0 ) {
	eptr->errDetails = (char *) calloc( 1, fSize );
	if ( eptr->errDetails == (char *) 0 ) {
	    fatalerr("Can't alloc error details.\n");
	}
    }
    return( eptr );
}

void HRPCFreeErr ( fErrRec )
    HRPCErrRec *fErrRec;
{
    free( fErrRec );
}

/*
 * These error messages should actually be in a database a la Eden.
 * As a matter of fact, the error message stuff was one of the
 * nicer things about Eden, but since I never made a big deal about
 * it, it has fallen into disuse, the brilliant technology rusting
 * away.  Oh well, it was just a VMS ripoff anyhow.
 *
 * Better yet, we should have a message server.  But that brings
 * up all sorts of other problems.
 *
 * Confronted by this multitude of possibilities, we will just do
 * our usual shoddy job.
 */

char *hrpcErrMsgs[] = {
    "no error",
    "timed out waiting for reply",
    "remote peer died",
    "no such program number",
    "no such procedure number",
    "bad parameter",
    "version number mismatch",
    "authentication error",
    "stub/binding mismatch",
    "RPC protocol version mismatch",
    "remote system error",
    "unspecified error",
    "server not running"
};

char *importErrMsgs[] = {
    "no error",
    "can't talk to name server",
    "can't talk to binding agent",
    "name not registered in name service",
    "bad SPEAK type",
    "bad stringname syntax",
    "interface not exported",
    "unknown host name"
};

char *exportErrMsgs[] = {
    "no error",
    "can't talk to name server",
    "can't talk to binding agent",
    "interface already registered",
    "bad SPEAK type",
    "bad stringname syntax"
};

typedef struct {
    char *facilityName;
    char **facilityMsgs;
} Facilities;

#define MIN_FACIL 1
#define MAX_FACIL 3

Facilities allFacilities[] = {
    {0, 0},
    {"HRPC", hrpcErrMsgs},
    {"HRPC Import", importErrMsgs},
    {"HRPC Export", exportErrMsgs}
};

extern int errno;

void HRPCperror( fErr )
    HRPCErrRec *fErr;
{
    int facil, errcode;
    
    if ( mIsSysErr(fErr) ) {
	facil = errno;
	errno = (int) fErr->hrpcErr;
	perror("HRPC");
	errno = facil;
	return;
    }

    facil = (fErr->hrpcErr & 0xFFFF0000) >> 16;
    errcode = (fErr->hrpcErr & 0xFFFF);

    if ( (facil < MIN_FACIL) || (facil > MAX_FACIL) ) {
	fprintf(stderr,"HRPC undifferentiated error code %d (0x%x).\n",
		fErr->hrpcErr,fErr->hrpcErr);
    }
    else {
	/* take a chance on low order bits */
	fprintf(stderr,"%s: %s.\n",allFacilities[facil].facilityName,
		allFacilities[facil].facilityMsgs[errcode]);
    }
}

	    
