/*****
 *
 * File: cellscr.c
 *
 * Cellsim, cellular automata simulator
 *
 * SunView-specific routines
 *
 *****/


#include "cellscr.h"

/* Note: Original black & white code was supplied by Carl Uhrik, and
 * subsequently modified by D. Hiebeler for better speed and
 * functionality.
 */

/*
 *
 * Cellsim copyright 1989, 1990 by Chris Langton and Dave Hiebeler
 * (cgl@lanl.gov, hiebeler@heretic.lanl.gov)
 *
 * This package may be freely distributed, as long as you don't:
 * - remove this notice
 * - try to make money by doing so
 * - prevent others from copying it freely
 * - distribute modified versions without clearly documenting your changes
 *   and notifying us
 *
 * Please contact either of the authors listed above if you have questions
 * or feel an exception to any of the above restrictions is in order.
 *
 * If you make changes to the code, or have suggestions for changes,
 * let us know!  If we use your suggestion, you will receive full credit
 * of course.
 */

/*****
 * Cellsim history:
 *
 * Cellsim was originally written on Apollo workstations by Chris Langton.
 *
 * Sun versions:
 *
 * - version 1.0
 *   by C. Ferenbaugh and C. Langton
 *   released 09/02/88
 *
 * - version 1.5
 *   by Dave Hiebeler and C. Langton  May - June 1989
 *   released 07/03/89
 *
 * - version 2.0
 *   by Dave Hiebeler and C. Langton  July - August 1989
 *   never officially released (unofficially released 09/08/89)
 *
 * - version 2.5
 *   by Dave Hiebeler and C. Langton  September '89 - February 1990
 *   released 02/26/90
 *****/


    
    
/****************************** initialization ******************************/

remove_frame_args(argc, argv)
    int    *argc;
    char  **argv;
{
    frame = window_create(NULL, FRAME, FRAME_LABEL, "Cellsim 2.5",
			  FRAME_ARGC_PTR_ARGV, argc, argv, 0);
    if (frame == NULL) {
	perror("cellsim");
	exit(1);
    }
}


init_scr()
{
    char   *str;
    int     i, button_vsize=20;
    /* Panel_item item; */
    Menu tmp_menu;
    Pixfont *screenr11, *screenr12;
    char CM_port_str[20];

    screenr11 = pf_open("/usr/lib/fonts/fixedwidthfonts/screen.r.11");
    if (screenr11 == NULL) {
	fprintf(stderr, "Couldn't open /usr/lib/fonts/fixedwidthfonts/screen.r.11\n");
	fprintf(stderr, "Using default system font.  The control panel may suffer problems\n");
	fprintf(stderr, "because of this.\n");
	screenr11 = pf_default();
    }
    screenr12 = pf_open("/usr/lib/fonts/fixedwidthfonts/screen.r.12");
    if (screenr12 == NULL) {
	fprintf(stderr, "Couldn't open /usr/lib/fonts/fixedwidthfonts/screen.r.12\n");
	fprintf(stderr, "Using default system font.  The default popups may suffer\n");
	fprintf(stderr, "because of this.\n");
	screenr12 = pf_default();
    }
    panel = window_create(frame, PANEL, WIN_FONT, screenr11, 0);
    pan_time = panel_create_item(panel, PANEL_MESSAGE,
				 PANEL_ITEM_X, ATTR_COL(5),
				 PANEL_ITEM_Y, ATTR_ROW(0),
				 PANEL_LABEL_STRING, "         ", 0);
    panel_create_item(panel, PANEL_MESSAGE,
		      PANEL_LABEL_STRING, "Time:",
		      PANEL_ITEM_X, ATTR_COL(0),
		      PANEL_ITEM_Y, ATTR_ROW(0),
		      0);
    show_time(0);
    pan_pos = panel_create_item(panel, PANEL_MESSAGE,
				PANEL_ITEM_X, ATTR_COL(19),
				PANEL_ITEM_Y, ATTR_ROW(0),
				PANEL_LABEL_STRING, "         ", 0);
    panel_create_item(panel, PANEL_MESSAGE,
		      PANEL_LABEL_STRING, "Pos:",
		      PANEL_ITEM_X, ATTR_COL(15),
		      PANEL_ITEM_Y, ATTR_ROW(0),
		      0);
    pan_state = panel_create_item(panel, PANEL_MESSAGE,
				  PANEL_ITEM_X, ATTR_COL(7),
				  PANEL_ITEM_Y, ATTR_ROW(1),
				  PANEL_LABEL_STRING, "    ",0);
    panel_create_item(panel, PANEL_MESSAGE,
		      PANEL_LABEL_STRING, "State:",
		      PANEL_ITEM_X, ATTR_COL(0),
		      PANEL_ITEM_Y, ATTR_ROW(1),
		      0);
    pan_planes = panel_create_item(panel, PANEL_MESSAGE,
				   PANEL_ITEM_X, ATTR_COL(19),
				   PANEL_ITEM_Y, ATTR_ROW(1),
				   PANEL_LABEL_STRING, "     ",
				   0);
    panel_create_item(panel, PANEL_MESSAGE,
		      PANEL_LABEL_STRING, "Planes:",
		      PANEL_ITEM_X, ATTR_COL(12),
		      PANEL_ITEM_Y, ATTR_ROW(1),
		      0);
    show_planes(AMASK);
    pan_mode = panel_create_item(panel, PANEL_CYCLE,
				 PANEL_LABEL_STRING, "Mode:",
				 PANEL_NOTIFY_PROC, mode_proc,
				 PANEL_ITEM_X, ATTR_COL(0),
				 PANEL_ITEM_Y, ATTR_ROW(2),
				 PANEL_LABEL_X, ATTR_COL(0),
				 PANEL_CHOICE_STRINGS,
				   "Edit", "Probe", "Plane Edit", 0,
				 PANEL_VALUE_X, pan_lvalue, 0);
    pan_stop_on = panel_create_item(panel, PANEL_CYCLE,
				    PANEL_LABEL_STRING, "Stop on:",
				    PANEL_NOTIFY_PROC, stop_on_proc,
				    PANEL_ITEM_X, ATTR_COL(0),
				    PANEL_ITEM_Y, ATTR_ROW(3),
				    PANEL_LABEL_X, ATTR_COL(0),
				    PANEL_CHOICE_STRINGS,
				   "Never", "No change", "All zero", 0,
				    PANEL_VALUE_X, pan_lvalue, 0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Run", run_proc,
			   MENU_STRING_ITEM, "Skip", skip_run_proc,
			   MENU_STRING_ITEM, "Bounded", bound_run_proc,
			   MENU_STRING_ITEM, "Skip/Bounded", sb_run_proc,
			   MENU_STRING_ITEM, "Screenful", screenful_proc,
			   MENU_STRING_ITEM, "Step", step_proc,
			   MENU_STRING_ITEM, "Single Skip", skip_proc,
			   MENU_STRING_ITEM, "Run Local", run_local_proc,
			   MENU_STRING_ITEM, "Step Local", step_local_proc,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Run", 8, NULL),
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 90,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    defaults_image_menu = menu_create(
				  MENU_STRING_ITEM, "Save in Image-dir", 1,
				  MENU_STRING_ITEM, "Save uncompressed", 2,
				  MENU_STRING_ITEM, "Set Image-dir", 3,
				  MENU_STRING_ITEM, "Disable auto-change", 4,
				  MENU_ACTION_PROC, defaults_image_menu_proc,
				  0);
    defaults_rule_menu = menu_create(
				  MENU_STRING_ITEM, "Save in Table-dir", 1,
				  MENU_STRING_ITEM, "Save uncompressed", 2,
				  MENU_STRING_ITEM, "Set Table-dir", 3,
				  MENU_STRING_ITEM, "Set Fcn-dir", 4,
				  MENU_STRING_ITEM, "Disable auto-change", 5,
				  MENU_ACTION_PROC, defaults_rule_menu_proc,
				  0);
    defaults_cmap_menu = menu_create(
				 MENU_STRING_ITEM, "Save in Cmap-dir", 1,
				 MENU_STRING_ITEM, "Set Cmap-dir", 2,
				 MENU_ACTION_PROC, defaults_cmap_menu_proc,
				  0);
    defaults_CM_menu = menu_create(
				   MENU_STRING_ITEM, "host", 1,
				   MENU_STRING_ITEM, "port", 2,
				   MENU_STRING_ITEM, "run-delay", 3,
				   MENU_STRING_ITEM, "Enable FB display", 4,
				   MENU_STRING_ITEM, "Disable Sun display", 5,
				   MENU_STRING_ITEM, "Set CM Image-dir", 6,
				   MENU_STRING_ITEM, "Set CM Fcn-dir", 7,
				   MENU_STRING_ITEM, "Set CM LT-dir", 8,
				   MENU_STRING_ITEM, "Set zoom factor", 9,
				   MENU_STRING_ITEM, "Set pan", 10,
				   MENU_ACTION_PROC, defaults_CM_menu_proc,
				   0);
    defaults_menu = menu_create(
				MENU_ACTION_ITEM, "Image", def_im_popup_proc,
				MENU_ACTION_ITEM, "Rule", def_rule_popup_proc,
				MENU_ACTION_ITEM, "Cmap", def_cmap_popup_proc,
				MENU_ACTION_ITEM, "CM", def_CM_popup_proc,
				MENU_ACTION_ITEM, "Change nhood / #states",
				   change_stuff_proc,
				MENU_ACTION_ITEM, "Change image-size",
				   change_image_size_proc,
				MENU_ACTION_ITEM, "Change working directory",
				   chdir_proc,
				0);
    defaults_button = panel_create_item(panel, PANEL_BUTTON,
				PANEL_LABEL_IMAGE,
				  panel_button_image(panel,"Defaults",8,NULL),
				PANEL_ITEM_X, 110,
				PANEL_ITEM_Y, 90,
				PANEL_EVENT_PROC, button_event_proc,
				PANEL_CLIENT_DATA, defaults_menu,
				0);
    image_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			     MENU_STRING_ITEM, "Load", iload,
			     MENU_STRING_ITEM, "Save", isave_proc,
			     MENU_STRING_ITEM, "Load/Resize", iload_resize,
			     MENU_STRING_ITEM, "Save/Resize", isave_resize,
			     MENU_STRING_ITEM, "Save/Raster", isave_raster,
			     MENU_STRING_ITEM, "Load/CAM6", iload_CAM6,
			     MENU_STRING_ITEM, "Save/CAM6", isave_CAM6,
			     MENU_STRING_ITEM, "Enter", ienter,
			     MENU_STRING_ITEM, "Clear", iclear,
			     MENU_STRING_ITEM, "Swap", iswap,
			     MENU_STRING_ITEM, "Copy", icopy,
			     MENU_STRING_ITEM, "Invert", iinvert,
			     0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Image", 8, NULL),
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 90+button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, image_menu,
		      0);
    rules_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			     MENU_STRING_ITEM, "Load", load_rule,
			     MENU_STRING_ITEM, "Save", tsave_proc,
			     MENU_STRING_ITEM, "Clear", tclear_proc,
			     MENU_STRING_ITEM, "Alter", talter,
			     MENU_STRING_ITEM, "Disable symmetry", tsymmetry,
			     MENU_STRING_ITEM, "Generate", tgenerate,
			     MENU_STRING_ITEM, "Lambda", lambda_proc,
			     MENU_STRING_ITEM, "Lambda Step", lambda_step_proc,
			     MENU_STRING_ITEM, "Rho", rho_proc,
			     MENU_STRING_ITEM, "Set parm1", edit_parm1,
			     MENU_STRING_ITEM, "Set parm2", edit_parm2,
			     0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Rules", 8, NULL),
		      PANEL_ITEM_X, 110,
		      PANEL_ITEM_Y, 90+button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, rules_menu,
		      0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Load", bload,
			   MENU_STRING_ITEM, "Save", bsave,
			   MENU_STRING_ITEM, "Swap", bswap,
			   MENU_STRING_ITEM, "Xor", bxor,
			   MENU_STRING_ITEM, "Or", bor,
			   MENU_STRING_ITEM, "And", band,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Buffer", 8, NULL),
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 90+2*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Array", clear_proc,
			   MENU_STRING_ITEM, "Zero time", zero_proc,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Clear", 8, NULL),
		      PANEL_ITEM_X, 110,
		      PANEL_ITEM_Y, 90+2*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Learn", learn_proc,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Learn", 8, NULL),
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 90+3*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Set", set_proc,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Set", 8, NULL),
		      PANEL_ITEM_X, 110,
		      PANEL_ITEM_Y, 90+3*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Toggle", zoom_proc,
			   MENU_STRING_ITEM, "Magnify", magnify_proc,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Zoom", 8, NULL),
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 90+4*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Closeup", closeup_proc,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Closeup", 8, NULL),
		      PANEL_ITEM_X, 110,
		      PANEL_ITEM_Y, 90+4*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Zoom in", zoom_in_proc,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Zoom in", 8, NULL),
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 90+5*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Zoom out", zoom_out_proc,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Zoom out", 8, NULL),
		      PANEL_ITEM_X, 110,
		      PANEL_ITEM_Y, 90+5*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Quick random", quick_random_proc,
			   MENU_STRING_ITEM, "General random", general_random_proc,
			   MENU_STRING_ITEM, "Horizontal line", horiz_line_proc,
			   MENU_STRING_ITEM, "Vertical line", vert_line_proc,
			   MENU_STRING_ITEM, "Arbitrary line", arb_line_proc,
			   MENU_STRING_ITEM, "Hollow_circle", hollow_circ_proc,
			   MENU_STRING_ITEM, "Shaded circle", shaded_circ_proc,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Draw", 8, NULL),
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 90+6*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Load", cmap_load,
			   MENU_STRING_ITEM, "Save", cmap_save_proc,
			   MENU_STRING_ITEM, "Alter", cmap_alter,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Color", 8, NULL),
		      PANEL_ITEM_X, 110,
		      PANEL_ITEM_Y, 90+6*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Reset", plane_reset_proc,
			   MENU_STRING_ITEM, "Display", plane_display_proc,
			   MENU_STRING_ITEM, "Clear", pclear,
			   MENU_STRING_ITEM, "Invert", pinvert,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Planes", 8, NULL),
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 90+7*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    CM_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Connect to CM", CM_connect,
			   MENU_STRING_ITEM, "Send image", CM_send_image,
			   MENU_STRING_ITEM, "Get image", CM_get_image_proc,
			   MENU_STRING_ITEM, "Send LT", CM_send_LT,
			   MENU_STRING_ITEM, "Send cmap", CM_send_cmap,
			   MENU_STRING_ITEM, "Disconnect", CM_disconnect,
			   MENU_STRING_ITEM, "Load image from CMFE", CM_load_image_proc,
			   MENU_STRING_ITEM, "Load rule from CMFE", CM_load_rule_proc,
			   MENU_STRING_ITEM, "Clear CM array", CM_clear,
			   MENU_STRING_ITEM, "CM quick random", CM_quick_random_proc,
			   MENU_STRING_ITEM, "CM general random", CM_general_random_proc,
			   0);
     panel_create_item(panel, PANEL_BUTTON,
		       PANEL_LABEL_IMAGE, panel_button_image(panel, "CM", 8, NULL),
		       PANEL_ITEM_X, 110,
		       PANEL_ITEM_Y, 90+7*button_vsize,
		       PANEL_EVENT_PROC, button_event_proc,
		       PANEL_NOTIFY_PROC, do_first_item_proc,
		       PANEL_CLIENT_DATA, CM_menu,
		       0);
     tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Sequence 1", sequence1_proc,
			   MENU_STRING_ITEM, "Sequence 2", sequence2_proc,
			   MENU_STRING_ITEM, "Sequence 3", sequence3_proc,
			   MENU_STRING_ITEM, "Sequence 4", sequence4_proc,
			   MENU_STRING_ITEM, "Sequence 5", sequence5_proc,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Sequence", 8, NULL),
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 90+8*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    tmp_menu = menu_create(MENU_NOTIFY_PROC, menu_notify_proc,
			   MENU_STRING_ITEM, "Quit", quit_proc,
			   0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel, "Quit", 8, NULL),
		      PANEL_ITEM_X, 110,
		      PANEL_ITEM_Y, 90+8*button_vsize,
		      PANEL_EVENT_PROC, button_event_proc,
		      PANEL_NOTIFY_PROC, do_first_item_proc,
		      PANEL_CLIENT_DATA, tmp_menu,
		      0);
    for (i=0; i < NUM_MSGS; i++) {
	pan_msg[i] = panel_create_item(panel, PANEL_MESSAGE,
				       PANEL_ITEM_X, 10,
				       PANEL_ITEM_Y, 270+i*20,
				       PANEL_LABEL_STRING, "                                ",
				       0);
    }
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel,"Up",6,NULL),
		      PANEL_ITEM_X, 80,
		      PANEL_ITEM_Y, 450,
		      PANEL_MENU_CHOICE_STRINGS, "Up", 0,
		      PANEL_SHOW_MENU, TRUE,
		      PANEL_NOTIFY_PROC, up_proc,
		      0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel,"Left",6,NULL),
		      PANEL_ITEM_X, 20,
		      PANEL_ITEM_Y, 470,
		      PANEL_MENU_CHOICE_STRINGS, "Left", 0,
		      PANEL_SHOW_MENU, TRUE,
		      PANEL_NOTIFY_PROC, left_proc,
		      0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel,"Shift",6,NULL),
		      PANEL_ITEM_X, 80,
		      PANEL_ITEM_Y, 470,
		      PANEL_MENU_CHOICE_STRINGS, "Shift", 0,
		      PANEL_SHOW_MENU, TRUE,
		      PANEL_NOTIFY_PROC, shift_proc,
		      0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel,"Right",6,NULL),
		      PANEL_ITEM_X, 140,
		      PANEL_ITEM_Y, 470,
		      PANEL_MENU_CHOICE_STRINGS, "Right", 0,
		      PANEL_SHOW_MENU, TRUE,
		      PANEL_NOTIFY_PROC, right_proc,
		      0);
    panel_create_item(panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(panel,"Down",6,NULL),
		      PANEL_ITEM_X, 80,
		      PANEL_ITEM_Y, 490,
		      PANEL_MENU_CHOICE_STRINGS, "Down", 0,
		      PANEL_SHOW_MENU, TRUE,
		      PANEL_NOTIFY_PROC, down_proc,
		      0);
    window_fit_width(panel);
    cursor = cursor_create(CURSOR_SHOW_CROSSHAIRS, TRUE,
			   CURSOR_IMAGE, &cursor_pixrect,
			   CURSOR_OP, PIX_SRC ^ PIX_DST,
			   CURSOR_CROSSHAIR_OP, PIX_SRC ^ PIX_DST,
			   CURSOR_XHOT, 8, CURSOR_YHOT, 8,
			   CURSOR_CROSSHAIR_COLOR, curr_color,
			   CURSOR_CROSSHAIR_THICKNESS, 2,
			   CURSOR_CROSSHAIR_LENGTH, 12,
			   CURSOR_CROSSHAIR_GAP, 5, 0);
    canvas = window_create(frame, CANVAS,
			   CANVAS_WIDTH, MAXSIZE, CANVAS_HEIGHT, MAXSIZE,
			   WIN_HEIGHT, MAXSIZE, WIN_WIDTH, MAXSIZE,
			   WIN_EVENT_PROC, screen_event_proc,
			   WIN_CONSUME_KBD_EVENT, WIN_ASCII_EVENTS,
			   WIN_CONSUME_PICK_EVENT, WIN_IN_TRANSIT_EVENTS,
			   CANVAS_RETAINED, FALSE,
			   CANVAS_REPAINT_PROC, display_image,
			   WIN_CURSOR, cursor, 0);
    window_fit(frame);
    window_set(panel, WIN_HEIGHT, WIN_EXTEND_TO_EDGE, 0);
    pw = canvas_pixwin(canvas);
    window_pw = (Pixwin *)window_get(frame, WIN_PIXWIN);
    panel_pw = (Pixwin *)window_get(panel, WIN_PIXWIN);

    icon = icon_create(ICON_IMAGE, &icon_pixrect, 0);
    window_set(frame, FRAME_ICON, icon, 0);

    edit_menu = menu_create(MENU_NROWS, 4, 0);
    for (i = 1; i < 8; i++) {
	str = (char *) malloc(2);
	str[0] = to_char(i);
	str[1] = NULL;
	menu_set(edit_menu, MENU_STRING_ITEM, str, i, 0);
    }
    probe_menu = menu_create(MENU_STRINGS, "Alter", 0, 0);
    planeedit_menu = menu_create(MENU_NROWS, 8, 0);
    for (i = 0; i < 8; i++) {
	str = (char *) malloc(10);
	sprintf(str, "Plane %d", i);
	menu_set(planeedit_menu, MENU_STRING_ITEM, str, (1 << i), 0);
    }
    change_canvas_menus();
    popup = window_create(frame, FRAME, FRAME_SHOW_LABEL, FALSE,
			  WIN_FONT, screenr11,
			  WIN_SHOW, FALSE, 0);
    pop_panel = window_create(popup, PANEL, WIN_FONT, screenr11, 0);
    pop_input = panel_create_item(pop_panel, PANEL_TEXT,
				  PANEL_NOTIFY_PROC, return_proc,
				  PANEL_VALUE_STORED_LENGTH, BUFLEN,
				  PANEL_VALUE_DISPLAY_LENGTH, POP_DISP_LEN,
				  PANEL_VALUE, "", 0);
    window_fit(pop_panel);
    window_fit(popup);

    def_im_popup = window_create(frame, FRAME,
				 /* FRAME_SHOW_LABEL, FALSE, */
				 FRAME_LABEL, "Image defaults",
				 WIN_SHOW, FALSE,
				 WIN_FONT, screenr12,
				 0);
    def_im_panel = window_create(def_im_popup, PANEL, 0);
    def_im_save_item = two_item_cycle(def_im_panel, "Save in:",
				      "Current working dir", "Image-dir",
				      1, 2);
    def_im_compress_item = two_item_cycle(def_im_panel, "Save:",
					  "Compressed", "Uncompressed",
					  1, 4);
    def_im_imagedir_item = panel_create_item(def_im_panel, PANEL_TEXT,
					     PANEL_LABEL_STRING, "Image-dir:",
					     PANEL_VALUE, Image_dir,
					     PANEL_ITEM_X, ATTR_COL(1),
					     PANEL_ITEM_Y, ATTR_ROW(6),
					     0);
    def_im_autochange_item = two_item_cycle(def_im_panel, "Auto change:",
					    "Enabled", "Disabled",
					    1, 8);
    panel_create_item(def_im_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(def_im_panel,"Done",0,0),
		      PANEL_NOTIFY_PROC, def_im_done_proc,
		      PANEL_ITEM_X, ATTR_COL(1),
		      PANEL_ITEM_Y, ATTR_ROW(10),
		      0);
    window_fit(def_im_panel);
    window_fit(def_im_popup);

    def_rule_popup = window_create(frame, FRAME,
				   /* FRAME_SHOW_LABEL, FALSE, */
				   FRAME_LABEL, "Rule defaults",
				   WIN_SHOW, FALSE,
				   WIN_FONT, screenr12,
				   0);
    def_rule_panel = window_create(def_rule_popup, PANEL, 0);
    def_rule_save_item = two_item_cycle(def_rule_panel, "Save in:",
				      "Current working dir", "Table-dir",
				      1, 2);
    def_rule_compress_item = two_item_cycle(def_rule_panel, "Save:",
					  "Compressed", "Uncompressed",
					  1, 4);
    def_rule_tabledir_item = panel_create_item(def_rule_panel, PANEL_TEXT,
					     PANEL_LABEL_STRING, "Table-dir:",
					     PANEL_VALUE, Table_dir,
					     PANEL_ITEM_X, ATTR_COL(1),
					     PANEL_ITEM_Y, ATTR_ROW(6),
					     0);
    def_rule_fcndir_item = panel_create_item(def_rule_panel, PANEL_TEXT,
					     PANEL_LABEL_STRING, "Fcn-dir:",
					     PANEL_VALUE, Fcn_dir,
					     PANEL_ITEM_X, ATTR_COL(1),
					     PANEL_ITEM_Y, ATTR_ROW(8),
					     0);
    def_rule_autochange_item = two_item_cycle(def_rule_panel, "Auto change:",
					    "Enabled", "Disabled",
					    1, 10);
    panel_create_item(def_rule_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(def_rule_panel,"Done",0,0),
		      PANEL_NOTIFY_PROC, def_rule_done_proc,
		      PANEL_ITEM_X, ATTR_COL(1),
		      PANEL_ITEM_Y, ATTR_ROW(12),
		      0);
    window_fit(def_rule_panel);
    window_fit(def_rule_popup);

    def_cmap_popup = window_create(frame, FRAME,
				   /* FRAME_SHOW_LABEL, FALSE, */
				   FRAME_LABEL, "Cmap defaults",
				   WIN_SHOW, FALSE,
				   WIN_FONT, screenr12,
				   0);
    def_cmap_panel = window_create(def_cmap_popup, PANEL, 0);
    def_cmap_save_item = two_item_cycle(def_cmap_panel, "Save in:",
				      "Current working dir", "Cmap-dir",
				      1, 2);
    def_cmap_compress_item = two_item_cycle(def_cmap_panel, "Save:",
					  "Compressed", "Uncompressed",
					  1, 4);
    def_cmap_cmapdir_item = panel_create_item(def_cmap_panel, PANEL_TEXT,
					     PANEL_LABEL_STRING, "Cmap-dir:",
					     PANEL_VALUE, Cmap_dir,
					     PANEL_ITEM_X, ATTR_COL(1),
					     PANEL_ITEM_Y, ATTR_ROW(6),
					     0);
    panel_create_item(def_cmap_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(def_cmap_panel,"Done",0,0),
		      PANEL_NOTIFY_PROC, def_cmap_done_proc,
		      PANEL_ITEM_X, ATTR_COL(1),
		      PANEL_ITEM_Y, ATTR_ROW(8),
		      0);
    window_fit(def_cmap_panel);
    window_fit(def_cmap_popup);

    def_CM_popup = window_create(frame, FRAME,
				 /* FRAME_SHOW_LABEL, FALSE,*/
				 FRAME_LABEL, "CM defaults",
				 WIN_SHOW, FALSE,
				 WIN_FONT, screenr12,
				 0);
    def_CM_panel = window_create(def_CM_popup, PANEL, 0);
    def_CM_host_item = panel_create_item(def_CM_panel, PANEL_TEXT,
					     PANEL_LABEL_STRING, "CMFE host:",
					     PANEL_VALUE, CM_hostname,
					     PANEL_ITEM_X, ATTR_COL(1),
					     PANEL_ITEM_Y, ATTR_ROW(2),
					     0);
    sprintf(CM_port_str, "%d", CM_port);
    def_CM_port_item = panel_create_item(def_CM_panel, PANEL_TEXT,
					     PANEL_LABEL_STRING, "CMFE port:",
					     PANEL_VALUE, CM_port_str,
					     PANEL_ITEM_X, ATTR_COL(1),
					     PANEL_ITEM_Y, ATTR_ROW(4),
					     0);
    def_CM_rundelay_item = panel_create_item(def_CM_panel, PANEL_SLIDER,
					     PANEL_LABEL_STRING, "Run delay:",
					     PANEL_VALUE, 0,
					     PANEL_MIN_VALUE, 0,
					     PANEL_MAX_VALUE, 1000000,
					     PANEL_SLIDER_WIDTH, 300,
					     PANEL_ITEM_X, ATTR_COL(1),
					     PANEL_ITEM_Y, ATTR_ROW(6),
					     0);
    def_CM_display_item = panel_create_item(def_CM_panel, PANEL_TOGGLE,
					    PANEL_LABEL_STRING, "Display on:",
					    PANEL_LAYOUT, PANEL_HORIZONTAL,
					    PANEL_CHOICE_STRINGS,
					      "CM Frame-buffer",
					      "Sun display",
					      0,
					    PANEL_TOGGLE_VALUE, 1, TRUE,
					    PANEL_ITEM_X, ATTR_COL(1),
					    PANEL_ITEM_Y, ATTR_ROW(8),
					    0);
    def_CM_imagedir_item = panel_create_item(def_CM_panel, PANEL_TEXT,
					     PANEL_LABEL_STRING, "Image-dir:",
					     PANEL_ITEM_X, ATTR_COL(1),
					     PANEL_ITEM_Y, ATTR_ROW(10),
					     0);
    def_CM_fcndir_item = panel_create_item(def_CM_panel, PANEL_TEXT,
					     PANEL_LABEL_STRING, "Fcn-dir:",
					     PANEL_ITEM_X, ATTR_COL(1),
					     PANEL_ITEM_Y, ATTR_ROW(12),
					     0);
    def_CM_LTdir_item = panel_create_item(def_CM_panel, PANEL_TEXT,
					     PANEL_LABEL_STRING, "LT-dir:",
					     PANEL_ITEM_X, ATTR_COL(1),
					     PANEL_ITEM_Y, ATTR_ROW(14),
					     0);
    def_CM_zoom_item = panel_create_item(def_CM_panel, PANEL_SLIDER,
					 PANEL_LABEL_STRING, "Zoom factor:",
					 PANEL_VALUE, CM_zoom,
					 PANEL_MIN_VALUE, 0,
					 PANEL_MAX_VALUE, 127,
					 PANEL_SLIDER_WIDTH, 300,
					 PANEL_ITEM_X, ATTR_COL(1),
					 PANEL_ITEM_Y, ATTR_ROW(16),
					 PANEL_NOTIFY_LEVEL, PANEL_ALL,
					 PANEL_NOTIFY_PROC, CM_send_zoom_proc,
					 0);
    def_CM_panx_item = panel_create_item(def_CM_panel, PANEL_SLIDER,
					 PANEL_LABEL_STRING, "X pan:",
					 PANEL_VALUE, CM_panx,
					 PANEL_MIN_VALUE, 0,
					 PANEL_MAX_VALUE, 512,
					 PANEL_SLIDER_WIDTH, 300,
					 PANEL_ITEM_X, ATTR_COL(1),
					 PANEL_ITEM_Y, ATTR_ROW(18),
					 PANEL_NOTIFY_LEVEL, PANEL_ALL,
					 PANEL_NOTIFY_PROC, CM_send_panx_proc,
					 0);
    def_CM_pany_item = panel_create_item(def_CM_panel, PANEL_SLIDER,
					 PANEL_LABEL_STRING, "Y pan:",
					 PANEL_VALUE, CM_pany,
					 PANEL_MIN_VALUE, 0,
					 PANEL_MAX_VALUE, 512,
					 PANEL_SLIDER_WIDTH, 300,
					 PANEL_ITEM_X, ATTR_COL(1),
					 PANEL_ITEM_Y, ATTR_ROW(20),
					 PANEL_NOTIFY_LEVEL, PANEL_ALL,
					 PANEL_NOTIFY_PROC, CM_send_pany_proc,
					 0);
    panel_create_item(def_CM_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE, panel_button_image(def_CM_panel,"Done",0,0),
		      PANEL_NOTIFY_PROC, def_CM_done_proc,
		      PANEL_ITEM_X, ATTR_COL(1),
		      PANEL_ITEM_Y, ATTR_ROW(22),
		      0);
    window_fit(def_rule_panel);
    window_fit(def_rule_popup);

    if (blackwhite)
	tmpmag = mem_create(MAXSIZE, MAXSIZE, 1);
    else
	tmpmag = mem_create(MAXSIZE, MAXSIZE, 8);
    mag = MAXSIZE / B;		/* initialize zoom pixrect & variables */
    saved_mag_factor = mag;
    side = B;

    if (blackwhite)
	init_textures();		/* set up textures */
    else
	init_cmap();			/* set up colormap */

} /* init_scr */


static int
def_im_popup_proc()
{
    window_set(def_im_popup, WIN_SHOW, TRUE, 0);
}


void
def_im_get_settings()
{
    save_in_image_dir = (int)panel_get_value(def_im_save_item);
    save_images_compressed = 1 - (int)panel_get_value(def_im_compress_item);
    strcpy(Image_dir, (char *)panel_get_value(def_im_imagedir_item));
    auto_image_change = 1 - (int)panel_get_value(def_im_autochange_item);
}    

static void
def_im_done_proc()
{
    def_im_get_settings();
    window_set(def_im_popup, WIN_SHOW, FALSE, 0);
}


static int
def_rule_popup_proc()
{
    window_set(def_rule_popup, WIN_SHOW, TRUE, 0);
}


void
def_rule_get_settings()
{
    save_in_LT_dir = (int)panel_get_value(def_rule_save_item);
    save_LT_compressed = 1 - (int)panel_get_value(def_rule_compress_item);
    strcpy(Table_dir, (char *)panel_get_value(def_rule_tabledir_item));
    strcpy(Fcn_dir, (char *)panel_get_value(def_rule_fcndir_item));
    auto_nhood_change = 1 - (int)panel_get_value(def_rule_autochange_item);
}


static void
def_rule_done_proc()
{
    def_rule_get_settings();
    window_set(def_rule_popup, WIN_SHOW, FALSE, 0);
}


static int
def_cmap_popup_proc()
{
    /* window_loop(def_cmap_popup); */
    window_set(def_cmap_popup, WIN_SHOW, TRUE, 0);
}


void
def_cmap_get_settings()
{
    save_in_cmap_dir = (int)panel_get_value(def_cmap_save_item);
    save_cmap_compressed = 1 - (int)panel_get_value(def_cmap_compress_item);
    strcpy(Cmap_dir, (char *)panel_get_value(def_cmap_cmapdir_item));
}


static void
def_cmap_done_proc()
{
    def_cmap_get_settings();
    window_set(def_cmap_popup, WIN_SHOW, FALSE, 0);
}


static int
def_CM_popup_proc()
{
    window_set(def_CM_popup, WIN_SHOW, TRUE, 0);
}


void
def_CM_get_settings()
{
    strcpy(CM_hostname, (char *)panel_get_value(def_CM_host_item));
    sscanf((char *)panel_get_value(def_CM_port_item), "%hd", &CM_port);
    CM_delay = (int)panel_get_value(def_CM_rundelay_item);
    use_FB = (int)panel_get(def_CM_display_item, PANEL_TOGGLE_VALUE, 0);
    use_Sun_disp = (int)panel_get(def_CM_display_item, PANEL_TOGGLE_VALUE, 1);
    strcpy(CM_image_dir, (char *)panel_get_value(def_CM_imagedir_item));
    strcpy(CM_fcn_dir, (char *)panel_get_value(def_CM_fcndir_item));
    strcpy(CM_LT_dir, (char *)panel_get_value(def_CM_LTdir_item));
    if (use_CM) {
	CM_set_displays();
	CM_send_delay();
	CM_send_dirs();
    }
    CM_zoom = (int)panel_get_value(def_CM_zoom_item);
    CM_panx = (int)panel_get_value(def_CM_panx_item);
    CM_pany = (int)panel_get_value(def_CM_pany_item);
}
    

static void
def_CM_done_proc()
{
    def_CM_get_settings();
    window_set(def_CM_popup, WIN_SHOW, FALSE, 0);
}


/*
 * Call one of the routines from the "Image" submenu of the "Defaults menu
 */
static int
defaults_image_menu_proc(m,mi)
Menu m;
Menu_item mi;
{
    int val;

    val = (int)menu_get(mi, MENU_VALUE);
    switch (val) {
      case 1:
	toggle_save_in_image_dir_proc();
	break;
      case 2:
	toggle_save_images_compressed();
	break;
      case 3:
	set_image_dir();
	break;
      case 4:
	toggle_auto_image_change_proc();
	break;
      default:
	clear_msg();
	show_msg("Invalid menu selection");
	break;
    }
}


/*
 * Call one of the routines from the "Rules" submenu of the "Defaults" menu
 */
static int
defaults_rule_menu_proc(m,mi)
Menu m;
Menu_item mi;
{
    int val;

    val = (int)menu_get(mi, MENU_VALUE);
    switch (val) {
      case 1:
	toggle_save_in_LT_dir_proc();
	break;
      case 2:
	toggle_save_LT_compressed();
	break;
      case 3:
	set_table_dir();
	break;
      case 4:
	set_fcn_dir();
	break;
      case 5:
	toggle_auto_nhood_change_proc();
	break;
      default:
	clear_msg();
	show_msg("Invalid menu selection");
	break;
    }
}


/*
 * "Cmap" submenu
 */
static int
defaults_cmap_menu_proc(m,mi)
Menu m;
Menu_item mi;
{
    int val;

    val = (int)menu_get(mi, MENU_VALUE);
    switch (val) {
      case 1:
	toggle_save_in_cmap_dir_proc();
	break;
      case 2:
	set_cmap_dir();
	break;
      default:
	clear_msg();
	show_msg("Invalid menu selection");
	break;
    }
}



/*
 * "CM" submenu
 */
static int
defaults_CM_menu_proc(m,mi)
Menu m;
Menu_item mi;
{
    int val;

    val = (int)menu_get(mi, MENU_VALUE);
    switch (val) {
      case 1:
	CM_change_host();
	break;
      case 2:
	CM_change_port();
	break;
      case 3:
	CM_change_delay_proc();
	break;
      case 4:
	FB_toggle();
	break;
      case 5:
	Sun_disp_toggle();
	break;
	/*
      case 6:
	CM_image_dir_proc();
	break;
      case 7:
	CM_fcn_dir_proc();
	break;
      case 8:
	CM_LT_dir_proc();
	break;
	*/
      case 9:
	CM_set_zoom();
	break;
      case 10:
	/* CM_set_pan(); */
	break;
      default:
	clear_msg();
	show_msg("Invalid menu selection");
	break;
    }
}


/*
 * This gets called when the image-size changes, to reset a few vars
 */
change_scr()
{
    char tmp_str[80];
    
    mag = MAXSIZE / B;
    saved_mag_factor = mag;
    side = B;
    if (rule_shown) {
	/* strcpy(tmp_str, saved_rule_file);
	show_rule(tmp_str); */
	show_rule(saved_rule_file);
    }
    else
	clear_rule();
    panel_set_value(def_CM_zoom_item, CM_zoom);
    panel_set_value(def_CM_panx_item, B/2);
    panel_set_value(def_CM_pany_item, B/2);
}


void
set_CM_sliders()
{
    panel_set_value(def_CM_zoom_item, CM_zoom);
    panel_set_value(def_CM_panx_item, B/2);
    panel_set_value(def_CM_pany_item, B/2);
}


/*
 * Change the menus in the canvas, when the number of states gets changed
 */
change_canvas_menus()
{
    Menu_item item;
    int i;

    for (i = 0; i < S; i++) {
	item = menu_get(edit_menu, MENU_NTH_ITEM, i);
	menu_set(item, MENU_INACTIVE, FALSE, 0);
    }
    for (i = S; i < 16; i++) {
	item = menu_get(edit_menu, MENU_NTH_ITEM, i);
	menu_set(item, MENU_INACTIVE, TRUE, 0);
    }
    for (i=1; i <= L; i++) {
	item = menu_get(planeedit_menu, MENU_NTH_ITEM, i);
	menu_set(item, MENU_INACTIVE, FALSE, 0);
    }
    for (i = L+1; i <= 8; i++) {
	item = menu_get(planeedit_menu, MENU_NTH_ITEM, i);
	menu_set(item, MENU_INACTIVE, TRUE, 0);
    }
}

/*
 * Set up the cursor based on current mode
 */
static int
mode_proc(item, value, event)
    Panel_item item;
    int     value;
    Event  *event;
{
    screen_mode = value;
    switch (value) {			/* change cursor according to mode */
    case MODE_EDIT:
	cursor_set(cursor, CURSOR_CROSSHAIR_COLOR, curr_color,
		   CURSOR_CROSSHAIR_OP, /* PIX_SRC, */ PIX_SRC ^ PIX_DST, 0);
	break;
    case MODE_PROBE:
	cursor_set(cursor, CURSOR_CROSSHAIR_COLOR, AMASK,
		   CURSOR_CROSSHAIR_OP, PIX_SRC ^ PIX_DST, 0);
	break;
    case MODE_PLANEEDIT:
	cursor_set(cursor, CURSOR_CROSSHAIR_COLOR, curr_planemask,
		   CURSOR_CROSSHAIR_OP, PIX_SRC ^ PIX_DST, 0);
	break;
    }
    window_set(canvas, WIN_CURSOR, cursor, 0);
}


/*
 * Set the "stop_on" variable according to teh current setting of the
 * switch in the control panel
 */
static int
stop_on_proc(item, value, event)
    Panel_item item;
    int     value;
    Event  *event;
{
    stop_on_mode = value;
}


/*
 * Set magnification
 */
set_mag(m)
    int     m;
{
    saved_mag_factor = m;
    pw_writebackground(pw, 0, 0, MAXSIZE, MAXSIZE, PIX_SRC);
    mag = m;
    display_image();
    if (B * mag < MAXSIZE) {
	pw_vector(pw, B * mag, 0, B * mag, B * mag, PIX_SRC, -1);
	pw_vector(pw, 0, B * mag, B * mag, B * mag, PIX_SRC, -1);
    }
} /* set_mag */



/*
 * Do necessary SunView stuff when plane mask is reset
 */
plane_reset_proc()
{
    int     m = AMASK;

    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(plane_reset_proc);
    pw_putattributes(pw, &m);
    display_image();
    show_planes(AMASK);
}



/*
 * Set plane display-mask
 */
plane_display_proc()
{
    int     m;
    char    buf[BUFLEN];
    
    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(plane_display_proc);
    get_msg("Display mask:", buf, BUFLEN, saved_display_mask);
    if (buf[0] == 0)
	return;
    strcpy(saved_display_mask, buf);
    if (sscanf(buf, "%x", &m) != 1) {
	show_msg("Illegal mask");
	return;
    }
    m &= AMASK;
    pw_writebackground(pw, 0, 0, MAXSIZE, MAXSIZE, PIX_SRC);
    pw_putattributes(pw, &m);
    display_image();
    show_planes(m);
}


/*
 * Quit the tool.  We have to be careful so that, if the user starts
 * to quit but then cancels, we won't detach from the CMFE if we're attached.
 */
quit_proc()
{
    int ret_val;
    
    clear_msg();
    if (invalid_set_pressed())
	return;
    ret_val = window_destroy(frame);
    if (ret_val) {
	if (use_CM)
	    CM_disconnect();
	window_set(frame, FRAME_NO_CONFIRM, TRUE, 0);
	window_destroy(frame);
    }
}


/*
 * Save current image in a Sun rasterfile.  If in color mode, it supposedly
 * dumps the proper colormap into the rasterfile (this doesn't always
 * seem to work right, though, probably because of Sun's problems with
 * colormaps of length 256).  In B&W mode, the current textures and
 * magnification are used.
 */
isave_raster()
{
    State *ap;
    int ras_type = RT_STANDARD, x, y;
    colormap_t *colormap;
    Pixrect *pr;
    FILE *fp;

    ap = ca + B;
    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(isave_raster);
    if (closeup) {
	show_msg("Doesn't work in closeup yet");
	return;
    }
    get_msg("Save raster file:", buf1, BUFLEN, saved_image_file);
    if (buf1[0] == NULL)
	return;
    if ((buf1[0] == '/') || (!strncmp(buf1, "./", 2)) || (!save_in_image_dir))
	;
    else {
	strcpy(buf2, buf1);
	sprintf(buf1, "%s/%s", Image_dir, buf2);
    }
    if (expand_fname(buf1, buf1))
	return;
    strcpy(saved_image_file, buf1);
    if ((fp = fopen(buf1, "r")) != NULL) {
	get_msg("Overwrite file?", buf2, BUFLEN, "");
	if (buf2[0] != 'y') {
	    show_msg("Save cancelled.");
	    return;
	}
	fclose(fp);
    }
    if ((fp = fopen(buf1,"w")) == NULL) {
	show_msg("Could not open file");
	return;
    }
    if (blackwhite) {
	if ((pr = mem_create(B*mag, B*mag, 1)) == NULL) {
	    show_msg("Couldn't create tmp pixrect");
	    return;
	}
	colormap = NULL;
    }
    else {
	if ((pr = mem_create(B, B, 8)) == NULL) {
	    show_msg("Couldn't create tmp pixrect");
	    return;
	}
	if ((colormap = (colormap_t *)malloc(sizeof(colormap_t))) == NULL) {
	    show_msg("Couldn't create colormap");
	    return;
	}
	colormap->type = RMT_EQUAL_RGB;
	colormap->length = 256;
	colormap->map[0] = red;
	colormap->map[1] = green;
	colormap->map[2] = blue;
    }
    if ((!blackwhite) || (mag == 1)) {
	for (y=0; y < B; y++)
	    for (x=0; x < B; x++) {
		pr_put(pr, x, y, *ap);
		ap++;
	    }
    }
    else {
	pr_rop(pr, 0, 0, side*mag, side*mag, PIX_SRC, NULL, 0, 0);
	for (y=0; y < B; y++)
	    for (x=0; x < B; x++) {
		if (color[*ap])
		    pr_rop(pr, x*mag, y*mag, mag, mag, PIX_SRC,
			   &textures_pr[color[*ap]], 0, 0);
		ap++;
	    }
    }
		
    if (pr_dump(pr, fp, colormap, ras_type, FALSE))
	show_msg("Error trying to dump raster");
    else
	show_msg("Rasterfile written.");
    fclose(fp);
    free(pr);
    if (colormap)
	free(colormap);
}
	    
    

/***** canvas event procedures *****/

/*
 * Main event procedure for the canvas
 */
static int
screen_event_proc(canvas, event)
    Canvas  canvas;
    Event  *event;
{
    int     x, y, k, a;

    pixel_to_cell(event_x(event), event_y(event), &x, &y);
    if (x >= B || y >= B) {
	clear_pos();
	clear_state();
	return;
    }
    k = event_id(event);
    if (!drawing) {
	switch (screen_mode) {
	  case MODE_EDIT:
	    edit_event(canvas, event, x, y, k);
	    break;
	  case MODE_PROBE:
	    probe_event(canvas, event, x, y, k);
	    break;
	  case MODE_PLANEEDIT:
	    planeedit_event(canvas, event, x, y, k);
	    break;
	}
    }
    switch (k) {		/* actions common to all screen modes */
      case LOC_MOVE:
      case LOC_RGNENTER:
	show_pos(x, y);
	show_state(ca[(y * B) + ABASE + x]);
	break;
      case LOC_RGNEXIT:
	clear_pos();
	clear_state();
	break;
      default:
	break;
    }
    if (drawing)
	drawing_event(canvas, event, x, y, k);
}


/*
 * Event-proc when in edit-mode
 */
edit_event(canvas, event, x, y, k)
    Canvas  canvas;
    Event  *event;
    int     x, y, k;
{
    switch (k) {
    case LOC_MOVE:
    case LOC_RGNENTER:
	if ((int) window_get(canvas, WIN_EVENT_STATE, MS_LEFT))
	    set_site(x, y, curr_color);
	else if ((int) window_get(canvas, WIN_EVENT_STATE, MS_MIDDLE))
	    set_site(x, y, 0);
	break;
    case MS_LEFT:
	set_site(x, y, curr_color);
	break;
    case MS_MIDDLE:
	set_site(x, y, 0);
	break;
    case MS_RIGHT:
	if (function) return;
	menu_set((Menu_item) menu_get(edit_menu, MENU_NTH_ITEM, curr_color),
		 MENU_BOXED, TRUE, 0);
	k = (int) menu_show(edit_menu, canvas, event, 0);
	menu_set((Menu_item) menu_get(edit_menu, MENU_NTH_ITEM, curr_color),
		 MENU_BOXED, FALSE, 0);
	if (k != 0) {
	    set_color(k);
	}
	break;
    default:
	if (is_state(k))
	    set_color(to_state(k));
	break;
    }
}


/*
 * Event-proc for probe mode
 */
probe_event(canvas, event, x, y, k)
    Canvas  canvas;
    Event  *event;
    int     x, y, k;
{
    int     a;

    if (function)
	return;
    switch (k) {
    case MS_LEFT:
	if (event_is_down(event))
	    show_next_state(x, y);
	break;
    case MS_RIGHT:
	if (use_CM) {
	    clear_msg();
	    show_msg("Not implement for CM yet");
	    return;
	}
	if (event_is_down(event)) {
	    show_next_state(x, y);
	    k = (int) menu_show(probe_menu, canvas, event, 0);
	    if (k != 0) {
		a = get_address(x, y);
		to_nhood(a, buf1);
		alter_nhood(buf1);
	    }
	}
	break;
    }
}

/*
 * Event-proc for plane-edit mode
 */
planeedit_event(canvas, event, x, y, k)
    Canvas  canvas;
    Event  *event;
    int     x, y, k;
{
    switch (k) {
    case LOC_MOVE:
    case LOC_RGNENTER:
	if ((int) window_get(canvas, WIN_EVENT_STATE, MS_LEFT))
	    set_site_bit(x, y, curr_planemask, TRUE);
	else if ((int) window_get(canvas, WIN_EVENT_STATE, MS_MIDDLE))
	    set_site_bit(x, y, curr_planemask, FALSE);
	break;
    case MS_LEFT:
	set_site_bit(x, y, curr_planemask, TRUE);
	break;
    case MS_MIDDLE:
	set_site_bit(x, y, curr_planemask, FALSE);
	break;
    case MS_RIGHT:
	menu_set(menu_find(planeedit_menu, MENU_VALUE, curr_planemask, 0),
		 MENU_BOXED, TRUE, 0);
	k = (int) menu_show(planeedit_menu, canvas, event, 0);
	menu_set(menu_find(planeedit_menu, MENU_VALUE, curr_planemask, 0),
		 MENU_BOXED, FALSE, 0);
	if (k != 0) {
	    set_planemask(k);
	}
	break;
    default:
	if (k >= ASCZ && k < ASCZ + L)
	    set_planemask(1 << (k - ASCZ));
	break;
    }
}


/*
 * Event-proc when user is drawing a line or circle
 */
drawing_event(canvas, event, x, y, k)
    Canvas  canvas;
    Event  *event;
    int     x, y, k;
{
    int left_down;
    static int
	col, row, start_row, start_col, line_x, line_y,
	center_x, center_y;

    left_down = 0;
    switch (k) {
    /* case LOC_MOVE: */
    case LOC_RGNENTER:
	if ((int) window_get(canvas, WIN_EVENT_STATE, MS_LEFT))
	    left_down = 1;
	else if ((int) window_get(canvas, WIN_EVENT_STATE, MS_MIDDLE)) {
	    drawing = 0;
	    clear_msg();
	    show_msg("Aborted");
	    return;
	}
	break;
    case MS_LEFT:
	if event_is_down(event) {
	    left_down = 1;
	}
	else {
	    if (drawing == DRAWING_DONE) {
		drawing = 0;
		return;
	    }
	}
	break;
    case MS_MIDDLE:
	drawing = 0;
	clear_msg();
	show_msg("Aborted");
	return;
	break;
    }
    if (left_down) {
	switch (drawing) {
	  case HORIZ_LINE:
	    clear_msg();
	    show_msg("Click left button at end col");
	    drawing = HORIZ_END;
	    start_col = x;
	    row = y;
	    set_site(x, y, curr_color);
	    display_site(x, y, curr_color);
	    break;
	  case HORIZ_END:
	    draw_horizontal_line(row, start_col, x, curr_color);
	    clear_msg();
	    show_msg("Done.");
	    drawing = DRAWING_DONE;
	    break;
	  case VERT_LINE:
	    clear_msg();
	    show_msg("Click left button at end row");
	    drawing = VERT_END;
	    col = x;
	    start_row = y;
	    set_site(x, y, curr_color);
	    display_site(x, y, curr_color);
	    break;
	  case VERT_END:
	    draw_vertical_line(col, start_row, y, curr_color);
	    clear_msg();
	    show_msg("Done.");
	    drawing = DRAWING_DONE;
	    break;
	  case ARB_LINE:
	    clear_msg();
	    show_msg("Click left button at other end");
	    line_x = x;
	    line_y = y;
	    drawing = ARB_END;
	    set_site(line_x, line_y, curr_color);
	    display_site(line_x, line_y, curr_color);
	    break;
	  case ARB_END:
	    draw_arbitrary_line(line_x, line_y, x, y, curr_color);
	    clear_msg();
	    show_msg("Done.");
	    drawing = DRAWING_DONE;
	    return;
	  case HOLLOW_CIRC:
	    drawing = HOLLOW_EDGE;
	    center_x = x;
	    center_y = y;
	    clear_msg();
	    show_msg("Click left button on edge");
	    show_msg("of circle");
	    set_site(center_x, center_y, curr_color);
	    display_site(center_x, center_y, curr_color);
	    break;
	  case HOLLOW_EDGE:
	    draw_hollow_circ(center_x, center_y, x, y, curr_color);
	    clear_msg();
	    show_msg("Done.");
	    drawing = DRAWING_DONE;
	    return;
	  case SHADED_CIRC:
	    drawing = SHADED_EDGE;
	    center_x = x;
	    center_y = y;
	    clear_msg();
	    show_msg("Click left button on edge");
	    show_msg("of circle");
	    set_site(center_x, center_y, curr_color);
	    display_site(center_x, center_y, curr_color);
	    break;
	  case SHADED_EDGE:
	    draw_shaded_circ(center_x, center_y, x, y, saved_xinc,
			    saved_yinc, curr_color);
	    clear_msg();
	    show_msg("Done.");
	    drawing = DRAWING_DONE;
	    return;
	}
    }
}


/***** general notify & event procedures *****/

/*
 * Pop up menu if right button, call top item if left, and call previous
 * item if middle button.
 */
static int
button_event_proc(item, event)
    Panel_item item;
    Event  *event;
{
    Menu_item m_item;
    func_ptr item_action_proc;

    if (((event_id(event) == MS_RIGHT) || (event_id(event) == MS_MIDDLE)) &&
	(event_is_down(event))) {
	def_im_get_settings();
	def_rule_get_settings();
	def_cmap_get_settings();
	/* def_CM_get_settings(); */
    }
	
    if (event_id(event) == MS_RIGHT && event_is_down(event))
	menu_show((Menu) panel_get(item, PANEL_CLIENT_DATA), panel, event, 0);
    /* else if ((event_id(event) == MS_MIDDLE) && event_is_down(event) &&
	     (item != defaults_button)) { */
    else if ((event_id(event) == MS_MIDDLE) && event_is_down(event)) {
	m_item = menu_get((Menu) panel_get(item, PANEL_CLIENT_DATA),
			MENU_SELECTED_ITEM);
	if (((char *)m_item) == NULL) {
	    clear_msg();
	    show_msg("no default menu selection found");
	    return;
	}
	item_action_proc = (func_ptr)(menu_get(m_item, MENU_VALUE));
	if (item_action_proc == NULL) {
	    clear_msg();
	    show_msg("No default menu selection found");
	    return;
	}
	item_action_proc();
    }
    else
	panel_default_handle_event(item, event);
}


/*
 * Call routine from menu
 */
static int
menu_notify_proc(m, mi)
    Menu    m;
    Menu_item mi;
{
    ((int (*) ()) menu_get(mi, MENU_VALUE)) ();
}


/*
 * Call the first routine in a menu
 */
static int
do_first_item_proc(item, event)
    Panel_item item;
    Event  *event;
{
    Menu    menu = (Menu) panel_get(item, PANEL_CLIENT_DATA);
    Menu_item mi = menu_get(menu, MENU_NTH_ITEM, 1);
    def_im_get_settings();
    def_rule_get_settings();
    def_cmap_get_settings();
    /* def_CM_get_settings(); */
    ((int (*) ()) menu_get(mi, MENU_VALUE)) ();
}


/***** display-related utilities *****/

void
command_loop()
{					/* wait for commands and process them
					 * as they come */
    window_main_loop(frame);		/* handled by the SunView Notifier */
}


set_run_mode()
{					/* disable panel & canvas input */
    int     fd;				/* window file descriptor */

    frame_mask = *((Inputmask *) window_get(frame, WIN_PICK_INPUT_MASK));
    window_set(frame, WIN_CONSUME_PICK_EVENT, MS_LEFT, 0);
    panel_mask = *((Inputmask *) window_get(panel, WIN_PICK_INPUT_MASK));
    window_set(panel, WIN_CONSUME_PICK_EVENT, WIN_NO_EVENTS, 0);
    canvas_mask = *((Inputmask *) window_get(canvas, WIN_PICK_INPUT_MASK));
    window_set(canvas, WIN_CONSUME_PICK_EVENT, WIN_NO_EVENTS, 0);
    fd = (int) window_get(frame, WIN_FD);
    fcntl(fd, F_SETFL, FNDELAY);
    /* set non-blocking mode */
}


unset_run_mode()
{					/* re-enable panel & canvas input */
    window_set(frame, WIN_PICK_INPUT_MASK, &frame_mask, 0);
    window_set(panel, WIN_PICK_INPUT_MASK, &panel_mask, 0);
    window_set(canvas, WIN_PICK_INPUT_MASK, &canvas_mask, 0);
}


check_button()
{					/* return TRUE iff the left button has
					 * been pressed */
    int     err, button;
    Event   event;
    do {				/* check all events in pending queue */
	err = window_read_event(frame, &event);
	button = ((err == 0) && (event_is_up(&event)) &&
		  (event_id(&event) == MS_LEFT));
    } while ((err == 0) && !button);
    /* if no events remain, err will equal -1 */
    return (button);
}


/* The following routines update various control panel items... */
/* Most are self-explanatory. */
show_time(stime)
{
    sprintf(buf1, "%-5d ", stime);
    panel_set(pan_time, PANEL_LABEL_STRING, buf1, 0);
}


show_pos(x, y)
    int     x, y;
{
    sprintf(buf1, "(%3d,%3d)", x, y);
    panel_set(pan_pos, PANEL_LABEL_STRING, buf1, 0);
}


clear_pos()
{
    panel_set(pan_pos, PANEL_LABEL_STRING, "         ", 0);
}


show_state(s)
    State   s;
{
    buf1[0] = to_char(s/16);
    buf1[1] = to_char(s%16);
    buf1[2] = NULL;
    panel_set(pan_state, PANEL_LABEL_STRING, buf1, 0);
}


clear_state()
{
    panel_set(pan_state, PANEL_LABEL_STRING, "   ", 0);
}


show_planes(mask)			/* show which bitplanes are currently
					 * displayed */
    int     mask;
{
    char    str[6];
    if (mask == AMASK)
	strcpy(str, "ALL");
    else
	sprintf(str, "%X", mask);
    panel_set(pan_planes, PANEL_LABEL_STRING, str, 0);
}


show_rule(str)
char   *str;
{
    char lbl[128], nhstr[10], istr[10];

    sprintf(istr, "b%d", B);
    switch(NHOOD) {
      case NH_MOORE:
	sprintf(nhstr, "m%d", S);
	break;
      case NH_MARG:
	sprintf(nhstr, "M%d", S);
	break;
      case NH_VONN:
	sprintf(nhstr, "v%d", S);
	break;
      case NH_LIN:
	sprintf(nhstr, "l%dr%d", S, R);
	break;
    }
    if (str == (char *)0)
	sprintf(lbl, "Cellsim 2.5: %s %s   rule: %s", nhstr, istr,
		saved_rule_file);
    else
	sprintf(lbl, "Cellsim 2.5: %s %s   rule: %s", nhstr, istr, str);
    window_set(frame, FRAME_LABEL, lbl, 0);
    rule_shown = 1;
}


clear_rule()
{
    char lbl[80], nhstr[10], istr[10];

    sprintf(istr, "b%d", B);
    switch(NHOOD) {
      case NH_MOORE:
	sprintf(nhstr, "m%d", S);
	break;
      case NH_MARG:
	sprintf(nhstr, "M%d", S);
	break;
      case NH_VONN:
	sprintf(nhstr, "v%d", S);
	break;
      case NH_LIN:
	sprintf(nhstr, "l%dr%d", S, R);
	break;
    }
    sprintf(lbl, "Cellsim 2.5: %s %s", nhstr, istr);
    window_set(frame, FRAME_LABEL, lbl, 0);
    rule_shown = 0;
}


show_msg(str)				/* display string in message area */
    char   *str;
{
    panel_set(pan_msg[msg_line++], PANEL_LABEL_STRING, str, 0);
    if (msg_line == NUM_MSGS)
	msg_line = 0;
}


get_msg(prompt, str, len, default_val)	/* input message in message area */
    char   *prompt, *str, *default_val;
    int     len;
{
    int i, j;
    
    show_msg(prompt);
    panel_set(pop_input,
	      PANEL_VALUE, default_val, 0);
    window_set(popup, WIN_X, 0,
    WIN_Y, (int) panel_get(pan_msg[msg_line], PANEL_ITEM_Y) + pan_tmargin, 0);
    window_loop(popup);
    strncpy(str, (char *) panel_get_value(pop_input), len);
    str[len] = '\0';
    i = 0;
    j = 0;
    while ((str[i] == ' ') || (str[i] == '\t'))
	i++;
    while (str[i] != '\0')
	str[j++] = str[i++];
    str[j] = '\0';
    show_msg(str);
}


static Panel_setting
return_proc()
{					/* return procedure for popup window
					 * loop */
    window_return(NULL);
}


int
get_num(prompt, n, nmin, nmax, default_val)  /* input (decimal or hex)
					      * integer in message area */
    char   *prompt;
    int    *n, nmin, nmax, default_val;
{
    int     err;
    char    buf[BUFLEN], numbuf[20];

    if (default_val != INTUNDEF)
	sprintf(numbuf,"%d",default_val);
    else
	numbuf[0] = '\0';
    get_msg(prompt, buf, BUFLEN, numbuf);
    if (buf[0] == NULL)
	return (1);
    if ((!strncmp(buf, "0x", 2)) || (!strncmp(buf, "0X", 2)))
	err = sscanf(&buf[2], "%x", n);
    else
	err = sscanf(buf, "%d", n);
    if (err != 1 || *n < nmin || *n > nmax) {
	show_msg("Illegal value");
	return (1);
    } else
	return (0);
}


/* Get 2 integers in message area */
int
get_2num(prompt, n1, n2, n1min, n1max, n2min, n2max, def1, def2)
    char   *prompt;
    int    *n1, *n2, n1min, n1max, n2min, n2max, def1, def2;
{
    int     err;
    char    buf[BUFLEN], numbuf[20], numbuf2[20];

    if ((def1 != INTUNDEF) && (def2 != INTUNDEF))
	sprintf(numbuf,"%d %d", def1, def2);
    else
	numbuf[0] = '\0';
    get_msg(prompt, buf, BUFLEN, numbuf);
    if (buf[0] == NULL)
	return (1);
    err = sscanf(buf, "%s %s", numbuf, numbuf2);
    if (err != 2) {
	show_msg("Not enough parameters entered");
	return 1;
    }
    if ((!strncmp(numbuf, "0x", 2)) || (!strncmp(numbuf, "0X", 2)))
	err = sscanf(&numbuf[2], "%x", n1);
    else
	err = sscanf(numbuf, "%d", n1);
    if (err != 1 || *n1 < n1min || *n1 > n1max) {
	show_msg("Illegal value");
	return (1);
    }
    if ((!strncmp(numbuf2, "0x", 2)) || (!strncmp(numbuf2, "0X", 2)))
	err = sscanf(&numbuf2[2], "%x", n2);
    else
	err = sscanf(numbuf2, "%d", n2);
    if (err != 1 || *n2 < n2min || *n2 > n2max) {
	show_msg("Illegal value");
	return (1);
    }
    return 0;
}


clear_msg()
{					/* clear message area */
    int     i;
    for (i = 0; i < NUM_MSGS; i++)
	panel_set(pan_msg[i], PANEL_LABEL_STRING, "", 0);
    msg_line = 0;
}


set_color(col)				/* set current insertion color for edit
					 * mode */
    int     col;
{
    static int odd=1;
    static int prev;

    if (function) {
	odd = 1-odd;
	if (!odd) {
	    prev = col;
	    return;
	}
    }
    else
	prev = 0;
    curr_color = col + 16*prev;
    clear_msg();
    sprintf(buf1,"current color = %02x\n",col + 16*prev);
    show_msg(buf1);
    cursor_set(cursor, CURSOR_CROSSHAIR_COLOR, col + 16*prev, 0);
    window_set(canvas, WIN_CURSOR, cursor, 0);
}


and_color(mask)   /* 'AND' the current color with the new state-mask */
short mask;
{
    set_color(curr_color & (int)mask);
}


set_planemask(m)			/* set current edit planemask for plane
					 * edit mode */
    int     m;
{
    curr_planemask = m;
    cursor_set(cursor, CURSOR_CROSSHAIR_COLOR, m, 0);
    window_set(canvas, WIN_CURSOR, cursor, 0);
}


and_planemask(mask)
short mask;
{
    set_planemask(curr_planemask & (int)mask);
}


set_site(x, y, value)			/* set given ca site to given value */
    int     x, y, value;
{
    ca[(y * B) + ABASE + x] = value;
    display_site(x, y, value);
    show_state(value);
}


set_site_bit(x, y, mask, on)	/* set/unset given bit in given ca site */
    int     x, y, mask, on;
{
    int     value;
    if (on)
	value = ca[(y * B) + ABASE + x] | mask;
    else
	value = ca[(y * B) + ABASE + x] & ~mask;
    set_site(x, y, value);
}


mark_site(x, y)				/* mark given site in ca display */
    int     x, y;
{
    int     x0, y0, x1, y1;
    x0 = (x - xstart) * mag;
    y0 = (y - ystart) * mag;
    x1 = x0 + mag - 1;
    y1 = y0 + mag - 1;
    display_site(x, y, 0);		/* clear site */
    pw_vector(pw, x0, y0, x1, y1, PIX_SRC, -1);
    pw_vector(pw, x0, y1, x1, y0, PIX_SRC, -1);
}


display_site(x, y, value)		/* show given value at given site in ca
					 * display */
    int     x, y;
    State   value;
{
    if (blackwhite)
	pw_rop(pw, (x-xstart) * mag, (y-ystart) * mag,
	       mag, mag, PIX_SRC, &textures_pr[color[value]], 0, 0);
    else
	pw_write(pw, (x - xstart) * mag, (y - ystart) * mag,
		 mag, mag, PIX_COLOR(value) | PIX_SRC, NULL, 0, 0);
}


display_image()
{					/* display entire ca array */
    unsigned char *cptr, *tptr;
    int x,y;

    pr_rop(tmpmag,0,0,B*mag,B*mag,PIX_SRC,NULL,0,0);
    if (blackwhite) {
	if (mag == 1) {
	    cptr = ca;
	    tptr = (unsigned char *)(mpr_d(tmpmag)->md_image);
	    for (y=0; y<B; y++)
		for (x=0; x<B; x++)
		    /* *(tptr++) = (*(cptr++) != 0); */
		    if (*(cptr++)) pr_put(tmpmag,x,y,1);
	    pw_write(pw,0,0,B*mag,B*mag,PIX_SRC,tmpmag,0,1);
	}
	else
	    magnify_image(mag,side,xstart,ystart);
    }
    else {
	if (mag == 1)
	    pw_write(pw, 0, 0, B * mag, B * mag, PIX_SRC, ca_handle, 0, 1);
	else
	    magnify_image(mag, side, xstart, ystart);
    }
}


pixel_to_cell(xp, yp, xc, yc)		/* convert from pixel coords. to cell
					 * coords. */
    int     xp, yp, *xc, *yc;
{
    *xc = (xp / mag);
    *yc = (yp / mag);
    if ((mag != 1) && closeup) {
	*xc += xstart;
	*yc += ystart;
    }
}


init_cmap()
{					/* initialize color map */
    int     k;


    color[0] = 0x000000;		/* black */

    color[1] = 0xFF0000;		/* red */
    color[2] = 0x0000FF;		/* blue */
    color[3] = 0xFF00FF;		/* magenta */
    color[4] = 0x00D000;		/* green */
    color[5] = 0xFFFF00;		/* yellow */
    color[6] = 0x00D0FF;		/* cyan */
    color[7] = 0xC0C0C0;		/* light gray */
    color[8] = 0x808080;		/* dark gray */
    color[9] = 0xFFC0C0;		/* light red */
    color[10] = 0xC0C0FF;		/* light blue */
    color[11] = 0xFFC0FF;		/* light magenta */
    color[12] = 0xC0FFC0;		/* light green */
    color[13] = 0xFFFFC0;		/* light yellow */
    color[14] = 0xC0FFFF;		/* light cyan */
    color[15] = 0xFFFFFF;		/* white */

    /* greyscale above 16 for now */
    for (k=16; k<256; k++)
	color[k] = k + k*256 + k*65536;
    color[255] = 0xFFFFFF;
    
    for (k = 0; k < 256; k++)
	set_cmap_entry(k, color[k]);

    sprintf(cmap_name, "cell%d", getpid());
    pw_setcmsname(pw, cmap_name);
    pw_putcolormap(pw, 0, 256, red, green, blue);
    
    sprintf(cmap_name2, "cell2%d", getpid());
    pw_setcmsname(window_pw, cmap_name2);
    pw_putcolormap(window_pw, 0, 256, red, green, blue);
    
    sprintf(cmap_name3, "cell3%d", getpid());
    pw_setcmsname(panel_pw, cmap_name3);
    pw_putcolormap(panel_pw, 0, 256, red, green, blue);
    

} /* init_cmap() */



init_textures()
{
    int i,j,j1,k,sz;
    char fname[40];
    struct mpr_data data[CMAPLEN];

    sz = MAXSIZE/B + 1;
    for (k=0; k < CMAPLEN; k++)
	color[k] = k%16;
}

	
set_cmap_entry(n, c)
    int     n, c;
{
    color[n] = c;
    red[n] = (c >> (2 * CLEN)) & CMASK;
    green[n] = (c >> CLEN) & CMASK;
    blue[n] = c & CMASK;
}


set_rgb_arrays(r,g,b)
unsigned char *r,*g,*b;
{
    int i;

    for (i=0; i<256; i++) {
	r[i] = (color[i] >> (2 * CLEN)) & CMASK;
	g[i] = (color[i] >> CLEN) & CMASK;
	b[i] = color[i] & CMASK;
    }
}


cmap_alter()
{					/* alter a colormap entry */
    int     e, c;
    char    buf2[6];
    
    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(cmap_alter);
    if (get_num("Change color of entry:", &e, 0, CMAPLEN - 1, saved_cmap_entry))
	return;
    saved_cmap_entry = e;
    sprintf(buf1, "Currently %06X; change to:", color[e]);
    get_msg(buf1, buf2, 6, "");
    if (buf2[0] == NULL)
	return;
    if (sscanf(buf2, "%6x", &c) != 1) {
	show_msg("Illegal value");
	return;
    }
    if (e == 0) {
	if (c == color[255]) {
	    clear_msg();
	    show_msg("Invalid -- cannot set entry 0");
	    show_msg("to same color as entry 255");
	    return;
	}
    }
    if (e == 255) {
	if (c == color[0]) {
	    clear_msg();
	    show_msg("Invalid -- cannot set entry 255");
	    show_msg("to same color as entry 0");
	    return;
	}
    }
    set_cmap_entry(e, c);
    if (blackwhite)
	display_image();
    else {
	pw_setcmsname(pw, cmap_name);
	pw_putcolormap(pw, 0, 256, red, green, blue);
	pw_setcmsname(window_pw, cmap_name2);
	pw_putcolormap(window_pw, 0, 256, red, green, blue);
	pw_setcmsname(panel_pw, cmap_name3);
	pw_putcolormap(panel_pw, 0, 256, red, green, blue);
    }
}



cmap_save_proc()
{
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(cmap_save_proc);
    if (save_cmap_compressed)
	cmap_save_compressed();
    else
	cmap_save();
}


cmap_save()
{					/* save current colormap */
    FILE   *fp;

    /* write out array */

    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(cmap_save);
    get_msg("Save color file:", buf1, BUFLEN, saved_cmap_file);
    if (buf1[0] == NULL) {
	show_msg("");
	return;
    }
    if ((buf1[0] == '/') || (!strncmp(buf1, "./", 2)) || (!strncmp(buf1, "../", 3)) || (!save_in_cmap_dir))
	;
    else {
	strcpy(buf2, buf1);
	sprintf(buf1, "%s/%s", Cmap_dir, buf2);
    }
    if (expand_fname(buf1, buf1))
	return;
    strcpy(saved_cmap_file, buf1);
    if ((fp = fopen(buf1, "r")) != NULL) {
	if (!confirm_overwrite()) {
	    fclose(fp);
	    return;
	}
    }
    fclose(fp);
    fp = fopen(buf1, "w");
    fwrite(color, sizeof(*color), CMAPLEN, fp);
    fclose(fp);
    strcat(buf1, " saved");
    show_msg(buf1);
}


cmap_save_compressed()
{
    FILE *fp;
    int err;

    clear_msg();
    get_msg("Save cmap file:", buf1, BUFLEN, saved_cmap_file);
    if (buf1[0] == NULL) {
	show_msg("");
	return;
    }
    if ((buf1[0] == '/') || (!strncmp(buf1, "./", 2)) || (!strncmp(buf1, "../", 3)) || (!save_in_cmap_dir))
	;
    else {
	strcpy(buf2, buf1);
	sprintf(buf1, "%s/%s", Cmap_dir, buf2);
    }
    if (expand_fname(buf1, buf1))
	return;
    if (strcmp(&buf1[strlen(buf1)-2], ".Z"))
	strcat(buf1, ".Z");
    strcpy(saved_cmap_file, buf1);
    if ((fp = fopen(buf1, "r")) != NULL) {
	if (!confirm_overwrite()) {
	    fclose(fp);
	    return;
	}
    }
    fclose(fp);
    sprintf(buf2, "compress > %s", saved_cmap_file);
    fp = popen(buf2, "w");
    fwrite(color, sizeof(*color), CMAPLEN, fp);
    err = fclose(fp);
    if (err) {
	sprintf(buf1, "pclose returned %d", err);
	show_msg(buf1);
	return;
    }
    strcat(buf1, " saved");
    show_msg(buf1);
}


cmap_load()
{					/* load colormap */
    FILE   *fp;
    int     k, file_len;

    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(cmap_load);
    get_msg("Load color file:", buf1, BUFLEN, saved_cmap_file);
    if (buf1[0] == NULL)
	return;
    if (expand_fname(buf1, buf1))
	return;
    strcpy(saved_cmap_file,buf1);
    if (!strcmp(&buf1[strlen(buf1)-2], ".Z")) {
	cmap_load_compressed(buf1);
	return;
    }
    if (!(fp = fopen(buf1, "r"))) {
	sprintf(buf2,"%s/%s",Cmap_dir,buf1);
	if (!(fp = fopen(buf2, "r"))) {
	    cmap_load_compressed(buf1);
	    return;
	}
    }
    for (k=0; k < 256; k++)
	color2[k] = color[k];	/* copy old colormap into new one first */
    k = 0;
    while (fread(&color2[k], sizeof(int), 1, fp) == 1) {
	k++;
	if (k > 256) {
	    fclose(fp);
	    return;
	}
    }
    if ((k != 16) && (k != 256)) {
	clear_msg();
	sprintf(buf1,"Bad colormap length: %d",file_len);
	show_msg(buf1);
	return;
    }
    fclose(fp);
    if (color2[0] == color2[255]) {
	clear_msg();
	show_msg("Invalid colormap");
	show_msg("Entry 0 same as entry 255");
	return;
    }

    if (blackwhite) {
	for (k=0; k < 256; k++) {
	    if (color2[k] > CMAPLEN) {
		show_msg("Error in colormap data");
		return;
	    }
	    else
		color[k] = color2[k];
	}
	display_image();
    }
    else {
	for (k = 0; k < 256; k++)
	    set_cmap_entry(k, color2[k]);
	pw_setcmsname(pw, cmap_name);
	pw_putcolormap(pw, 0, 256, red, green, blue);
	pw_setcmsname(window_pw, cmap_name2);
	pw_putcolormap(window_pw, 0, 256, red, green, blue);
	pw_setcmsname(panel_pw, cmap_name3);
	pw_putcolormap(panel_pw, 0, 256, red, green, blue);
    }
}


cmap_load_compressed(fname)
char *fname;
{					/* load colormap */
    FILE   *fp;
    int     k, file_len;
    struct stat statbuf;
    int err;

    if (!strcmp(&fname[strlen(fname)-2],".Z"))
	strcpy(buf1, fname);
    else
	sprintf(buf1, "%s.Z", fname);
    if (!stat(buf1, &statbuf)) {
	sprintf(buf2, "uncompress -c %s", buf1);
	if (!(fp = popen(buf2, "r"))) {
	    show_msg("Error trying to open pipe");
	    return;
	}
    }
    else {
	sprintf(buf2, "%s/%s", Cmap_dir, buf1);
	if (!stat(buf2, &statbuf)) {
	    sprintf(buf2, "uncompress -c %s/%s",Cmap_dir, buf1);
	    if (!(fp = popen(buf2, "r"))) {
		show_msg("Error trying to open pipe");
		return;
	    }
	}
	else {
	    show_msg("No such cmap");
	    return;
	}
    }
    strcpy(saved_cmap_file,buf1);
    for (k=0; k < 256; k++)
	color2[k] = color[k];	/* copy old colormap into new one first */
    k = 0;
    while (fread(&color2[k], sizeof(int), 1, fp) == 1) {
	k++;
	if (k > 256) {
	    pclose(fp);
	    return;
	}
    }
    if ((k != 16) && (k != 256)) {
	clear_msg();
	sprintf(buf1,"Bad colormap length: %d",file_len);
	show_msg(buf1);
	return;
    }
    err = pclose(fp);
    if (err) {
	sprintf(buf2, "pclose returned %d", err);
	show_msg(err);
	return;
    }
    if (color2[0] == color2[255]) {
	clear_msg();
	show_msg("Invalid colormap");
	show_msg("Entry 0 same as entry 255");
	return;
    }

    if (blackwhite) {
	for (k=0; k < 256; k++) {
	    if (color2[k] > CMAPLEN) {
		show_msg("Error in colormap data");
		return;
	    }
	    else
		color[k] = color2[k];
	}
	display_image();
    }
    else {
	for (k = 0; k < 256; k++)
	    set_cmap_entry(k, color2[k]);
	pw_setcmsname(pw, cmap_name);
	pw_putcolormap(pw, 0, 256, red, green, blue);
	pw_setcmsname(window_pw, cmap_name2);
	pw_putcolormap(window_pw, 0, 256, red, green, blue);
	pw_setcmsname(panel_pw, cmap_name3);
	pw_putcolormap(panel_pw, 0, 256, red, green, blue);
    }
}



magnify_image(m, side, xstart, ystart)	/* magnify given portion of image by 
					 * factor of m */
    register short m;
    int     side, xstart, ystart;
{
    register short x, y, delx, dely;
    register unsigned char value;
    register unsigned char *srow, *trow, *sptr, *tptr, *tgroup;
    unsigned int size;

    size = side * m / 32;
    srow = ca + ABASE + ystart * B + xstart;
    if (blackwhite) {
	pr_rop(tmpmag,0,0,side*m,side*m,PIX_SRC,NULL,0,0);
	for (y=0; y<m*side; y+=m) {
	    sptr = srow;
	    for (x=0; x<m*side; x+=m) {
		value = *(sptr++);
		if (color[value])
		    pr_rop(tmpmag,x,y,m,m,PIX_SRC,&textures_pr[color[value]],0,0);
	    }
	    srow += B;
	}
	pw_rop(pw,0,0,side*m,side*m,PIX_SRC,tmpmag,0,0);
    }
    else {
	trow = (unsigned char *) (mpr_d(tmpmag)->md_image);
	for (y = 0; y < side; y++) {
	    tptr = trow;
	    sptr = srow;
	    for (x = 0; x < side; x++) {
		value = *(sptr++);
		for (delx = 0; delx < m; delx++) {
		    *(tptr++) = value;
		}
	    }
	    tgroup = trow;
	    trow += MAXSIZE;
	    x = size;
	    for (dely = 1; dely < m; dely++) {
		tptr = trow;
		sptr = tgroup;
		for (delx = 0; delx < x; delx++) {
		    *(((long *) tptr)++) = *(((long *) sptr)++);
		    *(((long *) tptr)++) = *(((long *) sptr)++);
		    *(((long *) tptr)++) = *(((long *) sptr)++);
		    *(((long *) tptr)++) = *(((long *) sptr)++);
		    *(((long *) tptr)++) = *(((long *) sptr)++);
		    *(((long *) tptr)++) = *(((long *) sptr)++);
		    *(((long *) tptr)++) = *(((long *) sptr)++);
		    *(((long *) tptr)++) = *(((long *) sptr)++);
		}
		trow += MAXSIZE;
	    }
	    srow += B;
	}
	pw_rop(pw, 0, 0, side * m, side * m, PIX_SRC, tmpmag, 0, 0);
    }
    /* draw tmpmag on canvas */
}


/*
 * The following "toggle" routines are here in the SunView-specific file
 * because they have to alter the strings in the control-panel menus.
 */
toggle_save_in_image_dir_proc()
{
    Menu_item item;
    
    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(toggle_save_in_image_dir_proc);
    save_in_image_dir = 1 - save_in_image_dir;
    item = menu_get(defaults_image_menu, MENU_NTH_ITEM, 1);
    if (save_in_image_dir) {	/* is enabled now */
	menu_set(item, MENU_STRING, "Save in current working dir", 0);
	show_msg("Images will be saved");
	show_msg("in Image-dir");
    }
    else {
	menu_set(item, MENU_STRING, "Save in Image-dir", 0);
	show_msg("Images will be saved in");
	show_msg("current working dir");
    }
}


toggle_save_images_compressed()
{
    Menu_item item;
    
    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(toggle_save_images_compressed);
    save_images_compressed = 1 - save_images_compressed;
    item = menu_get(defaults_image_menu, MENU_NTH_ITEM, 2);
    if (save_images_compressed) {
	menu_set(item, MENU_STRING, "Save uncompressed", 0);
	show_msg("Images will be saved");
	show_msg("in compressed form");
    }
    else {
	menu_set(item, MENU_STRING, "Save compressed", 0);
	show_msg("Images will be saved in");
	show_msg("uncompressed form");
    }
}


toggle_save_in_LT_dir_proc()
{
    Menu_item item;
    
    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(toggle_save_in_LT_dir_proc);
    save_in_LT_dir = 1 - save_in_LT_dir;
    item = menu_get(defaults_rule_menu, MENU_NTH_ITEM, 1);
    if (save_in_LT_dir) {	/* is enabled now */
	menu_set(item, MENU_STRING, "Save in current working dir", 0);
	show_msg("LT's will be saved");
	show_msg("in Table-dir");
    }
    else {
	menu_set(item, MENU_STRING, "Save in Table-dir", 0);
	show_msg("LT's will be saved in");
	show_msg("current working dir");
    }
}


toggle_save_LT_compressed()
{
    Menu_item item;
    
    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(toggle_save_LT_compressed);
    save_LT_compressed = 1 - save_LT_compressed;
    item = menu_get(defaults_rule_menu, MENU_NTH_ITEM, 2);
    if (save_LT_compressed) {
	menu_set(item, MENU_STRING, "Save uncompressed", 0);
	show_msg("LT's will be saved");
	show_msg("in compressed form");
    }
    else {
	menu_set(item, MENU_STRING, "Save compressed", 0);
	show_msg("LT's will be saved");
	show_msg("in uncompressed form");
    }
}


toggle_save_in_cmap_dir_proc()
{
    Menu_item item;
    
    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(toggle_save_in_cmap_dir_proc);
    save_in_cmap_dir = 1 - save_in_cmap_dir;
    item = menu_get(defaults_cmap_menu, MENU_NTH_ITEM, 1);
    if (save_in_cmap_dir) {	/* is enabled now */
	menu_set(item, MENU_STRING, "Save in current working dir", 0);
	show_msg("Cmaps will be saved");
	show_msg("in Cmap-dir");
    }
    else {
	menu_set(item, MENU_STRING, "Save in Cmap-dir", 0);
	show_msg("Cmaps will be saved in");
	show_msg("current working dir");
    }
}
    

toggle_auto_image_change_proc()
{
    Menu_item item;

    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(toggle_auto_image_change_proc);
    auto_image_change = 1 - auto_image_change;
    item = menu_get(defaults_image_menu, MENU_NTH_ITEM, 4);
    if (auto_image_change) {	/* it is enabled now */
	menu_set(item, MENU_STRING, "Disable auto-change", 0);
	show_msg("Auto image-change enabled");
    }
    else {
	menu_set(item, MENU_STRING, "Enable auto-change", 0);
	show_msg("Auto image-change disabled");
    }
}


toggle_auto_nhood_change_proc()
{
    Menu_item item;

    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(toggle_auto_nhood_change_proc);
    auto_nhood_change = 1 - auto_nhood_change;
    item = menu_get(defaults_rule_menu, MENU_NTH_ITEM, 5);
    if (auto_nhood_change) {
	menu_set(item, MENU_STRING, "Disable auto-change", 0);
	show_msg("Auto nhood-change enabled");
    }
    else {
	menu_set(item, MENU_STRING, "Enable auto-change", 0);
	show_msg("Auto nhood-change disabled");
    }
}


tsymmetry()
{					/* toggle table symmetry */
    Menu_item item;
    
    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(tsymmetry);
    table_symmetry = 1-table_symmetry;
    item = menu_get(rules_menu, MENU_NTH_ITEM, 5);
    if (table_symmetry) {
	show_msg("Symmetry enabled");
	menu_set(item, MENU_STRING, "Disable symmetry", 0);
    }
    else {
	show_msg("Symmetry disabled");
	menu_set(item, MENU_STRING, "Enable symmetry", 0);
    }
}


FB_toggle()
{
    Menu_item item;

    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(FB_toggle);
    use_FB = 1 - use_FB;
    item = menu_get(defaults_CM_menu, MENU_NTH_ITEM, 4);
    if (use_FB)
	menu_set(item, MENU_STRING, "Disable FB display", 0);
    else
	menu_set(item, MENU_STRING, "Enable FB display", 0);
    CM_set_displays();
}



Sun_disp_toggle()
{
    Menu_item item;

    clear_msg();
    if (invalid_set_pressed())
	return;
    if (setting_sequence)
	add_sequence(Sun_disp_toggle);
    use_Sun_disp = 1 - use_Sun_disp;
    item = menu_get(defaults_CM_menu, MENU_NTH_ITEM, 5);
    if (use_Sun_disp)
	menu_set(item, MENU_STRING, "Disable Sun display", 0);
    else
	menu_set(item, MENU_STRING, "Enable Sun display", 0);
    CM_set_displays();
}


/***** CM-related stuff *****/

static void
CM_send_zoom_proc(item, value, event)
Panel_item item;
int value;
Event *event;
{
    CM_zoom = value;
    if (use_CM)
	CM_send_zoom();
}


static void
CM_send_panx_proc(item, value, event)
Panel_item item;
int value;
Event *event;
{
    CM_panx = value;
    if (use_CM)
	CM_send_pan();
}


static void
CM_send_pany_proc(item, value, event)
Panel_item item;
int value;
Event *event;
{
    CM_pany = value;
    if (use_CM)
	CM_send_pan();
}   


void
CM_get_dirs()
{
    CM_get_fcn_dir();
    CM_get_LT_dir();
    CM_get_image_dir();
    panel_set_value(def_CM_imagedir_item, CM_image_dir);
    panel_set_value(def_CM_fcndir_item, CM_fcn_dir);
    panel_set_value(def_CM_LTdir_item, CM_LT_dir);
}


