/* ------------------------------ TARGET.C ---------------------------
 *	Target control Window Procedure
 * ---------------------------------------------------------------- */

#include "Windows.h"
#include "Target_.h"

// ---- Extra bytes indexes
#define XCUR		0
#define YCUR		2
#define FCAPTURE	4
#define VBBMP		6				// Bitmap handle (for VB)

/* -------------------------- DrawOurFrame ----------------------------
 *	Draws the frame
 * ---------------------------------------------------------------- */

static VOID DrawOurFrame (HWND hWnd, HDC hDC, BOOL fOn)
{
RECT			Rect;
HBRUSH			hBr;

	GetClientRect (hWnd, &Rect);

	if (fOn)
		FrameRect (hDC, &Rect, GetStockObject (GRAY_BRUSH));
	else
	{
		hBr = CreateSolidBrush (
			GetSysColor (GetWindowLong (hWnd, GWL_STYLE) & TGS_BORDER?
									COLOR_WINDOWFRAME: COLOR_WINDOW));
		FrameRect (hDC, &Rect, hBr);
		DeleteObject (hBr);
	}
}

/* -------------------------- DrawCross -------------------------------
 *	Draws the pointer
 * ---------------------------------------------------------------- */

static VOID DrawCross (HWND hWnd)
{
HDC				hDC;
HPEN			hPen, hPenOld;
int				xCur,
				yCur;				// Current position

	xCur = GetWindowWord (hWnd, XCUR);
	yCur = GetWindowWord (hWnd, YCUR);

	hDC = GetDC (hWnd);
	SetROP2 (hDC, R2_NOT);
	hPen = CreatePen (PS_SOLID, 3, RGB (0, 0, 0));
	hPenOld = SelectObject (hDC, hPen);

	MoveTo (hDC, xCur-10, yCur-10);
	LineTo (hDC, xCur+10, yCur+10);
	MoveTo (hDC, xCur-10, yCur+10);
	LineTo (hDC, xCur+10, yCur-10);

	SelectObject (hDC, hPenOld);
	DeleteObject (hPen);
	ReleaseDC (hWnd, hDC);

	SetWindowWord (hWnd, XCUR, xCur);
	SetWindowWord (hWnd, YCUR, yCur);
	return;
}

/* -------------------------- wmKeyboard ----------------------------
 *	Processes WM_CHAR, WM_KEYDOWN et WM_KEYUP messages
 *		wMsg is the message
 *		wParam is the caracter (WM_CHAR) or virtual key (others)
 *		lParam : cf doc
 * ---------------------------------------------------------------- */

static VOID wmKeyboard (HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
static int		nInc;				// Move increment
RECT			Rect;
int				r;
int				xCur,
				yCur;				// Current position

	if (wMsg == WM_KEYDOWN &&
		(wParam == VK_DOWN ||
		 wParam == VK_UP ||
		 wParam == VK_RIGHT ||
		 wParam == VK_LEFT))
	{
		DrawCross (hWnd);			// Erases cross

		if (lParam & (1L<<30))		// Key was down
			nInc++;
		else
			nInc = 1;

		r = LOWORD (lParam);		// Repeat count, is 1 or more
		if (r)
			while (--r)
				nInc++;

		xCur = GetWindowWord (hWnd, XCUR);
		yCur = GetWindowWord (hWnd, YCUR);

		switch (wParam)
		{
		case VK_DOWN:
			yCur += nInc;
			break;

		case VK_UP:
			yCur -= nInc;
			break;

		case VK_RIGHT:
			xCur += nInc;
			break;

		case VK_LEFT:
			xCur -= nInc;
			break;
		}

		GetClientRect (hWnd, &Rect);	// Test max/min
		yCur = max (yCur, 0);
		yCur = min (yCur, Rect.bottom);
		xCur = max (xCur, 0);
		xCur = min (xCur, Rect.right);
		SetWindowWord (hWnd, XCUR, xCur);
		SetWindowWord (hWnd, YCUR, yCur);

		DrawCross (hWnd);

// ---- Sends notification
		SendMessage (GetParent (hWnd), WM_COMMAND,
					GetWindowWord (hWnd, GWW_ID), MAKELONG (hWnd, TGN_CHANGE));
	}

	return;
}

/* -------------------------- wmCreate ------------------------------
 *	Processes WM_CREATE message
 *		lpCrst (lParam) points to a CREATESTRUCT structure
 * ---------------------------------------------------------------- */

static VOID wmCreate (HWND hWnd, LPCREATESTRUCT lpCrst)
{
	SetWindowWord (hWnd, XCUR, lpCrst->cx/2);
	SetWindowWord (hWnd, YCUR, lpCrst->cy/2);
	SetWindowWord (hWnd, FCAPTURE, FALSE);

// ---- Sends notification
	SendMessage (GetParent (hWnd), WM_COMMAND,
					GetWindowWord (hWnd, GWW_ID), MAKELONG (hWnd, TGN_CHANGE));
	return;
}

/* -------------------------- wmMouse -------------------------------
 *	Processes WM_MOUSEMOVE and WM_?BUTTON* messages
 *		wMsg is the message
 *		wParam indicates keys or buttons pressed
 *		x (LO lParam) is x coordinate of pointer
 *		y (HI lParam) is y coordinate of pointer
 * ---------------------------------------------------------------- */

static VOID wmMouse (HWND hWnd, WORD wMsg, WORD wParam, int x, int y)
{
RECT			Rect;

// ---- Keeps cros within rectangle

	switch (wMsg)
	{
	case WM_LBUTTONDOWN:
		SetFocus (hWnd);

		SetCapture (hWnd);
		SetWindowWord (hWnd, FCAPTURE, TRUE);
		ShowCursor (FALSE);

		DrawCross (hWnd);
		SetWindowWord (hWnd, XCUR, x);
		SetWindowWord (hWnd, YCUR, y);
		DrawCross (hWnd);

		GetClientRect (hWnd, &Rect);
		ClientToScreen (hWnd, (LPPOINT)&Rect);
		ClientToScreen (hWnd, (LPPOINT)&Rect.right);
		ClipCursor (&Rect);			// Clips cursor

		break;

	case WM_MOUSEMOVE:
		if (GetWindowWord (hWnd, FCAPTURE))
		{
			DrawCross (hWnd);
			SetWindowWord (hWnd, XCUR, x);
			SetWindowWord (hWnd, YCUR, y);
			DrawCross (hWnd);
		}

		break;

	case WM_LBUTTONUP:
		if (GetWindowWord (hWnd, FCAPTURE))
		{
			ClipCursor (NULL);		// Frees cursor
			ReleaseCapture ();
			SetWindowWord (hWnd, FCAPTURE, FALSE);
			ShowCursor (TRUE);

// ---- Sends notification
			SendMessage (GetParent (hWnd), WM_COMMAND,
					GetWindowWord (hWnd, GWW_ID), MAKELONG (hWnd, TGN_CHANGE));
		}
		break;
	}
	return;
}

/* -------------------------- wmPaint -------------------------------
 *	Processes WM_PAINT message
 * ---------------------------------------------------------------- */

static VOID wmPaint (HWND hWnd)
{
PAINTSTRUCT		ps;
HDC				hDC;
RECT			Rect;
HBRUSH			hBr, hBrOld;
int				X, Y;
register		i;
char			szText[256];
HBITMAP			hBmp, hBmpOld;
HDC				hDCMem;
BITMAP			Bmp;
static COLORREF	Color [] =
{
	0x00000000L, 0x000000ffL, 0x0000ff00L, 0x0000ffffL,
	0x00ff0000L, 0x00ff00ffL, 0x00ffff00L, 0x00ffffffL
};

// ---- Hides cross
	DrawCross (hWnd);

	hDC = BeginPaint (hWnd, &ps);

// ---- Draws background
	GetClientRect (hWnd, &Rect);
	switch (GetWindowLong (hWnd, GWL_STYLE) & 7)	// 3 lower bits
	{
	case TGS_CIRCLES:
		InflateRect (&Rect, -1, -1);
		X = -Rect.right/16;
		Y = -Rect.bottom/16;
		for (i = 0; i < 8; ++i)
		{							// Draws 5 ellipses
			hBr = CreateSolidBrush (Color[i]);
			hBrOld = SelectObject (hDC, hBr);
			Ellipse (hDC, Rect.left, Rect.top, Rect.right, Rect.bottom);
			InflateRect (&Rect, X, Y);
			SelectObject (hDC, hBrOld);
			DeleteObject (hBr);
		}
		break;

	case TGS_PLAIN:
		hBr = (HBRUSH)SendMessage (GetParent (hWnd), WM_CTLCOLOR,
									hDC, MAKELONG (hWnd, 0));
		if (hBr)
			FillRect (hDC, &Rect, hBr);
		break;

	case TGS_BITMAP:
		if (GetWindowLong (hWnd, GWL_STYLE) & TGS_VB)
			hBmp = GetWindowWord (hWnd, VBBMP);
		else
		{
			GetWindowText (hWnd, szText, sizeof szText);
			hBmp = LoadBitmap (GetWindowWord (hWnd, GWW_HINSTANCE), szText);
		}

		if (hBmp)
		{
			hDCMem = CreateCompatibleDC (hDC);
			hBmpOld = SelectObject (hDCMem, hBmp);
			GetObject (hBmp, sizeof (BITMAP), (LPSTR)&Bmp);
//			BitBlt (hDC, 0, 0, Bmp.bmWidth, Bmp.bmHeight, hDCMem, 0, 0, SRCCOPY);
			StretchBlt (hDC, 0, 0, Rect.right, Rect.bottom, hDCMem, 0, 0, Bmp.bmWidth, Bmp.bmHeight, SRCCOPY);
			SelectObject (hDCMem, hBmpOld);
			DeleteDC (hDCMem);
		}
		break;
	}

// ---- Draws frame
	DrawOurFrame (hWnd, hDC, GetFocus () == hWnd);

	EndPaint (hWnd, &ps);

// ---- Restores cross
	DrawCross (hWnd);

	return;
}

/* -------------------------- wmSize --------------------------------
 *	Processes WM_SIZE message
 *		wParam is type of size change
 *		cx (LO lParam) is new width
 *		cy (HI lParam) is new height
 * ---------------------------------------------------------------- */

static VOID wmSize (HWND hWnd, WORD wParam, int cx, int cy)
{
// ---- Keeps cross within window
	if (wParam != SIZEICONIC)
	{
		if ((int)GetWindowWord (hWnd, XCUR) > cx)
			SetWindowWord (hWnd, XCUR, cx);
		if ((int)GetWindowWord (hWnd, YCUR) > cy)
			SetWindowWord (hWnd, YCUR, cy);
	}

	return;
}

/* -------------------------- wmSetFocus ----------------------------
 *	Processes WM_SETFOCUS message
 *		hWndOther (wParam) is window loosing focus
 * ---------------------------------------------------------------- */

static VOID wmSetFocus (HWND hWnd, HWND hWndOther)
{
HDC				hDC;

// ---- Draws a gray frame
	hDC = GetDC (hWnd);
	DrawOurFrame (hWnd, hDC, TRUE);
	ReleaseDC (hWnd, hDC);
	return;
}

/* -------------------------- wmKillFocus ---------------------------
 *	Processes WM_KILLFOCUS message
 *		hWndOther (wParam) is window getting focus
 * ---------------------------------------------------------------- */

static VOID wmKillFocus (HWND hWnd, HWND hWndOther)
{
HDC				hDC;

// ---- Draws a normal frame
	hDC = GetDC (hWnd);
	DrawOurFrame (hWnd, hDC, FALSE);
	ReleaseDC (hWnd, hDC);
	return;
}

/* -------------------------- tgmSetPos -----------------------------
 *	Processes TGM_SETPOS message
 *		x (LO lParam) is x pos
 *		y (HI lParam) is y pos
 * ---------------------------------------------------------------- */

static VOID tgmSetPos (HWND hWnd, int x, int y)
{
	DrawCross (hWnd);				// Erases cross
	SetWindowWord (hWnd, XCUR, x);
	SetWindowWord (hWnd, YCUR, y);
	DrawCross (hWnd);				// Draws cross

// ---- Sends notification
	SendMessage (GetParent (hWnd), WM_COMMAND,
					GetWindowWord (hWnd, GWW_ID), MAKELONG (hWnd, TGN_CHANGE));
}

/* -------------------------- TargetWndProc ---------------------------
 *	Target WndProc - Processes messages
 * ---------------------------------------------------------------- */

LRESULT CALLBACK _export TargetWndProc (HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
long			lRet = 0;			// Return value
BOOL			fProcessed = TRUE;	// Indicates command processed

	switch (wMsg)
	{
// ---- Specific TARGET messages
	case TGM_SETPOS:
		tgmSetPos (hWnd, LOWORD (lParam), HIWORD (lParam));
		break;

	case TGM_GETPOS:
		lRet = MAKELONG (GetWindowWord (hWnd, XCUR),
						 GetWindowWord (hWnd, YCUR));
		break;

	case TGM_SETBMP:
		SetWindowWord (hWnd, VBBMP, wParam);
		break;

// ---- Windows messages
	case WM_GETDLGCODE:
		lRet = DLGC_WANTARROWS;		// Want arrows
		break;

	case WM_CREATE:
		wmCreate (hWnd, (LPCREATESTRUCT)lParam);
		break;

	case WM_SETFOCUS:
		wmSetFocus (hWnd, wParam);
		break;

	case WM_KILLFOCUS:
		wmKillFocus (hWnd, wParam);
		break;

	case WM_PAINT:
		wmPaint (hWnd);
		break;

	case WM_SIZE:
		wmSize (hWnd, wParam, LOWORD (lParam), HIWORD (lParam));
		break;

	case WM_CHAR:
	case WM_KEYDOWN:
	case WM_KEYUP:
		wmKeyboard (hWnd, wMsg, wParam, lParam);
		break;

	case WM_MOUSEMOVE:
	case WM_LBUTTONDOWN:
	case WM_LBUTTONUP:
	case WM_LBUTTONDBLCLK:
	case WM_RBUTTONDOWN:
	case WM_RBUTTONUP:
	case WM_RBUTTONDBLCLK:
	case WM_MBUTTONDOWN:
	case WM_MBUTTONUP:
	case WM_MBUTTONDBLCLK:
		wmMouse (hWnd, wMsg, wParam, LOWORD (lParam), HIWORD (lParam));
		break;

	default:
		fProcessed = FALSE;
		break;
	}

// ---- Default processing
	if (!fProcessed)
		lRet = DefWindowProc (hWnd, wMsg, wParam, lParam);

	return lRet;
}
