/*
** This file is the std.t replacement for "Common Ground". As such it
** contains a lot of the basic definitions for the adventure.
**
** Copyright (c) 1999, Stephen Granade. All rights reserved.
*/

/* Use C operators */
#pragma C+

/*
 *   Pre-declare all functions, so the compiler knows they are functions.
 *   (This is only really necessary when a function will be referenced
 *   as a daemon or fuse before it is defined; however, it doesn't hurt
 *   anything to pre-declare all of them.)
 */
die: function;
scoreRank: function;
init: function;
terminate: function;
pardon: function;
darkTravel: function;
mainRestore: function;
commonInit: function;
initRestore: function;
getSysInfo: function;    // This function is in checks.t
changeQuotes: function;  // This function is in gndparse.t

/*
** The die() function is called when the player dies. Since you can't
** really die in "Common Ground," this is just what is called when the
** story ends.
*/
die: function
{
    if (global.isHTMLRuntime)
        "\b<CENTER><FONT SIZE=\"+1\">The End</FONT></CENTER>\b";
    else "\b\(The End\)\b";

    "\bYou may restore a saved game, start over, quit, or undo
    the current command.\n";
    while (true)
    {
        local resp;

	"\nPlease enter (R)ESTORE, RE(S)TART, (Q)UIT, or (U)NDO: >";
        resp = upper(input());
        if (resp == 'RESTORE' || resp == 'R') {
            resp = askfile('File to restore',
                            ASKFILE_PROMPT_OPEN, FILE_TYPE_SAVE);
            if (resp == nil)
                "Restore failed. ";
            else if (restore(resp))
                "Restore failed. ";
            else {
                parserGetMe().location.lookAround(true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
        }
        else if (resp == 'RESTART' || resp == 'S') {
            scoreStatus(0, 0);
            restart();
        }
        else if (resp == 'QUIT' || resp == 'Q') {
            terminate();
            quit();
            abort;
        }
        else if (resp == 'UNDO' || resp == 'U') {
            if (undo())
            {
                "(Undoing one command)\b";
                parserGetMe().location.lookAround(true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
            else
                "Sorry, no undo information is available. ";
        }
    }
}

/*
** scoreRank() shows how well the player is doing.
*/
scoreRank: function
{
    "You have achieved a score of <<global.score>> points out of a
     possible <<global.maxscore>>.\n";
}

/*
** I'm going to have to modify restoreVerb to call getSysInfo if
** the restored game is okay.
** [N.B. This code comes from adv.t]
*/
modify restoreVerb
    restoreGame(actor) = {
        local savefile, flag;
        
        savefile = askfile('File to restore game from',
			   ASKFILE_PROMPT_OPEN, FILE_TYPE_SAVE);
        if (savefile == nil || savefile == '') {
            "Failed. ";
            return nil;
        }
        else {
            flag = mainRestore(savefile);
            if (flag) {
                getSysInfo();
                "\H+";
            }
            return flag;
        }
    }
;

/*
** I'm also going to modify quitVerb to keep it from trying to print
** a score, and I'll get rid of scoreVerb.
** [N.B. This code comes from adv.t]
*/

modify quitVerb
    quitGame(actor) = {
        local yesno;

        "You sure you want to quit? (YES or NO) > ";
        yesno = yorn();
        "\b";
        if (yesno == 1) {
            terminate();    // allow user good-bye message
            quit();
        }
        else "Okay. ";
    }
;

replace scoreVerb: object
;

/*
 *   commonInit() - this is not a system-required function; this is simply
 *   a helper function that we define so that we can put common
 *   initialization code in a single place.  Both init() and initRestore()
 *   call this.  Since the system may call either init() or initRestore()
 *   during game startup, but not both, it's desirable to have a single
 *   function that both of those functions call for code that must be
 *   performed at startup regardless of how we're starting. 
 */
commonInit: function
{
    "\H+";             // On goes the HTML formatting

    clearscreen();     // Clear the screen
    getSysInfo();      // Set up system-dependent information
    randomize();       // Re-seed the RNG
    if (global.isHTMLRuntime) {
        "<ABOUTBOX><center>
	 <img src=\"title.png\" alt=\"Common Ground\"></center>
	 Copyright &copy; 1999, Stephen Granade<br>
         http://www.phy.duke.edu/~sgranade/ground.html</ABOUTBOX>";
        "<TITLE>Common Ground</TITLE>";
	setOutputFilter(changeQuotes);
	global.curlyQuotesOn = true;
    }
}

/*
 *   The init() function is run at the very beginning of the game.
 *   It should display the introductory text for the game, start
 *   any needed daemons and fuses, and move the player's actor ("Me")
 *   to the initial room
 */
init: function
{
    /* perform common initializations */
    commonInit();
    
    version.sdesc;              // display the game's name and version number

    if (global.canDoGraphics)
	gprintf('one.png', 'alt="One"');
    else "\(One\)\b";
    "\nGreat, just great. First you oversleep, end up being late for school.
     Then the whole day sucks so bad you can barely wait until seventh
     period's over and you're free. Now you've wasted so
     much time watching TV that you've gotta rush through your makeup
     or you won't be ready for tonight before April gets here with
     her wheels. Mississippi may say you're old enough to drive, but
     Frank sure doesn't agree.
     \bMan, you can't wait for all this to be over.\b";

    parserSetMe(jeanieMe);

    setdaemon(turncount, nil);               // start the turn counter daemon
    parserGetMe().location = jeanie_chair;   // move player to initial room
    jeanie_bedroom.lookAround(true);         // show player where he is
    jeanie_bedroom.isseen = true;            // note that we've seen the room
    scoreStatus(0, 0);                       // initialize the score display
}

/*
 *   initRestore() - the system calls this function automatically at game
 *   startup if the player specifies a saved game to restore on the
 *   run-time command line (or through another appropriate
 *   system-dependent mechanism).  We'll simply restore the game in the
 *   normal way.
 */
initRestore: function(fname)
{
    /* perform common initializations */
    commonInit();

    /* tell the player we're restoring a game */
    "\b[Restoring saved position...]\b";

    /* go restore the game in the normal way */
    mainRestore(fname);
}


/*
 *   preinit() is called after compiling the game, before it is written
 *   to the binary game file.  It performs all the initialization that can
 *   be done statically before storing the game in the file, which speeds
 *   loading the game, since all this work has been done ahead of time.
 *
 *   This routine puts all lamp objects (those objects with islamp = true) into
 *   the list global.lamplist.  This list is consulted when determining whether
 *   a dark room contains any light sources.
 */
preinit: function
{
    initSearch();
}

/*
 *   The terminate() function is called just before the game ends.  It
 *   generally displays a good-bye message.  The default version does
 *   nothing.  Note that this function is called only when the game is
 *   about to exit, NOT after dying, before a restart, or anywhere else.
 */
terminate: function
{
    "Thank you for playing <b>Common Ground</b>. ";
}

/*
 *   The pardon() function is called any time the player enters a blank
 *   line.  The function generally just prints a message ("Speak up" or
 *   some such).  This default version just says "I beg your pardon?"
 */
pardon: function
{
    "You do nothing. ";
}

/*
 *   The numObj object is used to convey a number to the game whenever
 *   the player uses a number in his command.  For example, "turn dial
 *   to 621" results in an indirect object of numObj, with its "value"
 *   property set to 621.
 */
numObj: basicNumObj  // use default definition from adv.t
;

/*
 *   strObj works like numObj, but for strings.  So, a player command of
 *     type "hello" on the keyboard
 *   will result in a direct object of strObj, with its "value" property
 *   set to the string 'hello'.
 *
 *   Note that, because a string direct object is used in the save, restore,
 *   and script commands, this object must handle those commands.
 *
 *   Also, I've changed it to tell the player to use TALK TO if they
 *   try to say something.
 */
strObj: basicStrObj     // use default definition from adv.t
    doSay(actor) = {
	"If you want to talk to someone, use TALK TO. ";
	abort;
    }
;

/*
 *   The "global" object is the dumping ground for any data items that
 *   don't fit very well into any other objects.  The properties of this
 *   object that are particularly important to the objects and functions
 *   are defined here; if you replace this object, but keep other parts
 *   of this file, be sure to include the properties defined here.
 *
 *   Note that awakeTime is set to zero; if you wish the player to start
 *   out tired, just move it up around the sleepTime value (which specifies
 *   the interval between sleeping).  The same goes for lastMealTime; move
 *   it up to around eatTime if you want the player to start out hungry.
 *   With both of these values, the player only starts getting warnings
 *   when the elapsed time (awakeTime, lastMealTime) reaches the interval
 *   (sleepTime, eatTime); the player isn't actually required to eat or
 *   sleep until several warnings have been issued.  Look at the eatDaemon
 *   and sleepDaemon functions for details of the timing.
 */
global: object
    turnsofar = 0               // no turns have transpired so far
    score = 0                   // no points have been accumulated yet
    maxscore = 100              // maximum possible score
    verbose = true              // Begin in verbose mode
    pauseRecorder = nil         // True if we aren't to record the next
                                //  action Jeanie takes
    minVersionNumber = '2.5.1'  // See checks.t
    curlyQuotesOn = nil         // Do we change single quotes to curly ones?
;

/*
 *   The "version" object defines, via its "sdesc" property, the name and
 *   version number of the game.  Change this to a suitable name for your
 *   game.
 */
version: object
    compile_time = __DATE__
    sdesc = {
	"\b";
	if (global.isHTMLRuntime) {
	    if (graphicsOn())
		gprintf('title.png', 'alt="Common Ground" align="right"');
	    else "<h3>Common Ground</h3>";
	}
	else "\(Common Ground\)";
	"\nCopyright &copy; 1999 by Stephen Granade. All rights reserved.
	 \bType <q><<link('about', 'about')>></q> for more information.
	 \nVersion <<VERSION>> (<<self.compile_time>>)
	 \nDeveloped with TADS, the Text Adventure Development System.\b";
    }
;

/*
 *   "Me" is the initial player's actor.
 */
Me: basicMe
;

// Empty defs of darkTravel() and goToSleep().
darkTravel: function {}
goToSleep: function {}

/*
 *   commandPrompt - this displays the command prompt.  For HTML games, we
 *   switch to the special TADS-Input font, which lets the player choose
 *   the font for command input through the preferences dialog.
 */
commandPrompt: function(code)
{
    if (global.isHTMLRuntime)
	"\b&gt;<font face='TADS-Input'>";
    else "\b>";
}

/*
 *   commandAfterRead - this is called just after each command is entered.
 *   For HTML games, we'll switch back to the original font that was in
 *   effect before commandPrompt switched to TADS-Input. 
 */
commandAfterRead: function(code)
{
    if (global.isHTMLRuntime)
	"</font>";
}
