/*
 * Copyright (c) 1995, 1994, 1993, 1992, 1991, 1990  
 * Open Software Foundation, Inc. 
 *  
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appears in all copies and 
 * that both the copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of ("OSF") or Open Software 
 * Foundation not be used in advertising or publicity pertaining to 
 * distribution of the software without specific, written prior permission. 
 *  
 * OSF DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE 
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
 * FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL OSF BE LIABLE FOR ANY 
 * SPECIAL, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES 
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN 
 * ACTION OF CONTRACT, NEGLIGENCE, OR OTHER TORTIOUS ACTION, ARISING 
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE 
 */
/*
 * OSF Research Institute MK6.1 (unencumbered) 1/31/1995
 */
/* CMU_HIST */
/*
 * Revision 2.3  92/08/03  17:16:40  jfriedl
 * 	Hacked boot_load_program and read_emulator_symbols to print
 * 	the location of unix and emulator symbols.  This is to make
 * 	sure startup loads the emulator text from the same place as
 * 	the emulator symbol table.
 * 	[92/08/02            cmaeda]
 * 
 * Revision 2.2  92/01/03  19:57:44  dbg
 *      Close startup file after loading it.
 * 	[91/09/25            dbg]
 * 
 * 	Move outside of kernel.
 * 	[91/09/04            dbg]
 * 
 * Revision 2.14  91/09/04  11:27:54  jsb
 * 	Fixed a printf in read_symtab_from_file (following failure of
 * 	get_symtab) by not printing unset variable ``result''.
 * 	[91/09/01  07:48:45  jsb]
 * 
 * Revision 2.13  91/08/28  11:09:49  jsb
 * 	Fixed to check vm_allocate return codes, esp in read_symtab_from_file.
 * 	[91/08/26            rpd]
 * 
 * Revision 2.12  91/07/31  17:25:29  dbg
 * 	Replace blkclr with bzero.
 * 	[91/07/26            dbg]
 * 
 * 	Moved read_symtab_from_file here and made independent of load
 * 	format.
 * 	[91/06/25            dbg]
 * 
 * 
 * Revision 2.11  91/05/14  15:23:41  mrt
 * 	Correcting copyright
 * 
 * Revision 2.10  91/03/16  14:41:37  rpd
 * 	Fixed read_emulator_symbols to not panic if there is no emulator.
 * 	[91/03/13            rpd]
 * 
 * Revision 2.9  91/02/05  17:01:45  mrt
 * 	Changed to new copyright
 * 	[91/01/28  14:55:09  mrt]
 * 
 * Revision 2.8  90/10/25  14:42:05  rwd
 * 	Modified boot_load_program and read_emulator_symbols
 * 	to allocate arrays from heap, not stack.
 * 	[90/10/23            rpd]
 * 
 * Revision 2.7  90/08/27  21:46:02  dbg
 * 	Reduce lint.
 * 	[90/08/13            dbg]
 * 
 * 	Use new error codes.  Use new file_io package.
 * 	[90/07/16            dbg]
 * 
 * Revision 2.6  90/06/02  14:45:31  rpd
 * 	Added read_emulator_symbols.
 * 	[90/05/11  17:08:19  rpd]
 * 
 * 	From jsb: to support debugging, read in the server's symbol table.
 * 	[90/04/23            rpd]
 * 	Converted to new IPC.
 * 	[90/03/26  21:31:33  rpd]
 * 
 * Revision 2.5  90/01/11  11:41:20  dbg
 * 	Use bootstrap-task print routines.
 * 	[89/12/20            dbg]
 * 
 * Revision 2.4  89/11/14  10:28:13  dbg
 * 	Restore read-only protection for text segment.
 * 	[89/11/01            dbg]
 * 
 * Revision 2.3  89/09/08  11:22:13  dbg
 * 	Use new open_file routine.  Change 'struct inode' to
 * 	'struct file' for new stand-alone IO package.
 * 	[89/08/31            dbg]
 * 
 * 26-May-89  Randall Dean (rwd) at Carnegie-Mellon University
 *	Fixed separate text/data case.  Added copyright to version
 *	generated by dbg
 *
 */
/* CMU_ENDHIST */
/* 
 * Mach Operating System
 * Copyright (c) 1991,1990,1989,1988 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Software Distribution Coordinator  or  Software.Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 * 
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

#include "bootstrap.h"

#include <stdarg.h>
#include <limits.h>

#include <ddb/nlist.h>

int load_program_file(struct file *, task_port_t, thread_port_t, objfmt_t);
void read_symtab_from_file(struct file *, mach_port_t, task_port_t,
			   const char *, objfmt_t);

/*
 * Loads the file into the user space of the specified task,
 * and sets up the starting thread's registers.
 */

boolean_t	load_protect_text = TRUE;

int
load_program_file(register struct file	*fp,
		  task_port_t		user_task,
		  thread_port_t		user_thread,
		  objfmt_t		ofmt)
{
	struct loader_info	*lp = &ofmt->info;
	kern_return_t		result;
	vm_offset_t		text_page_start,
				text_page_end,
				data_page_start,
				data_page_end,
				area_start;

	if ((fp->f_di.di_mode & IFMT) != IFREG ||
	    (fp->f_di.di_mode & (IEXEC|IEXEC>>3|IEXEC>>6)) == 0) {
		printf("server not executable\n");
		return (EX_NOT_EXECUTABLE);
	}

	/*
	 * Allocate space.
	 */
	text_page_start = trunc_page(lp->text_start);
	text_page_end   = round_page(lp->text_start + lp->text_size);
	data_page_start = trunc_page(lp->data_start);
	data_page_end   = round_page(lp->data_start + lp->data_size)
		+ round_page(lp->bss_size);

	if (text_page_end >= data_page_start) {
	    /*
	     * One contiguous area for text and data.
	     */
	    result = vm_allocate(mach_task_self(),
				&area_start,
				(vm_size_t)(data_page_end - text_page_start),
				TRUE);
	    if (result) {
		printf("allocate text+data failed\n");
		return (result);
	    }

	    result = read_file(fp,
				lp->text_offset,
				area_start + (lp->text_start - text_page_start),
				lp->text_size);
	    if (result) {
		printf("read text into text+data failed\n");
		return (result);
	    }

	    result = read_file(fp,
				lp->data_offset,
				area_start + (lp->data_start - text_page_start),
				lp->data_size);
	    if (result) {
		printf("read data into text+data failed\n");
		return (result);
	    }

	    bzero((char *)(area_start +	(lp->data_start + lp->data_size - text_page_start)),
		  lp->bss_size);

	    result = vm_allocate(user_task,
				&text_page_start,
				(vm_size_t)(data_page_end - text_page_start),
				FALSE);
	    if (result) {
		printf("allocate text+data (%x,%x) failed\n", text_page_start,
		       data_page_end);
		return (result);
	    }

	    result = vm_write(user_task,
				text_page_start,
				(pointer_t)area_start,
				(vm_size_t)(data_page_end - text_page_start));
	    if (result) {
		printf("write text+data failed\n");
		return (result);
	    }

	    result = vm_deallocate(mach_task_self(),
				area_start,
				(vm_size_t)(data_page_end - text_page_start));
	    if (result) {
		printf("deallocate text+data failed\n");
		return (result);
	    }

	}
	else {
	    /*
	     * Separated text and data areas.
	     */
	    /*
	     * First read in the text
	     */
	    result = vm_allocate(mach_task_self(),
				&area_start,
				(vm_size_t)(text_page_end - text_page_start), TRUE);
	    if (result) {
		printf("allocate text buffer failed\n");
		return (result);
	    }

	    result = read_file(fp,
			       lp->text_offset,
			       area_start + (lp->text_start - text_page_start),
			       lp->text_size);
	    if (result) {
		printf("read text failed\n");
		return (result);
	    }

	    result = vm_allocate(user_task,
				&text_page_start,
				(vm_size_t)(text_page_end - text_page_start),
				FALSE);
	    if (result) {
		printf("allocate text (%x,%x) failed\n", text_page_start,
		       data_page_end);
		return (result);
	    }

	    result = vm_write(user_task,
				text_page_start,
				(pointer_t)area_start,
				(vm_size_t)(text_page_end - text_page_start));
	    if (result) {
		printf("write text failed\n");
		return (result);
	    }

	    result = vm_deallocate(mach_task_self(),
				area_start,
				(vm_size_t)(text_page_end - text_page_start));
	    if (result) {
		printf("deallocate text failed\n");
		return (result);
	    }

	    /*
	     * Then the data
	     */
	    result = vm_allocate(mach_task_self(),
				&area_start,
				(vm_size_t)(data_page_end - data_page_start),
				TRUE);
	    if (result) {
		printf("allocate data buffer failed\n");
		return (result);
	    }

	    result = read_file(fp,
			       lp->data_offset,
			       area_start + (lp->data_start - data_page_start),
			       lp->data_size);
	    if (result) {
		printf("read data failed\n");
		return (result);
	    }

	    bzero((char *)(area_start + (lp->data_start + lp->data_size - data_page_start)), lp->bss_size);

	    result = vm_allocate(user_task,
				&data_page_start,
				(vm_size_t)(data_page_end - data_page_start),
				FALSE);
	    if (result) {
		printf("allocate data (%x,%x) failed\n", data_page_start,
		       data_page_end);
		return (result);
	    }

	    result = vm_write(user_task,
				data_page_start,
				(pointer_t)area_start,
				(vm_size_t)(data_page_end - data_page_start));
	    if (result) {
		printf("write data failed\n");
		return (result);
	    }

	    result = vm_deallocate(mach_task_self(),
				area_start,
				(vm_size_t)(data_page_end - data_page_start));
	    if (result) {
		printf("deallocate data failed\n");
		return (result);
	    }

	}
	/*
	 * Protect the text.
	 */
	if (load_protect_text) {
	    result = vm_protect(user_task,
			    text_page_start,
			    (vm_size_t)(trunc_page(lp->text_start+lp->text_size) - text_page_start),
			    FALSE,
			    VM_PROT_READ|VM_PROT_EXECUTE);
	    if (result) {
		printf("protect text failed\n");
		return (result);
	    }
	}

	return (KERN_SUCCESS);
}

#define	isspace(c)	((c) == ' ' || (c) == '\t')

/*
 * Uses code similar to boot_load_program, but merely determines whether
 * or not the file is kernel-loadable, i.e. is linked at legal kernel
 * addresses.  Will become obsolete when we can relocate servers at load
 * time.
 */
boolean_t
is_kernel_loadable(mach_port_t master_device_port,
		  const char *server_dir_name,
		  const char *file_name)
{
	kern_return_t	result;
	boolean_t	kernel_loadable;
	struct file	file;
	char		*namebuf;
	struct objfmt	ofmt;
	vm_offset_t	text_page_start;

	/*
	 * Build file name: "$server_dir_name/$filename"
	 * Dont insert mach_servers/ if startup name is absolute.
	 */
	namebuf = (char *)malloc((size_t)PATH_MAX+1);
	if (*file_name != '/')
	    (void) strbuild(namebuf,
			    server_dir_name, "/", file_name, (char *)0);
	else 
	    (void) strbuild(namebuf,
			    "/dev/boot_device", file_name, (char *)0);

	/*
	 * Open the file
	 */
	bzero((char *)&file, sizeof(file));
	result = open_file(master_device_port, namebuf, &file);
	if (result != 0) {
	    panic("openi %d", result);
	}

	/*
	 * Read in the header and get the file pointers.
	 */
	result = ex_get_header(&file, &ofmt);
	if (result)
		kernel_loadable = FALSE;
	else {
		text_page_start = trunc_page(ofmt.info.text_start);
		kernel_loadable = text_page_start >= unix_mapbase &&
				  text_page_start < unix_mapend;
		/* By default, load it at the base address it was linked for: */
		if (kernel_loadable && !unix_mapbase_specified)
			unix_mapbase = text_page_start;
	}

	close_file(&file);
	return kernel_loadable;
}

/*VARARGS4*/
int
boot_load_program(mach_port_t master_host_port,
		  mach_port_t master_device_port,
		  task_port_t user_task,
		  thread_port_t user_thread,
		  const char *symtab_name,
		  const char *server_dir_name,
		  const char *file_name)
{
	int		arg_len;
	char		*arg_pos;
	kern_return_t	result;
	struct file	file;
	char		*namebuf;
	struct objfmt	ofmt;
	unsigned long	mapend;

	/*
	 * Build file name: "$server_dir_name/$filename"
	 * Dont insert mach_servers/ if startup name is absolute.
	 */
	namebuf = (char *)malloc((size_t)PATH_MAX+1);
	if (*file_name != '/')
	    (void) strbuild(namebuf,
			    server_dir_name, "/", file_name, (char *)0);
	else 
	    (void) strbuild(namebuf,
			    "/dev/boot_device", file_name, (char *)0);

	/*
	 * Open the file
	 */
	bzero((char *)&file, sizeof(file));
	result = open_file(master_device_port, namebuf, &file);
	if (result != 0) {
	    panic("openi %d", result);
	}

	/*
	 * Allocate space for:
	 *    dummy 0 argument count
	 *    dummy 0 pointer to arguments
	 *    dummy 0 pointer to environment variables
	 *    and align to integer boundary
	 */
	arg_len = sizeof(int) + 2 * sizeof(char *);
	arg_len = (arg_len + (sizeof(int) - 1)) & ~(sizeof(int)-1);

	/*
	 * Read in the header and get the file pointers.
	 */
	result = ex_get_header(&file, &ofmt);
	if (result)
		return (result);

	/*
	 * Load the file
	 */
	result = load_program_file(&file, user_task, user_thread, &ofmt);
	if (result)
	    panic("load_program_file %d", result);

	/*
	 * Set up the stack and user registers.
	 */
	if (!unix_in_kernel || strcmp(symtab_name, "unix") != 0)
		mapend = 0;
	else
		mapend = unix_mapbase + unix_mapsize;
	arg_pos = set_regs(master_host_port, user_task,
	    user_thread, &ofmt.info, mapend, arg_len);

	/*
	 * Read symbols from file
	 */
	if (symtab_name != (char *) 0) {
		printf("(bootstrap): loading symbols from %s", namebuf);
		printf("(%s)\n", ofmt.fmt->name);
		(*ofmt.fmt->symload)(&file, master_host_port, user_task,
				     symtab_name, &ofmt);
	}

	/*
	 * Copy out the arguments.
	 */
	{
	    vm_offset_t	u_arg_start;
				/* user start of argument list block */
	    vm_offset_t	k_arg_start;
				/* kernel start of argument list block */
	    vm_offset_t u_arg_page_start;
				/* user start of args, page-aligned */
	    vm_size_t	arg_page_size;
				/* page_aligned size of args */
	    vm_offset_t	k_arg_page_start;
				/* kernel start of args, page-aligned */

	    register
	    char **	k_ap;	/* kernel arglist address */

	    /*
	     * Get address of argument list in user space
	     */
	    u_arg_start = (vm_offset_t)arg_pos;

	    /*
	     * Round to page boundaries, and allocate kernel copy
	     */
	    u_arg_page_start = trunc_page(u_arg_start);
	    arg_page_size = (vm_size_t)(round_page(u_arg_start + arg_len)
					- u_arg_page_start);

	    result = vm_allocate(mach_task_self(),
				 &k_arg_page_start,
				 (vm_size_t)arg_page_size,
				 TRUE);
	    if (result)
		panic("boot_load_program: arg size");

	    /*
	     * Set up addresses corresponding to user pointers
	     * in the kernel block
	     */
	    k_arg_start = k_arg_page_start + (u_arg_start - u_arg_page_start);

	    k_ap = (char **)k_arg_start;

	    /*
	     * first the argument count
	     */
	    *k_ap++ = (char *)0;

	    /*
	     * then the null pointer to arguments
	     */
	    *k_ap++ = (char *)0;

	    /*
	     * last, the null environment pointer.
	     */
	    *k_ap   = (char *)0;

	    /*
	     * Now write all of this to user space.
	     */
	    (void) vm_write(user_task,
			    u_arg_page_start,
			    k_arg_page_start,
			    arg_page_size);

	    (void) vm_deallocate(mach_task_self(),
				 k_arg_page_start,
				 arg_page_size);
	}

	free(namebuf);

	/*
	 * Close the file.
	 */
	close_file(&file);
	return (0);
}

int
ex_get_header(struct file *fp, objfmt_t ofmt)
{
    int i;
    objfmt_switch_t fmt;
    char buf[HEADER_MAX], *bufptr = buf;
    int result;

    result = read_file(fp, 0, (vm_offset_t)buf, sizeof(buf));
    if (result)
	return (result);

    for (i = 0; formats[i] != 0; i++) {
	fmt = formats[i];
	if (!(*fmt->recog)(fp, ofmt, &bufptr))
	    continue;
	ofmt->fmt = fmt;
	result = (*fmt->load)(fp, ofmt, bufptr);
	if (bufptr != buf)
	    free(bufptr);
	return result;
    }
    return EX_NOT_EXECUTABLE;
}
