/*
 * RPC AARP daemon for CAP services via Native EtherTalk
 * (also maintains information about the current bridge)
 *
 * Created: Charles Hedrick, Rutgers University <hedrick@rutgers.edu>
 * Modified: David Hornsby, Melbourne University <djh@munnari.OZ.AU>
 *	16/02/91: add support for CAP 6.0 atalkdbm routines, back out
 *	of shared memory in favour of RPC based [SG]etBridgeAddress()
 *	Add the SVC descriptors to the low level scheduler.
 */

#include <stdio.h>
#include <errno.h>
#include <sys/file.h>
#include <sys/types.h>
#include <netat/appletalk.h>
#include <rpc/rpc.h>
#include <rpc/svc.h>
#include "../uab/ethertalk.h"		/* iso: level 1 */
#include "../uab/if_desc.h"
#include "../uab/log.h"
#include "aarpd.h"

extern struct lap_description ethertalk_lap_description;
extern u_char *etalk_resolve();		/* address resolver		*/
extern void aarpdprog();		/* the RPC program interface	*/
extern char interface[50];		/* which ethernet device	*/
extern char this_zone[34];		/* zone for this host		*/
extern byte bridge_node;		/* the local bridge		*/
extern word bridge_net;			/* the local bridge		*/
extern byte this_node;			/* this host node		*/
extern word this_net;			/* this host node		*/
extern byte nis_node;			/* atis running here		*/
extern word nis_net;			/* atis running here		*/

extern short lap_proto;			/* our LAP mechanism		*/

byte pzone[INTZONESIZE+1];		/* zone (pascal string)		*/
char intrfc[INTFSIZE];			/* interface description	*/
u_char null_ether[6];			/* null ethernet address	*/
AddrBlock baddr;			/* current bridge address	*/
IDESC_TYPE id;				/* our node information		*/
int dlevel=0;				/* debug level			*/

u_char *aarp_resolve_svc();		/* the aarpd resolver		*/
u_char *rtmp_getbaddr_svc();		/* get the bridge address	*/
u_char *rtmp_setbaddr_svc();		/* set the bridge address	*/

main(argc, argv)
int argc;
char *argv[];
{
	int c;
	fd_set rdfds;
	char *cp, *ep;
	SVCXPRT *transp;
	extern int optind;
	extern char *optarg;
	void init_enet(), svc_run(), set_svc_listen(), run();

	while ((c = getopt(argc, argv, "D:d:l:")) != EOF) {
	  switch (c) {
	  case 'D':
	    dlevel = atoi(optarg);
	    if (dlevel > L_LVLMAX)
	      dlevel = L_LVLMAX;
	    break;
	  case 'd':
	    dbugarg(optarg);
	    dlevel = 1;
	    break;
	  case 'l':
	    logitfileis(optarg, "w");
	    break;
	  }
	}
	set_debug_level(dlevel);
	
	openetalkdb(NULL);		/* open/create etalk.local */

	nis_net = this_net = 0;		/* assume that we don't know */

	baddr.node = bridge_node;	/* set bridge addr hint */
	baddr.net = bridge_net;
	baddr.skt = 0;

	if (argc == (optind+2)) {
	  /* arg list supplied interface & zone names */
	  strncpy(interface, argv[optind++], sizeof(interface));
	  strncpy(this_zone, argv[optind++], sizeof(this_zone));
	}

	if (optind != argc) {
	  fprintf(stderr,
	    "usage: aarpd [-D level] [-d opt] [-l log] [interface zone]\n");
	  exit(1);
	}

	if (*interface == '\0') {
	  fprintf(stderr, "No ethernet interface specified\n");
	  exit(1);
	}
	ep = NULL;
	strncpy(intrfc, interface, sizeof(intrfc));
	for (cp = intrfc; *cp != '\0'; cp++) {
	  if (*cp >= '0' && *cp <= '9')
	    ep = cp;
	}
	if (ep == NULL) { /* interface, but no number */
	  fprintf(stderr, "Specified interface invalid: %s\n", interface);
	  exit(1);
	}
	id.id_intfno = *ep - '0';
	id.id_intf = intrfc;
	*ep = '\0';

	if (*this_zone == '\0') {
	  fprintf(stderr, "No zone name specified\n");
	  exit(1);
	}
	/* convert zone to pascal string */
	strncpy(pzone+1, this_zone, sizeof(pzone)-1);
	pzone[0] = strlen(this_zone);
	id.id_zone = pzone;

	init_enet();		/* init & get a node number */
	etalkdbupdate(NULL);	/* rewrite gleaned information	*/

	bzero(null_ether, sizeof(null_ether));

	/* set up aarpd RPC services */

	(void)pmap_unset(AARPDPROG, AARPDVERS);

	transp = svcudp_create(RPC_ANYSOCK);
	if (transp == NULL) {
		(void)fprintf(stderr, "aarpd: cannot create udp service.\n");
		exit(1);
	}
	if (!svc_register(transp,AARPDPROG,AARPDVERS,aarpdprog,IPPROTO_UDP)) {
		(void)fprintf(stderr,
			"unable to register (AARPDPROG, AARPDVERS, udp).\n");
		exit(1);
	}

	transp = svctcp_create(RPC_ANYSOCK, 0, 0);
	if (transp == NULL) {
		(void)fprintf(stderr, "aarpd: cannot create tcp service.\n");
		exit(1);
	}
	if (!svc_register(transp,AARPDPROG,AARPDVERS,aarpdprog,IPPROTO_TCP)) {
		(void)fprintf(stderr,
			"unable to register (AARPDPROG, AARPDVERS, tcp).\n");
		exit(1);
	}

	set_svc_listen(); /* add RPC descriptors to scheduler */

	if (!dbug.db_flgs && (dlevel == 0))
		disassociate();

	run(); /* do all the CAP and RPC work */

	(void)fprintf(stderr, "aarpd: run() returned!\n");
	exit(1);
}

/*
 * initialise network and get node number
 *
 */

void
init_enet()
{
  id.id_ld = &ethertalk_lap_description;
  id.id_local = NULL;
  id.id_isabridge = 0;
  id.id_network = 0;
  id.id_next = NULL;
  id.id_ifuse = 0;

  if (dbug.db_flgs || (dlevel != 0))
    printf("Interface %s%d, zone %s, ", id.id_intf, id.id_intfno, id.id_zone+1);

  init_fdlistening();  		/* low level scheduler	*/
  etalk_init(&id, FALSE);	/* set up EtherTalk	*/

  this_node = nis_node = etalk_mynode(&id);
  if (dbug.db_flgs || (dlevel != 0))
    printf("acquired node number %d\n", this_node);
}

disassociate()
{
  int i;
  /* disassociate */
  if (fork())
    _exit(0);			/* kill parent */
  for (i=0; i < 3; i++) close(i); /* kill */
  (void)open("/",0);
#ifdef NODUP2
  (void)dup(0);			/* slot 1 */
  (void)dup(0);			/* slot 2 */
#else  NODUP2
  (void)dup2(0,1);
  (void)dup2(0,2);
#endif NODUP2
#ifdef TIOCNOTTY
  if ((i = open("/dev/tty",2)) > 0) {
    (void)ioctl(i, TIOCNOTTY, (caddr_t)0);
    (void)close(i);
  }
#endif TIOCNTTY
}

/*
 * these routines do the work for the RPC calls
 *
 */

u_char *
aarp_resolve_svc(node, rqstp)
  int *node;
  struct svc_req *rqstp;
{
  u_char *ea;

  logit(2, "request for %d", *node);
  ea = etalk_resolve(&id, *node);
  return((ea) ? ea : null_ether);
}

/*
 * return the current bridge address
 *
 */

u_char *
rtmp_getbaddr_svc(i, rqstp)
  int *i;
  struct svc_req *rqstp;
{
  logit(2, "request for bridge address");
  return((u_char *) &baddr);
}

/*
 * set the bridge address
 * (normally called from atis, which is listening to RTMP packets)
 *
 */

u_char *
rtmp_setbaddr_svc(bad, rqstp)
  AddrBlock *bad;
  struct svc_req *rqstp;
{
  logit(2, "setting bridge address");
  bcopy(bad, &baddr, sizeof(baddr));
  bridge_node = baddr.node;
  bridge_net = this_net = nis_net = baddr.net;
  etalkdbupdate(NULL);	/* write the info. back out */
  return((u_char *) &baddr);
}

/*
 * add the RPC descriptors to the low level scheduler
 *
 */

void
set_svc_listen()
{
	int i, svc_listener();

	for(i = 0 ; i < NFDBITS ; i++)
		if (FD_ISSET(i, &svc_fdset))
			fdlistener(i, svc_listener, NULL, 0);
}

/*
 * service a waiting SVC call
 *
 */

private int
svc_listener(fd, addr, n)
int fd;
caddr_t addr;
int n;
{
	fd_set nfds;

	FD_ZERO(&nfds);
	FD_SET(fd, &nfds);
	svc_getreqset(&nfds);
}

/*
 * all the really hard work happens here
 *
 */

void
run()
{
  	for (;;)
	  abSleep(sectotick(30), TRUE);
}
