/*
 * $Author: djh $ $Date: 91/03/13 20:28:29 $
 * $Header: afposncs.c,v 2.2 91/03/13 20:28:29 djh Exp $
 * $Revision: 2.2 $
*/

/*
 *  afposncs.c - Appletalk Filing Protocol OS Normalizing for Character Sets
 *
 *
 *  afposncs contains the logic to normalize between an OS character
 *  set and a Macintosh character set.  It decides when a translation
 *  is necessary (based on file extentions and file types) and does
 *  the actual translation.  It is particularly useful for normalizing
 *  between unix files on machines that support an international
 *  character set and the macintosh character set.
 *  Note: logic to map between EOL chars was moved here as well.
 *
 *
 *  The intent is to provide some "transparency".  Translation is done
 *  based on a the ncs_transtable which defines which file
 *  type/creators get translated.  In addition, it possible to specify
 *  that files with particular suffixes (and file type/creators) get
 *  translated.
 *
 *  Caveat: much more massive changes must be made to allow for mapping
 *  from a single character to multiple (or vice versa).
 *
 *  Jan 1988 Dan Sahlin    Transliteration tables for .ascii, .swe,
 *                               .fin and .latin1
 *  Feb 1988 Charlie C. Kim  Rewrite all but tables.
 *
 */

/* PATCH: PC.aufs/afposncs.c.diffs, djh@munnari.OZ.AU, 15/11/90 */

#include <stdio.h>
#include <sys/types.h>
#include <netat/appletalk.h>
#include <netat/afp.h>
#include "afposncs.h"
#include "afps.h"

/* to_ tables are to unix from mac */
/* from_ tables are from unix to mac */

/* ASCII: only a CR-LF mapping is performed */
/* These are actually dummy table since it faster to compare than */
/* index and store for the simple translation involved */
private byte to_ascii[1];	/* mac to unix: dummy table */
private byte from_ascii[1];	/* unix to mac: dummy table */
#ifdef SHORT_NAMES
private byte pc_to_ascii[1];
private byte pc_from_ascii[1];
#endif SHORT_NAMES
#ifdef notdef
private byte to_ascii[] = {
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  0x08, 0x09, 0x0a, 0x0b, 0x0c, '\n', 0x0e, 0x0f, /*!*/
  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
  0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
  0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
  0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
  0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
  0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
  0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
  0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
  0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
  0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
  0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
  0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
};

private byte from_ascii[] = {
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  0x08, 0x09, '\r', 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, /*!*/
  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
  0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
  0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
  0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
  0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
  0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
  0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
  0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
  0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
  0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
  0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
  0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff 
};
#endif

#ifdef FULL_NCS_SUPPORT
/*
 * SWEdish standard D47, national characters {}|[]\ are mapped,
 * characters with bit 8 set are untouched
*/
private byte to_swe[] = {
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  0x08, 0x09, 0x0a, 0x0b, 0x0c, '\n', 0x0e, 0x0f, /*!*/
  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
  0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
  0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
  0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
  0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
  0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
  0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
  0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
  0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
  '[' , ']' , 0x82, 0x83, 0x84, '\\', 0x86, 0x87, /*!*/
  0x88, 0x89, '{' , 0x8b, '}' , 0x8d, 0x8e, 0x8f, /*!*/
  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
  0x98, 0x99, '|' , 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, /*!*/
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
};

private byte from_swe[] = {
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  0x08, 0x09, '\r', 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, /*!*/
  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
  0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
  0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
  0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
  0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
  0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
  0x58, 0x59, 0x5a, 0x80, 0x85, 0x81, 0x5e, 0x5f, /*!*/
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
  0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
  0x78, 0x79, 0x7a, 0x8a, 0x9a, 0x8c, 0x7e, 0x7f, /*!*/
  0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
  0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
  0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
};

/*
 * FINnish/Swedish standard D47, national characters {}|[]\~`@^ are mapped,
 *   characters with bit 8 set are untouched
*/
byte to_fin[] = {
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  0x08, 0x09, 0x0a, 0x0b, 0x0c, '\n', 0x0e, 0x0f, /*!*/
  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
  0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
  0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
  0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
  0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
  0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
  0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
  0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
  0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
  '[' , ']' , 0x82, 0x40, 0x84, '\\', 0x5e, 0x87, /*!*/
  0x88, 0x89, '{' , 0x8b, '}' , 0x8d, 0x60, 0x8f, /*!*/
  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
  0x98, 0x99, '|' , 0x9b, 0x9c, 0x9d, 0x9e, 0x7e, /*!*/
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
};

private byte from_fin[] = {
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  0x08, 0x09, '\r', 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, /*!*/
  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
  0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
  0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
  0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
  0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
  0x83, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
  0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
  0x58, 0x59, 0x5a, 0x80, 0x85, 0x81, 0x86, 0x5f, /*!*/
  0x8e, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
  0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
  0x78, 0x79, 0x7a, 0x8a, 0x9a, 0x8c, 0x9f, 0x7f, /*!*/
  0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
  0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
  0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
};

/*
 * ISO 8859 Latin1, 8-bit code,
 * an attempt is made to map all characters into something similar
 * but some characters without a similar character in latin1
 * are mapped into SUB (0x1a).
 * Since it is not possible to map one character into two, no correct
 * mapping CR <-> CRLF can be done here.
 * This table has not been tested agains other software using latin1,
 * so there might be some errors in the table.
*/
private byte to_latin1[] = {
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  0x08, 0x09, 0x0a, 0x0b, 0x0c, '\n', 0x0e, 0x0f, /*!*//*? really CRLF */
  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
  0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
  0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
  0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
  0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
  0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
  0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
  0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
  0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
  0xc4, 0xc5, 0xc7, 0xc9, 0xd1, 0xd6, 0xdc, 0xe1, /* ! */
  0xe0, 0xe2, 0xe4, 0xe3, 0xe5, 0xe7, 0xe9, 0xe8, /* ! */
  0xea, 0xeb, 0xed, 0xec, 0xee, 0xef, 0xf1, 0xf3, /* ! */
  0xf2, 0xf4, 0xf6, 0xf5, 0xfa, 0xf9, 0xfb, 0xfc, /* ! */
  0x1a, 0xb0, 0xa2, 0xa3, 0xa7, 0xb7, 0xb6, 0xdf, /* ! */
  0xae, 0xa9, 0x1a, 0xb4, 0xa8, 0x1a, 0xc6, 0xd8, /* ! */
  0x1a, 0xb1, 0x1a, 0x1a, 0xa5, 0xb5, 0x1a, 0x1a, /* ! */
  0x1a, 0x1a, 0x1a, 0xaa, 0xba, 0x1a, 0xe6, 0xf8, /* ! */
  0xbf, 0xa1, 0xac, 0x1a, 0x1a, 0x1a, 0x1a, 0xab, /* ! */
  0xbb, 0x1a, 0xa0, 0xc0, 0xc3, 0xd5, 0x1a, 0x1a, /* ! */
  0x2d, 0x2d, 0x22, 0x22, 0x60, 0x27, 0xf7, 0x1a, /* ! */
  0xff, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, /* ! */
  0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, /* ! */
  0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, /* ! */
  0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, /* ! */
  0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1a /* ! */
};
/*
 * if no similar character is found 0xff is returned
 * 0x1a is also defined as 0xff
*/
private byte from_latin1[] = {
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  0x08, 0x09, '\r', 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, /*!*/
  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
  0x18, 0x19, 0xff, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
  0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
  0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
  0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
  0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
  0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
  0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
  0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
  0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
  0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
  0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
  0xca, 0xc1, 0xa2, 0xa3, 0xff, 0xb4, 0x7c, 0xa4,
  0xac, 0xa9, 0xbb, 0xc7, 0xc2, 0xff, 0xa8, 0xff,
  0xa1, 0xb1, 0xff, 0xff, 0x27, 0xb5, 0xa6, 0xa5,
  0x2c, 0xff, 0xbc, 0xc8, 0xff, 0xff, 0xff, 0xc0,
  0xcb, 'A' , 'A' , 0xcc, 0x80, 0x81, 0xae, 0x82,
  'E' , 0x83, 'E' , 'E' , 'I' , 'I' , 'I' , 'I' ,
  'D' , 0x84, 'O' , 'O' , 'O' , 0xcd, 0x85, 0xff,
  0xaf, 'U' , 'U' , 'U' , 0x86, 'Y' , 0xff, 0xa7,
  0x88, 0x87, 0x89, 0x8b, 0x8a, 0x8c, 0xbe, 0x8d,
  0x8f, 0x8e, 0x90, 0x91, 0x93, 0x92, 0x94, 0x95,
  0xff, 0x96, 0x98, 0x97, 0x99, 0x9b, 0x9a, 0xd6,
  0xbf, 0x9d, 0x9c, 0x9e, 0x9f, 'y' , 0xff, 0xd8
};

#endif /* FULL_NCS_SUPPORT */

#define NCS_entry(name, suf, conj, creator, type, totable, fromtable) \
   { name, suf, sizeof(suf)-1, conj, creator, sizeof(creator)-1, \
     type, sizeof(type)-1, totable, fromtable }

/*
 * NCS_entry:
 *  suffix to map on
 *  conj: AND for suffix and creator/type, OR for suffix or creator/type
 *  creator/type: macintosh file creator/type
 *  totable - translate to unix from macintosh
 *  fromtable - translate from unix to macintosh
 *
 * NULL entries for suffix or file type/creator always match
 * Table is searched sequentially so put the more specific items near the
 * start.
 *
*/
NCS_TransTable ncs_transtables[] = {
#ifdef FULL_NCS_SUPPORT
  /* Swedish standard D47 translation */
  NCS_entry("Swedish D47",".swe",NCS_AND,NULL,NULL, to_swe,from_swe),
  /* Swedish-Finnish standard E47 translation */
  NCS_entry("Swedish-Finnish E47",".fin",NCS_AND,NULL, NULL, to_fin,from_fin),
  /* ISO 8859-1 latin1 */
  NCS_entry("ISO 8859-1 Latin 1", ".latin1", NCS_AND, NULL, NULL,
	    to_latin1,from_latin1),
#endif
  /* ASCII */
  NCS_entry("ascii", NULL, NCS_AND, (byte *)"unix", (byte *)"TEXT", 
	    to_ascii, from_ascii),
#ifdef SHORT_NAMES
  NCS_entry("pc-ascii", NULL, NCS_AND, (byte *)"unix", (byte *)"TEXT", 
	    pc_to_ascii, pc_from_ascii),
#endif SHORT_NAMES
  {NULL}
};

#define NCS_TRANSTABLELEN ((sizeof(ncs_transtables)/sizeof(NCS_TransTable))-1)


/*
 * find the translation table and return index.  return -1 o.w.
*/
int
#ifdef SHORT_NAMES
ncs_istrans(finderinfo, file, ptype)
byte ptype;
#else SHORT_NAMES
ncs_istrans(finderinfo, file)
#endif SHORT_NAMES
fileFinderInfo *finderinfo;
char *file;
{
  register NCS_TransTable *tt;
  int i, idx;
  int suffixmatch, finderinfomatch;

  for (idx = 0, tt = ncs_transtables; idx < NCS_TRANSTABLELEN; tt++, idx++) {
    suffixmatch = (tt->ncs_suffix == NULL) || 
      (file && (i = strlen(file)) >= tt->ncs_suffix_len &&
       strcmp(tt->ncs_suffix, file+i-tt->ncs_suffix_len)==0);
    if (suffixmatch) {
      if (tt->ncs_conj == NCS_OR)
	return(idx);
    } else {
      if (tt->ncs_conj == NCS_AND)
	continue;
    }
    /* see if file type and file creator match (except if null, then */
    /* ignored).  Partial matches are acceptable -- depends on what */
    /* is specified */
    finderinfomatch = FALSE;
    if (tt->ncs_file_creator == NULL ||
	bcmp(tt->ncs_file_creator, finderinfo->fdCreator, tt->ncs_fc_len) == 0)
      if (tt->ncs_file_type == NULL || 
	  bcmp(tt->ncs_file_type, finderinfo->fdType, tt->ncs_ft_len) == 0)
	finderinfomatch = TRUE;
#ifdef SHORT_NAMES
printf("suffixmatch %d, finderinfomatch %d, ptype 0x%x\n",suffixmatch,finderinfomatch,ptype);
#endif SHORT_NAMES
    switch (tt->ncs_conj) {
    case NCS_OR:
#ifdef SHORT_NAMES
      if ((suffixmatch || finderinfomatch) && (ptype == 0x01)) 
         return(idx+1);     
/* so,,,, this is bad, and will only work when there 
   are 2 entries in the trans table. */
#endif SHORT_NAMES
      if (suffixmatch || finderinfomatch)
	return(idx);
      break;
    case NCS_AND:
#ifdef SHORT_NAMES
      if (suffixmatch && finderinfomatch && (ptype == 0x01)) {
printf("returning idx+1 = %d\n",idx+1);
         return(idx+1);     
      }
#endif SHORT_NAMES
      if (suffixmatch && finderinfomatch)
	return(idx);
      break;
    }
  }
  return(-1);			/* no match */
}

/*
 * return the name of a translation table given its index
 *
*/
char *
ncs_tt_name(ttidx)
int ttidx;
{
  if (ttidx < 0 || ttidx >= NCS_TRANSTABLELEN)
    return("no translation");
  return(ncs_transtables[ttidx].ncs_trans_name);
}

ncs_translate(which, ttidx, buf, cnt)
int which;
int ttidx;
byte *buf;
register int cnt;
{
  register byte *bp;
  register byte *tt;
  register byte tc, fc;

  if (ttidx < 0 || ttidx >= NCS_TRANSTABLELEN)
    return;
  
  switch (which) {
#ifdef SHORT_NAMES
/* the NCS_TRANS_UNIXTOMAC and NCS_TRANS_MACTOUNIX are not good
   names anymore because they will also be used for the PC */
#endif SHORT_NAMES
  case NCS_TRANS_UNIXTOMAC:
    tt = ncs_transtables[ttidx].ncs_fromtable;
    break;
  case NCS_TRANS_MACTOUNIX:
    tt = ncs_transtables[ttidx].ncs_totable;
    break;
  default:
    return;
  }
  bp = buf;
  if (tt == to_ascii || tt == from_ascii) {
    /* by pass for to_ascii, and from_ascii to speed things up */
    if (tt == to_ascii) fc = ENEWLINE, tc = INEWLINE;
    else fc = INEWLINE, tc = ENEWLINE;
    while (cnt--) {
      if (*bp == fc)
	*bp = tc;
      bp++;
    }
#ifdef SHORT_NAMES
  } else if ((tt == pc_to_ascii) || (tt == pc_from_ascii)) {
     while (cnt--)
          bp++;
#endif SHORT_NAMES
  } else {
    /* standard */
    while (cnt--) {
      *bp = tt[*bp];
      bp++;
    }
  }
}

#define strval(x) ((x) == NULL ? "<any>" : (x))

ncs_table_dump()
{
  register NCS_TransTable *tt;
  int idx;

  for (idx = 0, tt = ncs_transtables; idx < NCS_TRANSTABLELEN; tt++, idx++) {
    logit(0,"  %s: xlate on: suffix %s %s file type %s, creator %s",
	tt->ncs_trans_name,
	strval(tt->ncs_suffix),
	(tt->ncs_conj == NCS_AND ? "and" : "or"),
	strval((char *)tt->ncs_file_type),
	strval((char *)tt->ncs_file_creator));
  }
}
