(* Unification of structures in Modules;
   See the Definition, Sections 5.9 -- 5.12 
   and the Commentary, Sections 11.4 and 11.5 *)

(*
$File: Common/ModuleUnify.sml $
$Date: 1993/02/24 07:58:27 $
$Revision: 1.6 $
$Locker: birkedal $
*)

(*$ModuleUnify:
      STRID TYCON FINMAP MODULE_STATOBJECT MODULE_ENVIRONMENTS
      ERROR_INFO LIST_HACKS CRASH MODULE_UNIFY
*)

functor ModuleUnify(structure StrId  : STRID

		    structure TyCon  : TYCON
		      sharing type TyCon.strid = StrId.strid

		    structure FinMap : FINMAP

		    structure O : MODULE_STATOBJECT
		      sharing type O.longstrid = StrId.longstrid

		    structure Env : MODULE_ENVIRONMENTS
		      sharing type Env.Realisation = O.Realisation
			  and type Env.longstrid   = StrId.longstrid
			  and type Env.TypeFcn     = O.TypeFcn
			  and type Env.ConEnv      = O.ConEnv
			  and type Env.StrName     = O.StrName
			  and type Env.TyStr       = O.TyStr
			  and type Env.strid       = StrId.strid
			  and type Env.tycon       = TyCon.tycon

		    structure ErrorInfo: ERROR_INFO
		      sharing type ErrorInfo.strid = StrId.strid
			  and type ErrorInfo.longstrid = StrId.longstrid
			  and type ErrorInfo.tycon = TyCon.tycon
			  and type ErrorInfo.longtycon = TyCon.longtycon
			  and type ErrorInfo.TyName = O.TyName
			  and type ErrorInfo.TypeFcn = O.TypeFcn

		    structure ListHacks: LIST_HACKS
		    structure Crash  : CRASH
		   ): MODULE_UNIFY =
  struct
    type StrName = O.StrName

    type NameSet = O.NameSet

    type tycon     = TyCon.tycon
     and longtycon = TyCon.longtycon

    type TyStr = O.TyStr

    type TyName  = O.TyName
     and TypeFcn = O.TypeFcn

    type con    = Env.con
     and ConEnv = Env.ConEnv

    type Assembly    = Env.Assembly
    type Realisation = O.Realisation

    type strid     = StrId.strid
     and longstrid = StrId.longstrid

    type 'a set = 'a list

    type ('name,'idclass)deco = 'name * strid list * 'idclass
    (* Type of decorated names (or type structures);
       for example, (m,[A,B,C], D) represents the information
       that, starting from the point of the sharing constraint,
       the structure A.B.C.D has name m.  Similarly for type
       structures.  Decoration is required for error messages
       only. *)

    type Mset = (StrName, strid)deco set;
    (* Sets of structure names together with information about
       how to reach them. *)

    type strsetenv = (strid, Mset) FinMap.map
    (* A mapping from structure identifiers to sets of structure
       names, namely, for each strid, the set of structure names 
       that can be reached via strid, starting from the current 
       equivalence class *)

    type Tset = (TyStr, tycon)deco set
    type tysetenv = (tycon, Tset) FinMap.map;

    type strrel = Mset list
    (* Relations on structure names are represented by lists
       of sets of structure names. If strrel is an equivalence
       relation, then every set in the list is an equivalence
       class, i.e. all the sets in the list will be non-empty
       and pairwise disjoint. *)

    type tyrel  = Tset list
    (* Similarly, but for types *)

    (********
    Possible results of unification.
    ********)

    open ErrorInfo
    type ErrorInfo = info

    datatype UnifyResult = OK of Realisation
			 | ERROR of ErrorInfo

    (********
    Local exception raised when unification fails.
    ********)

    exception UNIFY of UnifyResult

   (* MEMO: a lot of lists should be sets. I'm not going to do this right
      now, since I don't know this code well, and besides it's Friday
      afternoon and I have to go home and watch Thunderbirds. *)

    local
      (********
      Insert (info, path, id) into E(id).
      ********)

      fun insert ((id, info), (E, path)) =
	let
	  val newlist =
	    case (FinMap.lookup E id) of
	      None =>
		[(info, path, id)]
	    | Some(list) =>
		(info, path, id) :: list
	in 
	  (FinMap.plus(E, FinMap.singleton(id, newlist)), path)
	end

      (********
      The result of classes (M, A) is the set of classes of info that must be
      identified given that the names in M are identified.
      *********
      For each decorated name m in M we look up the arcs going out from a structure
      with name m in the assembly A. We add the information from each arc to the
      environment, E, which is being built. After this, we extract from the range of
      the environment the lists of equivalence classes.
      ********)

      fun classes Alookup Afold (M : Mset, A : Assembly) =
	let 
	  fun f (m, path, strid) E =
	    let
	      val arcs_out = Alookup(A, m)
	      val (E', _)  = Afold insert (E, path @ [strid]) arcs_out
	    in
	      E'
	    end

	  val E = List.foldL f FinMap.empty M
	in 
	  FinMap.fold (op ::) [] E
	end

      (********
      Produces the lists of names that must be identified given the identification
      of names expressed in R.
      *********)

      fun F Alookup Afold (A : Assembly, R : strrel) =
	let
	  fun f Mset rel = classes Alookup Afold (Mset, A) @ rel
	in
	  List.foldL f [] R
	end
    in
      (* NICK: deal with lookup failures caused by previous elaboration
         errors. *)

      fun safe (Alookup, default) (A, strName) =
	if strName = O.bogus_StrName then
	  default
	else
	  Alookup(A, strName)

      val safe_Alookup_Str = safe(Env.Alookup_Str, Env.bogus_OffspringStr)
      val safe_Alookup_Ty = safe(Env.Alookup_Ty, Env.bogus_OffspringTy)

      (********
      Produces the lists of structure names that must be identified given the
      identification of structure names expressed in the argument.
      *********)

      val F_Str = F safe_Alookup_Str Env.Aoffspring_Str_Fold

      (********
      Produces the lists of type names that must be identified given the
      identification of structure names expressed in the argument.
      *********)

      val F_Ty  = F safe_Alookup_Ty Env.Aoffspring_Ty_Fold
    end 

    fun transClos (eq: 'a * 'a -> bool)
		  (insert_when_eq: bool)
		  (newsets: 'a set list,
		   oldsets: 'a set list): bool * 'a set list =
  (* The lists in oldsets are pairwise disjoint, but nothing is known about the
  lists in newsets. The resulting sets are pairwise disjoint,
  and moreover, the resulting boolean is true if and only if 
  every member (i.e. list) of newsets, is contained in some member (i.e., list)
  of oldsets *)
    let
      val any: bool = true (* could be false, for that matter *)

      fun isin(x,[]) = false
	| isin(x,x'::rest) = eq(x,x') orelse isin(x,rest)

     (* In the following function, L is a list of pairwise disjoint
	sets. S' is the set currently under construction;
	it is also disjoint from the sets in L.  x 
	will be added to S' if x is not in any set in L.  *)

      fun add(x:'a,
		(found_fixpt: bool, found_class: bool, L as [], S')):
		 bool * bool * 'a list list * 'a list =
	     if isin(x,S') 
	     then if insert_when_eq
		  then
		    (found_fixpt andalso found_class, found_class,[],x::S')
		  else
		    (found_fixpt andalso found_class, found_class,[],S')
	     else
		   (false, any, [], x::S')

	| add(x,(found_fixpt,found_class,(S::L), S')) =
	     if isin(x,S) 
	     then
	       if insert_when_eq 
	       then
		 (found_fixpt andalso not found_class,true, L, x::(S @ S'))
	       else
		 (found_fixpt andalso not found_class,true, L, S @ S')
	     else
		let val (found_fixpt1, found_class1, L1, S1)=
		      add(x,(found_fixpt,found_class,L,S'))
		 in 
		   (found_fixpt andalso found_fixpt1,
		    found_class orelse found_class1,
		    S::L1,
		    S1
		   )
		end

      fun add_S (S,(found_fixpt, oldsets)) : bool * 'a set list =
	let
	  val (found_fixpt1, _, L1, S1) = 
	    List.foldL (General.curry add)
		       (found_fixpt, false, oldsets, nil) S
	 in 
	   case S1 of 
	     [] => (found_fixpt andalso found_fixpt1, L1)
	   | _  => (found_fixpt andalso found_fixpt1, S1::L1)
	end

     in
       List.foldL (General.curry add_S) (true, oldsets) newsets
    end

    val transClos_Str = 
       transClos(fn((m1,_,_), (m2,_,_))=> m1=m2) false
    val transClos_Ty  =
       transClos
	  (fn((tystr1,_,_),(tystr2,_,_))=> O.TyStr_shares(tystr1,tystr2)) 
	  true

    fun fixedPoint (M : Mset, A : Assembly) : strrel = 
      let
	fun iterate (true , L) = L
	  | iterate (false, L) = iterate(transClos_Str(F_Str(A, L), L))
      in 
	iterate(false, [M])
      end

    (* The following function checks that each equivalence class
       contains at most one rigid structure; if not, a unification
       error has been found *)

    type collapseStrResult = (StrName set * StrName)list
		       (*  ^class        ^representative *)
    fun collapseStr(
		 ML: strrel, (* equivalence classes *)
		 NofB: NameSet    (* rigid names *)
		): collapseStrResult=
      let 
	fun check([]: Mset, rep) = rep
	  | check((m,path,strid)::rest, None) =
	      if O.isIn_StrName(m,NofB) then
	       check(rest, Some(m,path,strid))
	      else
	       check(rest,None)
	  | check((deco_m as (m,path,strid))::rest, 
		   Some (deco_m' as (m',path',strid'))) =
	      if O.isIn_StrName(m,NofB) then
	       if m=m' then check(rest,Some deco_m')
	       else
		 raise UNIFY(ERROR(U_RIGIDSTRCLASH(
				     StrId.implode_longstrid(path, strid),
				     StrId.implode_longstrid(path', strid')
				   )
			          )
			    )
	      else check(rest, Some deco_m')
	fun getname(m,_,_) = m
	fun Check(M: Mset) = 
	  case check(M,None) of
	    Some(m,_,_) => (map getname M, m)
	  | None => case map getname M of 
		      []=> Crash.impossible "empty equivalence class in fun Check"
		    | names as m::_ => (names, m)
      in
	 map Check ML
     end


    type collapseTyResult = (TyName set * TypeFcn)list
			 (*  ^class       ^representative *)
    fun collapseTy(
		 TL: tyrel, (* equivalence classes *)
		 NofB: NameSet    (* rigid names *)
		): collapseTyResult=
    let 
       fun error(constructor, rep, rep')=
       let fun errordata (rep as (_, path, tycon)) =
		 TyCon.implode_LongTyCon(path,tycon)
	in 
	   raise UNIFY(ERROR(constructor(errordata rep, errordata rep')))
       end

       fun CheckArity(T: Tset): unit = 
       (* CheckArity T checks whether all type functions in T have the
	  same arity. *)
       let 
	  fun check([], None) = Crash.impossible "Empty equivalence\
                                            \ class (CheckArity)"
	    | check([], Some _) = ()
	    | check((rep as (tystr, path, tycon))::T, None) =
		     check(T, Some rep)
	    | check((rep as (tystr, path, tycon))::T, 
		    Some (rep' as (tystr', path', tycon'))) =
	       if O.arity_TypeFcn(O.Theta_of(tystr)) =
		  O.arity_TypeFcn(O.Theta_of(tystr'))
	       then 
		  check(T,Some rep')
	       else
		 error(U_CONFLICTINGARITY,rep,rep')
	in check(T, None)
       end 

       fun CheckTyStrWF(T: Tset): unit = 
       (* CheckTyStrWF T checks makes sure that T does not contain
	  a type structure whose type function is not a type name
	  and contains a type structure with a non-empty constructor
	  environment.  It also checks that there are no two 
	  non-empty constructor environments with different domains. *)
       let 
	  fun CheckCEdom(T: Tset)(*: (con list*(TyStr,tycon) deco) Option *)= 
	  (* Checks that there are not two non-empty constructor environments
	     with different domains. *)
	  let
	     fun checkdom([]:Tset, rep) = rep
	       | checkdom((rep as (tystr, _, _)):: T', None) =
		  (case Env.domCE(O.CE_of_TyStr(tystr)) of
		     [] => checkdom(T',None)
		   | dom=> checkdom(T',Some(dom,rep)))
	       | checkdom((rep as (tystr, _, _)):: T', 
		     Some(d', rep' as (tystr',_,_))) =
		  (case Env.domCE(O.CE_of_TyStr(tystr)) of
		     [] => checkdom(T',Some(d',rep'))
		   | dom=> if ListHacks.eqSet(dom, d') then checkdom(T',Some(d',rep'))
			   else error(U_CONFLICTING_DOMCE, rep, rep'))
	  in
	     checkdom(T, None)
	 end

	in (* CheckTyStrWF *)
	  case CheckCEdom(T) of
	    None => ()  (* all constructor environments are empty *)
	  | Some(_, rep as (tystr, _,_)) =>
	      (* rep is a type structure with a non-empty constructor
		 environment; now check that all type functions are
		 type names: *)
	      let
		 fun allTyNames([]: Tset): unit = ()
		   | allTyNames((rep' as (tystr', _, _)):: T') =
		      if O.isTyName tystr' then allTyNames T'
		      else error(TYPESTRILLFORMEDNESS, rep, rep')
	       in 
		 allTyNames(T)
	      end
       end (* CheckTyStrWF *)

       fun CheckEqAttr(T: Tset): TyName list * TypeFcn =
       (* Makes sure that it is not the case that T contains a rigid
	  type structure that does not admit equality and a type structure
	  that does admit equality. It also looks for the theta0 mentioned
	  in the proof of the theorem of principal admissifiers in the
	  Commentary *)
       let 
	  fun LookForTheta0(T: Tset) : 
	    (TyStr,tycon)deco Option * Tset = 
	  (* Returns theta0, if it exists (hence the option), and
	     a list of flexible type names. It is checked that either
	     T is a list of flexible type names or T contains a single
	     grounded type function and all other members of T are
	     flexible type names *)
	  let
	     fun check([]:Tset, rep_opt, acc) = (rep_opt, acc)
	       | check((rep as (tystr,_,_))::T', rep_opt, acc) =
		   if O.grounded_TypeFcn(O.Theta_of tystr,NofB) then
		      case rep_opt of
			None => check(T',Some rep, acc)
		      | Some (rep' as (tystr', _,_)) =>
			   if (O.Theta_of tystr) = (O.Theta_of tystr')
			   then check(T', rep_opt, acc)
			   else error(U_RIGIDTYCLASH,rep,rep')
		   else (case O.unTyName_TypeFcn(O.Theta_of tystr) of
			  Some _ => check(T', rep_opt, rep :: acc)
			| None  => Crash.impossible "I found a type function\
                               \ which is not grounded and which is not\
                               \ a type name (LookForTheta0)")
	   in
	      check(T,None,[])
	  end

	  fun pick_eq_name T : TyName list * TypeFcn =
	  (* pick_eq_name(T) picks an equality type name from T, if
	     one is present  and otherwise it picks any type name from T *)
	  let fun loop([],tlist, Some t) = (tlist, O.TyName_in_TypeFcn t)
		| loop([],[], None) = Crash.impossible "empty equivalence\
                                       \ class (pick_eq_name)"
		| loop([],t::rest,None) = (rest, O.TyName_in_TypeFcn t)
		| loop((rep as (tystr,_,_))::rest, acc, t_opt) =
		   (case O.unTyName_TypeFcn (O.Theta_of tystr) of
		      Some t' => 
			  if O.admits_equality(O.Theta_of tystr)
			  then 
			    case t_opt of
			      None => loop(rest, acc, Some t')
			    | Some t => loop(rest, t'::acc, t_opt)
			  else loop(rest, t'::acc, t_opt)
		    | None    => Crash.impossible "found type function which\
                             \ is not a type name (pick_eq_name)")
	   in loop(T,[],None)
	  end (* pick_eq_name *)

	  (* tynames(T), where T is a list of decorated type structures
	     with no non-trivial type functions, extracts the type names *)
	  fun tynames [] = []
	    | tynames ((hd as (tystr, _,_))::rest) =
		case O.unTyName_TypeFcn (O.Theta_of tystr) of
		  Some t => t :: tynames rest
		| None    => Crash.impossible "found type function which\
                             \ is not a type name (tynames, CheckEqAttr)"

	  (* all_noneq(T) check whether all type functions in T do not
	     admit equality *)
	  fun all_noneq [] = None
	    | all_noneq ((hd as (tystr, _,_))::T') =
		if O.admits_equality (O.Theta_of tystr) then Some hd
		else all_noneq T'
	in 
	  case LookForTheta0 T of
	    (None, T') => pick_eq_name(T')
	  | (Some (rep as (tystr0,_,_)), T')=>
	       let val theta0 = O.Theta_of tystr0
		in
		  if O.admits_equality(theta0)  
		  then (tynames T', theta0)
		  else case all_noneq(T') of
			 None => (tynames T', theta0)
		       | Some rep' => error(RIGIDTYFUNEQERROR,rep,rep')
	       end
       end (* CheckEqAttr *)

       fun CheckAll(T:Tset) = (CheckArity T; CheckTyStrWF T; CheckEqAttr T)
     in
       map CheckAll TL
    end

    (* unification *)

    fun mkreastar(NofB, A, ML: strrel, TL: tyrel): UnifyResult =
    let val strL = collapseStr (ML,NofB)
	val tyL  = collapseTy  (TL,NofB)
	val rea  = O.mkRea(strL,tyL)
    in
      case Env.cyclic(Env.onA(rea, A)) of
	Some(longstrid) => ERROR(CYCLE longstrid)
      | None => OK rea
    end
    handle UNIFY error => error

    (********
    Unify a list of type constructors.
    *********
    We replace the CE of each type structure by a CE obtained from the assembly.
    ********)

    fun unifyTy
      (NofB : NameSet, A : Assembly, L : (TyStr * longtycon) list) : UnifyResult =
      let 
	fun CEassembly (A, tystr) =
	  if tystr = O.bogus_TyStr then
	    O.bogus_TyStr
	  else
	    Env.Alookup_TypeFcn(A, O.Theta_of tystr) 

	fun f (tystr, longtycon) =
	  let
	    val (path, tycon) = TyCon.explode_LongTyCon longtycon
	  in
	    (CEassembly(A, tystr), path, tycon)
	  end

	val shared = map f L
      in
	mkreastar(NofB, A, [], [shared])
      end

    (********
    Unify a list of structures.
    ********)

    fun unifyStr
      (NofB : NameSet, A : Assembly, L : (StrName * longstrid) list) : UnifyResult =
      let 
	val M = map 
		 (fn(m,longstrid)=> 
		    let val (path,strid) = StrId.explode_longstrid longstrid
		     in (m,path,strid)
		    end) 
		 L
	val strrel = fixedPoint(M,A)
	val (_,tyrel)  = transClos_Ty(F_Ty(A,strrel), [])
     in 
	mkreastar(NofB,A,strrel,tyrel)
    end
  end;
