/* array.c      Distribution 1.1   89/3/30   Scry */

/*   The Scry system is copyright (C) 1988, 1989, Regents  of  the
University  of  California.   Anyone may reproduce ``Scry'',
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  Scry  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  Scry  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     Scry is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
Scry, or any part thereof.     */

/* array.c	Preprocessing of a dense array of data
 * This is part of the marching cubes implementation.  This routine finds the
 * interesting parts of an array of data to save time on the actual tessellation
 * and finds the number of vertices and triangles that will be used, saving
 * a great deal of memory allocation time.
 *
 * Max Rible	Lawrence Berkeley Laboratories
 * 		Advanced Development group
 *		Computer Graphics Laboratory
 *		1988
 */

#include "march.h"

/* The process_array() subroutine scans the input file and assembles
 * a list of vectors of cubes that the surface in question intersects.
 * This list is then passed to the triangle-fitting algorithm.
 * process_array() accurately predicts the number of triangles and
 * inaccurately predicts the number of vertices; however, error is
 * on the side of more vertices, so we only have to worry about
 * minimal waste.
 */

long
process_array(dx, dy, dz, grid, cubes, depths, pred_edges, num_sides)
    int dx, dy, dz;
    u_char ***grid;
    struct cube **cubes;
    long     *depths, *pred_edges, *num_sides;
{
    register int i, j, k, tmp;
    int       tx, ty, pd;
    register int tz;
    register Pix *curslice, *prevslice;
    Pix      *tmptr;
    long      num_tri = 0, edges_predicted = 0;
    register Cube ***array;
    unsigned  num_cubes = 0;
    Vectors  *cubearr, *cubearrj, *cubearrk;
#ifdef PARANOIA
    static int worry[15] = {0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1, 1, 0};
    static int fret[15] = {0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0};
    int       hole, dir, side, holes = 0;
    Cube      temp;
#ifdef DEBUG
    int       scan_triangles = 0, false = 0;
#endif
#endif

    tx = dx - 1;
    ty = dy - 1;
    tz = dz - 1;
    pd = (dz / 32) + ((dz % 32) ? 1 : 0);

    if ((prevslice = Calloc(dy * pd, Pix)) == NULL)
	perror("prevslice");
    if ((curslice = Calloc(dy * pd, Pix)) == NULL)
	perror("curslice");

    if ((array = Calloc(tx, Cube **)) == NULL)
	perror("array");
    if ((array[0] = Calloc(2 * ty, Cube *)) == NULL)
	perror("array[]");
    if ((array[0][0] = Calloc(2 * ty * tz, Cube)) == NULL)
	perror("array[][]");

    array[1] = *array + ty;
    array[1][0] = **array + ty * tz;
    for (j = 1; j < ty; j++) {
	array[0][j] = **array + tz * j;
	array[1][j] = **array + (ty + j) * tz;
    }

#define SETPIX(y,z) (curslice[((y)*pd)+((z)/32)] |= BIT((z) & 31))
#define GETCUR(y,z) (curslice[((y)*pd)+((z)/32)] & BIT((z) & 31))
#define GETPREV(y,z) (prevslice[((y)*pd)+((z)/32)] & BIT((z) & 31))

    for (j = 0; j < dy; j++)
	for (k = 0; k < dz; k++)
	    if (grid[0][j][k] > 0)
		SETPIX(j, k);
    tmptr = curslice;
    curslice = prevslice;
    prevslice = tmptr;
    Memset(curslice, 0, sizeof(Pix) * dy * pd);

    cubearr = edgecorns;	/* For most of the cube, use the standard
				 * array */
    cubearrj = edgecornsk;	/* Holding z constant */
    cubearrk = edgecornsj;	/* Holding y constant */
    for (i = 0; i < tx; i++) {
	num_cubes = 0;
	if (i > 1) {
	    array[i] = array[i & 1];
	    Memset(array[i][0], 0, sizeof(Cube) * ty * tz);
	}
	for (j = 0; j < dy; j++)
	    if (grid[i + 1][j][0] > 0)
		SETPIX(j, 0);
	for (k = 1; k < dz; k++)
	    if (grid[i + 1][0][k] > 0)
		SETPIX(0, k);
	for (j = 0; j < ty; j++) {
	    for (k = 0; k < tz; k++) {
		if (grid[i + 1][j + 1][k + 1] > 0)
		    SETPIX(j + 1, k + 1);
		tmp = 0;
		tmp |= (GETPREV(j, k) ? 01 : 0);	/* BIT(0) */
		tmp |= (GETCUR(j, k) ? 02 : 0);	/* BIT(1) */
		tmp |= (GETPREV(j + 1, k) ? 04 : 0);	/* BIT(2) */
		tmp |= (GETCUR(j + 1, k) ? 010 : 0);	/* BIT(3) */
		tmp |= (GETPREV(j, k + 1) ? 020 : 0);	/* BIT(4) */
		tmp |= (GETCUR(j, k + 1) ? 040 : 0);	/* BIT(5) */
		tmp |= (GETPREV(j + 1, k + 1) ? 0100 : 0);	/* BIT(6) */
		tmp |= (GETCUR(j + 1, k + 1) ? 0200 : 0);	/* BIT(7) */
		array[i][j][k] = tmp;
		if ((tmp != 0) && (tmp != 255)) {
		    num_cubes++;
#if defined(DEBUG) && defined(PARANOIA)
		    scan_triangles += numtri[tmp];
#endif
		    edges_predicted += cubearr[tmp];
		}
	    }
	}
	depths[i] = num_cubes;
	if (num_cubes > 0) {
	    if ((cubes[i] = Calloc(num_cubes, struct cube)) == NULL)
		perror("cubes[]");
	    num_cubes = 0;
	    for (j = 0; j < ty; j++)
		for (k = 0; k < tz; k++) {
		    if (((tmp = array[i][j][k]) != 0) && (tmp != 255)) {
			num_tri += numtri[tmp];
#ifdef PARANOIA
			if (worry[pattern[tmp]]) {
			    if (i > 0)
				if (fret[pattern[temp = array[i - 1][j][k]]] &&
				    morebits[temp]) {
				    hole = 1;
				    dir = EX;
				    side = 0;
				    goto save;
				}
			    if (j > 0)
				if (fret[pattern[temp = array[i][j - 1][k]]] &&
				    morebits[temp]) {
				    hole = 1;
				    dir = EY;
				    side = 0;
				    goto save;
				}
			    if (k > 0)
				if (fret[pattern[temp = array[i][j][k - 1]]] &&
				    morebits[temp]) {
				    hole = 1;
				    dir = EZ;
				    side = 0;
				    goto save;
				}
#ifdef DEBUG
			    false++;
#endif
			    goto nohole;
		    save:
			    cubes[i][num_cubes].hole = hole;
			    cubes[i][num_cubes].dir = dir;
			    cubes[i][num_cubes].side = side;
			    holes++;
			}
		nohole:
#endif				/* PARANOIA */
			cubes[i][num_cubes].cube = (unsigned char) tmp;
			cubes[i][num_cubes].y = (unsigned) j;
			cubes[i][num_cubes].z = (unsigned) k;
			num_cubes++;
		    }
		}
	    /* Calculate the number of edges on the boundaries */
	    for (j = 0; j < ty - 1; j++)
		edges_predicted += cubearrj[array[i][j][tz - 1]];
	    for (k = 0; k < tz - 1; k++)
		edges_predicted += cubearrk[array[i][ty - 1][k]];
	    edges_predicted += numedg[array[i][ty - 1][tz - 1]];
	}
	if (i == tx - 2) {	/* About to do the far side of the array */
	    cubearr = edgecornst;
	    cubearrj = numedg;
	    cubearrk = numedg;
	}
	tmptr = curslice;
	curslice = prevslice;
	prevslice = tmptr;
	Memset(curslice, 0, sizeof(Pix) * dy * pd);
    }

    Cfree(curslice);
    Cfree(prevslice);
    free_3d_array((char ***) array);

    *pred_edges = edges_predicted;

#ifdef DEBUG
    printf("Cube vertices calculated, %d triangles, %d vertices.\n", num_tri, edges_predicted);
#ifdef PARANOIA
    printf("Filled in %d holes, %d triangles scanned, %d false diagnoses.\n", holes, scan_triangles, false);
#endif
#endif

    *num_sides = 3 * num_tri;
#ifdef PARANOIA
    *num_sides += 4 * holes;
#endif

    return (num_tri);
}
