/*******************************************************************************
+
+  LEDA  3.0
+
+
+  basic.h
+
+
+  Copyright (c) 1992  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 6600 Saarbruecken, FRG     
+  All rights reserved.
+ 
*******************************************************************************/


#ifndef LEDA_BASIC_H
#define LEDA_BASIC_H


//------------------------------------------------------------------------------
// compiler/system dependent definitions and includes
//------------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>

#if defined(__ZTC__)
#define __MSDOS__
#define MAXINT          (int(0x7FFFFFFF))
#define MAXFLOAT	(float(3.37E+38))
#define MAXDOUBLE       (double(1.797693E+308))
#include <generic.hpp>
#include <iostream.hpp>
#else
#include <values.h>
#include <generic.h>
#if __GNUG__ == 1
#include <stream.h>
#else
#include <iostream.h>
#endif
#endif

#if defined(__TURBOC__)
#define name2 _Paste2
#define name3 _Paste3
#define name4 _Paste4
#else
#define _CLASSTYPE
#endif


#define declare1(_a,t)           name2(_a,declare1)(t)
#define declare3(_a,t1,t2,t3)    name2(_a,declare3)(t1,t2,t3)


// __NO_RANDOM__ : random number generator "random" not available

#if defined(__TURBOC__) || defined(__ZTC__)
#define __NO_RANDOM__
#endif


/*
#if defined(unix)
typedef daddr_t PointerDiff	
#elif defined(__TURBOC__)
typedef ptrdiff_t PointerDiff;
#endif
*/


//------------------------------------------------------------------------------
// Global Types
//------------------------------------------------------------------------------

typedef void* GenPtr;    // generic pointer type 

typedef int  (*CMP_PTR)(GenPtr&,GenPtr&);
typedef void (*APP_PTR)(GenPtr&);
typedef int  (*ORD_PTR)(GenPtr&);



//------------------------------------------------------------------------------
// Error Handling
//------------------------------------------------------------------------------

typedef void (*PEH)(int,char*);   // Pointer to Error Handler

extern PEH  p_error_handler;
extern PEH  set_error_handler(PEH);
extern void default_error_handler(int,char*);
inline void error_handler(int i, char* s)  { p_error_handler(i,s); }


//------------------------------------------------------------------------------
// Templates
//------------------------------------------------------------------------------

#include <LEDA/template.h>


//------------------------------------------------------------------------------
// LEDA INIT
//------------------------------------------------------------------------------


struct LEDA {

 LEDA();
~LEDA();

char* init_list;
static const char* version_string;
};

extern LEDA L_E_D_A;


//------------------------------------------------------------------------------
// Memory Management
//------------------------------------------------------------------------------

struct  memory_elem_type { memory_elem_type* next; };
typedef memory_elem_type* memory_elem_ptr;

extern memory_elem_ptr memory_free_list[];

extern memory_elem_ptr memory_allocate_block(int);
extern memory_elem_ptr allocate_bytes(int);
extern memory_elem_ptr allocate_words(int);

extern void            deallocate_bytes(void*,int);
extern void            deallocate_bytes_with_check(void*,int);
extern void            deallocate_words(void*,int);
extern void            memory_clear();
extern void            memory_kill();
extern void            print_statistics();

inline void deallocate_list(void* head,void* tail, int bytes)
{ memory_elem_ptr(tail)->next = memory_free_list[bytes];
  memory_free_list[bytes] = memory_elem_ptr(head);
 }

#define OPERATOR_NEW(bytes)\
void* operator new(size_t)\
{ memory_elem_ptr* q = memory_free_list+bytes;\
  if (*q==0) *q = memory_allocate_block(bytes);\
  memory_elem_ptr p = *q;\
  *q = p->next;\
  return p;\
 }

#define OPERATOR_DEL(bytes)\
void  operator delete(void* p)\
{ memory_elem_ptr* q = memory_free_list+bytes;\
  memory_elem_ptr(p)->next = *q;\
  *q = memory_elem_ptr(p);\
 }

#define OPERATOR_DEL_WITH_CHECK(bytes)\
void  operator delete(void* p) { deallocate_bytes_with_check(p,bytes); }


#define LEDA_MEMORY(type)\
OPERATOR_NEW(sizeof(type))\
OPERATOR_DEL(sizeof(type))

#define LEDA_MEMORY_WITH_CHECK(type)\
OPERATOR_NEW(sizeof(type))\
OPERATOR_DEL_WITH_CHECK(sizeof(type))


//------------------------------------------------------------------------------
// handle_base/rep: base classes for handle types string, point, segment,...
//------------------------------------------------------------------------------


class handle_rep  {

friend class handle_base;
   
protected:

   int  count;
    
         handle_rep()  { count = 1; } 
virtual ~handle_rep()  {}

LEDA_MEMORY(handle_rep);
   
};


class handle_base { 

protected:

handle_rep* PTR;

public:

 handle_base()                     {}
 handle_base(const handle_base& x) { PTR = x.PTR;  PTR->count++; }

~handle_base()                     {}


handle_base& operator=(const handle_base& x)
{ x.PTR->count++;
  if (--PTR->count == 0)  delete PTR;
  PTR = x.PTR;
  return *this;
 }

public:

void clear()        { if (--(PTR->count)==0) delete PTR; }
GenPtr copy() const { PTR->count++; return PTR; }


friend int    compare(const handle_base&, const handle_base&);
friend void   Print(const handle_base&, ostream& = cout);
friend void   Read(handle_base&, istream& = cin);
friend void   Init(handle_base&);
friend void   Clear(handle_base&);
friend GenPtr Copy(handle_base&);
friend GenPtr Convert(const handle_base&);

};

inline int compare(const handle_base&, const handle_base&)
{ error_handler(1,"compare undefined");  
  return 0; }

inline void Print(const handle_base&, ostream&)
{ error_handler(1,"Print undefined");  }

inline void Read(handle_base&, istream&)
{ error_handler(1,"Read undefined");  }


inline void   Init(handle_base&)            {}
inline void   Clear(handle_base& y)         { y.clear(); }
inline GenPtr Copy(handle_base& y)          { return y.copy();}
inline GenPtr Convert(const handle_base& y) { return y.PTR; }





#if defined(__ZTC__)
#define LEDA_HANDLE_TYPE(type)\
inline void   Clear(type& y)  { y.clear(); }\
inline GenPtr Copy(type& y)   { return y.copy();}\
inline const GenPtr& Access(const type&, const GenPtr& p) { return p; }\
inline char*  Type_Name(type*) { return STRINGIZE(type); }
#else
#define LEDA_HANDLE_TYPE(type)\
inline void   Clear(type& y)  { y.clear(); }\
inline GenPtr Copy(type& y)   { return y.copy();}\
inline char*  Type_Name(type*) { return STRINGIZE(type); }
#endif





//------------------------------------------------------------------------------
// compare, Print, Read, Init, Copy, Clear, Convert, ...  for built-in types
//------------------------------------------------------------------------------

// defining a linear order

inline int compare(GenPtr x, GenPtr y) { return (char*)x-(char*)y; }
inline int compare(char x, char y)     { return x-y; }
inline int compare(int  x, int  y)     { return x-y; }

inline int compare(long x, long y) 
{ if (x < y)  return -1;  
  else if (x > y)  return  1;  
  else return 0; 
}

inline int compare(double x, double y) 
{ if (x < y)  return -1;  
  else if (x > y)  return  1;  
  else return 0; 
}

inline int compare(float x, float y) 
{ if (x < y)  return -1;  
  else if (x > y)  return  1;  
  else return 0; 
}


// stream output

/*
inline ostream& operator<<(ostream& out, GenPtr x)     { return out<<long(x); }
*/

inline void Print(const GenPtr x, ostream& out = cout) { out << x; }
inline void Print(char x, ostream& out = cout)         { out << x; }
inline void Print(short x, ostream& out = cout)        { out << x; }
inline void Print(int  x, ostream& out = cout)         { out << x; }
inline void Print(long x, ostream& out = cout)         { out << x; }
inline void Print(float x, ostream& out = cout)        { out << x; }
inline void Print(double x, ostream& out = cout)       { out << x; }

// stream input

inline istream& operator>>(istream& in, GenPtr)     { return in; }

inline void Read(GenPtr,   istream&)          {}
inline void Read(char& x,  istream& in = cin) { in >> x; }
inline void Read(short& x, istream& in = cin) { in >> x; }
inline void Read(int&  x,  istream& in = cin) { in >> x; }
inline void Read(long& x,  istream& in = cin) { in >> x; }
inline void Read(float& x, istream& in = cin) { in >> x;}
inline void Read(double& x,istream& in = cin) { in >> x;}


// initialization

inline void Init(GenPtr)    {}
inline void Init(char&   x) { x=0; }
inline void Init(short&  x) { x=0; }
inline void Init(int&    x) { x=0; }
inline void Init(long&   x) { x=0; }
inline void Init(float&  x) { x=0; }
inline void Init(double& x) { x=0; }


// destruction

inline void Clear(GenPtr)    {}
inline void Clear(char  )    {}
inline void Clear(short )    {}
inline void Clear(int   )    {}
inline void Clear(long  )    {}
inline void Clear(float )    {}
inline void Clear(double& x) { deallocate_bytes(&x,sizeof(double)); }


// copying

inline GenPtr Copy(GenPtr x)  { return x; }
inline GenPtr Copy(char x)    { GenPtr p; *((char*)&p) =x; return p; }
inline GenPtr Copy(short x)   { GenPtr p; *((short*)&p)=x; return p; }
inline GenPtr Copy(int  x)    { return *(GenPtr*)&x; }
inline GenPtr Copy(long x)    { return *(GenPtr*)&x; }
inline GenPtr Copy(float x)   { return *(GenPtr*)&x; }
inline GenPtr Copy(double x) 
{ double* p = (double*)allocate_bytes(sizeof(double)); 
  *p = x;
  return p; 
 }


// converting to a generic pointer

inline GenPtr Convert(GenPtr x)     { return x; }
inline GenPtr Convert(char  x)      { GenPtr p; *((char*)&p) =x; return p; }
inline GenPtr Convert(short x)      { GenPtr p; *((short*)&p)=x; return p; }
inline GenPtr Convert(int   x)      { return GenPtr(x); }
inline GenPtr Convert(long  x)      { return GenPtr(x); }
inline GenPtr Convert(float x)      { return *(GenPtr*)&x; }

#if __GNUG__ == 2
inline GenPtr Convert(double& x)       { return GenPtr(&x); }
#else
inline GenPtr Convert(const double& x) { return GenPtr(&x); }
#endif


// access through a generic pointer

inline const GenPtr& Access(GenPtr, const GenPtr& p) { return p; }

#if defined(__ZTC__)
inline const GenPtr& Access(char,  const GenPtr& p) { return p; }
inline const GenPtr& Access(short, const GenPtr& p) { return p; }
inline const GenPtr& Access(int,   const GenPtr& p) { return p; }
inline const GenPtr& Access(long,  const GenPtr& p) { return p; }
inline const GenPtr& Access(float, const GenPtr& p) { return p; }
inline double& Access(double, const GenPtr& p){ return *(double*)p;  }
#else
inline double& Access(double*, const GenPtr& p){ return *(double*)p;  }
#endif



// integer type or not ?

inline int Int_Type(GenPtr) { return 0; }

#if !defined(__MSDOS__)
//inline int Int_Type(char*)  { return 1; }
inline int Int_Type(int*)   { return 1; }
inline int Int_Type(long*)  { return 1; }
#endif

inline char* Type_Name(GenPtr)     { return "unknown"; }
inline char* Type_Name(char*  )    { return "char"; }
inline char* Type_Name(short* )    { return "short"; }
inline char* Type_Name(int*   )    { return "int"; }
inline char* Type_Name(long*  )    { return "long"; }
inline char* Type_Name(float* )    { return "float"; }
inline char* Type_Name(double*)    { return "double"; } 


// maximal and minimal values for some numerical types

inline int    Max_Value(int& x)     { return x =  MAXINT;   }
inline int    Min_Value(int& x)     { return x = -MAXINT;   }
inline float  Max_Value(float& x)   { return x =  MAXFLOAT; }
inline float  Min_Value(float& x)   { return x = -MAXFLOAT; }
inline double Max_Value(double& x)  { return x =  MAXDOUBLE;}
inline double Min_Value(double& x)  { return x = -MAXDOUBLE;}


/* 
  type arguments for the LEDA_TYPE_PARAMETER macro must define:

  a constructor taking no arguments: type::type()
  a copy constructor:                type::type(const type&)
  a Read function:                   void Read(type&, istream&)
  a Print function:                  void Print(const type&, ostream&)
  a compare function:                int compare(const type&, const type&)

*/

#if defined(__ZTC__)
#define LEDA_TYPE_PARAMETER(type)\
inline void   Init(type&)                        {}\
inline void   Clear(type& x)                     { delete (type*)&x; }\
inline GenPtr Copy(type& x)                      { return new type(x);}\
inline GenPtr Convert(type& x)                   { return GenPtr(&x);}\
inline type&  Access(const type&, GenPtr p)      { return *(type*)p; }\
inline char*  Type_Name(type*)                   { return STRINGIZE(type); }
#else
#define LEDA_TYPE_PARAMETER(type)\
inline void   Init(type&)                        {}\
inline void   Clear(type& x)                     { delete (type*)&x; }\
inline GenPtr Copy(type& x)                      { return new type(x);}\
inline GenPtr Convert(type& x)                   { return GenPtr(&x);}\
inline type&  Access(type*, GenPtr p)            { return *(type*)p; }\
inline char*  Type_Name(type*)                   { return STRINGIZE(type); }
#endif


#define DEFINE_LINEAR_ORDER(type,cmp,new_type)\
struct new_type : public type\
{ new_type(type s)            : type(s) {}\
  new_type(const new_type& s) : type(s) {}\
  new_type() {}\
 ~new_type() {}\
};\
inline int compare(const new_type& x, const new_type& y) { return cmp(x,y); }


//------------------------------------------------------------------------------
// Macros
//------------------------------------------------------------------------------

// nil pointer

#define nil 0


// ACCESS(T,x): access a reference of type T through a generic pointer p
// we have to create a dummy first argument of type T* for Access(T*,x)

#if !defined(__ZTC__)

#define ACCESS(type,x) (type&)(Access((type*)0,x))

#else
double ztc_access_dummy;
#define ACCESS(type,x) (type&)(Access(*(type*)&ztc_access_dummy,x))
#endif

#define INT_TYPE(type)  Int_Type((type*)0)
#define TYPE_NAME(type) Type_Name((type*)0)


//turning a macro argument into a string

#if defined(__STDC__)  || defined(__MSDOS__)
#define STRINGIZE(x) #x
#else
#define STRINGIZE(x) "x"
#endif



// iteration macros

#define forall_items(x,S) for(x = (S).first_item(); x; x = (S).next_item(x) )
#define forall(x,S)       for((S).init_iterator(); (S).next_element(x); )
#define Forall(x,S)       for((S).Init_iterator(); (S).prev_element(x); )



// macros for declaring cmp, apply, ord function parameters (array,list)
// (necessary because of a limitation in GNU g++ version 1)

#if __GNUG__ == 1
#define CMP_FUNC_DECL(T)   typedef int  (*name2(T,_cmp))(T&,T&);
#define APP_FUNC_DECL(T)   typedef void (*name2(T,_apply))(T&);
#define ORD_FUNC_DECL(T)   typedef int  (*name2(T,_ord))(T&);
#define CMP_FUNC_TYPE(T,F) name2(T,_cmp) F
#define APP_FUNC_TYPE(T,F) name2(T,_apply) F
#define ORD_FUNC_TYPE(T,F) name2(T,_ord) F
#else
#define CMP_FUNC_DECL(T)
#define APP_FUNC_DECL(T)
#define ORD_FUNC_DECL(T)
#define CMP_FUNC_TYPE(T,F) int  (*F)(T&,T&)
#define APP_FUNC_TYPE(T,F) void (*F)(T&)
#define ORD_FUNC_TYPE(T,F) int  (*F)(T&)
#endif



// miscellaneous

#define Main            main(int argc, char** argv)
#define newline         (cout << "\n").flush()
#define forever         for(;;)
#define loop(a,b,c)     for (a=b;a<=c;a++)

#define Max(a,b) ( (a>b) ? a : b )
#define Min(a,b) ( (a>b) ? b : a )



//------------------------------------------------------------------------------
// enumerations
//------------------------------------------------------------------------------

enum rel_pos { before = 1, after = 0 };

enum direction { forward = 0, backward = 1 };


//------------------------------------------------------------------------------
// boolean
//------------------------------------------------------------------------------

typedef int bool;       // bool = int  (should bool be a class ?)

enum {false=0, true=1};


//------------------------------------------------------------------------------
// strings
//------------------------------------------------------------------------------


class string_rep : public handle_rep { 

friend class string;


      char*   s;

 string_rep(const char*);
 string_rep(char);

~string_rep() { delete s; }

 LEDA_MEMORY(string_rep)

};



class format_string     
{ //class used for resolving ambiguity in string constructor (see below)
  friend class string;
  const char* str;
  public:
  format_string(const char* s) { str = s; }
 };


class string  : public handle_base 
{

friend class string_rep;

static char* str_dup(const char*);
static char* str_cat(const char*,const char*);
static char* str_cat(int, char**);

string_rep*  ptr() const { return (string_rep*)PTR; }

public:

 string()                { PTR = new string_rep(""); }
 string(char c)          { PTR = new string_rep(c);  }


//
// string(const char*);
// string(const char*, ...); // printf-like constructor
//
// that's want we want, but it is ambiguous (see ARM, page 326)
//
// we use the dummy class "format_string" to resolve the ambiguity:

#if !defined(sparc) || defined(__GNUG__)
// The following does not work with some sparcstation cfront compilers ?
// There seems to be a bug in stdarg if the first argument is a class object.

    string(format_string,...);
    string(const char* s)      { PTR = new string_rep(s);}

#else
// in this case we only provide the printf-like constructor

    string(const char*,...);

#endif


 string(int argc, char** argv) { PTR = new string_rep(str_cat(argc,argv)); }

 string(const string& x)  : handle_base(x)  {}

~string()     { clear(); }

string& operator=(const string& x) { handle_base::operator=(x); return *this; }

operator char*()    const   { return str_dup(ptr()->s); }

char*  operator~()  const   { return str_dup(ptr()->s); }
char*  cstring()    const   { return ptr()->s; }

int    length()          const;
string sub(int,int)      const;
int    pos(string,int=0) const;

string operator()(int i, int j)  const { return sub(i,j); }
string head(int i)               const { return sub(0,i-1); }
string tail(int i)               const { return sub(length()-i,length()-1); }

string insert(string, int=0)     const;
string insert(int, string)       const;

string replace(const string&, const string&, int=1) const; 
string replace(int, int, const string&) const; 

string del(const string&, int=1)    const; 
string del(int, int)         const; 

void   read(istream&, char=' ');
void   read(char delim=' ')     { read(cin,delim); }

string replace_all(const string& s1, const string& s2) 
                                         const  { return replace(s1,s2,0); } 
string replace(int i, const string& s)   const  { return replace(i,i,s);  } 

string del_all(const string& s)          const  { return del(s,0); }
string del(int i)                        const  { return del(i,i); }  

void   read_line(istream& I = cin)              { read(I,'\n'); }

string format(string) const;

char  operator[](int) const;
char& operator[](int);

string operator+(const string& x)  const;
string operator+(const char* s1)   const;
string& operator+=(const char* s1);
string& operator+=(const string& x);


friend int operator==(const string& x, const char* s);
friend int operator==(const string& x, const string& y);
friend int operator!=(const string& x, const char* s);
friend int operator!=(const string& x, const string& y);
friend int operator< (const string& x, const string& y);
friend int operator> (const string& x, const string& y);
friend int operator<=(const string& x, const string& y);
friend int operator>=(const string& x, const string& y);


friend istream& operator>>(istream&, string&); 
friend ostream& operator<<(ostream&, const string&) ;

friend char* Type_Name(string*);
friend void Print(const string&, ostream& = cout);
friend void Read(string&, istream& = cin);
friend int  compare(const string& x, const string& y);

};

inline void Print(const string& x, ostream& out) { out << x; }
inline void Read(string&  x, istream& in)        { in  >> x; }

LEDA_HANDLE_TYPE(string)


//------------------------------------------------------------------------------
// STRING<cmp>: string with user defined linear order cmp
//------------------------------------------------------------------------------

typedef int  (*CMP_STRING_TYPE)(const string&, const string&);


template<CMP_STRING_TYPE cmp>

struct _CLASSTYPE STRING : public string 
{  STRING(string s)        : string(s) {}
   STRING(const STRING& s) : string(s) {}
   STRING() {}
 ~ STRING() {}
friend int compare(const STRING<cmp>& x, const STRING<cmp>& y) 
{ return cmp(x,y); }
};




//------------------------------------------------------------------------------
// INT<cmp>: int with user defined linear order cmp
//------------------------------------------------------------------------------

typedef int   (*CMP_INT_TYPE)(int,int);


template<CMP_INT_TYPE cmp>

struct _CLASSTYPE INT
{int p;
 INT(const int i=0) { p = i;}
 operator int()     { return p; }

friend void Print(const INT<cmp>& x, ostream& out = cout) { out << x.p; }
friend void Read(INT<cmp>&  x, istream& in = cin)  { in  >> x.p; }
friend void Init(INT<cmp>&  x)            { x.p=0; }
friend GenPtr Copy(const INT<cmp>& x)     { return GenPtr(x); }
friend void Clear(INT<cmp>& x)            { x.p = 0; }
friend GenPtr Convert(const INT<cmp>& x)  { return GenPtr(x.p); }
friend INT<cmp>& Access(INT<cmp>*, const GenPtr& p) { return *(INT<cmp>*)&p; }

friend int compare(const INT<cmp>& x, const INT<cmp>& y) { return cmp(x.p,y.p);}
};





//------------------------------------------------------------------------------
// some useful functions
//------------------------------------------------------------------------------

typedef int (*SIG_PF) (...);

extern SIG_PF signal(int, SIG_PF);

extern int interrupt_handler(...);

inline SIG_PF catch_interrupts(SIG_PF handler = interrupt_handler)
{ return signal(2,handler); }



#if defined(__NO_RANDOM__)
inline void srandom(int x) { srand(x); }
inline int  random()       { return rand(); }
#endif

extern void init_random(int=0);

inline int    random(int a, int b)  { return a + random()%(b-a+1); }
inline double rrandom()             { return double(random())/MAXINT; }

inline unsigned random(unsigned a, unsigned b)  
{ return a + unsigned(random())%(b-a+1); }



extern float used_time();
extern float used_time(float&);
extern void  print_time(string s);

inline void  print_time() { print_time(""); }

extern void  wait(unsigned int seconds);


//------------------------------------------------------------------------------
// input/output
//------------------------------------------------------------------------------

// problems with cfront2.1's "stdarg" on SPARCs
// call save_sparc_registers() before va_start !

#if !defined(__GNUG__) && defined(sparc)
extern  "C" void  save_sparc_registers();
#define Va_Start  save_sparc_registers(); va_start
#else
#define Va_Start  va_start
#endif


extern int    Yes(string s);
extern int    Yes();

extern int    read_int(string s);
extern int    read_int();

extern char   read_char(string s);
extern char   read_char();

extern double read_real(string s);
extern double read_real();

extern char*  read_string(string s);
extern char*  read_string();

extern char*  read_line(istream& s);
extern char*  read_line();


#endif
