
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/***********************************************************************
*
*  File: check.c
*
*  Contents:  Various integrity and consistency checks.
*
*/

#include "include.h"

/* maximum errors of any type before aborting */
#define MAXERR 10

/*********************************************************************
*
*  Function: run_checks()
*
*  Purpose:  Overall control of checking.
*/

int run_checks()
{
  int numerr = 0;

#ifdef TC
if ( farheapcheck() < 0 )
  error("Corrupt heap. Reboot! Memory is trashed.\n",UNRECOVERABLE);
#ifdef HEAPREPORT
else
  { struct farheapinfo hinfo;
    int b_use=0,b_free=0;
    long mem_use=0,mem_free=0;

    hinfo.ptr = NULL;
    while ( farheapwalk(&hinfo) == _HEAPOK )
      if (hinfo.in_use)
        { 
#ifdef HEAPLIST
          fprintf(outfd,"%p %5ld ",hinfo.ptr,hinfo.size);
          b_use++; mem_use+= hinfo.size;
#endif
        }
      else { b_free++; mem_free += hinfo.size; }
    fprintf(outfd,"\n");
    fprintf(outfd,"blocks in use: %d   memory in use: %ld \n",b_use,mem_use);
    fprintf(outfd,"blocks free:   %d   memory free:   %ld \n",b_free,mem_free);
  }
#endif
#endif

  numerr += list_check();
  if ( !web.simplex_flag )
   { numerr += facetedge_check(REGCHECK);
     numerr += facet_body_check();
     numerr += collapse_check();
   }
  return numerr;
}

/**********************************************************************
*
*  Function: list_check()
*
*  Purpose:  Check element linked lists.
*
*/

int list_check()
{
  int type;
  int freecount,usedcount,maxfree,maxused;
  element_id id,prev_id;
  int numerr = 0;

  for ( type = 0 ; type < 5 ; type++ )
    { 
#ifndef NOFREELIST
      if ( web.simplex_flag  && (type == EDGE) ) continue;
      /* check free list */
      maxfree = web.skel[type].maxcount - web.skel[type].count;
      id = web.skel[type].free;
      freecount = 0;
      while ( id != NULLID )
        { freecount++;
          if ( freecount > maxfree )
            { fprintf(stderr,"Type %d freelist has too many elements: %d.\n",
                  type,freecount);
              if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
              break;
            }
          if ( id_type(id) != type )
            { fprintf(stderr,"Type %d freelist has bad id %lX\n",type,id);
              if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
              break;
            }
          id = elptr(id)->forechain;
        }
      if ( freecount != maxfree )
        fprintf(stderr,"Type %d freelist has %d elements instead of %d.\n",
	   type,freecount,maxfree);
      if ( !equal_id(id,NULLID) )
        fprintf(stderr,"Type %d freelist last id is non-null: %lX\n",type,id);
#endif

      /* check used list */
      maxused = web.skel[type].count;
      id = web.skel[type].used;
      prev_id = NULLID;
      usedcount = 0;
      while ( valid_id(id) )
        { usedcount++;
          if ( usedcount > maxused )
            { fprintf(stderr,"Type %d usedlist has too many elements: %d.\n",
                  type,usedcount);
              if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
              break;
            }
          if ( id_type(id) != type )
            { fprintf(stderr,"Type %d usedlist has bad id %lX of type %d\n",
                  type,id,id_type(id));
              if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
              break;
            }
          if ( !equal_id(prev_id,elptr(id)->backchain) )
            { fprintf(stderr,"Type %d used list id %lX has backchain %lX instead of %lX\n",
                 type,id,elptr(id)->backchain,prev_id);
              if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
	    }
          prev_id = id;
          id = elptr(id)->forechain;
        }
      if ( usedcount != maxused )
        fprintf(stderr,"Type %d usedlist has %d elements.\n",type,usedcount);
      if ( !equal_id(id,NULLID) )
        fprintf(stderr,"Type %d usedlist last id is non-null: %lX\n",type,id);
    }

  /* check body fixed volume count */
  { int count = 0;
    body_id b_id;
    FOR_ALL_BODIES(b_id)
      if ( get_battr(b_id) & FIXEDVOL ) count++;
    if ( count != fixed_volumes )
     fprintf(stderr,"fixed_volumes is %d instead of %d.\n",fixed_volumes,count);
  }
  return numerr;
}



/**********************************************************************
*
*  Function: facetedge_check()
*
*  Purpose:  Check integrity of facetedge chains both ways.
*/

int facetedge_check(flag)
int flag;  /* PRELIMCHECK for check before subdivision, else REGCHECK */
{
  vertex_id v_id;
  edge_id e_id;
  facet_id f_id;
  facetedge_id fe_id,last_fe;
  int count;
  int numerr = 0;
  int nullfacets = 0 ;   /* number of fe's on null facet */
  
  /* check facetedge chain consistencies */
  if ( numerr >= MAXERR ) numerr = 0;
  count = 0;
  FOR_ALL_FACETEDGES(fe_id)
    { facetedge_id fe;
      if ( !valid_id(get_fe_facet(fe_id)) ) nullfacets++;
      fe = get_prev_edge(fe_id);
      if ( valid_id(fe) && !equal_id(get_next_edge(fe),fe_id) )
        { fprintf(stderr,"Facetedge %d has bad next edge link\n",ordinal(fe_id)+1);
          if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
        }
      fe = get_next_edge(fe_id);
      if ( valid_id(fe) && !equal_id(get_prev_edge(fe),fe_id) )
        { fprintf(stderr,"Facetedge %d has bad prev edge link\n",ordinal(fe_id)+1);
          if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
        }
      fe = get_next_facet(fe_id);
      if ( valid_id(fe) && !equal_id(get_prev_facet(fe),fe_id) )
        { fprintf(stderr,"Facetedge %d has bad next facet link\n",ordinal(fe_id)+1);
          if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
        }
      fe = get_prev_facet(fe_id);
      if ( valid_id(fe) && !equal_id(get_next_facet(fe),fe_id) )
        { fprintf(stderr,"Facetedge %d has bad prev facet link\n",ordinal(fe_id)+1);
          if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
        }
      fe = get_next_edge(fe_id);
      if ( valid_id(fe) && !equal_id(get_fe_headv(fe_id),get_fe_tailv(fe)) )
        { fprintf(stderr,"Facetedge %d head vertex disagrees with next tail.\n",
            ordinal(fe_id)+1);
          if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
        }
      fe = get_prev_edge(fe_id);
      if ( valid_id(fe) && !equal_id(get_fe_tailv(fe_id),get_fe_headv(fe)) )
        { fprintf(stderr,"Facetedge %d tail vertex disagrees with prev head.\n",
            ordinal(fe_id)+1);
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
        }
      count++;
    }
  if ( count != web.skel[FACETEDGE].count )
   fprintf(stderr,"Only %d facetedges out of %d generated.\n",
       count,web.skel[FACETEDGE].count);

   /* check that vertices have legit facetedge link */
   FOR_ALL_VERTICES(v_id)
    { fe_id = get_vertex_fe(v_id);
      if ( !valid_id(fe_id) ) 
       { sprintf(errmsg,"Vertex %d has no facetedge link.\n",ordinal(v_id)+1);
	 error(errmsg,WARNING);
       }
      else if ( !equal_id(v_id,get_fe_tailv(fe_id)) )
       { sprintf(errmsg,"Vertex %d has bad facetedge link.\n",ordinal(v_id)+1);
	 check_vertex_fe(v_id);
	 error(errmsg,WARNING);
       }
    }
  count = 0;
  FOR_ALL_EDGES(e_id)
   {
     last_fe = NULLID;
     generate_edge_fe_init();
     while ( generate_edge_fe(e_id,&fe_id) )
      {
        if ( !equal_id(e_id,get_fe_edge(fe_id)) )
          { fprintf(stderr,"Facetedge %d on edge %d instead of %d.\n",
               ordinal(fe_id)+1,ordinal(get_fe_edge(fe_id))+1,ordinal(e_id)+1);
            if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); goto facet_check;}
	  }
        count++;
        last_fe = fe_id;
      }
     if ( last_fe == NULLID )
       { if ( (web.dimension == SOAPFILM) && (flag == REGCHECK) ) 
	   {
             fprintf(stderr,"Edge %d has no facets.\n",ordinal(e_id)+1);
	     if ( get_original(e_id)  != ordinal(e_id)+1 )
	       fprintf(stderr,"     (originally edge %d)\n",get_original(e_id));
	   }
       }
     else if ( !equal_id(get_edge_fe(e_id),get_next_facet(last_fe)) )
       { fprintf(stderr,"Facets around edge %d do not link up.\n",
           ordinal(e_id)+1);
	 if ( get_original(e_id)  != ordinal(e_id)+1 )
	   fprintf(stderr,"     (originally edge %d)\n",get_original(e_id));
       }
   }
  if ( count != web.skel[FACETEDGE].count )
    { fprintf(stderr,"Edges have %d facetedges out of %d used.\n",
       count,web.skel[FACETEDGE].count);
       ++numerr;
    }

facet_check:
  if ( numerr >= MAXERR ) numerr = 0;
  count = 0;
  FOR_ALL_FACETS(f_id)
   {
     int thiscount = 0;
     last_fe = NULLID;
     generate_facet_fe_init();
     while ( generate_facet_fe(f_id,&fe_id) )
      {
        if ( !equal_id(f_id,get_fe_facet(fe_id)) )
          { fprintf(stderr,"Facetedge %d on facet %d instead of %d.\n",
             ordinal(fe_id)+1,ordinal(get_fe_facet(fe_id))+1,ordinal(f_id)+1);
            if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); break;}
	  }
        count++;
        if ( ++thiscount > web.skel[FACETEDGE].count )
	  { fprintf(stderr,"Facetedge loop not closed on facet %d.\n",
		ordinal(f_id)+1);
	    if ( get_original(f_id) != ordinal(f_id)+1  )
	      fprintf(stderr,"     (originally facet %d)\n",get_original(f_id));
	    break;
	  }
        last_fe = fe_id;
      }
     if ( web.dimension == SOAPFILM ) 
      if ( !equal_id(get_facet_fe(f_id),get_next_edge(last_fe)) )
       { fprintf(stderr,"Edges around facet %d do not link up.\n",
           ordinal(f_id)+1);
	 if ( get_original(f_id) != ordinal(f_id)+1  )
	   fprintf(stderr,"     (originally facet %d)\n",get_original(f_id));
         if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); break;}
       }
     if ( (web.dimension == SOAPFILM ) && (thiscount != 3) 
				       && (flag == REGCHECK)  )
       { fprintf(stderr,"Facet %d has %d edges.\n",ordinal(f_id)+1,thiscount);
	 if ( get_original(f_id) != ordinal(f_id)+1 )
	   fprintf(stderr,"     (originally facet %d)\n",get_original(f_id));
         if ( ++numerr > MAXERR )
	   {fprintf(stderr,"Too many errors.\n"); break;}
       }
   }
  if ( count+nullfacets != web.skel[FACETEDGE].count )
    { fprintf(stderr,"Facets have %d facetedges out of %d used.\n",
         count,web.skel[FACETEDGE].count);
      ++numerr;
    }
  return numerr;
}


/***********************************************
*
*  Function: facet_body_check()
*
*  Purpose:  Check whether adjacent facets have same bodies.
*            Should be used only for complete bodies.
*/

int facet_body_check()
{
  facetedge_id fe;
  int numerr = 0;
  int orig;
     
  if ( web.dimension != SOAPFILM ) return 0;

  FOR_ALL_FACETEDGES(fe)
   { body_id b_id,bb_id;
     facet_id f_id,ff_id;
     edge_id e_id;
    
     f_id = get_fe_facet(fe);
     b_id = get_facet_body(f_id);
     if ( equal_id(fe,get_prev_facet(fe)) ) continue;
     ff_id = get_fe_facet(fe_inverse(get_prev_facet(fe)));
     bb_id = get_facet_body(ff_id);
     e_id = get_fe_edge(fe);
     if ( !equal_id(b_id,bb_id) && !(get_attr(e_id)&CONSTRAINT) )
       { fprintf(stderr,"Inconsistent bodies for facets on edge %d.",
                    ordinal(e_id)+1 );
	 orig = get_original(e_id);
	 if ( orig && (orig != ordinal(e_id)+1) )
	    fprintf(stderr,"     (originally edge %d)",orig);
         fprintf(stderr,"\n    facet %d",ordinal(f_id)+1);
	 orig = get_original(f_id);
	 if ( orig && (orig != ordinal(f_id)+1) )
	    fprintf(stderr," (orig. %d)\n",orig);
	 if ( valid_id(b_id) )
	  {
	     fprintf(stderr," has body %d",ordinal(b_id)+1);
             orig = get_original(b_id);
	     if ( orig && (orig != ordinal(b_id)+1) )
	        fprintf(stderr," (orig. %d)",orig);
	  }
	 else fprintf(stderr," has no body");
	 fprintf(stderr,"; facet %d",ordinal(ff_id)+1);
	 orig = get_original(ff_id);
	 if ( orig && (orig != ordinal(ff_id)+1) )
	    fprintf(stderr," (orig. %d)",orig);
         if ( valid_id(bb_id) )
	   {
	     fprintf(stderr,"has body %d",ordinal(bb_id)+1);
	     orig = get_original(bb_id);
	     if ( orig && (orig != ordinal(bb_id)+1) )
	        fprintf(stderr," (orig. %d)",orig);
           }
         else fprintf(stderr," has no body");
	 fprintf(stderr,".\n");
         if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
        }
    }
  return numerr;
}


/**********************************************************************
*
*  Function: collapse_check()
*
*  Purpose:  Checks whether adjacent facets have collapsed.
*            Tests whether they have the same third vertex.
*            Also checks for edges that have the same endpoints.
*/

int vvvvcomp(a,b)
struct vvvv *a,*b;
{ int i;
  for ( i = 0 ; i < 3 ; i++ )
    { if ( a->vord[i] < b->vord[i] ) return -1;
      if ( a->vord[i] > b->vord[i] ) return 1;
    }
  return 0;
}

int collapse_check()
{
  struct vvvv *list,*current;
  facet_id f_id;
  edge_id e_id;
  int ord1,ord2,ord3,tmp;
  int i;
  int count;
  int numerr = 0;
  int orig;

  /* check edges */
  if ( web.skel[EDGE].count == 0 ) return numerr;
  list = (struct vvvv *)temp_calloc(web.skel[EDGE].count,sizeof(struct vvvv));

  current = list;
  count = 0;
  FOR_ALL_EDGES(e_id)
    { 
      ord1 = ordinal(get_edge_tailv(e_id));
      ord2 = ordinal(get_edge_headv(e_id));
      if ( ord1 == ord2 )
        { fprintf(stderr,"Edge %d",ordinal(e_id)+1);
	  orig = get_original(e_id);
	  if ( orig && (orig != ordinal(e_id)+1) )
	    fprintf(stderr," (orig. %d)\n",orig);
	  fprintf(stderr," is loop on vertex %d",ord1+1);
	  orig = get_original(get_edge_tailv(e_id));
	  if ( orig && (orig != ord1+1) )
	    fprintf(stderr," (orig. %d)\n",orig);
	  fprintf(stderr,".\n");
	}
      if ( ord1 <= ord2 )
        { current->id = e_id;
          current->vord[0] = ord1;
          current->vord[1] = ord2;
        }
      else
        { current->id = inverse_id(e_id);
          current->vord[0] = ord2;
          current->vord[1] = ord1;
        }
      if ( ++count > web.skel[EDGE].count )
        { fprintf(stderr,"Edge count disagrees with supposed value.");
          count--;
          break;
        }
      current++;
    }
  qsort((char *)list,count,sizeof(struct vvvv),FCAST vvvvcomp);

  /* scan list for duplicates */
  for ( i = 0 ; i < count-1 ; i++ )
    { if ( vvvvcomp(list+i,list+i+1) == 0 )
        fprintf(stderr,"Edges %d and %d have same endpoints: %d %d.\n",
          ordinal(list[i].id)+1,ordinal(list[i+1].id)+1,list[i].vord[0]+1,
          list[i].vord[1]+1);
    }
  temp_free((char*)list);

  if ( web.dimension == STRING ) return numerr;

  /* check facets */
  if ( web.skel[FACET].count == 0 ) return numerr;
  list = (struct vvvv *)temp_calloc(web.skel[FACET].count,sizeof(struct vvvv));

  current = list;
  count = 0;
  FOR_ALL_FACETS(f_id)
    { 
      facetedge_id fe;
      fe = get_facet_fe(f_id);
      ord1 = ordinal(get_fe_tailv(fe));
      ord2 = ordinal(get_fe_headv(fe));
      ord3 = ordinal(get_fe_headv(get_next_edge(fe)));
      /* bubble sort */
      if ( ord1 > ord2 )
        { tmp = ord1; ord1 = ord2; ord2 = tmp; invert(f_id); }
      if ( ord2 > ord3 )
        { tmp = ord2; ord2 = ord3; ord3 = tmp; invert(f_id); }
      if ( ord1 > ord2 )
        { tmp = ord1; ord1 = ord2; ord2 = tmp; invert(f_id); }
      current->id = f_id;
      current->vord[0] = ord1;
      current->vord[1] = ord2;
      current->vord[2] = ord3;
      if ( ++count > web.skel[FACET].count )
        { fprintf(stderr,"Facet count disagrees with supposed value.");
          count--;
          if ( ++numerr > MAXERR )
		 {fprintf(stderr,"Too many errors.\n"); return numerr;}
          break;
        }
      current++;
    }
  qsort((char *)list,count,sizeof(struct vvvv),FCAST vvvvcomp);

  /* scan list for duplicates */
  for ( i = 0 ; i < count-1 ; i++ )
    { if ( vvvvcomp(list+i,list+i+1) == 0 )
        fprintf(stderr,"Facets %d and %d have same vertices: %d %d %d.\n",
          ordinal(list[i].id)+1,ordinal(list[i+1].id)+1,list[i].vord[0]+1,
          list[i].vord[1]+1,list[i].vord[2]+1);
    }
  temp_free((char*)list);
 return numerr;
}



/**********************************************************************
*
*  Function:  normal_change_check()
*
*  Purpose:   Checks how much the normal of facets have changed
*             during motion. Returns the largest (delta normal)/(normal)
*/

REAL normal_change_check()
{
  facet_id f_id;
  facetedge_id fe;
  vertex_id v_id;
  REAL side[2][MAXCOORD];
  REAL max_delta;
  REAL old_normal[MAXCOORD];
  REAL new_normal[MAXCOORD];
  REAL x[3][MAXCOORD];
  int ord_v;
  REAL diff;
  struct boundary *bdry;
  int i,n;

  max_delta = 0.0;

  FOR_ALL_FACETS(f_id)
   {
     if ( get_fattr(f_id) & FIXED ) continue;

     /* get old normal */
     generate_facet_fe_init();
     for ( n = 0 ; generate_facet_fe(f_id,&fe) ; n++ )
      {
        v_id = get_fe_tailv(fe);
        ord_v = ordinal(v_id);
        if ( get_vattr(v_id) & BOUNDARY )    
        { bdry = get_boundary(v_id);
          for ( i = 0 ; i < web.sdim ; i++ )
            x[n][i] = eval(bdry->coordf[i],oldcoord[ord_v]);
        }
        else
         {   
           for ( i = 0 ; i < web.sdim ; i++ )
             x[n][i] = oldcoord[ord_v][i];
         }
      }
     for ( i = 0 ; i < web.sdim ; i++ )
      { side[0][i] = x[1][i] - x[0][i];
        side[1][i] = x[2][i] - x[0][i];
      }
     cross_prod(side[0],side[1],old_normal);

     /* get new normal */
     generate_facet_fe_init();
     for ( n = 0 ; generate_facet_fe(f_id,&fe) ; n++ )
      {
        v_id = get_fe_tailv(fe);
        for ( i = 0 ; i < web.sdim ; i++ )
           x[n][i] = get_coord(v_id)[i];
      }
     for ( i = 0 ; i < web.sdim ; i++ )
      { side[0][i] = x[1][i] - x[0][i];
        side[1][i] = x[2][i] - x[0][i];
      }
     cross_prod(side[0],side[1],new_normal);

     /* test difference */
     for ( i = 0 ; i < web.sdim ; i++ )
       side[0][i] = new_normal[i] - old_normal[i];
     diff = sqrt(dot(side[0],side[0],web.sdim)/
                                  dot(old_normal,old_normal,web.sdim));
     if ( diff > max_delta ) 
       max_delta = diff;
   }

  return max_delta;
}


