(*---RCS--- $Log:	_convert.sml,v $
Revision 1.1  92/09/17  14:17:23  birkedal
Edinburgh Version 11Sep92
*)

(*$Convert: LAMBDA_EXP CEXP LVARS FINMAP LISTUTIL CRASH CONVERT*)
(* Convert is the translator from lambda-expressions to continuation
   expressions. Look to the Orbit/Scheme literature for details. *)

functor Convert(structure LambdaExp: LAMBDA_EXP

		structure CExp: CEXP
		  sharing type LambdaExp.prim = CExp.prim

		structure Lvars: LVARS
		  sharing type LambdaExp.lvar = CExp.lvar = Lvars.lvar

		structure FinMap: FINMAP
		  sharing type LambdaExp.map = CExp.map = FinMap.map

		structure ListUtil: LISTUTIL
		structure Crash: CRASH
	       ): CONVERT =
  struct
    structure L = LambdaExp and C = CExp
    type lvar = Lvars.lvar
    type LambdaExp = L.LambdaExp
    type prim = L.prim
    type CExp = C.CExp

   (* some short-hand: *)
    fun fix1(f, args, body, scope) =
      C.FIX([{f=f, formals=args, body=body}], scope)

    fun app(f, args) = C.APP{f=f, actuals=args}

   (* convertSwitch is outside convert because we want it to be fully
      polymorphic (we have three different types of switch). *)

    fun convertSwitch(cc: LambdaExp -> CExp,
		      L.SWITCH{arg: lvar,
			       selections: (''a, LambdaExp) FinMap.map,
			       wildcard: LambdaExp L.option
			      }
		     ) =
      C.SWITCH{arg=arg,
	       selections=FinMap.composemap cc selections,
	       wildcard=case wildcard of L.SOME exp => C.SOME(cc exp)
	     			       | L.NONE => C.NONE
	      }

   (* convert - given a continuation, convert an expression to CPS form on
      that continuation. The new expression must apply the continuation
      to any atomic expressions (integers, lvars, etc, and we also do
      vectors of lvars here), and do the sensible thing for the other
      lambda constructs. Note that the calling convention for functions
      changes: they become two-argument, where the second argument is their
      continuation. *)

    fun convert (gamma: lvar) (lexp: LambdaExp): CExp =
      case lexp			(* Basic atoms are transformed into
				   applications of the continuation
				   to the atom *)
	of L.VAR lvar  => app(gamma, [C.VAR lvar])
	 | L.INTEGER i => app(gamma, [C.INTEGER i])
	 | L.STRING s  => app(gamma, [C.STRING s])
	 | L.REAL r    => app(gamma, [C.REAL r])

	 | L.FN(lvar, body) =>	(* Transform this into an application of the
				   continuation to a new 2-argument lambda. *)
	     let
	       val g = Lvars.newLvar()
	       val f = Lvars.newLvar()
	     in
	       fix1(f, [lvar, g], convert g body,
		    app(gamma, [C.VAR f])
		   )
	     end

	 | L.FIX(fns, bodies, scope) =>
	     let
	       fun f(func, L.FN(formal, body)) =
		     let
		       val g = Lvars.newLvar()
		     in
		       {f=func, formals=[formal, g],
			body=convert g body
		       }
		     end
		 | f _ = Crash.impossible "convert(FIX): not FN"

	       val bindings = map f (ListUtil.zip(fns, bodies))
	     in
	       C.FIX(bindings, convert gamma scope)
	     end

	 | L.APP(exp1, exp2) =>
		(* In Steele's scheme, this would be:
		   convert (\r.convert (\x.r(x, gamma)) exp2) exp1
		   We have some extra junk because we don't have anonymous
		   lambda-expressions. *)
	     let
	       val c1 = Lvars.newLvar()
	       val c2 = Lvars.newLvar()
	       val r = Lvars.newLvar()
	       val x = Lvars.newLvar()
	     in
	       fix1(c1, [r], fix1(c2, [x], app(r, [C.VAR x, C.VAR gamma]),
				  convert c2 exp2
				 ),
		    convert c1 exp1
		   )
	     end

	 | L.PRIM_APP(prim, exp) =>
	     let
	       val f = Lvars.newLvar()
	       val r = Lvars.newLvar()
	     in
	       fix1(f, [r], C.PRIM_APP{cont=gamma, prim=prim, arg=r},
		    convert f exp
		   )
	     end

	 | L.VECTOR exps => convertVec(exps, gamma, nil)

	 | L.SELECT(int, lvar) => C.SELECT(int, lvar)

	 | L.SWITCH_I sw => C.SWITCH_I(convertSwitch(convert gamma, sw))
	 | L.SWITCH_S sw => C.SWITCH_S(convertSwitch(convert gamma, sw))
	 | L.SWITCH_R sw => C.SWITCH_R(convertSwitch(convert gamma, sw))

	 | L.RAISE exp => Crash.unimplemented "convert(RAISE)"

	 | L.HANDLE(exp1, exp2) => Crash.unimplemented "convert(HANDLE)"

	 | L.VOID => app(gamma, [C.VOID])

    and convertVec(x :: rest, gamma, vars) =
          let
	    val f = Lvars.newLvar()
	    val a = Lvars.newLvar()
	  in
	    fix1(f, [a], convertVec(rest, gamma, a :: vars),
		 convert f x
		)
	  end

      | convertVec(nil, gamma, vars) =
	  app(gamma, [C.VECTOR(rev vars)])

   (* cc - wrap up the top-level by converting an expression with the
      identity function as continuation. *)

    fun cc exp =
      let
	val id = Lvars.newLvar()
	val x = Lvars.newLvar()
      in
	fix1(id, [x], C.SIMPLE(C.VAR x), convert id exp)
      end
  end;
