/*
 * xtiff - view a TIFF file in an X window
 *
 * Dan Sears
 * Chris Sears
 *
 * Revsion 1.2  90/05/29
 *      Fixed a bug with MINISWHTE images.
 * Revsion 1.1  90/05/16
 *      Fixed a few minor bugs.
 * Revsion 1.0  90/05/07
 *      Initial release.
 *
 * Copyright 1990 by Digital Equipment Corporation, Maynard, Massachusetts.
 *
 *                        All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of Digital not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Notes:
 *
 * According to the TIFF 5.0 Specification, it is possible to have
 * both a TIFFTAG_COLORMAP and a TIFFTAG_COLORRESPONSECURVE.  This
 * doesn't make sense since a TIFFTAG_COLORMAP is 16 bits wide and
 * a TIFFTAG_COLORRESPONSECURVE is tfBitsPerSample bits wide for each
 * channel.  This is probably a bug in the specification.
 * In this case, TIFFTAG_COLORRESPONSECURVE is ignored.
 * This might make sense if TIFFTAG_COLORMAP was 8 bits wide.
 *
 * TIFFTAG_COLORMAP is often incorrectly written as ranging from
 * 0 to 255 rather than from 0 to 65535.  CheckAndCorrectColormap()
 * takes care of this.
 *
 * Only ORIENTATION_TOPLEFT is supported correctly.  This is the
 * default TIFF and X orientation.  Other orientations will be
 * displayed incorrectly.
 *
 * There is no support for or use of 3/3/2 DirectColor visuals.
 * TIFFTAG_MINSAMPLEVALUE and TIFFTAG_MAXSAMPLEVALUE are not supported.
 *
 * Only TIFFTAG_BITSPERSAMPLE values that are 1, 2, 4 or 8 are supported.
 */

#include <math.h>
#include <stdio.h>
#include <tiffio.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#include "xtifficon.h"

#define TIFF_GAMMA  2.2
#define ROUND(x)    (u_short) ((x) + 0.5)
#define SCALE(x, s) (((x) * 65535L) / (s))
#define MCHECK(m)   if (!m) { fprintf(stderr, "malloc failed\n"); exit(0); }

void                GetTIFFHeader();
void                CheckAndCorrectColormap();
void                SimpleGammaCorrection();
void                InitializeX();
void                GetTIFFImage();
void                CreateXImage();
int                 SearchVisualList();
void                EventLoop();
int                 XTiffErrorHandler();

/*
 * X data structures
 */
Colormap            xColormap;
Display *           xDisplay;
Window              xWindow;
Pixmap              xImagePixmap, xTiffIconPixmap;
Visual *            xVisual;
XImage *            xImage;
GC                  xWinGc, xBitmapGc;
XGCValues           xGcValues;
int                 xScreen, xSyncFlag = False, xImageDepth, xNVisuals,
                    xPixmapFlag = True, xRedMask, xGreenMask, xBlueMask;
char *              xGeometry = NULL, *xDisplayName = NULL;
XVisualInfo *       xVisualList;

/*
 * TIFF data structures
 */
TIFF *              tfFile;
u_short             tfBitsPerSample, tfSamplesPerPixel, tfPlanarConfiguration,
                    tfPhotometricInterpretation, tfGrayResponseUnit,
                    tfImageDepth, tfImageWidth, tfImageHeight, tfBytesPerRow;
int                 tfDirectory = 0;
double              tfUnitMap, tfGrayResponseUnitMap[] = {
                        -1, -10, -100, -1000, -10000, -100000
                    };

/*
 * display data structures
 */
double              dGamma = TIFF_GAMMA, *dRed, *dGreen, *dBlue;

/*
 * shared data structures
 */
u_short *           redMap = NULL, *greenMap = NULL, *blueMap = NULL,
                    *grayMap = NULL, colormapSize;
u_char *            imageMemory;
char *              fileName, *programName;

void
main(argc, argv)
    int argc;
    char ** argv;
{
    int i;

    programName = argv[0];

    if (argc == 1) {
        fprintf(stderr, "Usage: %s file\noptions:\n", programName);
        fprintf(stderr, "    [-directory directory]\n");
        fprintf(stderr, "    [-display display]\n");
        fprintf(stderr, "    [-gamma gamma]\n");
        fprintf(stderr, "    [-geometry geometry]\n");
        fprintf(stderr, "    [-nopixmap]\n");
        fprintf(stderr, "    [-sync]\n");
        exit(0);
    }

    for (i = 1; i < argc; i++) {
        if (strcmp(argv[i], "-display") == 0)
            xDisplayName = argv[++i];
        else if (strcmp(argv[i], "-gamma") == 0)
            dGamma = atof(argv[++i]);
        else if (strcmp(argv[i], "-geometry") == 0)
            xGeometry = argv[++i];
        else if (strcmp(argv[i], "-nopixmap") == 0)
            xPixmapFlag = False;
        else if (strcmp(argv[i], "-sync") == 0)
            xSyncFlag = True;
        else if (strcmp(argv[i], "-directory") == 0)
            tfDirectory = atoi(argv[++i]);
        else
            fileName = argv[i];
    }

    GetTIFFHeader();
    SimpleGammaCorrection();
    InitializeX(argc, argv);
    GetTIFFImage();
    CreateXImage();
    XMapWindow(xDisplay, xWindow);
    EventLoop();
}

void
GetTIFFHeader()
{
    register int i;

    if ((tfFile = TIFFOpen(fileName, "r")) == NULL) {
        fprintf(stderr, "%s: can't open %s as a TIFF file\n",
            programName, fileName);
        exit(0);
    }

    if (!TIFFSetDirectory(tfFile, tfDirectory)) {
        fprintf(stderr, "%s: can't seek to directory %d in %s\n",
            programName, tfDirectory, fileName);
        exit(0);
    }

    TIFFGetField(tfFile, TIFFTAG_IMAGEWIDTH, &tfImageWidth);
    TIFFGetField(tfFile, TIFFTAG_IMAGELENGTH, &tfImageHeight);

    /*
     * If the following tags aren't present then use the TIFF defaults.
     */
    if (!TIFFGetField(tfFile, TIFFTAG_BITSPERSAMPLE, &tfBitsPerSample))
        tfBitsPerSample = 1;
    if (!TIFFGetField(tfFile, TIFFTAG_SAMPLESPERPIXEL, &tfSamplesPerPixel))
        tfSamplesPerPixel = 1;
    if (!TIFFGetField(tfFile, TIFFTAG_PLANARCONFIG, &tfPlanarConfiguration))
        tfPlanarConfiguration = PLANARCONFIG_CONTIG;
    if (!TIFFGetField(tfFile, TIFFTAG_GRAYRESPONSEUNIT, &tfGrayResponseUnit))
        tfGrayResponseUnit = 2;

    tfUnitMap = tfGrayResponseUnitMap[tfGrayResponseUnit];
    colormapSize = 1 << tfBitsPerSample;
    tfImageDepth = tfBitsPerSample * tfSamplesPerPixel;

    dRed = (double *) malloc(colormapSize * sizeof(double));
    dGreen = (double *) malloc(colormapSize * sizeof(double));
    dBlue = (double *) malloc(colormapSize * sizeof(double));
    MCHECK(dRed); MCHECK(dGreen); MCHECK(dBlue);

    /*
     * If TIFFTAG_PHOTOMETRIC is not present then assign a reasonable default.
     * The TIFF 5.0 specification doesn't give a default.
     */
    if (!TIFFGetField(tfFile, TIFFTAG_PHOTOMETRIC,
            &tfPhotometricInterpretation)) {
        if (tfSamplesPerPixel != 1)
            tfPhotometricInterpretation = PHOTOMETRIC_RGB;
        else if (tfBitsPerSample == 1)
            tfPhotometricInterpretation = PHOTOMETRIC_MINISBLACK;
        else if (TIFFGetField(tfFile, TIFFTAG_COLORMAP,
                &redMap, &greenMap, &blueMap)) {
            tfPhotometricInterpretation = PHOTOMETRIC_PALETTE;
            free(redMap); free(greenMap); free(blueMap);
            redMap = greenMap = blueMap = NULL;
        } else
            tfPhotometricInterpretation = PHOTOMETRIC_MINISBLACK;
    }

    /*
     * Given TIFFTAG_PHOTOMETRIC extract or create the response curves.
     */
    switch (tfPhotometricInterpretation) {
    case PHOTOMETRIC_RGB:
        if (!TIFFGetField(tfFile, TIFFTAG_COLORRESPONSECURVE,
                &redMap, &greenMap, &blueMap)) {
            redMap = (u_short *) malloc(colormapSize * sizeof(u_short));
            greenMap = (u_short *) malloc(colormapSize * sizeof(u_short));
            blueMap = (u_short *) malloc(colormapSize * sizeof(u_short));
            MCHECK(redMap); MCHECK(greenMap); MCHECK(blueMap);
            for (i = 0; i < colormapSize; i++)
                dRed[i] = dGreen[i] = dBlue[i]
                    = (double) SCALE(i, colormapSize - 1);
        } else {
            CheckAndCorrectColormap();
            for (i = 0; i < colormapSize; i++) {
                dRed[i] = (double) redMap[i];
                dGreen[i] = (double) greenMap[i];
                dBlue[i] = (double) blueMap[i];
            }
        }
        break;
    case PHOTOMETRIC_PALETTE:
        if (!TIFFGetField(tfFile, TIFFTAG_COLORMAP,
                &redMap, &greenMap, &blueMap)) {
            redMap = (u_short *) malloc(colormapSize * sizeof(u_short));
            greenMap = (u_short *) malloc(colormapSize * sizeof(u_short));
            blueMap = (u_short *) malloc(colormapSize * sizeof(u_short));
            MCHECK(redMap); MCHECK(greenMap); MCHECK(blueMap);
            for (i = 0; i < colormapSize; i++)
                dRed[i] = dGreen[i] = dBlue[i]
                    = (double) SCALE(i, colormapSize - 1);
        } else {
            CheckAndCorrectColormap();
            for (i = 0; i < colormapSize; i++) {
                dRed[i] = (double) redMap[i];
                dGreen[i] = (double) greenMap[i];
                dBlue[i] = (double) blueMap[i];
            }
        }
        break;
    case PHOTOMETRIC_MINISWHITE:
        redMap = (u_short *) malloc(colormapSize * sizeof(u_short));
        greenMap = (u_short *) malloc(colormapSize * sizeof(u_short));
        blueMap = (u_short *) malloc(colormapSize * sizeof(u_short));
        MCHECK(redMap); MCHECK(greenMap); MCHECK(blueMap);
        if (!TIFFGetField(tfFile, TIFFTAG_GRAYRESPONSECURVE, &grayMap))
            for (i = 0; i < colormapSize; i++)
                dRed[i] = dGreen[i] = dBlue[i]
                    = (double) SCALE(colormapSize - 1 - i, colormapSize - 1);
        else {
            dRed[colormapSize - 1] = dGreen[colormapSize - 1]
                = dBlue[colormapSize - 1] = 0;
            for (i = 0; i < colormapSize - 1; i++)
                dRed[i] = dGreen[i] = dBlue[i] = 65535.0 -
                    (65535.0 * pow(10.0, (double) grayMap[i] / tfUnitMap));
        }
        break;
    case PHOTOMETRIC_MINISBLACK:
        redMap = (u_short *) malloc(colormapSize * sizeof(u_short));
        greenMap = (u_short *) malloc(colormapSize * sizeof(u_short));
        blueMap = (u_short *) malloc(colormapSize * sizeof(u_short));
        MCHECK(redMap); MCHECK(greenMap); MCHECK(blueMap);
        if (!TIFFGetField(tfFile, TIFFTAG_GRAYRESPONSECURVE, &grayMap))
            for (i = 0; i < colormapSize; i++)
                dRed[i] = dGreen[i] = dBlue[i]
                    = (double) SCALE(i, colormapSize - 1);
        else {
            dRed[0] = dGreen[0] = dBlue[0] = 0;
            for (i = 1; i < colormapSize; i++)
                dRed[i] = dGreen[i] = dBlue[i] =
                    (65535.0 * pow(10.0, (double) grayMap[i] / tfUnitMap));
        }
        break;
    default:
        fprintf(stderr,
            "%s: can't display photometric interpretation type %d\n",
            programName, tfPhotometricInterpretation);
        exit(0);
    }

    return;
}

/*
 * Many programs get TIFF colormaps wrong.  They use 8-bit colormaps instead of
 * 16-bit colormaps.  This function is a heuristic to detect and correct this.
 */
void
CheckAndCorrectColormap()
{
    register int i;

    for (i = 0; i < colormapSize; i++)
        if ((redMap[i] > 255) || (greenMap[i] > 255) || (blueMap[i] > 255))
            return;

    for (i = 0; i < colormapSize; i++) {
       redMap[i] = SCALE(redMap[i], 255);
       greenMap[i] = SCALE(greenMap[i], 255);
       blueMap[i] = SCALE(blueMap[i], 255);
    }

    return;
}

void
SimpleGammaCorrection()
{
    register int i;

    for (i = 0; i < colormapSize; i++) {
        if (((tfPhotometricInterpretation == PHOTOMETRIC_MINISWHITE)
            && (i == colormapSize - 1))
            || ((tfPhotometricInterpretation == PHOTOMETRIC_MINISBLACK)
            && (i == 0)))
            redMap[i] = greenMap[i] = blueMap[i] = 0;
        else {
            redMap[i] = ROUND((pow(dRed[i] / 65535.0, 1.0 / dGamma) * 65535.0));
            greenMap[i] = ROUND((pow(dGreen[i] / 65535.0, 1.0 / dGamma) * 65535.0));
            blueMap[i] = ROUND((pow(dBlue[i] / 65535.0, 1.0 / dGamma) * 65535.0));
        }
    }

    free(dRed); free(dGreen); free(dBlue);
    return;
}

void
InitializeX(xargc, xargv)
    int xargc;
    char **xargv;
{
    register int i;
    int win_x = 0, win_y = 0, win_w = tfImageWidth, win_h = tfImageHeight;
    u_int border_width = 3;
    XSizeHints size_hints;
    XSetWindowAttributes window_attributes;
    XVisualInfo template_visual;
    register XColor *colors = NULL;

    if ((xDisplay = XOpenDisplay(xDisplayName)) == NULL) {
        fprintf(stderr, "%s: cannot connect to X server %s\n",
            programName, XDisplayName(xDisplayName));
        exit(0);
    }

    XSetErrorHandler(XTiffErrorHandler);
    XSynchronize(xDisplay, xSyncFlag);
    xScreen = DefaultScreen(xDisplay);

    if (tfImageDepth != 1) {
        template_visual.screen = xScreen;
        xVisualList = XGetVisualInfo(xDisplay, VisualScreenMask,
            &template_visual, &xNVisuals);

        if (xNVisuals == 0) {
            fprintf(stderr, "%s: visual list not available\n", programName);
            exit(0);
        }
    }

    switch (tfImageDepth) {
    /*
     * X really wants a 32-bit image with the fourth channel unused,
     * but the visual structure thinks it's 24-bit.  bitmap_unit is 32.
     */
    case 32:
    case 24:
        if (!SearchVisualList(24, DirectColor, &xVisual)) {
            fprintf(stderr, "%s: 24-bit DirectColor visual not available\n",
                programName);
            exit(0);
        }

        colors = (XColor *) malloc(3 * colormapSize * sizeof(XColor));
        MCHECK(colors);

        for (i = 0; i < colormapSize; i++) {
            colors[i].pixel = (u_long) (i << 16) + (i << 8) + i;
            colors[i].red = redMap[i];
            colors[i].green = greenMap[i];
            colors[i].blue = blueMap[i];
            colors[i].flags = DoRed | DoGreen | DoBlue;
        }

        xColormap = XCreateColormap(xDisplay, RootWindow(xDisplay, xScreen),
            xVisual, AllocAll);
        XStoreColors(xDisplay, xColormap, colors, colormapSize);
        break;
    case 8:
    case 4:
    case 2:
        /*
         * We assume that systems with 24-bit visuals also have 8-bit visuals.
         * We don't promote from 8-bit PseudoColor to 24/32 bit DirectColor.
         */
        switch (tfPhotometricInterpretation) {
        case PHOTOMETRIC_MINISWHITE:
        case PHOTOMETRIC_MINISBLACK:
            if (SearchVisualList((int) tfImageDepth, GrayScale, &xVisual))
                break;
        case PHOTOMETRIC_PALETTE:
            if (SearchVisualList((int) tfImageDepth, PseudoColor, &xVisual))
                break;
        default:
            fprintf(stderr, "%s: Unsupported TIFF/X configuration\n",
                programName);
            exit(0);
        }

        colors = (XColor *) malloc(colormapSize * sizeof(XColor));
        MCHECK(colors);

        for (i = 0; i < colormapSize; i++) {
            colors[i].pixel = (u_long) i;
            colors[i].red = redMap[i];
            colors[i].green = greenMap[i];
            colors[i].blue = blueMap[i];
            colors[i].flags = DoRed | DoGreen | DoBlue;
        }

        /*
         * xtiff's colormap allocation is private.  It does not attempt
         * to detect whether any existing colormap entries are suitable
         * for its use.  This causes colormap flashing.
         */
        xColormap = XCreateColormap(xDisplay, RootWindow(xDisplay, xScreen),
            xVisual, AllocAll);
        XStoreColors(xDisplay, xColormap, colors, colormapSize);
        break;
    case 1:
        xImageDepth = 1;
        xVisual = DefaultVisual(xDisplay, xScreen);
        xColormap = DefaultColormap(xDisplay, xScreen);
        break;
    default:
        fprintf(stderr, "%s: unsupported image depth %d\n",
            programName, tfImageDepth);
        exit(0);
    }

    window_attributes.colormap = xColormap;
    window_attributes.background_pixel = XWhitePixel(xDisplay, xScreen);
    window_attributes.border_pixel = XBlackPixel(xDisplay, xScreen);
    window_attributes.event_mask = ExposureMask | ButtonPressMask;

    /*
     * win_w and win_h are not currently used.
     */
    if (xGeometry != NULL)
        XParseGeometry(xGeometry, &win_x, &win_y, &win_w, &win_h);

    xWindow = XCreateWindow(xDisplay, RootWindow(xDisplay, xScreen),
        win_x, win_y, tfImageWidth, tfImageHeight, border_width,
        xImageDepth == 1 ? CopyFromParent : xImageDepth, InputOutput,
        xVisual, CWColormap | CWBackPixel | CWBorderPixel | CWEventMask,
        &window_attributes);

    xGcValues.function = GXcopy;
    xGcValues.plane_mask = AllPlanes;
    if (tfPhotometricInterpretation == PHOTOMETRIC_MINISWHITE) {
        xGcValues.foreground = XWhitePixel(xDisplay, xScreen);
        xGcValues.background = XBlackPixel(xDisplay, xScreen);
    } else {
        xGcValues.foreground = XBlackPixel(xDisplay, xScreen);
        xGcValues.background = XWhitePixel(xDisplay, xScreen);
    }

    xWinGc = XCreateGC(xDisplay, xWindow, GCFunction | GCPlaneMask
        | GCForeground | GCBackground, &xGcValues);

    xTiffIconPixmap = XCreateBitmapFromData(xDisplay, xWindow,
        xtifficon_bits, xtifficon_width, xtifficon_height);

    size_hints.flags = PPosition | PSize | PMinSize | PMaxSize;
    size_hints.x = win_x; size_hints.y = win_y;
    size_hints.width = size_hints.min_width =
        size_hints.max_width = tfImageWidth;
    size_hints.height = size_hints.min_height =
        size_hints.max_height = tfImageHeight;

    XSetStandardProperties(xDisplay, xWindow, fileName, fileName,
        xTiffIconPixmap, xargv, xargc, &size_hints);

    if (colors != NULL)
        free(colors);

    return;
}

int
SearchVisualList(image_depth, visual_class, visual)
    register int image_depth, visual_class;
    Visual **visual;
{
    register XVisualInfo *vis;
    register int i;

    for (vis = xVisualList, i = 0; i < xNVisuals; vis++, i++) {
        if ((vis->class == visual_class) && (vis->depth >= image_depth)) {
            *visual = vis->visual;
            xImageDepth = vis->depth;
            xRedMask = vis->red_mask;
            xGreenMask = vis->green_mask;
            xBlueMask = vis->blue_mask;
            return True;
        }
    }

    return False;
}

void
GetTIFFImage()
{
    register u_char *scan_line, *output_p, *input_p;
    register int i, j, s;
    int pixel_map[3], red_shift, green_shift, blue_shift;

    tfBytesPerRow = TIFFScanlineSize(tfFile);
    scan_line = (u_char *) malloc(tfBytesPerRow);
    MCHECK(scan_line);

    if ((tfImageDepth == 32) || (tfImageDepth == 24)) {
        output_p = imageMemory = (u_char *)
            malloc(tfImageWidth * tfImageHeight * 4);
        MCHECK(imageMemory);

        /*
         * Handle different color masks for different frame buffers.
         */
        red_shift = pixel_map[0] = xRedMask == 0xFF000000 ? 3
            : (xRedMask == 0xFF0000 ? 2 : (xRedMask == 0xFF00 ? 1 : 0));
        green_shift = pixel_map[1] = xGreenMask == 0xFF000000 ? 3
            : (xGreenMask == 0xFF0000 ? 2 : (xGreenMask == 0xFF00 ? 1 : 0));
        blue_shift = pixel_map[2] = xBlueMask == 0xFF000000 ? 3
            : (xBlueMask == 0xFF0000 ? 2 : (xBlueMask == 0xFF00 ? 1 : 0));

        if (tfPlanarConfiguration == PLANARCONFIG_CONTIG) {
            for (i = 0; i < tfImageHeight; i++) {
                if (TIFFReadScanline(tfFile, scan_line, i, 0) < 0)
                    break;
                for (input_p = scan_line, j = 0; j < tfImageWidth; j++) {
                    *(output_p + red_shift) = *input_p++;
                    *(output_p + green_shift) = *input_p++;
                    *(output_p + blue_shift) = *input_p++;
                    output_p += 4;
                    if (tfSamplesPerPixel == 4) /* skip the fourth channel */
                        input_p++;
                }
            }
        } else {
            for (s = 0; s < tfSamplesPerPixel; s++) {
                if (s == 3)                     /* skip the fourth channel */
                    continue;
                for (i = 0; i < tfImageHeight; i++) {
                    if (TIFFReadScanline(tfFile, scan_line, i, s) < 0)
                        break;
                    input_p = scan_line;
                    output_p = imageMemory + (i*tfImageWidth*4) + pixel_map[s];
                    for (j = 0; j < tfImageWidth; j++, output_p += 4)
                        *output_p = *input_p++;
                }
            }
        }
    } else {
        if (xImageDepth == tfImageDepth) {
            output_p = imageMemory = (u_char *)
                malloc(tfBytesPerRow * tfImageHeight);
            MCHECK(imageMemory);

            for (i = 0; i < tfImageHeight; i++, output_p += tfBytesPerRow)
                if (TIFFReadScanline(tfFile, output_p, i, 0) < 0)
                    break;
        } else if ((xImageDepth == 8) && (tfImageDepth == 4)) {
            output_p = imageMemory = (u_char *)
                malloc(tfBytesPerRow * 2 * tfImageHeight + 2);
            MCHECK(imageMemory);

            /*
             * If a scanline is of odd size the inner loop below will overshoot.
             * This is handled very simply by recalculating the start point at
             * each scanline and padding imageMemory a little at the end.
             */
            for (i = 0; i < tfImageHeight; i++) {
                if (TIFFReadScanline(tfFile, scan_line, i, 0) < 0)
                    break;
                output_p = &imageMemory[i * tfImageWidth];
                input_p = scan_line;
                for (j = 0; j < tfImageWidth; j += 2, input_p++) {
                    *output_p++ = *input_p >> 4;
                    *output_p++ = *input_p & 0xf;
                }
            }
        } else if ((xImageDepth == 8) && (tfImageDepth == 2)) {
            output_p = imageMemory = (u_char *)
                malloc(tfBytesPerRow * 4 * tfImageHeight + 4);
            MCHECK(imageMemory);

            for (i = 0; i < tfImageHeight; i++) {
                if (TIFFReadScanline(tfFile, scan_line, i, 0) < 0)
                    break;
                output_p = &imageMemory[i * tfImageWidth];
                input_p = scan_line;
                for (j = 0; j < tfImageWidth; j += 4, input_p++) {
                    *output_p++ = *input_p >> 6;
                    *output_p++ = (*input_p >> 4) & 3;
                    *output_p++ = (*input_p >> 2) & 3;
                    *output_p++ = *input_p & 3;
                }
            }
        } else if ((xImageDepth == 4) && (tfImageDepth == 2)) {
            output_p = imageMemory = (u_char *)
                malloc(tfBytesPerRow * 2 * tfImageHeight + 2);
            MCHECK(imageMemory);

            for (i = 0; i < tfImageHeight; i++) {
                if (TIFFReadScanline(tfFile, scan_line, i, 0) < 0)
                    break;
                output_p = &imageMemory[i * tfImageWidth];
                input_p = scan_line;
                for (j = 0; j < tfImageWidth; j += 4, input_p++) {
                    *output_p++ = ((*input_p>>6) << 4) | ((*input_p >> 4) & 3);
                    *output_p++ = (((*input_p>>2) & 3) << 4) | (*input_p & 3);
                }
            }
        } else {
            fprintf(stderr,
                "%s: can't handle %d-bit TIFF file on an %d-bit display\n",
                programName, tfImageDepth, xImageDepth);
            exit(0);
        }
    }

    TIFFClose(tfFile);
    free(scan_line);

    return;
}

void
CreateXImage()
{
    xImage = XCreateImage(xDisplay, xVisual, xImageDepth,
        xImageDepth == 1 ? XYBitmap : ZPixmap, /* offset */ 0,
        (char *) imageMemory, tfImageWidth, tfImageHeight,
        /* bitmap_pad */ 8, /* bytes_per_line */ 0);

    /*
     * libtiff converts LSB data into MSB but doesn't change the FillOrder tag.
     */
    if (xImageDepth == 1)
        xImage->bitmap_bit_order = MSBFirst;
    if (xImageDepth <= 8)
        xImage->byte_order = MSBFirst;

    if (xPixmapFlag == True) {
        xImagePixmap = XCreatePixmap(xDisplay, RootWindow(xDisplay, xScreen),
            xImage->width, xImage->height, xImageDepth);

        /*
         * According to the O'Reilly X Protocol Reference Manual, page 53,
         * "A pixmap depth of one is always supported and listed, but windows
         * of depth one might not be supported."  Therefore we create a pixmap
         * of depth one and use XCopyPlane().  This is idiomatic.
         */
        if (xImageDepth == 1) {
            xBitmapGc = XCreateGC(xDisplay, xImagePixmap, 0, &xGcValues);
            XPutImage(xDisplay, xImagePixmap, xBitmapGc, xImage,
                0, 0, 0, 0, xImage->width, xImage->height);
        } else
            XPutImage(xDisplay, xImagePixmap, xWinGc, xImage,
                0, 0, 0, 0, xImage->width, xImage->height);
        XDestroyImage(xImage);
    }

    if (grayMap != NULL)
        free(grayMap);
    if (redMap != NULL)
        free(redMap);
    if (greenMap != NULL)
        free(greenMap);
    if (blueMap != NULL)
        free(blueMap);

    return;
}

void
EventLoop()
{
    XEvent event;

    for (;;) {
        XNextEvent(xDisplay, &event);

        switch (event.type) {
        case Expose:
            if (xPixmapFlag == True) {
                if (xImageDepth == 1)
                    XCopyPlane(xDisplay, xImagePixmap, xWindow, xWinGc,
                        event.xexpose.x, event.xexpose.y,
                        event.xexpose.width, event.xexpose.height,
                        event.xexpose.x, event.xexpose.y, 1);
                else
                    XCopyArea(xDisplay, xImagePixmap, xWindow, xWinGc,
                        event.xexpose.x, event.xexpose.y,
                        event.xexpose.width, event.xexpose.height,
                        event.xexpose.x, event.xexpose.y);
            } else
                XPutImage(xDisplay, xWindow, xWinGc, xImage,
                    event.xexpose.x, event.xexpose.y,
                    event.xexpose.x, event.xexpose.y,
                    event.xexpose.width, event.xexpose.height);
            break;
        case ButtonPress:
            if (xPixmapFlag == False)
                free(imageMemory);
            XFree(xVisualList);
            XFreeGC(xDisplay, xWinGc);
            if (xPixmapFlag == True) {
                if (xImageDepth == 1)
                    XFreeGC(xDisplay, xBitmapGc);
                else
                    XFreeColormap(xDisplay, xColormap);
                XFreePixmap(xDisplay, xImagePixmap);
            } else {
                if (xImageDepth != 1)
                    XFreeColormap(xDisplay, xColormap);
                XDestroyImage(xImage);
            }
            XFreePixmap(xDisplay, xTiffIconPixmap);
            XDestroyWindow(xDisplay, xWindow);
            XCloseDisplay(xDisplay);
            exit(0);
        default:
            break;
        }
    }
}

int
XTiffErrorHandler(display, error_event)
    Display *display;
    XErrorEvent *error_event;
{
    char message[80];

    /*
     * Some X implementations have limitations on the size of pixmaps.
     */
    if ((error_event->error_code == BadAlloc)
            && (error_event->request_code == X_CreatePixmap))
        fprintf(stderr, "%s: requested pixmap too big for display\n",
            programName);
    else {
        XGetErrorText(display, error_event->error_code, message, 80);
        fprintf(stderr, "%s: error code %s\n", programName, message);
    }

    exit(0);
}
