/***************************************************************************
 * Class:  Sketchpad implementation
 * Author: Mark Roseman
 * 
 * Bitmap sketchpad glyph.  Based on original prototype.
 * 
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 06/15/92 MR        initial version
 * 10/22/92 MR        various modifications to get fixed, added color
 * 
 ****************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <gk/sketchpad.h>

#include <InterViews/bitmap.h>
#include <InterViews/stencil.h>
#include <InterViews/style.h>
#include <InterViews/patch.h>
#include <InterViews/layout.h>
#include <InterViews/event.h>
#include <InterViews/session.h>
#include <InterViews/display.h>
#include <InterViews/canvas.h>
#include <stdio.h>
#include <gk/groupkit.h>
#include <gk/conference.h>
#include <gk/infoconn.h>
#include <gk/reader.h>
#include <gk/straction.h>
#include <gk/msgsender.h>
#include <IV-look/kit.h>
#include <OS/string.h>
#include <InterViews/color.h>

declareStrActionCallback(Sketchpad);
implementStrActionCallback(Sketchpad);
	
/**************************************************************************
 * 
 * constructor.  create a bitmap the size of the inside glyph and wrap it
 * in a stencil.  the stencil is then placed on top of the inside glyph
 * using the InterViews overlay, and the whole thing wrapped in a 
 * fixed_span so the size can't be changed (hard to expand a bitmap
 * unfortunately).  note the vcenter around the inside glyph -- this changes
 * the alignment so that the bitmap and inside glyph actually line up on
 * top of each other.
 *
 **************************************************************************/

Sketchpad::Sketchpad(ActiveHandler* inside, Style* style,
		     Conference* conf) : Overlay(inside, style, conf) {
  LayoutKit& layout = *LayoutKit::instance();
  display_ = Session::instance()->default_display();
  Requisition req;
  inside->request(req);

  bitmap_ = new Bitmap( (const void*)0L, 
		       display_->to_pixels(req.x_requirement().natural()), 
		       display_->to_pixels(req.y_requirement().natural()),
		       0, 0);
  
  String sketch_color;
  style->find_attribute( "SketchOverlay-foreground", sketch_color );
  const Color* clr = Color::lookup( Session::instance()->default_display(), 
				   sketch_color );
  patch_ = new Patch ( new Stencil ( bitmap_, 
				    (clr==nil? new Color(1,1,1) : clr)));

  body( layout.fixed_span(  
			  layout.overlay( layout.vcenter(inside,1.0), patch_),
			  req.x_requirement().natural(),
			  req.y_requirement().natural() 
			  ));

  conf->connections()->callbacks()->insert( SCRIBBLE, 
					   new StrActionCallback(Sketchpad)
					   (this, &Sketchpad::remoteScribble));
  lastx_ = lasty_ = -1.0;
}


/**************************************************************************
 * 
 * dX and dY return the glyph's displacement from the origin (fudge factor
 * we need to get things lined up right)
 *
 **************************************************************************/

Coord Sketchpad::dX(void) {
	return patch_->allocation().allotment(Dimension_X).begin();
}

Coord Sketchpad::dY(void) {
	return patch_->allocation().allotment(Dimension_Y).begin();
}

#define MOUSEX  display_->to_pixels(e.pointer_x() - dX())
#define MOUSEY  display_->to_pixels(e.pointer_y() - dY())

/**************************************************************************
 * 
 * "draw" the line into the bitmap - call the bresenham's routine and then 
 * damage that area of the canvas so it will be redrawn
 *
 **************************************************************************/

void Sketchpad::doLine(int x0, int y0, int x1, int y1) {
  bresline(x0,y0,x1,y1);
  patch_->canvas()->damage( 
			   display_->to_coord((x0<x1?x0:x1))+dX()-1,
			   display_->to_coord((y0<y1?y0:y1))+dY()-1,
			   display_->to_coord((x0<x1?x1:x0))+dX()+1,
			   display_->to_coord((y0<y1?y1:y0))+dY()+1);
}

/**************************************************************************
 * 
 * remote user has drawn a line
 *
 **************************************************************************/

void Sketchpad::remoteScribble(char *s) {
  int x0, y0, x1, y1;
  sscanf(s, "%d:%d:%d:%d", &x0, &y0, &x1, &y1);
  doLine(x0, y0, x1, y1);
}


/**************************************************************************
 * 
 * mouse button pressed
 *
 **************************************************************************/

void Sketchpad::press(const Event &e) {
  lastx_ = MOUSEX;
  lasty_ = MOUSEY;
}


/**************************************************************************
 * 
 * drag event - draw the line and tell remote users about it.  note we're
 *   just using sprintf/sscanf rather than attribute lists for transmitting
 *
 **************************************************************************/

void Sketchpad::drag(const Event &e) { 
  char s[30];
  if( lastx_ != -1)  {
    doLine((int)lastx_, (int)lasty_, MOUSEX, MOUSEY);
    sprintf(s, "%d:%d:%d:%d", (int)lastx_, (int)lasty_, (int)MOUSEX, 
	    (int)MOUSEY);
    conference()->connections()->toAll( new StrMsgSender( SCRIBBLE, s) );
  }
  lastx_ = MOUSEX;
  lasty_ = MOUSEY;
}


/**************************************************************************
 * 
 * release mouse button
 *
 **************************************************************************/

void Sketchpad::release(const Event &e) { 
  lastx_ = lasty_ = -1.0;
}



/**************************************************************************
 * 
 * draw the line into the bitmap
 *
 **************************************************************************/

void Sketchpad::bresline( int x1, int y1, int x2, int y2 )
{
#define PLOTPOINT(x,y)  bitmap_->poke(1,x,y)

    register int p, dy, dx, sign;
    int ddy, dydx, err;
    int xs=0, ys=0;

    if ( (y2-y1) < 0 ) {		/* if line is being drawn downwards */
	err=x1; x1=x2; x2=err;		/* reverse coords so it is being    */
	err=y1; y1=y2; y2=err;		/* drawn upwards		    */
    }
    dx = x2 - x1;
    dy = y2 - y1;
    sign = dx;
    dx = ((dx<0) ? -dx:dx);

    if ( dx > dy ) {

	if ( (dx+1) & 1 ) {		/* if odd number of pidxls	*/
	    p = (dy << 1) - dx;		/* initial p value		*/
	     PLOTPOINT( x1, y1 );
	    if (sign > 0) x1++;
	    else	  x1--;
	    dx--;
	    if ( p>=0 ) {		/* adjust values accordingly to */
	    	dy--;  y1++;		/* make a line with an even     */
	    }				/* number of pixels		*/
	}

	ddy = (dy << 1);		/* constant 2*dy		*/
	dydx = (dy - dx) << 1;		/* constant 2*(dy-dx)		*/
	p = (dy << 1) - dx;		/* initial p value		*/

	for( ; xs < dx ; xs++, dx-- ) {
	    if (sign > 0) {
		 PLOTPOINT( x1+xs, y1+ys );
		 PLOTPOINT( x2-xs, y2-ys );
	    } else {
		 PLOTPOINT( x1-xs, y1+ys );
		 PLOTPOINT( x2+xs, y2-ys );
	    }
	    if ( p<0 )
	    	p += ddy;
	    else {
	    	ys++;
	    	p += dydx;
	    }
    	}
    }
    else {
/*____________________________________________________________________________
              reflect the code above about this line to get the code below  */

	if ( (dy+1) & 1 ) {		/* if odd number of pidxls	*/
	    p = (dx << 1) - dy;		/* initial p value		*/
	     PLOTPOINT( x1, y1 );
	    if ( p>=0 ) {
		if (sign > 0) {  x1++; dx--;  }
		else          {  x1--; dx--;  }
	    }
	    y1++; dy--;
	}

	ddy = (dx << 1);
	dydx = (dx - dy) << 1;
	p = (dx << 1) - dy;
	for( ; ys < dy ; ys++, dy-- ) {
	    if (sign > 0) {
		 PLOTPOINT( x1+xs, y1+ys );
		 PLOTPOINT( x2-xs, y2-ys );
	    } else {
		 PLOTPOINT( x1-xs, y1+ys );
		 PLOTPOINT( x2+xs, y2-ys );
	    }

	    if ( p<0 )
	    	p += ddy;
	    else {
	    	xs++;
	    	p += dydx;
	    }
    	}
    }
}




