/***************************************************************************
 *
 * Class:  Conference implementation
 * Author: Mark Roseman
 *
 *
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 02/18/92 MR        initial version
 * 05/17/92 MR        finished up joining / leaving routines
 * 05/29/92 MR        redid to use CallbackRpcReader etc.
 * 06/08/92 MR        heavily redone to run as part of separate process not
 *                    connected to coordinator; connects up to coordinator
 *                    via communications now
 * 08/14/92 MR        clean up
 * 09/17/92 MR        added callback for reader connection close
 * 10/22/92 MR        fixed bug where our info message was needed before
 *                       we got the info - now such messages are queued in
 *                       the "info_needed_" list
 *
 ***************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <gk/reader.h>
#include <gk/writer.h>
#include <gk/conference.h>
#include <gk/infoconn.h>
#include <gk/msgsender.h>
#include <gk/groupkit.h>
#include <gk/straction.h>
#include <gk/rpcaction.h>
#include <gk/connaction.h>
#include <gk/confglyph.h>
#include <gk/groupsession.h>
#include <gk/attrlist.h>

#include <Dispatch/rpcstream.h>
#include <Dispatch/rpchdr.h>
#include <InterViews/session.h>
#include <InterViews/window.h>
#include <OS/list.h>

#include <iostream.h>
#include <stdio.h>
#include <strings.h>
#include <osfcn.h>
#include <stdlib.h>

declareStrActionCallback(Conference);
implementStrActionCallback(Conference);

declareConnActionCallback(Conference);
implementConnActionCallback(Conference);

declareRpcActionCallback(Conference);
implementRpcActionCallback(Conference);

implementPtrList(ConferenceMonitorList, ConferenceMonitor);

declareList(IntList,int);
implementList(IntList,int);


/****************************************************************************
 *
 * Constructor.  Create a list of connections for other conferences to hook
 * up to, and begin listening for connections.  Connect up to the coordinator
 * that spawned us.  Also, create a list of monitors (objects that wish to
 * receive info about users joining and leaving).
 *
 ****************************************************************************/

Conference::Conference(char* /* name */, const char* host, int port, 
		       int confnum) : 
      RpcPeer("/dev/null"), conferenceID_(confnum) 
{
  monitors_ = new ConferenceMonitorList();
  our_attributes_ = new AttributeList();
  info_needed_ = new IntList();
  startListening();
  conns_ = new InfoConnectionList( INFOMSG, new ConnActionCallback(Conference)
                                           (this, &Conference::info_callback));
  if(!createReaderAndWriter(host,port)) 
    exit(-1);
}


/****************************************************************************
 *
 * The first message that other conferences will send us after connecting up
 * should be an INFOMSG, which is caught by this routine.  The message
 * includes name, etc. as well as the ID of the remote conference user, which
 * is needed for other transactions.  When it is determined, we then 
 * "officially" have a new user.
 *
 ****************************************************************************/

void Conference::info_callback(char *s, Connection* c) {
  char id[80];
  AttributeList* al = AttributeList::read(s);
  if( al->find_attribute( "usernum", id))
    c->id_ = atoi(id);
  newUser(al);
}


/****************************************************************************
 *
 * Delete the conference.  
 *
 ****************************************************************************/

Conference::~Conference() {
  delete conns_;
  delete monitors_;
  delete our_attributes_;
  delete info_needed_;
}



/****************************************************************************
 * 
 * Connect up to the coordinator that originally created us.   Send our
 * info message giving our conference ID as well as address, and set up
 * callbacks for the messages we can receive from the coordinator.
 *
 ****************************************************************************/

boolean Conference::createReaderAndWriter(const char* host, int port) {
  char s[200], t[80];
  coord_ = new Connection(host,port);
  if (coord_->writer()->server()) {
    coord_->reader()->closeCallback( new RpcActionCallback(Conference)
				    (this, &Conference::connectionClose));

    AttributeList al;
    sprintf(t, "%d", conferenceID_ );
    al.attribute("confnum", t );
    al.attribute("host", GroupSession::host_name());
    sprintf(t, "%d", _lPort);
    al.attribute("port", t);
    al.write(s);
    coord_->writer()->sendMsg (INFOMSG, s);

    coord_->callbacks()->insert( CONNECTTO, new StrActionCallback(Conference)
				(this, &Conference::connectTo));
    coord_->callbacks()->insert( YOUAREID, new StrActionCallback(Conference)
				(this, &Conference::youAreID));
    coord_->callbacks()->insert( DELUSER, new StrActionCallback(Conference)
				(this, &Conference::removeUser));
    coord_->callbacks()->insert( DELCONF, new StrActionCallback(Conference)
				(this, &Conference::deleteConference));
    return true;
  } else {
    fprintf(stderr, "Conference: could not connect to Coordinator (%s, %d)\n",
	    host, port);
    return false;
  }
}


/****************************************************************************
 * 
 * Connect up to another conference object.  After doing the connection, we
 * also process the remote user (since we're given enough information to do 
 * so).
 *
 ****************************************************************************/

void Conference::connectTo(char *s) {
  char name[80], uid[80], host[80], t[200], port[80], id[80];
  AttributeList* al = AttributeList::read(s);
  al->find_attribute("host", host);
  al->find_attribute("port", port);
  al->find_attribute("id", id);
  al->find_attribute("username", name);
  al->find_attribute("userid", uid);
  Connection* c = conns_->add(host, atoi(port), atoi(id));

  if (c != nil) {
    c->reader()->closeCallback ( new RpcActionCallback(Conference)
				(this, &Conference::connectionClose));

    /*
     * send our info message to them - we can only do this if we've received
     * our info though.  if not stick them in a list, and when we receive our
     * info we can send it to them
     */
    
    our_attributes_->write(t);
    if(strlen(t)==0) 
      info_needed_->append( atoi(id) );
    else
      conns_->sendTo( atoi(id), new StrMsgSender( INFOMSG, t) );

    newUser(al);
  } else {
    fprintf(stderr, "Conference: could not connect to conference run by %s (%s) on port %d of %s\n", name, uid, atoi(port), host);
    fprintf(stderr, "Conference: will attempt to continue\n");
  }
}


/****************************************************************************
 *
 * The coordinator has given us some necessary info, including our ID that
 * the registrar assigned.  We're now "official".  If anyone is waiting
 * for our INFOMSG, send it to them now.
 *
 ****************************************************************************/

void Conference::youAreID(char *s) {
  char port[80];
  our_attributes_ = AttributeList::read(s);
  our_attributes_->attribute("host", GroupSession::host_name());
  our_attributes_->attribute("port", sprintf(port, "%d", _lPort));
  char t[1000];
  our_attributes_->write(t);
  newUser( our_attributes_);

  for ( ListItr(IntList) i(*info_needed_); i.more(); i.next()) {
    conns_->sendTo( i.cur(), new StrMsgSender( INFOMSG, t ));
  }
}


/****************************************************************************
 *
 * Remove the particular user from the conference.  Nuke the conference if
 * it is us.
 *
 ****************************************************************************/

void Conference::removeUser(char *s) {
  char id[80];
  AttributeList* al = AttributeList::read(s);
  if(al->find_attribute("usernum", id)) {
    userLeaving(atoi(id));
    if (atoi(id)==localID())
      deleteConference("");
  }
}


/****************************************************************************
 * 
 * Delete the conference object.  Could probably be a bit more graceful
 * about this.
 *
 ****************************************************************************/

void Conference::deleteConference(char *) {
  Session::instance()->quit();
}


/****************************************************************************
 *
 * A remote conference user has connected up to us.
 *
 ****************************************************************************/

void Conference::createReaderAndWriter(int fd) {
  Connection* c = conns_->add(fd);
  c->reader()->closeCallback ( new RpcActionCallback(Conference)
			      (this, &Conference::connectionClose));
}



/****************************************************************************
 *
 * A new user has joined the conference, and we have all the necessary info
 * to proceed (e.g. ID number from the registrar).  Go through the list of
 * conference monitors and inform them.  The user is already in the list from
 * the initial socket connections.
 *
 ****************************************************************************/

void Conference::newUser(AttributeList* al) {
  for ( ListItr(ConferenceMonitorList) i(*monitors_); i.more(); i.next()) {
    ConferenceMonitor* mon = i.cur();
    mon->newUser( al );
  }
}


/****************************************************************************
 *
 * A user is leaving.  Tell all the conference monitors about it, then remove
 * the user from the list. 
 *
 ****************************************************************************/

void Conference::userLeaving(int id) {
  for ( ListItr(ConferenceMonitorList) i(*monitors_); i.more(); i.next()) {
    ConferenceMonitor* mon = i.cur();
    mon->userLeaving(id);
  }
  conns_->list()->remove( conns_->find(id) );
}


/****************************************************************************
 *
 * A bunch of wrappers to return our instance variables.
 *
 ****************************************************************************/

int Conference::localID() { 
  char id[80];
  if(our_attributes_->find_attribute("usernum", id))
    return atoi(id);
  else
    return -1;
}

InfoConnectionList* Conference::connections() { return conns_; }
ConferenceMonitorList* Conference::monitors() { return monitors_; }


/****************************************************************************
 *
 * A connection (either to one of the other conferences or to the 
 * coordinator has died.  Do something sensible here!
 *
 ****************************************************************************/

void Conference::connectionClose(class CallbackRpcReader* r, int /* fd */) {
  if( r == coord_->reader() ) {
    coord_ = nil;
  } else {
    for ( ListItr(ConnList) i(*conns_->list()); i.more(); i.next()) {
      Connection* conn = i.cur();
      if( conn->reader() == r ) {
	userLeaving(conn->id_);
	break;
      }
    }
  }
}




