/****************************************************************************
 *
 * Classes: OpenRegClient implementation
 * Author:  Mark Roseman
 *
 * Do an "open" registrar client.  Such a client assumes anyone can join
 * any conference.
 *
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 05/22/92 MR        initial version
 * 08/14/92 MR        clean up
 * 09/20/92 MR        updated to use host/port to find registrar
 *
 ****************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include "openregclient.h"
#include <gk-reg/coordinator.h>
#include "rcdisplay.h"
#include <OS/host.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <gk/groupsession.h>

/****************************************************************************
 *
 * Constructor.
 *
 ****************************************************************************/


OpenRegClient::OpenRegClient(const char* host, int port, Coordinator* c) : 
                  RegistrarClient(host,port,c) {
  newConfPending = myselfJoinPending = false;
}

/****************************************************************************
 * 
 * A new conference has been found.  Update the conference list display,
 * and ask to see the list of users (which should be empty, but anyway..).
 * As well, see if this is a conference that we've asked to be created.
 * If so, we now have its ID and can therefore tell the coordinator to
 * create the conference.  After creation, we also join the conference,
 * since the "open" registration protocol assumes we'll join any conference
 * we create.  The actual join doesn't proceed until after being passed
 * through the registrar, again to get an ID.
 *
 ****************************************************************************/

void OpenRegClient::foundNewConference(AttributeList* al) {
  char conf[80];
  al->find_attribute("confnum", conf);
  conference_tbl_->insert( atoi(conf), al );
  users_tbl_->insert( atoi(conf), new AttrListTable(30));
  display_->updateConference();
  PollUsers( atoi(conf));
  if (newConfPending) {         /* better check would be nice */
    coord_->createConference(al);
    callJoinConference( atoi(conf));
    newConfPending = false;
    myselfJoinPending = true;
  }
}


/****************************************************************************
 *
 * Found a deleted conference.  Our protocol says to obey all delete  
 * conference requests, so after updating the display, the coordinator is
 * asked to remove the conference.
 *
 ****************************************************************************/

void OpenRegClient::foundDeletedConference(int conf) {
  conference_tbl_->remove(conf);
  users_tbl_->remove(conf);
  display_->updateConference();
  if(conf==currentConference) {
    currentConference = -1;
    display_->updateUsers(-1);
  }
  coord_->deleteConference(conf);
}


/****************************************************************************
 *
 * A new user for a conference has been found.  Update the users display if
 * it is displaying that particular conference.  If the new user is ourself,
 * (because we're waiting for the ID the registrar assigns), pass the 
 * information along to the conference via the coordinator.
 *
 ****************************************************************************/

void OpenRegClient::foundNewUser(AttributeList* al) {
  char conf[80], usernum[80];
  AttrListTable* usrs;
  al->find_attribute("confnum", conf);
  al->find_attribute("usernum", usernum);
  if( users_tbl_->find( usrs, atoi(conf))) {
    usrs->insert( atoi(usernum), al);
    if(atoi(conf)==currentConference)
      display_->updateUsers(atoi(conf));
    if (myselfJoinPending) {
      coord_->setLocalInfo(al);
      myselfJoinPending = false; 
    }
  }
}


/****************************************************************************
 *
 * A deleted user has been found.  Ask the coordinator to delete the user
 * from the conference (if we're running the conference).  Again, we're
 * going to blindly listen to all delete user requests.
 *
 ****************************************************************************/

void OpenRegClient::foundDeletedUser(int conf, int user) {
  AttrListTable* users;
  users_tbl_->find( users, conf );
  users->remove( user );
  if(conf==currentConference)
    display_->updateUsers(currentConference);
  coord_->deleteUser(conf,user);
}


/****************************************************************************
 *
 * The user has asked to create a new conference of a given name and type.  
 * Ask the registrar to create the conference.  We can't actually proceed
 * with creating it until the registrar assigns it an ID, so we set up the
 * newConfPending flag, which is used by the foundNewConference() method
 * to actually create the conference.
 *
 ****************************************************************************/

void OpenRegClient::requestNewConference(char* name, char* type) {
  AttributeList al;
  al.attribute("name", name);
  al.attribute("type", type);
  callNewConference( &al );
  newConfPending = true;
}


/****************************************************************************
 *
 * The user has asked to join the conference.  Under this scheme, assume
 * we can join any conference.  First, tell the registrar to join us to the
 * conference.  Second, actually create the conference component.  Finally,
 * initiate a connection to all existing users in the conference.
 *
 ****************************************************************************/

void OpenRegClient::requestJoinConference(int conf) {
  char s[80], name[80], type[80];
  callJoinConference(conf);
  myselfJoinPending = true;
  AttributeList al, *cl;
  if ( conference_tbl_->find( cl, conf) ) {
    cl->find_attribute( "name", name);   al.attribute("name", name);
    cl->find_attribute( "type", type);   al.attribute("type", type);
    al.attribute("confnum", sprintf(s, "%d", conf));
    coord_->createConference( &al );
    
    AttrListTable* users;
    users_tbl_->find(users, conf);
    for(TableIterator(AttrLstTbl) i(*users); i.more(); i.next()) 
      coord_->joinTo(i.cur_value());
  }
}

/****************************************************************************
 *
 * The user has asked to leave the conference.  Tell the registrar we want
 * to leave, and then also ask the coordinator to inform the conference.
 * We should check if we're the last user of the conference, and if so, tell
 * the registrar to delete the conference.
 *
 *   *** NEEDS TO BE REDONE ***
 *
 ****************************************************************************/

void OpenRegClient::requestLeaveConference(int conf) {
  char port[80], host[80], usernum[80];
  AttrListTable* usrs;
  users_tbl_->find( usrs, conf );
  for ( TableIterator(AttrLstTbl) i(*usrs); i.more(); i.next()) {
    i.cur_value()->find_attribute("host", host);
    i.cur_value()->find_attribute("port", port);
    if ( (atoi(port)==lPort()) && 
	(strcmp(host, GroupSession::host_name())==0)) {
      i.cur_value()->find_attribute("usernum", usernum);
      callLeaveConference( conf, atoi(usernum));
    }
  }
  PollUsers(conf);
}



/****************************************************************************
 *
 * The user has asked to view the user list for the indicated conference.
 * Update the display.
 *
 ****************************************************************************/

void OpenRegClient::requestViewConference(int conf) {
  currentConference = conf;
  display_->updateUsers(conf);
}



/****************************************************************************
 * 
 * Add a display which monitors the conference and user lists.
 *
 ****************************************************************************/

void OpenRegClient::addDisplay(class RegClientDisplay* dsp) {
  display_ = dsp;
}

/****************************************************************************
 * 
 * Hook called by the coordinator to report a user leaving a conference
 * controlled by the coordinator (or the conference crashing).
 *
 ****************************************************************************/

void OpenRegClient::userLeft(int conf_id, int /* user_id */) {
  requestLeaveConference(conf_id);
  AttrListTable* users;
  int numusers = 0;
  users_tbl_->find( users, conf_id );
  for( TableIterator(AttrLstTbl) i(*users); i.more(); i.next()) 
			     numusers++;
  if(numusers < 2)
    callDeleteConference(conf_id);
  PollConferences();			     
}





