/*
linux/kernel/chr_drv/joystick.c
   Copyright (C) 1992, 1993 Arthur C. Smith
   Joystick driver for Linux running on an IBM compatable computer.

VERSION INFO:
01/08/93	ACS	0.1: Works but needs multi-joystick support
01/13/93	ACS	0.2: Added multi-joysitck support (minor 0 and 1)
		  	     Added delay between measuring joystick axis
		   	     Added scaling ioctl
02/16/93	ACS	0.3: Modified scaling to use ints to prevent kernel
			     panics 8-)
02/28/93	ACS	0.4: Linux99.6 and fixed race condition in js_read.
			     After looking at a schmatic of a joystick card
                             it became apparent that any write to the joystick
			     port started ALL the joystick one shots. If the one
			     that we are reading is short enough and the first
		 	     one to be read, the second one will return bad data
			     if it's one shot has not expired when the joystick
			     port is written for the second time. Thus solves
			     the mystery delay problem in 0.2!
04/22/93	ACS/Eyal 0.5:Upgraded the driver to the 99.8 kernel, added
			     joystick support to the make config options,
			     updated the driver to return the buttons as
			     positive logic, and read both axis at once
			     (thanks Eyal)!
12 Feb 94	Eyal	Allow timelimit of zero.
			DO read first time!
			Return all 4 buttons for port 0
			DO return correct buttons for port 1
			Slight cleaning of analog reading loop.
			TIMEOUT is now in timer count units.
			Rename fields to lower case, remove js_ prefix.
			Use a private internal data structure.
			Allow long reads for both sticks at once.
13 Feb 94	Eyal	Fix header "linux/joystick.h".
			For compatibility, I still access 'corr' using the same
			  structure. I'd rather use the struct only for reads.
			js_init() now recognizes complex joysticks.
			Check timeout incrementally, to allow large timeouts
			  without locking the machine up it the stick is
			  disconnected.
		?	Should js_open() enforce exclusion? Just checking the
			  'busy' flag it not enough, I think.
			If js_read() times out, DO release the semaphore.
		?	Is there a need for a better 'corr' logic, like setting
			  a low/high/range value such that:
				reading = the_usual_timer_loop;
				if (reading > high)
					reading = high;
				else if (reading < low)
					reading = low;
				return (reading - low) * range / (high - low);
25 Apr 94	Eyal	Merge in the loadable stuff from 0.7 public release
			(maintained by Carlos Puchol?)
14 Oct 94	Eyal	minor cosmetics.
 4 Feb 95	Eyal	add <linux/mm.h> 1.1.88 needs it.
*/

#include <linux/mm.h>
#include <linux/joystick.h>
#include "release.h"

#define JS_VERSION	"0.7a (Eyal)"

static struct JS_DATA_SAVE_TYPE js_data[JS_MAX];	/* misc data */
static int js_exist = 0;		/* which joysticks' axis exist? */
static int js_read_semaphore = 0;	/* to prevent two processes */
					/* from trying to read different */
					/* joysticks at the same time */
/* First some utilities.
*/

/* get_timer0 ()
 *
 * Returns the current value of timer 0. This is a 16 bit counter that starts
 * at LATCH and counts down to 0
*/
inline int 
get_timer0 (void)
{
	int	t;

	outb (0, PIT_MODE);
	t  = (int)inb (PIT_COUNTER_0);
	t += (int)inb (PIT_COUNTER_0) << 8;
	return (t);
}

#define GET_USERMEM(kernel,user) \
	get_usermem ((char *)&kernel, (char *)user, sizeof(kernel))

#define PUT_USERMEM(kernel,user) \
	put_usermem ((char *)&kernel, (char *)user, sizeof(kernel))

static void
get_usermem (char *kernel, char *user, int size)
{
	verify_area (VERIFY_READ, (void *)user, size);
	while (size-- > 0)
		*kernel++ = get_fs_byte (user++);
}

static void
put_usermem (char *kernel, char *user, int size)
{
	verify_area (VERIFY_WRITE, (void *)user, size);
	while (size-- > 0)
		put_fs_byte (*kernel++, user++);
}

int 
js_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
	unsigned long arg)
{
	unsigned int	minor = MINOR(inode->i_rdev);

	if (MAJOR(inode->i_rdev) != JS_MAJOR)
		return -EINVAL;
	if (minor >= JS_MAX)
		return -ENODEV;
	if ((((inb(JS_PORT) & JS_ALL) >> (minor<<1)) & 0x03) != 0)
		return -ENODEV;		/* js minor does no exist */ 

	switch (cmd) {
	case JS_SET_CAL:	/* from struct *arg to js_data[minor] */
		GET_USERMEM (js_data[minor].corr, arg);
		break;
	case JS_GET_CAL:	/* to struct *arg from js_data[minor] */
		PUT_USERMEM (js_data[minor].corr, arg);
		break;
	case JS_SET_TIMEOUT:
		GET_USERMEM (js_data[minor].timeout, arg);
		break;
	case JS_GET_TIMEOUT:
		PUT_USERMEM (js_data[minor].timeout, arg);
		break;
	case JS_SET_TIMELIMIT:
		GET_USERMEM (js_data[minor].timelimit, arg);
		break;
	case JS_GET_TIMELIMIT:
		PUT_USERMEM (js_data[minor].timelimit, arg);
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

static int
js_test (int mask, int timeout)
{
	int	t, t0, t1;

	cli ();
	t = 0;
	t0 = get_timer0 ();
	while (mask & inb (JS_PORT)) {
		t1 = get_timer0 ();
		t += DELTA_TIME (t0, t1);
		t0 = t1;
		if (t > timeout)
			break;
	}
	t = inb (JS_PORT);
	sti ();

	return (t);
}

static int
js_open (struct inode * inode, struct file * file)
{
	unsigned int	minor = MINOR(inode->i_rdev);
	int		t, mask;

	if (minor >= JS_MAX)
		return -ENODEV;

	t = js_test (JS_ALL, JS_DEF_TIMEOUT);

	mask = minor ? (JS_X_1 | JS_Y_1) : (JS_X_0 | JS_Y_0);

	cli ();		/* block js_read while js_exist is being modified */
	js_exist = ~t & JS_ALL;
	if ((js_exist & mask) != mask) {
		sti ();
		return -ENODEV; 
	}
	sti ();

	if (JS_TRUE == js_data[minor].busy)
		return -EBUSY;
	js_data[minor].busy       = JS_TRUE;

	js_data[minor].corr.x     = JS_DEF_CORR;	/* default scale */
	js_data[minor].corr.y     = JS_DEF_CORR;
	js_data[minor].timeout    = JS_DEF_TIMEOUT;
	js_data[minor].timelimit  = JS_DEF_TIMELIMIT;
	js_data[minor].expiretime = CURRENT_JIFFIES;

	MOD_INC_USE_COUNT;

	return 0;
}

static void
js_release (struct inode * inode, struct file * file)
{
	unsigned int	minor = MINOR(inode->i_rdev);

	inode->i_atime = CURRENT_TIME;
	js_data[minor].busy = JS_FALSE;

	MOD_DEC_USE_COUNT;
}

/* js_read ()
 *
 * reads the buttons x, and y axis from both joysticks if a given interval has
 * expired since the last read or is equal to -1l. The buttons are in port
 * 0x201 in the high nibble. The axis are read by writing to 0x201 and then
 * measuring the time it takes the one shots to clear.
*/
static int
js_read (struct inode * inode, struct file * file, char * buf, int count)
{
	int		j, chk;
	int		t, t0, t1, t_x0, t_y0, t_x1, t_y1;
	char		*c;
	unsigned int	minor = MINOR(inode->i_rdev);
	struct JS_DATA_TYPE		js_short;
	struct JS_DATA_TYPE_LONG	js_long;

	if (!(count == JS_RETURN 
	      || (count == JS_RETURN_LONG && !minor && (js_exist & 0x0c))))
		return -EOVERFLOW;
	inode->i_atime = CURRENT_TIME;

/* Only read if the old data is stale.
*/
	if (CURRENT_JIFFIES >= js_data[minor].expiretime) { 
		j = js_data[minor].timeout;

/* This is just to verify that this port is still active. Most of the time
 * it will execute the loop just once. Pretty sure it is not necessary.
 * Note that the button reading depends on chk being set here.
*/
		chk = js_test (js_exist, j);
		if (chk & js_exist)
			return -ENODEV;		/* joystick not present */

/* Make sure no other proc is using the port.
*/
		while (1) {
			cli ();
			if (!js_read_semaphore) {
				js_read_semaphore++;
				sti ();
				break;
			}
			sti ();
		}

/* Get the buttons.
*/
		chk = (~chk >> 4) & 0x0f;
		js_data[0].buttons = chk;		/* all four */
		js_data[1].buttons = chk >> 2;		/* only two */

/* Now read the stick.
*/
		t_x0 = t_y0 = t_x1 = t_y1 = 0;
		cli ();
		t = 0;
		t0 = get_timer0 ();
		outb (0, JS_PORT);			/* trigger one-shots */
		while (0 != (chk = inb (JS_PORT) & js_exist)) {
			t1 = get_timer0 ();
			t += DELTA_TIME (t0, t1);
			t0 = t1;
			if (t > j)
				break;			/* read timed out */
			if (chk & JS_X_0)
				t_x0 = t;
			if (chk & JS_Y_0)
				t_y0 = t;
			if (chk & JS_X_1)
				t_x1 = t;
			if (chk & JS_Y_1)
				t_y1 = t;
		}
		sti ();

/* Update local data register.
*/
		if (!chk) {
			js_data[0].expiretime = CURRENT_JIFFIES
							+ js_data[0].timelimit;
			js_data[0].x = t_x0 >> js_data[0].corr.x;
			js_data[0].y = t_y0 >> js_data[0].corr.y;
			js_data[1].expiretime = CURRENT_JIFFIES
							+ js_data[1].timelimit;
			js_data[1].x = t_x1 >> js_data[1].corr.x;
			js_data[1].y = t_y1 >> js_data[1].corr.y;
		}

/* Allow other reads to proceed.
*/
		js_read_semaphore = 0;

		if (chk)
			return -ENODEV;			/* read timed out */
	}

/* Deliver data to user memory.
*/
	if (JS_RETURN_LONG == count) {
		js_long.buttons = js_data[minor].buttons;
		js_long.x = js_data[0].x;
		js_long.y = js_data[0].y;
		js_long.x2 = js_data[1].x;
		js_long.y2 = js_data[1].y;
		c = (char *)&js_long;
	} else {
		js_short.buttons = js_data[minor].buttons;
		js_short.x = js_data[minor].x;
		js_short.y = js_data[minor].y;
		c = (char *)&js_short;
	}
	put_usermem (c, buf, count);

	return (count);
}


static struct file_operations js_fops = {
	NULL,			/* js_lseek	*/
	js_read,		/* js_read	*/
	NULL,			/* js_write	*/
	NULL,			/* js_readaddr	*/
	NULL,			/* js_select	*/
	js_ioctl,		/* js_ioctl	*/
	NULL,			/* js_mmap	*/
	js_open,		/* js_open	*/
	js_release,		/* js_release	*/
	NULL			/* js_sync	*/
};

/* this was the static part of this module
*/

#if 0

long
js_init (long kmem_start)
{
	int	t, js_num;

	t = js_test (JS_ALL, JS_DEF_TIMEOUT);

	js_exist = ~t & JS_ALL;	/* get joystick status byte */
	t = 0;
	switch (js_exist) {
	case JS_ALL:	
		js_num = 2;
		break;
	case JS_0 | JS_X_1:
	case JS_0 | JS_Y_1:
	case JS_1 | JS_X_0:
	case JS_1 | JS_Y_0:
		t = 1;
	case JS_0:
	case JS_1:
		js_num = 1;
		break;
	default:		/* No complete joystick */
		t = 1;
	case 0x00:		/* No joystick at all */
		js_num = 0;
		break;
	}

	printk ("js_init: Found %d%s Joystick%c\n",
		js_num, t ? "+" : "", (js_num == 1) ? ' ' : 's');

	if (register_chrdev (JS_MAJOR, "Joystick", &js_fops))
		printk ("Unable to get major=%d for Joystick\n");

	for (t = 0; t < JS_MAX; t++)
		js_data[t].busy = JS_FALSE;

	js_read_semaphore = 0;
	return kmem_start;
}
#endif

/* This is the loadable module appendage.
*/

#ifdef __cplusplus
extern "C" {
#endif

int init_module (void) {
	int	t, js_num;

	t = js_test (JS_ALL, JS_DEF_TIMEOUT);

	js_exist = ~t & JS_ALL;	/* get joystick status byte */
	t = 0;
	switch (js_exist) {
	case JS_ALL:	
		js_num = 2;
		break;
	case JS_0 | JS_X_1:
	case JS_0 | JS_Y_1:
	case JS_1 | JS_X_0:
	case JS_1 | JS_Y_0:
		t = 1;
	case JS_0:
	case JS_1:
		js_num = 1;
		break;
	default:		/* No complete joystick */
		t = 1;
	case 0x00:		/* No joystick at all */
		js_num = 0;
		break;
	}

	printk ("joystick %s: Found %d%s Joystick%c\n", JS_VERSION,
		js_num, t ? "+" : "", (js_num == 1) ? ' ' : 's');

	if (register_chrdev (JS_MAJOR, "Joystick", &js_fops))
		printk ("Unable to get major=%d for Joystick\n",
		JS_MAJOR);
	for (js_num = 0; js_num < JS_MAX; js_num++)
		  js_data[js_num].busy = JS_FALSE;
	js_read_semaphore = 0;
	return 0;
}

void cleanup_module (void)
{
	if (MOD_IN_USE)
		printk ("Joystick: device busy, remove delayed.\n");

	if (unregister_chrdev (JS_MAJOR, "Joystick") != 0) {
		printk ("cleanup_module failed.\n");
	}
	else {
		printk ("cleanup_module succeeded.\n");
	}
}

#ifdef __cplusplus
}
#endif
