/*
**  tout - a personal timeout program
**
**  Andrew Rudoff, University of Colorado, Boulder
**
**  usage: tout minutes
**
**  This is a voluntary timeout program.  A timeout occurs when the user
**  is idle for a time larger than "minutes."  The inaccuracy can be as
**  much as 1 minute (1 minute granularity).  The user is warned 1 minute
**  before the timeout; the timeout can be prevented by typing any key to
**  clear the idle status of the terminal.
*/

#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <sgtty.h>
#include <sys/stat.h>

#define BEEP '\007'

main(argc, argv)

int             argc;
char            **argv;

{

char            user[50];           /* name of current user */
char            invoker[50];        /* name of invoker */
struct          stat statbuf;       /* buffer for stats on tty */
long            current, timeout;   /* times in seconds */
int             pgrp, tmp = 0;

    if (argc != 2)
    {
        fprintf(stderr, "usage: %s minutes\n", argv[0]);
        exit(1);
    }

    nice(4);    /* run gracefully in the background */

    if ((timeout = 60 * atoi(argv[1])) == 0)
    {
        fprintf(stderr, "%s: incomprehensible or zero timeout\n", argv[0]);
        exit(1);
    }

    if (!isatty(0))
    {
        fprintf(stderr, "%s: stdin not a terminal\n", argv[0]);
        exit(1);
    }
        
    strcpy(invoker, getlogin());

    printf("Timeout set for %s (%s minutes)\n", invoker, argv[1]);
    fflush(stdout);

    if (fork())
    {
        exit(0);
    }

    while(1)
    {
        sleep(60);  /* spend most of my spare time sleeping */
        strcpy(user, getlogin());
        
        /* Be sure same person is logged in */
        
        if (strcmp(user, invoker))
        {
            exit(0);
        }

        if ((fstat(0, &statbuf)) != 0)
        {
            fprintf(stderr, "%s: couldn't stat tty\n", argv[0]);
            exit(1);
        }

        time(&current);

        if (current - statbuf.st_atime >= timeout)
        {
            fprintf(stderr, "\n\n%s: Timeout - you're outta here!\n\n",
                argv[0]);

            while(ioctl(0, TIOCGPGRP, &pgrp), pgrp != tmp)
            {
                tmp = pgrp;
                killpg(pgrp, SIGHUP);   /* Let editors save, etc */
                sleep(5);
            }
            
            vhangup();
            exit(0);
        }

        if (current + 60 - statbuf.st_atime >= timeout)
        {
            fprintf(stderr, "\n%s: WARNING - timeout in 1 minute.%c\n",
                argv[0], BEEP);
        }
    }
}
