/*------------------------------------------------------------------------

      File:  cmdModule.c
   Project:  Tcl Modules
   Created:  Tue Oct 22 23:45:57 1991
    Author:  John L. Furlani<john.furlani@East.Sun.COM>
  Revision:  1.3
  Last Mod:  20:19:57, 1/19/92

  Description of File:
		
	
	
	
	
	
	
	
------------------------------------------------------------------------*/
/***********************************************************************
* Copyright 1991 by John L. Furlani.  All rights reserved.
* 
* This material was written by John L. Furlani.
*
* Redistribution and use in source and binary forms are permitted
* provided that this entire copyright notice is duplicated in all such
* copies, and that any documentation, announcements, and other
* materials related to such distribution and use acknowledge that the
* software was developed by John Furlani.  No charge, other than an 
* "at-cost" distribution fee, may be charged for copies, derivations, 
* or distributions of this material without the express written 
* consent of the copyright holder.  The name of the author may not
* be used to endorse or promote products derived from this material 
* without specific prior written permission
*
* THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
* WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
* MERCHANTIBILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
************************************************************************/
static char SccsId[] = "@(#)cmdModule.c	1.3\t1/19/92";

#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include "global.h"
#include <regexp.h>


int
  cmdModule(ClientData client_data,
	    Tcl_Interp* interp,
	    int         argc,
	    char*       argv[])
{
  regexp* addPtr    = regcomp("^ad|^lo");
  regexp* rmPtr     = regcomp("^rm|^de|^er|^rem|^unl");
  regexp* swPtr     = regcomp("^sw");
  regexp* dispPtr   = regcomp("^di|^sh");
  regexp* listPtr   = regcomp("^l");
  regexp* availPtr  = regcomp("^av");
  regexp* helpPtr   = regcomp("^he");
  regexp* initPtr   = regcomp("^i");
  regexp* usePtr    = regcomp("^use");
  regexp* unusePtr  = regcomp("^unu");
  regexp* clearPtr  = regcomp("^cl");
  int     return_val = -1;
  int     store_flags = flags;
  char*   store_curmodule = NULL;

  if(current_module)
    store_curmodule = current_module;
  
  if(regexec(addPtr, argv[1])) {
    return_val = Load_Modules(interp, (argv + 2));
  } else if(regexec(rmPtr, argv[1])) {
    return_val = Unload_Modules(interp, (argv + 2));
  } else if(regexec(swPtr, argv[1])) {
    return_val = Switch_Modules(interp, (argc - 2), (argv + 2));
  } else if(regexec(dispPtr, argv[1])) {
    return_val = Display_Modules(interp, (argv + 2));
  } else if(regexec(listPtr, argv[1])) {
    return_val = List_Modules(interp, (argv + 2));
  } else if(regexec(availPtr, argv[1])) {
    return_val = Available_Modules(interp, (argv + 2));
  } else if(regexec(clearPtr, argv[1])) {
    return_val = Clear_LoadedModules(interp, (argc - 2), (argv + 2));
  } else if(regexec(initPtr, argv[1])) {

    initPtr = regcomp("^inita|^ia");
    if(regexec(initPtr, argv[1])) {
      flags |= M_LOAD;
      return_val = Init_Modules(interp, (argc - 2), (argv + 2));
      flags &= ~M_LOAD;
    }
    
    initPtr = regcomp("^initr|^ir");
    if(regexec(initPtr, argv[1])) {
      flags |= M_REMOVE;
      return_val = Init_Modules(interp, (argc - 2), (argv + 2));
      flags &= ~M_REMOVE;
    }

    initPtr = regcomp("^initl|^il");
    if(regexec(initPtr, argv[1])) {
      flags |= M_DISPLAY;
      return_val = Init_Modules(interp, (argc - 2), (argv + 2));
      flags &= ~M_DISPLAY;
    }

    initPtr = regcomp("^inits|^is");
    if(regexec(initPtr, argv[1])) {
      flags |= M_SWITCH;
      return_val = Init_Modules(interp, (argc - 2), (argv + 2));
      flags &= ~M_SWITCH;
    }

    initPtr = regcomp("^initc|^ic");
    if(regexec(initPtr, argv[1])) {
      flags |= M_CLEAR;
      return_val = Init_Modules(interp, (argc - 2), (argv + 2));
      flags &= ~M_CLEAR;
    }

    initPtr = regcomp("^initp|^ip");
    if(regexec(initPtr, argv[1])) {
      flags |= (M_PREPEND | M_LOAD);
      return_val = Init_Modules(interp, (argc - 2), (argv + 2));
      flags &= ~(M_PREPEND | M_LOAD);
    }
  } else if(regexec(usePtr, argv[1])) {
    return_val = Prepend_ModulePath(interp, (argc - 2), (argv + 2));
  } else if(regexec(unusePtr, argv[1])) {
    return_val = Remove_ModulePath(interp, (argc - 2), (argv + 2));
  } else if(regexec(helpPtr, argv[1])) {
    return_val = ProvideHelp(interp, (argc - 2), (argv + 2));
  }

  flags = store_flags;
  if(store_curmodule)
    current_module = store_curmodule;

  if(return_val < 0) {
    Tcl_AppendResult(interp,
		     "'", argv[1], "' is an Unrecognized subcommand", NULL);
    return TCL_ERROR;
  }

  return return_val;
}

int
  ProvideHelp(Tcl_Interp* interp,
	      int         argc,
	      char*       argv[])
{
    fprintf(stderr, "\n  Available Commands and Usage:\n\n");

    fprintf(stderr, "\t+  add|load\t[modulefile ...]\n");
    fprintf(stderr, "\t+  rm|unload\t[modulefile ...]\n");
    fprintf(stderr, "\t+  switch\toldmodulefile newmodulefile\n");
    fprintf(stderr, "\t+  display\t[modulefile ...]\n");
    fprintf(stderr, "\t+  avail\t[moduledir]\n");
    fprintf(stderr, "\t+  use\t\tdir [dir ...]\n");
    fprintf(stderr, "\t+  unuse\tdir [dir ...]\n");
    fprintf(stderr, "\t+  list\n");
    fprintf(stderr, "\t+  clear\n");
    fprintf(stderr, "\t+  help\n");
    fprintf(stderr, "\t+  initadd\t[modulefile ...]\n");
    fprintf(stderr, "\t+  initrm\t[modulefile ...]\n");
    fprintf(stderr, "\t+  initswitch\toldmodulefile newmodulefile\n");
    fprintf(stderr, "\t+  initlist\n");
    fprintf(stderr, "\t+  initclear\n\n");

/*

  fprintf(stderr, "    * Load the given modulefiles into the current working environment.\n\n");

  fprintf(stderr, "    * Remove the given modules from the current working environment.\n\n");

  fprintf(stderr, "    * Switch two similar modulefiles in place\n\n");

  fprintf(stderr, "    * Display the environment changes each given modulefile makes.\n");
  fprintf(stderr, "        May also display licensing or other usage issues\n\n");

  fprintf(stderr, "    * List the currently loaded modulefiles\n\n");

  fprintf(stderr, "    * Lists all of the modules that are currently available to be loaded\n");
  fprintf(stderr, "    * Modules with multple versions are denoted by a '/' after the name\n");
  fprintf(stderr, "    * Passing one of these as an argument shows the different versions\n\n");

  fprintf(stderr, "    * All of these commands are used to manipulate a 'module load' command\n");
  fprintf(stderr, "         in your shell's .startup file.  They work on the first 'module load'\n");
  fprintf(stderr, "         encountered.\n");
	  
*/

  return TCL_OK;
}


int
  Prepend_ModulePath(Tcl_Interp* interp,
		     int         argc,
		     char*       argv[])
{
  struct stat  stats;
  char*        pathargv[4];
  int          i = 0;
  
  if(argc < 1) {
    Tcl_AppendResult(interp,
		     "Wrong number of arguments:  usage is ", 
		     "'use dir [dir ...]'", NULL);
    return TCL_ERROR;
  }

  pathargv[0] = "prepend-path";
  pathargv[1] = "MODULEPATH";
  pathargv[3] = NULL;
  for(i = 0; i < argc; i++) {
    if(stat(argv[i], &stats) < 0) {
      Tcl_AppendResult(interp, argv[i], " is not found or unreadable", NULL);
      return TCL_ERROR;
    } else if(! S_ISDIR(stats.st_mode))  {
      Tcl_AppendResult(interp, argv[i], " is not a directory", NULL);
      return TCL_ERROR;
    }

    pathargv[2] = argv[i];
    if(cmdSetPath((ClientData) 0, interp, 3, pathargv) == TCL_ERROR)
      return TCL_ERROR;
  }
  
  return TCL_OK;
}

int
  Remove_ModulePath(Tcl_Interp* interp,
		    int         argc,
		    char*       argv[])
{
  struct stat  stats;
  char*        pathargv[4];
  int          i = 0;
  
  if(argc < 1) {
    Tcl_AppendResult(interp,
		     "Wrong number of arguments:  usage is ", 
		     "'unuse dir [dir ...]'", NULL);
    return TCL_ERROR;
  }
  
  pathargv[0] = "remove-path";
  pathargv[1] = "MODULEPATH";
  pathargv[3] = NULL;
  for(i = 0; i < argc; i++) {
    pathargv[2] = argv[i];
    if(cmdRemovePath((ClientData) 0, interp, 3, pathargv) == TCL_ERROR)
      return TCL_ERROR;
  }
  
  return TCL_OK;
}

int
  Clear_LoadedModules(Tcl_Interp* interp,
		      int         argc,
		      char*       argv[])
{
  char         buf[10];
  struct stat  stats;
  char*        clearargv[4];
  int          i = 0;

  if(argc < 1) {
    fprintf(stderr, "\nAre you sure you want to clear all \
loaded modules!? [n] ");
    fgets(buf, 10, stdin);
    if(buf[0] == 'y') {
      clearargv[0] = "setenv";
      clearargv[1] = "LOADEDMODULES";
      clearargv[2] = "";
      clearargv[3] = NULL;
      cmdSetEnv((ClientData) 0, interp, 3, clearargv);
    } else
      fprintf(stderr, "\nLOADEDMODULES was NOT cleared.\n");
  } else {
    fprintf(stderr, "\nAre you sure you want to clear modules out of \
loaded modules!? [n] ");
    fgets(buf, 10, stdin);
    if(buf[0] == 'y') {
      clearargv[0] = "remove-path";
      clearargv[1] = "LOADEDMODULES";
      clearargv[3] = NULL;
      for(i = 0; i < argc; i++) {
	clearargv[2] = argv[i];
	if(cmdRemovePath((ClientData) 0, interp, 3, clearargv) == TCL_ERROR)
	  return TCL_ERROR;
      }
    } else
      fprintf(stderr, "\nLOADEDMODULES was NOT cleared.\n");
  }

  return TCL_OK;
}

void
  SetStartupFiles(void)
{
  if((strcmp("csh", shell_name) == 0) ||
     (strcmp("tcsh", shell_name) == 0)) {
    shell_startups = (char**) malloc(4*sizeof(char*));
    shell_startups[0] = ".cshrc";
    shell_startups[1] = ".csh_variables";
    shell_startups[2] = ".login";
    shell_startups[3] = NULL;
  } else if((strcmp("sh", shell_name) == 0) ||
	    (strcmp("ksh", shell_name) == 0)) {
    shell_startups = (char**) malloc(2*sizeof(char*));
    shell_startups[0] = ".profile";
    shell_startups[1] = NULL;
  } else if((strcmp("bash", shell_name) == 0)) { 
    shell_startups = (char**) malloc(4*sizeof(char*));
    shell_startups[0] = ".bashrc";
    shell_startups[1] = ".bash_env";
    shell_startups[2] = ".bash_profile";
    shell_startups[3] = NULL;
  } else {
    shell_startups = (char**) malloc(1*sizeof(char*));
    shell_startups[0] = NULL;
  }
}

int
  Init_Modules(Tcl_Interp*    interp,
		  int            argc,
		  char**         argv)
{
  static char home_pathname[1064];  /* 1024 + 40 */
  static char home_pathname2[1064];  /* 1024 + 40 */
  regexp*     cookiePtr = regcomp("^[ \t]*module load .*\n$");
  regexp*     modcmdPtr = regcomp("^[ \t]*module load ");
  regexp*     modlstPtr = regcomp("^[^#^\n]*");
  char**      modlist;
  char*       home, *p;
  char*       buffer;
  char        ch, rch;
  FILE*       fileptr, *newfileptr;
  int         i, j;
  int         found_module_list = 0;
  int         found_modload_flag = 0;
  int         matched_module = 0;
  int         shell_num = 0;
  int         nummods, bufsiz = 8192;
  int         home_end, path_end;
  int         sw_found = 0, rm_found = 0;

  if(argc < 1 && !(flags & (M_DISPLAY | M_CLEAR)))
    return TCL_OK;
  
  if(flags & M_SWITCH) {
    argc--;
    if(argc != 1) {
      Tcl_AppendResult(interp,
		       "Wrong number of arguments:  usage is ", 
		       "'initswitch oldmodule newmodule'", NULL);
      return TCL_ERROR;
    }
  }
  
  if((buffer = (char*) malloc(bufsiz * sizeof(char))) == NULL) {
    Tcl_AppendResult(interp, "malloc() failed in Init_Modules()", NULL);
    return TCL_ERROR;
  }
  
  home = getenv("HOME");
  
  if(home) home_end = strlen(home);
  else {
    Tcl_AppendResult(interp, "HOME Environment variable is not set", NULL);
    return TCL_ERROR;
  }
  
  /*
    Put HOME into a buffer and store where the end of HOME is
    for quick contatination of the shell startup files.
    */
  strcpy(home_pathname, home);
  home_pathname[home_end++] = '/';
  home_pathname[home_end] = '\0';
  
  SetStartupFiles();
  
  while(shell_startups[shell_num]) {
    strcpy(&home_pathname[home_end], shell_startups[shell_num]);
    if((fileptr = fopen(home_pathname, "r")) == NULL) 
      {
	shell_num++;
	continue;
      }
    
    path_end = (home_end + strlen(shell_startups[shell_num]));
    strcpy(&home_pathname[path_end], "-NEW");
    
    if((newfileptr = fopen(home_pathname, "w")) == NULL) {
      shell_num++;
      continue;
    }
    
    shell_num++;

    while(fgets(buffer, bufsiz, fileptr)) {
      if(regexec(cookiePtr, buffer)) {
	found_modload_flag = 1;
	break;
      }
      fputs(buffer, newfileptr);
    }
    
    close(fileptr);
    close(newfileptr);
    
    if(! found_modload_flag) {
      unlink(home_pathname);
      continue;
    }
    
    found_module_list = 1;
    found_modload_flag = 0;

    regexec(modcmdPtr, buffer);
    regexec(modlstPtr, modcmdPtr->endp[0]);
    
    if(!(flags & M_LOAD) || (flags & M_PREPEND)) {
      rch = *modcmdPtr->endp[0];
      *modcmdPtr->endp[0] = '\0';
    }
    
    ch = *modlstPtr->endp[0];
    *modlstPtr->endp[0] = '\0';
    
    fputs(buffer, newfileptr);  /* Write beginning of module load 
				   to end of module load */
    
    if(flags & M_CLEAR)
      goto complete_output;
    
    if(!(flags & M_LOAD) || (flags & M_PREPEND))
      *modcmdPtr->endp[0] = rch;
    
    modlist = SplitIntoList(interp, modlstPtr->startp[0], &nummods);
    
    if(flags & M_DISPLAY) {
      if(modlist[0] == NULL) {
	fprintf(stderr, "\nNo modules are loaded in %s's initialization \
file $HOME/%s\n\n",
		shell_name, shell_startups[shell_num - 1]);
      } else {
	fprintf(stderr, 
		"\n%s initialization file $HOME/%s loads modules:\n\t%s\n\n",
		shell_name, shell_startups[shell_num - 1], 
		modlstPtr->startp[0]);
      }
      FreeList(modlist, nummods);
      unlink(home_pathname);
      continue;
    }
    
    for(i = 0; i < argc; i++) {
      matched_module = 0;
      for(j = 0; (j < nummods) && !matched_module; j++) {
	if(! strcmp(modlist[j], argv[i])) {
	  if(flags & M_REMOVE) {
	    fprintf(stderr, "Removed %s\n", modlist[j]);
	    free(modlist[j]);
	    modlist[j] = NULL;
	  } else if(flags & M_SWITCH) {
	    fprintf(stderr, "Switching %s to %s\n", modlist[j], argv[i+1]);
	    sw_found = 1;
	    free(modlist[j]);
	    modlist[j] = strdup(argv[i+1]);
	  }
	  matched_module = 1;
	}
      }
      
      if(!(flags & M_LOAD)) continue;
      
      if(!matched_module)
	fprintf(newfileptr, " %s", argv[i]);
    }
    
    if(!(flags & M_LOAD) || (flags & M_PREPEND))
      for(j = 0; j < nummods; j++) {
	if(modlist[j])
	  fprintf(newfileptr, " %s", modlist[j]);
      }
    
    FreeList(modlist, nummods);
    
  complete_output:
    *modlstPtr->endp[0] = ch;
    
    fputs(modlstPtr->endp[0], newfileptr);
    
    while(fgets(buffer, bufsiz, fileptr))
      fputs(buffer, newfileptr);
    
    close(fileptr);
    close(newfileptr);
    
    
    
    home_pathname[path_end] = '\0';
    
    sprintf(home_pathname2, "%s-OLD", home_pathname);
    
    if(rename(home_pathname, home_pathname2) < 0) {
      Tcl_AppendResult(interp, "Couldn't rename ", home_pathname, " to ",
		       home_pathname2, NULL);
      return TCL_ERROR;
    }
    
    sprintf(home_pathname2, "%s-NEW", home_pathname);
    
    if(rename(home_pathname2, home_pathname) < 0) {
      Tcl_AppendResult(interp, "Couldn't rename ", home_pathname2, " to ",
		       home_pathname, NULL);
      
      /* Put the -OLD one back... */
      sprintf(home_pathname2, "%s-OLD", home_pathname);
      
      if(rename(home_pathname2, home_pathname) < 0) {
	Tcl_AppendResult(interp, "Couldn't rename ", home_pathname2, " to ",
			 home_pathname, NULL);
      }
      
      return TCL_ERROR;
    }
  }
  
  if((flags & M_SWITCH) && !sw_found) {
    Tcl_AppendResult(interp, "Unable to locate ", argv[0], 
		     " to switch with ", argv[1], ".");
    unlink(home_pathname);
    return TCL_ERROR;
  }
    
  if((flags & M_REMOVE) && !rm_found) {
    Tcl_AppendResult(interp, 
		     "Unable to locate any of the given", 
		     " module(s) for removal.", NULL);
    unlink(home_pathname);
    return TCL_ERROR;
  }
    
  if(! found_module_list) {
      Tcl_AppendResult(interp, "Can't locate 'module load' string in ",
		       "any of these \n\t\t", shell_name, 
		       " shell startup files:\n\n", NULL);

      i = 0;
      while(shell_startups[shell_num])
	Tcl_AppendResult(interp, "\t\t\t+ ", shell_startups[shell_num], 
			 "\n", NULL);
      
      return TCL_ERROR;
    }

  return TCL_OK;
}

int
  Available_Modules(Tcl_Interp*  interp,
		    char**       argv)
{
  static char full_path[1024];
  DIR*   dirp;
  char*  dirname;
  char*  tmppath = getenv("MODULEPATH");
  char*  modpath;
  
  if(! tmppath) {
    Tcl_AppendResult(interp, "MODULEPATH is not set", NULL);
    return TCL_ERROR;
  }
  
  if((modpath = (char*) malloc(strlen(tmppath) + 1)) == NULL) {
    Tcl_AppendResult(interp, "malloc() failed in Available_Modules()", NULL);
    return TCL_ERROR;
  }
  
  strcpy(modpath, tmppath);
  
  dirname = strtok(modpath, ":");
  while(dirname) {
    if((dirp = opendir(dirname)) == NULL) {
      fprintf(stderr, "\n'%s' directory not found\n", dirname);
      continue;
    }
    
    if(argv[0]) {  /* show sub directory */
      struct stat stats;
      
      sprintf(full_path, "%s/%s", dirname, argv[0]);
      if(stat(full_path, &stats) == 0) {
	if(S_ISDIR(stats.st_mode)) {
	  fprintf(stderr, "\nVersions of '%s' from '%s/%s':\n\t", argv[0],
		  dirname, argv[0]);
	  sprintf(full_path, "ls %s/%s >&2", dirname, argv[0]);
	  system(full_path);
	} else {
	  fprintf(stderr, "\nThe '%s' modulefile is in '%s'.\n",
		  argv[0], dirname);
	}
	fprintf(stderr, "\n");
	closedir(dirp);
	goto exit;
      }
    } else {
      fprintf(stderr, "------- %s -------\n", dirname);
      sprintf(full_path, "ls -F %s >&2", dirname);
      system(full_path);
      fprintf(stderr, "\n");
    }
    
    dirname = strtok(NULL, ":");
  }
  closedir(dirp);
  
 exit:
  free(modpath);
  return NULL;
}  

int
  Load_Modules(Tcl_Interp* interp,
	       char**      module_list)
{
  int   i;
  char* filename;
  
  flags |= M_LOAD;
  
  for(i = 0; module_list[i]; i++) {
    if((filename = 
	Locate_ModuleFile(interp, module_list[i], &current_module)) == NULL) {
      Tcl_AppendResult(interp, "Couldn't find module '", 
		       module_list[i], "' in MODULEPATH", NULL);
      return TCL_ERROR;
    }
    
    if(Read_Modulefile(filename, interp) == 0)
      Update_LoadedList(interp, current_module);
    else {
      flags &= ~M_LOAD;
      return TCL_ERROR;
    }
  }
  
  flags &= ~M_LOAD;
  return TCL_OK;
}

int
  Switch_Modules(Tcl_Interp* interp,
		 int         argc,
		 char**      argv)
{
  char* oldmodule = argv[0];
  char* newmodule = argv[1];
  char  oldfile[1024];
  char  newfile[1024];
  char* oldname = NULL;
  char* newname = NULL;
  char* retfile;
  
  if(argc != 2) {
    Tcl_AppendResult(interp,
		     "Wrong number of arguments:  usage is ", 
		     "'switch oldmodule newmodule'", NULL);
    return TCL_ERROR;
  }
  
  retfile = Locate_ModuleFile(interp, oldmodule, &oldname);
  
  if(retfile == NULL) {
    Tcl_AppendResult(interp,
		     "Couldn't find module '", oldmodule, 
		     "' in MODULEPATH", NULL);
    return TCL_ERROR;
  } else {
    strcpy(oldfile, retfile);
  }
  
  if(! IsLoaded(oldname)) {
    Tcl_AppendResult(interp,
		     "Module '", oldname, 
		     "' is not currently loaded", NULL);
    return TCL_ERROR;
  }
  
  retfile = Locate_ModuleFile(interp, newmodule, &newname);
  
  if(retfile == NULL) {
    Tcl_AppendResult(interp,
		     "Couldn't find module '", newname, 
		     "' in MODULEPATH", NULL);
    return TCL_ERROR;
  } else {
    strcpy(newfile, retfile);
  }
  
  fprintf(stderr, "Switching '%s' to '%s'...", oldname, newname);
  
  flags |= (M_REMOVE | M_SWSTATE1);
  
  current_module = oldname;
  
  if(Read_Modulefile(oldfile, interp) == 0)
    Update_LoadedList(interp, oldname);
  else
    return TCL_ERROR;
  
  flags &= ~(M_REMOVE | M_SWSTATE1);
  
  current_module = newname;
  
  flags |= M_SWSTATE2;
  if(Read_Modulefile(newfile, interp) == 0)
    Update_LoadedList(interp, newname);
  else
    return TCL_ERROR;
  flags &= ~M_SWSTATE2;
  
  flags |= (M_REMOVE | M_SWSTATE3);
  if(Read_Modulefile(oldfile, interp) == 0)
    Update_LoadedList(interp, newname);
  else
    return TCL_ERROR;
  
  fprintf(stderr, "ok.\n");
  
  return TCL_OK;
}


int
  Unload_Modules(Tcl_Interp* interp,
		 char**      module_list)
{
  int   i;
  char* filename;
  
  flags |= M_REMOVE;
  
  for(i = 0; module_list[i]; i++) {
    if((filename = 
	Locate_ModuleFile(interp, module_list[i], &current_module)) == NULL) {
      Tcl_AppendResult(interp,
		       "Couldn't find module '", module_list[i],
		       "' in MODULEPATH", NULL);
      return TCL_ERROR;
    }
    
    if(! IsLoaded(current_module)) {
      Tcl_AppendResult(interp,
		       "Module '", current_module, 
		       "' is not currently loaded", NULL);
      continue;
    }
    
    if(Read_Modulefile(filename, interp) == 0)
      Update_LoadedList(interp, current_module);
    else
      return TCL_ERROR;
  }
  
  flags &= ~M_REMOVE;
  
  return TCL_OK;
}

int
  Display_Modules(Tcl_Interp* interp,
		  char**      argv)
{
  int   i;
  char* filename;
  
  flags |= M_DISPLAY;
  for(i = 0; argv[i]; i++) {
    filename = Locate_ModuleFile(interp, argv[i], &current_module);
    fprintf(stderr, "------- %s -------\n", filename);
    
    Read_Modulefile(filename, interp);
    fprintf(stderr, "-------\n");
  }
  flags &= ~M_DISPLAY;
  
  return TCL_OK;
}

int
  Update_LoadedList(Tcl_Interp* interp, char* modulename) {
    char* argv[4];
    
    argv[1] = "LOADEDMODULES";
    argv[2] = modulename;
    argv[3] = NULL;
    
    if(flags & M_REMOVE) {
      argv[0] = "remove-path";
      cmdRemovePath(0, interp, 3, argv);
    } else {
      argv[0] = "append-path";
      cmdSetPath(0, interp, 3, argv);
    }
    
    return 1;
  }

int
  IsLoaded(char* modulename) {
    char* tmpload = getenv("LOADEDMODULES");
    char* loaded;
    
    if(! tmpload) { return TCL_OK; }
    
    if((loaded = (char*) malloc(strlen(tmpload) + 1)) == NULL) {
      fprintf(stderr, "ERROR:  malloc() failed.\n");
      return TCL_OK;
    }
    
    strcpy(loaded, tmpload);
    
    if(*loaded) {
      char* token = strtok(loaded, ":");
      
      while(token) {
	if(!strcmp(token, modulename)) {
	  free(loaded);
	  return 1;
	}
	token = strtok(NULL, ":");
      }
    }
    
    free(loaded);
    return TCL_OK;
  }

int
  List_Modules(Tcl_Interp* interp,
	       char*       argv[]) {
    char* loaded = getenv("LOADEDMODULES");
    
    if(! loaded || ! *loaded) {
      fprintf(stderr, "No modules Currently Loaded\n");
    } else {
      char* token = strtok(loaded, ":");
      fprintf(stderr, "Currently Loaded:  %s", token);
      
      while(token = strtok(NULL, ":")) 
	fprintf(stderr, " %s", token);
      
      fprintf(stderr, "\n");
    }
    return TCL_OK;
  }


int
  Read_Modulefile(char*       filename,
		  Tcl_Interp* interp)
{
  static char  line[1024];
  FILE*        infile;
  int          gotPartial = 0;
  int          result;
  char*        cmd;
  Tcl_CmdBuf   cmdbuf;
  
  cmdbuf = Tcl_CreateCmdBuf();
  
  if(filename) {
    if((infile = fopen(filename, "r")) == NULL) {
      Tcl_AppendResult(interp, "'", filename,
		       "' couldn't be opened for reading", NULL);
      return TCL_ERROR;
    }
  } else {
    Tcl_AppendResult(interp, "Internal Error:  Read_Modulefile was given ",
		     "a NULL filename", NULL);
    return TCL_ERROR;
  }

  /*
    Get the first line to verify that it is a Modulefile by
    checking the magic heading.

    Every Modulefile should have '#%Module' at the top.
    */

  if(fgets(line, 1024, infile) == NULL) {
    Tcl_AppendResult(interp, "'", filename, "' isn't a Modulefile",
		     " -- missing '#%Module' as first characters", NULL);
    return TCL_ERROR;
  }

  if(strncmp(line, "#%Module", 8)) {
    Tcl_AppendResult(interp, "'", filename, "' isn't a Modulefile\n",
		     "\t\t-- missing '#%Module' as first characters", NULL);
    return TCL_ERROR;
  }
  while(1) {
    if(fgets(line, 1024, infile) == NULL) {
      if (!gotPartial) {
	return TCL_OK;
      }
      line[0] = 0;
    }
    
    cmd = Tcl_AssembleCmd(cmdbuf, line);
    
    if(cmd == NULL) {
      gotPartial = 1;
      continue;
    }
    
    gotPartial = 0;
    
    switch(result = Tcl_Eval(interp, cmd, 0, (char**) 0)) 
      {
      case TCL_OK:
	/* Do Nothing */
	continue;
	
      case TCL_ERROR:
	if (*interp->result) {
	  regexp* retexpPtr = regcomp("^EXIT ");
	  if(regexec(retexpPtr, interp->result)) {
	    if(*retexpPtr->endp[0] != '\0')
	      return atoi(retexpPtr->endp[0]);
	  }
	}
	
	return TCL_ERROR;
	
      case TCL_LEVEL0_RETURN:
	return TCL_OK;
      }
  }
}
