/* $Copyright:	$
 * Copyright (c) 1991,1992,1993 by Steve Baker
 * All rights reserved
 *  
 * This software is provided as is without any express or implied
 * warranties, including, without limitation, the implied warranties
 * of merchantability and fitness for a particular purpose.
 */
#include <errno.h>
#include <ctype.h>
#include "shell.h"
#include "cmds.h"

struct Limits {
  char *resource;
  long val;
} limits[6] = {
  "cpu",RLIMIT_CPU,
  "filesize",RLIMIT_FSIZE,
  "datasize",RLIMIT_DATA,
  "stacksize",RLIMIT_STACK,
  "coredumpsize",RLIMIT_CORE,
  "memoryuse",RLIMIT_RSS,
};

char *funcs[] = {
  0, "Previous history", "Next history", "Move Left", "Move Right",
  "Delete/Backspace", "Toggle insert/overstrike", "Move to beginning of line",
  "Delete to the right", "Move to EOL", "Kill to EOL", "Restore line",
  "Kill line", "Quoted insert", "Find matching history", "Kill to BOL",
  "Filename completion", "Backward one word", "Forward one word",
  "Expand line.", "Kill word", "Next character is a control character",
  "Bell", 0
};

extern char _restricted;
extern char *_home, *_statline, _loginshell, _glob, **_cdpath, _nofork;
extern char ***file, **history, *getenv();
extern char *_kbuf[MAX_GLVL], _kmax[MAX_GLVL];
extern char *exit_val[], *_term[11];
extern int _maxhist, curhist, fptr, _failat;
extern struct custom_keys **keys[MAX_GLVL];
extern struct proc_tab *proc;
extern int max_ent, errno, _pgrp, _string, _status;
extern int _source, err;
extern char path[1025], buf[1025];
extern struct _setvar *setvar, *find_var(), *makenvar();
extern unsigned long SIGMASK;

char lif;	/* result of last single line if */

char *string(), *UPPER(), *grab(), **evalw();
int setenv();
void unsetenv(), *malloc();

int a, b, c, d;

execute(cmd,arg,in,inf,out,outf,outa,err,errf,erra,forked)
char cmd,**arg,*inf,*outf,*errf,outa,erra,forked;
int in,out,err;
{
  FILE *fin = stdin, *fout = stdout, *ferr = stderr;
  int pid,n = nargs(arg), rc = 0;

  if (forked) {
    if (pid = fork()) return pid;
  }
  sh_redirect(&in,inf,&out,outf,outa,&err,errf,erra,forked);
  if (in > 0) fin = fdopen(in,"r");
  if (out > 1) {
    if (outa) fout = fdopen(out,"a");
    else fout = fdopen(out,"w");
  }
  if (err > 2 && err != out) {
    if (erra) ferr = fdopen(err,"a");
    else ferr = fdopen(err,"w");
  } else if (err == out) ferr = fout;

  switch(cmd) {
    case CMD_CD:
      rc = cd(n,arg,fin,fout,ferr);
      break;
    case CMD_HISTORY:
      rc = hist(n,arg,fin,fout,ferr);
      break;
    case CMD_JOBS:
      rc = jobs(n,arg,fin,fout,ferr);
      break;
    case CMD_KEY:
      rc = key(n,arg,fin,fout,ferr);
      break;
    case CMD_ALIAS:
      rc = ALIAS(n,arg,fin,fout,ferr);
      break;
    case CMD_LIMIT:
      rc = limit(n,arg,fin,fout,ferr);
      break;
    case CMD_UNLIMIT:
      rc = unlimit(n,arg,fin,fout,ferr);
      break;
    case CMD_FG:
      rc = fg(n,arg,fin,fout,ferr);
      break;
    case CMD_BG:
      rc = bg(n,arg,fin,fout,ferr);
      break;
    case CMD_STOP:
      rc = stop(n,arg,fin,fout,ferr);
      break;
    case CMD_SOURCE:
      rc = SOURCE(n,arg,fin,fout,ferr);
      break;
    case CMD_SET:
      rc = SET(n,arg,fin,fout,ferr);
      break;
    case CMD_SETENV:
      rc = SETENV(n,arg,fin,fout,ferr);
      break;
    case CMD_UNSETENV:
      rc = UNSETENV(n,arg,fin,fout,ferr);
      break;
    case CMD_UNSET:
      rc = UNSET(n,arg,fin,fout,ferr);
      break;
    case CMD_UNKEY:
      rc = unkey(n,arg,fin,fout,ferr);
      break;
    case CMD_UNALIAS:
      rc = UNALIAS(n,arg,fin,fout,ferr);
      break;
    case CMD_UMASK:
      rc = UMASK(n,arg,fin,fout,ferr);
      break;
    case CMD_EXIT:
      rc = EXIT(n,arg);
      break;
    case CMD_LOGOUT:
      rc = logout(n,arg,ferr);
      break;
    case CMD_VERSION:
      rc = version(fout);
      break;
    case CMD_INPUT:
      rc = INPUT(n,arg,fin,fout,ferr);
      break;
    case CMD_IF:
      rc = IF(n,arg,ferr);
      break;
    case CMD_ELSE:
      rc = ELSE(n,arg,ferr);
      break;
    case CMD_ENDIF:
      rc = ENDIF(ferr);
      break;
    case CMD_LOGIN:
      rc = login(n,arg,ferr);
      break;
    case CMD_SECHO:
      rc = secho(n,arg,fin,fout,ferr);
      break;
    case CMD_WHILE:
      rc = WHILE(n,arg,ferr);
      break;
    case CMD_REPEAT:
      rc = REPEAT(n,arg,ferr);
      break;
    case CMD_BREAK:
      rc = BREAK(ferr);
      break;
    case CMD_CONTINUE:
      rc = CONTINUE(ferr);
      break;
    case CMD_GOTO:
      rc = GOTO(n,arg,ferr);
      break;
    case CMD_FOREACH:
      rc = FOREACH(n,arg,ferr);
      break;
    case CMD_FOR:
      rc = FOR(n,arg,ferr);
      break;
    case CMD_INC:
      rc = INC(n,arg,ferr);
      break;
    case CMD_DEC:
      rc = DEC(n,arg,ferr);
      break;
    case CMD_LABEL:
      rc = LABEL(n,arg,ferr);
      break;
    case CMD_WEND:
      rc = WEND(ferr);
      break;
    case CMD_UNTIL:
      rc = UNTIL(ferr);
      break;
    case CMD_ENDFOR:
      rc = ENDFOR(ferr);
      break;
    case CMD_NEXT:
      rc = NEXT(ferr);
      break;
    case CMD_SWITCH:
      rc = SWITCH(n,arg,ferr);
      break;
    case CMD_ENDSW:
      rc = ENDSW(ferr);
      break;
    case CMD_EVAL:
      rc = EVAL(n,arg,ferr);
      break;
    case CMD_EXEC:
      rc = EXECUTE(n,arg,ferr);
      break;
    case CMD_USAGE:
      rc = usage(fout,ferr);
      break;
    case CMD_INTR:
      rc = INTR(n,arg,fout,ferr);
      break;
    case CMD_TERM:
      rc = TERM(n,arg,fout,ferr);
      break;
    case CMD_PROTECT:
      rc = PROTECT(n,arg,ferr);
      break;
    case CMD_ASSIGN:
      rc = ASSIGN(n,arg,fout,ferr);
      break;
    case CMD_UNASSIGN:
      rc = UNASSIGN(n,arg,fout,ferr);
      break;
    case CMD_SHIFT:
      rc = SHIFT(n,arg,ferr);
      break;
    case CMD_SOPEN:
      rc = SOPEN(n,arg,ferr);
      break;
    case CMD_SCLOSE:
      rc = SCLOSE(n,arg,ferr);
      break;
    case CMD_SREAD:
      rc = SREAD(n,arg,ferr);
      break;
    case CMD_SWRITE:
      rc = SWRITE(n,arg,ferr);
      break;
    case CMD_SSEEK:
      rc = SSEEK(n,arg,ferr);
      break;
  }
  _status = rc;
  if (forked) exit(rc);
  if (in > 0) { fclose(fin); close(in); }
  if (out > 1) { fclose(fout); close(out); }
  if (err > 2 && err != out) { fclose(ferr); close(err); }
  return 0;
}

nargs(arg)
char **arg;
{
  int i;

  for(i=0;arg[i];i++);
  return i;
}

cd(n,arg,in,out,err)
int n;
FILE *in,*out,*err;
char **arg;
{
  if (n > 2) fprintf(err,"cd: too many arguments\n");
  if (n == 1) {
    if (!_home) {
      fprintf(err,"cd: no home directory\n");
      return 0;
    }
    if ((a = chdir(_home)) < 0) c = errno;
  } else {
    if ((a = chdir(arg[1])) < 0) {
      c = errno;
      if (_cdpath) {
	for(b=0;_cdpath[b];b++) {
	  sprintf(path,"%s/%s",_cdpath[b],arg[1]);
	  if (chdir(path) < 0) continue;
	  a = 0;
	  break;
	}
      }
    }
  }
  if (a < 0) {
    switch(c) {
      case ENOTDIR:
	fprintf(err,"%s: Not a directory\n",arg[1]);
	break;
      case ENOENT:
	fprintf(err,"%s: No such file or directory\n",arg[1]);
	break;
      case ELOOP:
	fprintf(err,"%s: Too many symbolic links.\n",arg[1]);
	break;
      default:
	fprintf(err,"%s: Permission denied.\n",arg[1]);
	break;
    }
    return 1;
  } else {
    getwd(path);
    if (_restricted && strncmp(_home,path,strlen(_home))) {
      chdir(_home);
      getwd(path);
    }
    makeset("cwd",path);
  }
  return 0;
}

limit(n,arg,in,out,err)
int n;
FILE *in,*out,*err;
char **arg;
{
  struct rlimit rlm;

  b = -1;
  if (n > 1) {
    for(a=0;a<6;a++)
      if (!strcmp(arg[1],limits[a].resource)) {
	b = a;
	break;
      }
  }
  if (n > 1 && b == -1) {
    fprintf(err,"limit: undefined limit %s\n",arg[1]);
    return 1;
  }
  if (n < 3) {
    if (n == 1) {
      for(a=0;a<6;a++) {
	getrlimit(limits[a].val,&rlm);
	if (rlm.rlim_cur == RLIM_INFINITY) fprintf(out,"%-15s unlimited\n",limits[a].resource);
	else fprintf(out,"%-15s %d K\n",limits[a].resource,rlm.rlim_cur/1024);
      }
    } else {
      getrlimit(limits[b].val,&rlm);
      if (rlm.rlim_cur == RLIM_INFINITY) fprintf(out,"%-15s unlimited\n",arg[1]);
      else fprintf(out,"%-15s %d K\n",arg[1],rlm.rlim_cur/1024);
    }
  } else if (n == 3) {
    if (!strcmp(arg[2],"unlimit") || !strcmp(arg[2],"unlimited")) {
      getrlimit(limits[b].val,&rlm);
      rlm.rlim_cur = RLIM_INFINITY;
      setrlimit(limits[b].val,&rlm);
    } else {
      if (!isnum(arg[2])) {
        fprintf(err,"limit: invalid argument - %s\n",arg[2]);
	return 1;
      }
      getrlimit(limits[b].val,&rlm);
      rlm.rlim_cur = atoi(arg[2]) * 1024;
      setrlimit(limits[b].val,&rlm);
    }
  } else {
    fprintf(err,"limit: too many arguments\n");
    return 1;
  }
  return 0;
}

unlimit(n,arg,in,out,err)
int n;
FILE *in,*out,*err;
char **arg;
{
  struct rlimit rlm;

  rlm.rlim_cur = RLIM_INFINITY;
  if (n == 1) {
    fprintf(err,"unlimit: no limit specified\n");
    return 1;
  }
  if (n > 2) {
    fprintf(err,"unlimit: too many aguments\n");
    return 1;
  }
  for(a=1;arg[a];a++) {
    for(b=0;b<6;b++)
      if (!strcmp(arg[a],limits[b].resource)) {
	setrlimit(limits[b].val,&rlm);
	break;
      }
    if (b == 6) {
      fprintf(err,"unlimit: invalid limit\n");
      return 1;
    }
  }
  return 0;
}

hist(n,arg,in,out,err)
int n;
FILE *in,*out,*err;
char **arg;
{
  char inc = 1, nflg = TRUE;

  c = -1;
  d = 0;
  for(a=0;arg[a];a++) {
    if (arg[a][0] == '-') {
      switch(arg[a][1]) {
	case 'r':
	  d = curhist - 1;
	  inc = -1;
	  break;
	case 'n':
  	  nflg = FALSE;
	  break;
	default:
	  b = atoi(arg[a]+1);
	  if (b > 0) c = b;
	  break;
      }
    }
  }
  for(a=d;a > -1 && a<curhist && c;a+=inc) {
    if (nflg) fprintf(out,"%d %s\n",a,history[a]);
    else fprintf(out,"%s\n",history[a]);
    if (c != -1) c--;
  }
  return 0;
}

key(n,arg,in,out,err)
int n;
FILE *in,*out,*err;
char **arg;
{
  struct custom_keys *key;
  char lvl=0;

  if (n == 1) {
    for(b=0;b<MAX_GLVL;b++) {
      if (!keys[b]) continue;
      for(a=0;keys[b][a];a++) {
	fprt(out,keys[b][a]->key);
	fprintf(out,"\t[%d] ",b);
	if (keys[b][a]->gold) fprintf(out,"GOLD -> LEVEL %d",keys[b][a]->glvl);
	else if (keys[b][a]->func) fprintf(out,"Function %2d: %s",keys[b][a]->func,funcs[(keys[b][a]->func)<21?(keys[b][a]->func):21]);
	else {
	  fputc('"',out);
	  fprt(out,keys[b][a]->cmd);
	  fputc('"',out);
	  if (keys[b][a]->rtn) fprintf(out,"\t[RETURN]");
	  if (keys[b][a]->clr) fprintf(out,"%c[CLEAR]",keys[b][a]->rtn?' ':'\t');
	}
	fputc('\n',out);
      }
    }
    return 0;
  }
  key = (struct custom_keys *)malloc(sizeof(struct custom_keys));
  bzero(key,sizeof(struct custom_keys));
  for(a=1;a<n;a++) {
    if (arg[a][0] == '-') {
      switch(arg[a][1]) {
	case 'r':
	  key->rtn = 1;
	  break;
	case 'c':
	  key->clr = 1;
	  break;
	case 'g':
	  key->gold = 1;
	  if (arg[a][2] >= '0' && arg[a][2] <= '4' ) key->glvl = arg[a][2] - 48;
	  else key->glvl = 0;
	  break;
	case 'l':
	  if (isdigit(arg[a][2]) && arg[a][2] < '5') lvl = arg[a][2] - 48;
	  break;
	case 'f':
	  key->func = atoi(arg[a]+2);
	  break;
      }
    } else break;
  }
  if ((a+2 > n && (!key->gold && !key->func)) || (a+1 > n && (key->gold || key->func))) {
    fprintf(err,"not enough arguments\n");
    free(key);
    return 1;
  }
  key->key = (char *)strcpy(malloc(strlen(arg[a])+1),arg[a]);
  a++;
  if (!key->gold && !key->func) key->cmd = string(arg+a);
  if (keys[lvl]) {
    for(a=0;keys[lvl][a];a++) {
      if (!strcmp(keys[lvl][a]->key,key->key)) {
	free(keys[lvl][a]->key);
	if (keys[lvl][a]->cmd) free(keys[lvl][a]->cmd);
	free(keys[lvl][a]);
	keys[lvl][a] = key;
	return 0;
      }
    }
  }
  if (strlen(key->key) > _kmax[lvl]) _kmax[lvl] = strlen(key->key);
  a = strlen(_kbuf[lvl]);
  _kbuf[lvl] = (char *)realloc(_kbuf[lvl],a+2);
  _kbuf[lvl][a] = key->key[0];
  _kbuf[lvl][a+1] = 0;
  if (!keys[lvl]) keys[lvl] = (struct custom_keys **)calloc(2,sizeof(struct custom_keys *));
  else keys[lvl] = (struct custom_keys **)realloc(keys[lvl], sizeof(struct custom_keys *) * (a+2));
  keys[lvl][a++] = key;
  keys[lvl][a] = NULL;
  return 0;
}

jobs(n,arg,in,out,err)
int n;
char **arg;
FILE *in,*out,*err;
{
  unsigned long mask;

  mask = sigblock(sigmask(SIGCHLD));
  for(a=0;a<max_ent;a++) {
    if (proc[a].pid != 0) {
      if (proc[a].status) {
	fprintf(out," [%d] (%d) %-30s%s %s\n",a,proc[a].pid,exit_val[proc[a].status],(proc[a].bg?" (background)":""),proc[a].cmd);
      } else {
	if (proc[a].bg) fprintf(out," [%d] (%d) %-30s %s\n",a,proc[a].pid,"Running (background)",proc[a].cmd);
	else fprintf(out," [%d] (%d) %-30s %s\n",a,proc[a].pid,"Running",proc[a].cmd);
      }
    }
  }
  sigsetmask(mask);
  return 0;
}

SETENV(n,arg,in,out,err)
int n;
char **arg;
FILE *in,*out,*err;
{
  char *ptr;

  if (n < 2) {
    fprintf(err,"setenv: not enough arguments\n");
  } else if (n > 3) {
    fprintf(err,"setenv: too many arguments\n");
  } else if (n == 2) {
    ptr = getenv(arg[1]);
    if (ptr) {
      printf("%s=%s\n",arg[1],ptr);
      free(ptr);
    }
  } else {
    setenv(arg[1],arg[2],1);
  }
  return 0;
}

UNSETENV(n,arg,in,out,err)
int n;
char **arg;
FILE *in,*out,*err;
{
  if (n < 2) {
    fprintf(err,"unsetenv: variable identifier expected.\n");
  } else if (n > 2) {
    fprintf(err,"unsetenv: too many arguments.\n");
  } else {
    unsetenv(arg[1]);
  }
  return 0;
}

fg(n,arg,in,out,err)
int n;
char **arg;
FILE *in,*out,*err;
{
  unsigned long mask;

  if (n > 2) {
    fprintf(err,"%s: too many args\n",arg[0]);
    return 1;
  }
  if (n == 1) {
    mask = sigblock(sigmask(SIGCHLD));
    for(a=0;a<max_ent;a++) {
      if (proc[a].pid != 0 && (proc[a].bg || proc[a].status)) {
	if (proc[a].bg) {
	  proc[a].bg = 0;
	  setpgrp(proc[a].pid,proc[a].pid);
	  proc[a].pgrp = proc[a].pid;
	}
	if (proc[a].pgrp != _pgrp) tcsetpgrp(TTY,proc[a].pgrp);
	if (killpg(proc[a].pgrp,SIGCONT) < 0) return 1;
	proc[a].status = STAT_RUNNING;
	break;
      }
    }
    sigsetmask(mask);
    return 0;
  } else {
    a = atoi(arg[1]);
    if (a < 0 || a >= max_ent) {
      fprintf(err,"%s: No such job: %d\n",a);
      return 1;
    }
    mask = sigblock(sigmask(SIGCHLD));
    if (proc[a].pid != 0 && (proc[a].bg || proc[a].status)) {
      if (proc[a].bg) {
	proc[a].bg = 0;
	setpgrp(proc[a].pid,proc[a].pid);
	proc[a].pgrp = proc[a].pid;
      }
      if (proc[a].pgrp != _pgrp) tcsetpgrp(TTY,proc[a].pgrp);
      if (killpg(proc[a].pgrp,SIGCONT) < 0) return 1;
      proc[a].status = STAT_RUNNING;
    }
    sigsetmask(mask);
  }
  return 0;
}

bg(n,arg,in,out,err)
int n;
char **arg;
FILE *in,*out,*err;
{
  unsigned long mask;

  if (n > 2) {
    fprintf(err,"%s: too many args\n",arg[0]);
    return 1;
  }
  if (n == 1) {
    mask = sigblock(sigmask(SIGCHLD));
    for(a=0;a<max_ent;a++) {
      if (proc[a].pid != 0 && !proc[a].bg) {
	proc[a].bg = 1;
	setpgrp(proc[a].pid,_pgrp);
	proc[a].pgrp = _pgrp;
	tcsetpgrp(TTY,_pgrp);
	if (killpg(proc[a].pgrp,SIGCONT) < 0) return 1;
	proc[a].status = STAT_RUNNING;
	break;
      }
    }
    sigsetmask(mask);
    return 0;
  } else {
    a = atoi(arg[1]);
    if (a < 0 || a >= max_ent) {
      fprintf(err,"%s: No such job: %d\n",a);
      return 1;
    }
    mask = sigblock(sigmask(SIGCHLD));
    if (proc[a].pid != 0 && !proc[a].bg) {
      proc[a].bg = 1;
      setpgrp(proc[a].pid,_pgrp);
      proc[a].pgrp = _pgrp;
      tcsetpgrp(TTY,_pgrp);
      if (killpg(proc[a].pgrp,SIGCONT) < 0) return 1;
      proc[a].status = STAT_RUNNING;
    }
    sigsetmask(mask);
  }
  return 0;
}

stop(n,arg,in,out,err)
int n;
char **arg;
FILE *in,*out,*err;
{
  unsigned long mask;

  if (n > 2) {
    fprintf(err,"stop: too many args\n");
    return 1;
  }
  mask = sigblock(sigmask(SIGCHLD));
  if (n == 1) {
    for(a=0;a<max_ent;a++) {
      if (proc[a].pid != 0 && proc[a].bg) {
	if (killpg(proc[a].pgrp,SIGTSTP) < 0) return 1;
	break;
      }
    }
  } else {
    a = atoi(arg[1]);
    if (proc[a].pid != 0 && proc[a].bg) {
      if (killpg(proc[a].pgrp,SIGTSTP) < 0) return 1;
    }
  }
  sigsetmask(mask);
  return 0;
}

SOURCE(n,arg,in,out,err)
int n;
char **arg;
FILE *in,*out,*err;
{
  int i,j,f;
  char ***tfile;

  if (n < 2) {
    source(NULL);
    return 0;
  }
  for(i=1;arg[i];i++) {
    f = fptr;
    tfile = file;
    j = source(arg[i]);
    fptr = f;
    file = tfile;
    if (j < 0) {
      fprintf(err,"source: error opening file %s\n",arg[i]);
      return 1;
    }
  }
  return 0;
}

unkey(n,arg,in,out,err)
int n;
char **arg;
FILE *in,*out,*err;
{
  c = 0;
  if (n < 2) {
    fprintf(stderr,"unkey: macro definition expected.\n");
    return 1;
  }
  for(a=1;arg[a];a++) {
    if (arg[a][0] == '-') {
      if (arg[a][1] == 'l') c = atoi(arg[a]+2);
      else {
        fprintf(stderr,"unkey: %s invalid argument.\n",arg[a]);
	return 1;
      }
    } else break;
  }
  for(b=0;keys[c][b];b++) {
    if (!strcmp(arg[a],keys[c][b]->key)) {
      free(keys[c][b]->cmd);
      free(keys[c][b]->key);
      free(keys[c][b]);
      for(;keys[c][b];b++) {
        keys[c][b] = keys[c][b+1];
	_kbuf[c][b] = _kbuf[c][b+1];
      }
      return 0;
    }
  }
  return 0;
}

UMASK(n,arg,in,out,err)
int n;
char **arg;
FILE *in, *out, *err;
{
  char *ex;

  if (n == 1) {
    a = umask(0);    
    fprintf(out,"%03o\n",a);
    umask(a);
    return 0;
  }
  ex = grab(arg,1,NULL,&a);
  if (!ex) {
    fprintf(err,"umask: file protection mask expected.\n");
    return 1;
  }
  a = expr(ex);
  free(ex);
  umask(a);
  return 0;
}

EXIT(n,arg)
int n;
char **arg;
{
  char *ex;

  if (n > 1) {
    ex = grab(arg,1,NULL,&a);
    a = expr(ex);
  } else a = 0;
  if (_source) {
    err = ERR_EXIT;
    return a;
  }
  if (_statline) printf("%s%s%s",_term[TS],_term[CE],_term[FS]);
  exit(a);
}

logout(n,arg,err)
int n;
char **arg;
FILE *err;
{
  static char f = TRUE;
  unsigned long mask;
  FILE *fd;

  if (f) {
    mask = sigblock(sigmask(SIGCHLD));
    for(a=0;a<max_ent;a++)
      if (proc[a].pid != 0 && (proc[a].bg || proc[a].status)) {
	fprintf(err,"You have stopped jobs!\n");
	f = FALSE;
	break;
      }
    sigsetmask(mask);
    if (!f) return 1;
  }
  
  if (_loginshell) {
    for(a=0;a<n;a++) {
      sprintf(path,"%d",a);
      makeset(path,arg[a]);
    }
    sprintf(path,"%s/%s",_home,HIST_SAVE);
    b = open(path,O_WRONLY | O_CREAT | O_TRUNC,0600);
    fd = fdopen(b,"w");
    fprintf(fd,"%d\n",_maxhist);
    for(a=0;a < curhist;a++) fprintf(fd,"%s\n",history[a]);
    fclose(fd);
    close(b);
    sprintf(path,"%s/%s",_home,SHELL_EXIT);
    source(path);
  }
  if (_statline) printf("%s%s%s",_term[TS],_term[CE],_term[FS]);
  exit(0);
}

char *string(wrd)
char **wrd;
{
  int i,j;
  char *str;

  for(i=j=0;wrd[i];i++) j += strlen(wrd[i])+1;
  str = (char *)malloc(j);
  strcpy(str,wrd[0]);
  for(i=1;wrd[i];i++) {
    strcat(str," ");
    strcat(str,wrd[i]);
  }
  return str;
}

char *UPPER(wrd)
char *wrd;
{
  char *tmp;

  tmp = (char *)malloc(strlen(wrd)+1);
  for(a=0;*wrd;wrd++) tmp[a++] = (islower(*wrd)? toupper(*wrd) : *wrd);
  tmp[a] = 0;
  return tmp;
}

fprt(stream,str)
FILE *stream;
char *str;
{
  char c;

  while(*str) {
    c = *str++ & 127;
    if (c < 32) {
      if (_term[SO]) fprintf(stream,"%s%c%s",_term[SO],c+64,_term[SE]);
      else fprintf(stream,"^%c",c+64);
    } else if (c == 127) {
      if (_term[SO]) fprintf(stream,"%s?%s",_term[SO],_term[SE]);
      else fprintf(stream,"^?");
    } else fputc(c,stream);
  }
}

version(out)
FILE *out;
{
  fprintf(out," Ssh V1.7\n Copyright (C) 1991,1992,1993 by Steve Baker and Thomas Moore.\n All rights reserved.\n");
  return 0;
}

IF(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  long res;
  int i,nest;
  int start = fptr, stop = fptr+1, end = fptr+1;
  char *ex;

  ex = grab(arg,1,"then",&i);
  if (!ex) {
    fprintf(ferr,"if: expression expected.\n");
    return err = 1;
  }
  if (!arg[i++]) {
    fprintf(ferr,"if: keyword 'then' expected.\n");
    return err = 1;
  }
  err = 0;
  res = expr(ex);
  free(ex);
  if (err) return 1;

  if (!_source && !arg[i]) return 1;  
  nest = 0;
  if (!arg[i]) {
    while(file[stop]) {
      if (!nest && (!strcmp(file[stop][0],"else") || !strcmp(file[stop][0],"endif"))) break;
      else if (nest && !strcmp(file[stop][0],"endif")) nest--;
      else if (!strcmp(file[stop][0],"if")) {
	for(i=1;file[stop][i];i++) {
	  if (!strcmp(file[stop][i],"then"))
	    if (file[stop][i+1]) break;
	    else {
	      nest++;
	      break;
	    }
	}
      }
      stop++;
    }
    end = stop;
    while(file[end]) {
      if (!nest && !strcmp(file[end][0],"endif")) break;
      else if (nest && !strcmp(file[end][0],"endif")) nest--;
      else if (!strcmp(file[end][0],"if")) {
	for(i=1;file[end][i];i++) {
	  if (!strcmp(file[end][i],"then"))
	    if (file[end][i+1]) break;
	    else {
	      nest++;
	      break;
	    }
	}
      }
      end++;
    }
    if (!file[stop] || !file[end]) {
      fprintf(ferr,"if: unexpected end of file.\n");
      return err = 1;
    }
    if (!res) {
      start = stop;
      stop = end;
    }
    if (start == end) {
      fptr = end;
      return 0;
    }
    fptr = start+1;
    while(fptr < stop) {
      run(fptr);
      if (err || badstat(_status)) {
	fptr = end;
	return _status&RET_MASK;
      }
      fptr++;
    }
    fptr = end;
  } else if (res) {
    lif = TRUE;
    ex = string(arg+i);
    run2(ex);
    free(ex);
  } else lif = FALSE;
  return 0;
}

ELSE(n,arg,err)
int n;
char **arg;
FILE *err;
{
  char *cmd;

  if (!_source) {
    fprintf(err,"else: no previous if-then statement.\n");
    return 1;
  } else {
    if (!fptr || strcmp(file[fptr-1][0],"if")) {
      fprintf(err,"else: no previous if-then statement.\n");
      return 1;
    }
    if (n < 2) {
      fprintf(err,"else: no command for else condition.\n");
      return 1;
    }
    if (lif) return 0;
    cmd = string(arg+1);
    run2(cmd);
    free(cmd);
  }
  return 0;
}

ENDIF(err)
FILE *err;
{
  fprintf(err,"else: no previous if-then-else statement.\n");
  return 1;
}

INPUT(n,arg,in,out,err)
int n;
char **arg;
FILE *in, *out, *err;
{
  char *var = NULL, *prompt = NULL;

  c = d = 0;
  if (n < 2) {
    fprintf(err,"input: not enough arguments\n");
    return 1;
  }
  for(a=1;arg[a];a++) {
    if (prompt == arg[a]) continue;
    if (arg[a][0] == '-') {
      for(b=1;arg[a][b];b++) {
	switch(arg[a][b]) {
	  case 'n':	/* don't replace var if nothing entered. */
	    c = TRUE;
	    break;
	  case 'c':	/* read a single character immediately. */
	    d = 1;
	    break;
	  case 'w':	/* break line up into words. */
	    d = 2;
	    break;
	  case 'e':	/* read single character, echoing it to screen. */
	    d = 3;
	    break;
	  case 'p':	/* Next word is the prompt */
	    if (arg[a+1]) prompt = arg[a+1];
	    else {
	      fprintf(err,"input: no prompt specified.\n");
	      return 1;
	    }
	    break;
	}
      }
    } else if (!var) var = arg[a];
    else {
      fprintf(err,"input: too many arguments.\n");
      return 1;
    }
  }
  if (prompt) fprintf(out,"%s",prompt);
  switch (d) {
    case 0:
    case 2:
      fgets(buf,1024,in);
      buf[strlen(buf)-1] = 0;
      if (buf[0] == 0 && c) return 0;
      makeset(var,buf);
      return 0;
    case 1:
    case 3:
      noecho();
      buf[0] = getchar();
      if (d == 3) outch(buf[0]);
      buf[1] = 0;
      makeset(var,buf);
      echo();
      return 0;
  }
  /*NOTREACHED*/
}

login(n,arg,err)
int n;
char **arg;
FILE *err;
{
  execv("/bin/login",arg);
  fprintf(err,"login: command not found.\n");
  return 1;
}

secho(n,arg,in,out,err)
int n;
char **arg;
FILE *in, *out, *err;
{
  char *ex;
  char x,z;

  x = z = b = d = TRUE;
  c = FALSE;

  for(a=1;arg[a] && x;a++) {
    if (z && arg[a][0] == '-' && (!arg[a][1] || !arg[a][2])) {
      switch(arg[a][1]) {
	case 'n':
	  b = FALSE;
	  break;
	case 'w':
	  c = !c;
	  break;
	case 'e':
	  d = !d;
	  break;
	case 'x':
	  ex = string(arg+(a+1));
	  d = expr(ex);
	  fprintf(out,"%ld",d);
	  free(ex);
	  x = c = FALSE;
	  break;
	case 0:
	  z = FALSE;
	  break;
      }
    } else {
      if (d) fputs((char *)pline(arg[a],1023,1023),out);
      else fputs(arg[a],out);
      if (!c && arg[a+1]) fputc(' ',out);
      if (c) fputc('\n',out);
    }
  }
  if (b && !c) fputc('\n',out);
  return 0;
}

BREAK()
{
  err = ERR_BREAK;
  return 0;
}

CONTINUE()
{
  err = ERR_CONTINUE;
  return 0;
}

WHILE(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  int start = fptr+1, end = fptr+1;
  char *ex, nest = 0;

  if (!_source) return 1;
  while(file[end]) {
    if (!strcmp(file[end][0],"while")) {
      nest++;
      end++;
      continue;
    }
    if (!nest && !strcmp(file[end][0],"wend")) break;
    if (nest && !strcmp(file[end][0],"wend")) nest--;
    end++;
  }
  if (!file[end]) {
    fprintf(ferr,"while: while without matching wend.\n");
    return 1;
  }
  
  ex = string(arg+1);
  while (expr(ex)) {
    fptr = start;
    while(fptr < end) {
      run(fptr);
      if (err == ERR_BREAK) {
	fptr = end;
	free(ex);
        return err = 0;
      } else if (err == ERR_CONTINUE) {
        err = 0;
	break;
      }
      if (err || badstat(_status)) {
	fptr = end;
        free(ex);
        return _status&RET_MASK;
      }
      fptr++;
    }
  }
  fptr = end;
  free(ex);
  return 0;
}

WEND(ferr)
FILE *ferr;
{
  fprintf(ferr,"wend: wend without matching while.\n");
  return 1;
}

REPEAT(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  int start = fptr+1, end = fptr+1;
  char *ex = NULL, nest = 0;

  if (!_source) return 1;

  while(file[end]) {
    if (!strcmp(file[end][0],"repeat")) {
      nest++;
      end++;
      continue;
    }
    if (!nest && !strcmp(file[end][0],"until")) break;
    if (nest && !strcmp(file[end][0],"until")) nest--;
    end++;
  }
  if (!file[end]) {
    fprintf(ferr,"repeat: repeat without matching until.\n");
    err = 1;
    return 1;
  }
  ex = string(file[end]+1);

  do {
    fptr = start;
    while (fptr < end) {
      run(fptr);
      if (err == ERR_BREAK) {
	fptr = end;
	break;
      } else if (err == ERR_CONTINUE) {
	fptr = end;
	err = 0;
	break;
      }
      if (err || badstat(_status)) {
	fptr = end;
        if (ex) free(ex);
        return _status&RET_MASK;
      }
      fptr++;
    }
    if (err) break;
  } while (!expr(ex));
  if (err) err = 0;
  free(ex);
  return 0;
}

UNTIL(ferr)
FILE *ferr;
{
  fprintf(ferr,"until: until without matching repeat.\n");
  return 1;
}

FOR(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  long beg, en, step;
  int i,nest = 0, start = fptr+1, end = fptr+1;
  char *ex = NULL;
  struct _setvar *v;

  if (!_source) return 1;

  if (arg[1]) v = makenvar(arg[1],0);
  else {
    fprintf(ferr,"for: missing variable identifier.\n");
    return err = 1;
  }
  if (!arg[2] || arg[2][0] != '=' || arg[2][1] != 0) {
    fprintf(ferr,"for: '=' expected.\n");
    return err = 1;
  }
  ex = grab(arg,3,"to", &i);
  if (!ex) {
    fprintf(ferr,"for: assignment expected.\n");
    return err = 1;
  }
  if (!arg[i]) {
    fprintf(ferr,"for: keyword 'to' expected.\n");
    return err = 1;
  }
  beg = expr(ex);
  if (err) return 1;
  free(ex);
  i++;
  ex = grab(arg,i,"step",&i);
  if (!ex) {
    fprintf(ferr,"for: ending value expected.\n");
    return err = 1;
  }
  en = expr(ex);
  free(ex);
  if (arg[i++]) {
    ex = grab(arg,i,NULL,&i);
    if (!ex) {
      fprintf(ferr,"for: step value expected.\n");
      return err = 1;
    }
    step = expr(ex);
    free(ex);
  } else step = 1;

  while(file[end]) {
    if (!nest && !strcmp(file[end][0],"next")) break;
    else if (nest && !strcmp(file[end][0],"next")) nest--;
    else if (!strcmp(file[end][0],"for")) nest++;
    end++;
  }
  if (!file[end]) {
    fprintf(ferr,"for: for without next.\n");
    fptr = end-1;
    return 1;
  }
  v->sv.val = beg;

  while(1) {
    if (step < 0) {
      if (v->sv.val < en) break;
    } else {
      if (v->sv.val > en) break;
    }
    fptr = start;
    while(fptr < end) {
      run(fptr);
      if (err == ERR_BREAK) {
	fptr = end;
	return err = 0;
      } else if (err == ERR_CONTINUE) {
	fptr = end;
	err = 0;
	break;
      }
      if (err || badstat(_status)) {
	fptr = end;
        return _status&RET_MASK;
      }
      fptr++;
    }
    if (err == ERR_BREAK) {
    }
    v->sv.val += step;
  }
  fptr = end;
  return 0;
}

NEXT(ferr)
FILE *ferr;
{
  fprintf(ferr,"next: Next without matching for statement.\n");
  return 1;
}

LABEL(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  if (n < 2) {
    fprintf(ferr,"label: missing label\n");
    return 1;
  }
  if (n > 3) {
    fprintf(ferr,"label: too many labels\n");
    return 1;
  }
  return 0;
}

GOTO(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  char *lab = NULL;
  char dir = 2;
  int ptr = fptr;

  if (!_source) return 0;
  for(a=1;arg[a];a++) {
    if (arg[a][0] == '-') {
      switch (arg[a][1]) {
	case 'f':
	  dir = 0;
	  break;
	case 'r':
	  dir = 1;
	  break;
	case 't':
	  dir = 2;
	  break;
	default:
	  fprintf(ferr,"goto: invalid switch %s\n",arg[a]);
	  return 1;
      }
    } else if (!lab) lab = arg[a];
      else {
	fprintf(ferr,"goto: illegal argument: %s",arg[a]);
	return 1;
      }
  }
  if (dir == 2) dir = ptr = 0;
  else if (dir) ptr--;
  else ptr++;
  if (dir) {
    for(; ptr >= 0 ; ptr--) {
      if (!strcmp(file[ptr][0],"label")) {
        if (!strcmp(file[ptr][1],lab)) {
	  err = ERR_GOTO;
	  fptr = ptr;
	  return 0;
	}
      }
    }
  } else {
    for(; file[ptr] ; ptr++) {
      if (!strcmp(file[ptr][0],"label")) {
        if (!strcmp(file[ptr][1],lab)) {
	  err = ERR_GOTO;
	  fptr = ptr;
	  return 0;
	}
      }
    }
  }
  fprintf(ferr,"goto: label '%s' not found.\n",lab);
  return 1;
}

FOREACH(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  int start = fptr+1, end = fptr+1, p = 2;
  char *ex = arg[1], nest = 0;

  if (!_source) return 1;

  while(file[end]) {
    if (!strcmp(file[end][0],"foreach")) {
      nest++;
      end++;
      continue;
    }
    if (!nest && !strcmp(file[end][0],"endfor")) break;
    if (nest && !strcmp(file[end][0],"endfor")) nest--;
    end++;
  }
  if (!file[end]) {
    fprintf(ferr,"foreach: foreach without matching endfor.\n");
    fptr = end-1;
    return 1;
  }
  if (n < 3) {
    fprintf(ferr,"foreach: Not enough arguments.\n");
    fptr = end;
    return 1;
  }

  while(arg[p]) {
    fptr = start;
    makeset(ex,arg[p++]);
    while (fptr < end) {
      run(fptr);
      if (err == ERR_BREAK) {
	fptr = end;
	break;
      } else if (err == ERR_CONTINUE) {
	fptr = end;
	err = 0;
	break;
      }
      if (err || badstat(_status)) {
	fptr = end;
        return 0;
      }
      fptr++;
    }
    if (err) break;
  }
  fptr = end;
  return 0;
}

ENDFOR(ferr)
FILE *ferr;
{
  fprintf(ferr,"endfor: endfor without foreach.\n");
  return 1;
}

char *grab(arg,start,hit,pt)
char **arg;
char *hit;
int start,*pt;
{
  static int i,j,k,p;
  static char *ex;

  j = 0;
  ex = NULL;
  if (!hit) {
    for(i=start;arg[i];i++) j+= strlen(arg[i]);
  } else {
    for(i=start;arg[i] && strcmp(arg[i],hit);i++) j+= strlen(arg[i]);
  }
  if (!j) return NULL;
  *pt = i;
  ex = (char *)malloc(j+1);
  for(p=0,k=start;k<i;k++) {
    for(j=0;arg[k][j];j++) ex[p++] = arg[k][j];
  }
  ex[p] = 0;
  return ex;
}

INC(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  struct _setvar *s;

  if (n < 2) {
    fprintf(ferr,"inc: variable expected.\n");
    return 1;
  }
  for(a=1;arg[a];a++) {
    s = find_var(arg[a]);
    if (!s) makenvar(arg[a],1);
    else {
      s->sv.val++;
    }
  }
  return 0;
}

DEC(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  struct _setvar *s;

  if (n < 2) {
    fprintf(ferr,"dec: variable expected.\n");
    return 1;
  }
  for(a=1;arg[a];a++) {
    s = find_var(arg[a]);
    if (!s) makenvar(arg[a],1);
    else {
      s->sv.val--;
    }
  }
  return 0;
}

EVAL(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  char *ex;

  if (n < 2) {
    fprintf(ferr,"eval: argument expected.\n");
    return 1;
  }
  ex = string(arg+1);
  run2(ex);
  free(ex);
  return 0;
}

EXECUTE(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  char *ex;

  if (n < 2) {
    fprintf(ferr,"exec: argument expected.\n");
    return 1;
  }
  ex = string(arg+1);
  _nofork = TRUE;
  run2(ex);
  _nofork = FALSE;
  free(ex);
  return 1;
}

SWITCH(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  char *s = arg[1], l = 0, **etmp, *tmp;
  int ptr = fptr+1, start = 0, end = 0, i, j, k;

  if (n < 2) {
    fprintf(ferr,"switch: string argument expected.\n");
    return 1;
  }
  if (n > 2) {
    fprintf(ferr,"switch: Too many arguments.\n");
    return 1;
  }
  for(;file[ptr]; ptr++) {
    if (!l && !start && file[ptr][1] && !strcmp("case",file[ptr][0])) {
      for(i=1;i<n && start != ptr;i++) {
	if (file[ptr][i][0] == '"' || file[ptr][i][0] == '\'') {
	  j = file[ptr][i][(k=strlen(file[ptr][i]))-1];
	  file[ptr][i][k-1] = 0;
	  etmp = evalw(tmp=file[ptr][i]+1);
	  file[ptr][i][k-1] = j;
	} else etmp = evalw(tmp=file[ptr][i]);
	for(j=0;etmp[j];j++)
	  if (patmatch(s,etmp[j])) {
	    start = ptr;
	    break;
	  }
	if (tmp != etmp[0])
	  for(j=0;etmp[j];j++) free(etmp[j]);
	free(etmp);
      }
    } else if (!l && !start && !strcmp("default",file[ptr][0])) {
      start = ptr;
    } else if (!strcmp("switch",file[ptr][0])) l++;
    else if (l && !strcmp("endsw",file[ptr][0])) l--;
    else if (!l && !strcmp("endsw",file[ptr][0])) {
      end = ptr;
      break;
    }
  }
  if (!end) {
    fprintf(ferr,"switch: switch without matching endsw.\n");
    return 1;
  }
  if (!start) {
    fptr = end;
    return 0;
  }
  fptr = start+1;
  while(fptr < end) {
    if (!strcmp("case",file[fptr][0]) || !strcmp("default",file[fptr][0])) {
      fptr++;
      continue;
    }
    run(fptr);
    if (err == ERR_BREAK) {
      fptr = end;
      err = 0;
      return 0;
    }
    if (err || badstat(_status)) {
      fptr = end;
      return _status&RET_MASK;
    }
    fptr++;
  }
  return 0;
}

ENDSW(ferr)
FILE *ferr;
{
  fprintf(ferr,"endsw: endsw without matching switch.\n");
  return 1;
}

usage(fout,ferr)
FILE *fout, *ferr;
{
  static struct rusage ru;

  getrusage(RUSAGE_SELF,&ru);
  a = ru.ru_utime.tv_sec;
  b = a%3600;
  c = b%60;
  fprintf(fout,"USER time: %02d:%02d:%02d.%06d\t",a/3600,b/60,c,ru.ru_utime.tv_usec);
  a = ru.ru_stime.tv_sec;
  b = a%3600;
  c = b%60;
  fprintf(fout,"SYSTEM time: %02d:%02d:%02d.%06d\n",a/3600,b/60,c,ru.ru_stime.tv_usec);
  fprintf(fout,"MAX RSS: %dK\tPage Reclaims: %d\t Page faults: %d\n",(ru.ru_maxrss*getpagesize())/1024,ru.ru_minflt,ru.ru_majflt);
  fprintf(fout,"SWAPS: %d\tBlocks In: %d\tBlocks Out: %d\tSignals: %d\n",ru.ru_nswap,ru.ru_inblock,ru.ru_oublock,ru.ru_nsignals);
  return 0;
}

isnum(w)
char *w;
{
  while(*w) if (!isdigit(*w++)) return FALSE;
  return TRUE;
}

INTR(n,arg,fout,ferr)
int n;
char **arg;
FILE *fout, *ferr;
{
  static char *sigs[] = {
    "HUP","INT","QUIT","ILL","TRAP","IOT","EMT","FPE","KILL","BUS","SEGV",
    "SYS","PIPE","ALRM","TERM","URG","STOP","TSTP","CONT","CHLD","TTIN",
    "TTOU","IO","XCPU","XFSZ","VTALRM","PROF","WINCH","USR1","USR2",0
  };

  if (n == 1) {
    for(a=0;sigs[a];a++)
      if (SIGMASK&(1<<a))
	fprintf(fout,"%s%s",a?" ":"",sigs[a]);
    fputc('\n',fout);
    return 0;
  }
  for(a=1;arg[a];a++) {
    if (arg[a][0] == '-' && !arg[a][1]) SIGMASK = 0;
    else if (arg[a][0] == '+' && !arg[a][1]) SIGMASK = 0xFFFFFFFF;
    else {
      for(b=0;sigs[b];b++) {
	if (!strcmp(sigs[b],arg[a])) {
	  SIGMASK ^= (1<<b);
	  break;
	}
      }
      if (!sigs[b]) {
	fprintf(ferr,"intr: %s: invalid argument.\n",arg[a]);
	return 1;
      }
    }
  }
  return 0;
}

TERM(n,arg,fout,ferr)
int n;
char **arg;
FILE *fout, *ferr;
{
  static char *t[12] = {"SO","SE","CE","KS","KE","DC","IC","DS","TC","FS","HS",0};
  struct _setvar *VAR;

  if (n == 1) {
    if ((VAR = find_var("term")) != NULL) fprintf(fout,"term: %s\n",VAR->sv.wrd[0]);
    for(a=0;a<10;a++) {
      fprintf(fout,"%s\t",t[a]);
      if (_term[a]) {
        fputc('\042',fout);
        fprt(fout,_term[a]);
        fputc('\042',fout);
      } else fputs("<undefined>",fout);
      fputc('\n',fout);
    }
    fprintf(fout,"HS\t%d\n",_term[HS]);
    return 0;
  } else {
    if (n == 2) {
      for(a=0;t[a];a++) {
        if (!strcmp(t[a],arg[1])) {
	  if (a == HS) fprintf(fout,"HS\t%d\n",_term[HS]);
	  else {
	    fprintf(fout,"%s\t",t[a]);
	    if (_term[a]) {
	      fputc('\042',fout);
	      fprt(fout,_term[a]);
	      fputc('\042',fout);
	    } else fputs("<undefined>",fout);
	    fputc('\n',fout);
	  }
	  return 0;
	}
      }
      fprintf(ferr,"term: %s not found.\n",arg[1]);
      return 1;
    } else {
      for(a=0;t[a];a++) {
        if (!strcmp(t[a],arg[1])) {
	  if (a == HS) _term[HS] = (char *)atoi(arg[2]);
	  else {
	    free(_term[a]);
	    _term[a] = string(arg+2);
	    if (!strlen(_term[a])) {
	      free(_term[a]);
	      _term[a] = NULL;
	    }
	  }
	  return 0;
	}
      }
      fprintf(ferr,"term: %s not found.\n",arg[1]);
      return 1;
    }
  }
}

PROTECT(n,arg,ferr)
int n;
char **arg;
FILE *ferr;
{
  struct _setvar *VAR;

  if (n < 2) {
    fprintf(ferr,"protect: missing argument.\n");
    return 1;
  }
  if ((VAR = find_var(arg[1])) == NULL) {
    fprintf(ferr,"protect: %s not found.\n",arg[1]);
    return 1;
  }
  VAR->protect = 1;
  return 0;
}

#ifdef NOSETENV
extern char **environ;
extern int errno;

int setenv(name,value,overwrite)
char *name, *value;
int overwrite;
{
  int i, l = strlen(name);
  char **tmp, **se = environ;

  if (name[l-1] == '=') l--;
  for(i=0;environ[i];i++) {
    if (!strncmp(name,environ[i],l) && environ[i][l] == '=') {
      if (overwrite) {
        free(environ[i]);
	break;
      }
      return 0;
    }
  }
  if (!environ[i]) {
    if ((environ = (char **)realloc(environ,sizeof(char *)*(i+2))) == NULL) {
      if (errno == EINVAL) {
	if ((tmp = (char **)malloc(sizeof(char *) * (i+2))) == NULL) return -1;
	for(i=0;se[i];i++) tmp[i] = se[i];
	environ = tmp;
      } else return -1;
    }
    environ[i+1] = NULL;
  }
  if ((environ[i] = (char *)malloc(l+strlen(value)+2)) == NULL) return -1;
  sprintf(environ[i],"%.*s=%s",l,name,value[0]=='='?value+1:value);
  return 0;
}

void unsetenv(name)
char *name;
{
  int i = 0, j, f = 0, l = strlen(name);

  if (name[l-1] == '=') l--;
  while(environ[i]) {
    if (!strncmp(name,environ[i],l) && environ[i][l] == '=') {
      f = 1;
      free(environ[i]);
      for(j=i;environ[j];j++) environ[j] = environ[j+1];
    } else i++;
  }
/*  if (f) environ = (char **)realloc(environ,sizeof(char *) * i+1); */
}
#endif
