/*
 * Command Input Shell
 * Dave Clemans
 * 12/88-1/89
 *
 * "spiritually" based on Bourne, Korn shells
 *
 * Handle shell variables
 *
 * $Id: var.c,v 1.12 89/03/07 19:35:06 dclemans Exp $
 *
 * $Log:	var.c,v $
 * Revision 1.12  89/03/07  19:35:06  dclemans
 * fix typo in GEMDOS code
 * 
 * Revision 1.11  89/03/07  19:05:53  dclemans
 * portability fixes
 * 
 * Revision 1.10  89/02/25  17:40:24  dclemans
 * miscellaneous bug fixes/speedups
 * 
 * Revision 1.9  89/02/22  21:32:07  dclemans
 * Implement simple background job monitoring facility
 * 
 * Revision 1.8  89/02/22  13:18:53  dclemans
 * Implement $!
 * 
 * Revision 1.7  89/02/22  08:17:18  dclemans
 * implement left-justified, right-justified, etc. parameter attributes
 * 
 * Revision 1.6  89/02/21  21:47:22  dclemans
 * Fix bugs in % and %% variable expansions
 * 
 * Revision 1.5  89/02/21  19:40:13  dclemans
 * Implement RANDOM and SECONDS variables
 * 
 * Revision 1.4  89/02/20  20:07:19  dclemans
 * Add RCS identifiers
 * 
 */
#include <stdio.h>
#include "shell.h"
#include <time.h>
extern unsigned long time();
#ifdef  LINED
#include "reader\history.h"
#endif  /* LINED */
#ifdef  GEMDOS
#include <basepage.h>
#endif  /* GEMDOS */

/*
 * Positional arguments
 */
char    *var_arg0;
int     var_argc;
char    **var_argv;

/*
 * Initial values of variables
 */
char    *var_init[] =
{
    "PS1",      "$ ",
    "PS2",      "> ",
    "PS3",      "#? ",
    "PS4",      "+ ",
    "IFS",      " \t",
#ifdef  GEMDOS
    "PATH",     ",c:\\bin",
    "SUFF",     ".ttp,.tos,.prg,.app,,.sh",
    "TMPDIR",   "",
    "COLUMNS",  "80",
    "LINES",    "24",
#else
    "PATH",     ":/bin:/usr/bin",
#endif  /* GEMDOS */
    (char *)NULL,   (char *)NULL
};

/*
 * Set up "reserved"/special handling variables
 */
#define VAR_CDPATH      0
#define VAR_COLUMNS     1
#define VAR_ENV         2
#define VAR_HISTSIZE    3
#define VAR_HOME        4
#define VAR_IFS         5
#define VAR_LINES       6
#define VAR_PATH        7
#define VAR_PS1         8
#define VAR_PS2         9
#define VAR_PS3         10
#define VAR_PS4         11
#define VAR_PWD         12
#define VAR_SHELL       13
#ifdef  GEMDOS
#define VAR_SUFF        14
#define VAR_TMPDIR      15
#endif  /* GEMDOS */
struct {
    char    *name;
    int     flag;
} reserved_names[] =
{   /* keep this list sorted... */
    { "CDPATH",     VAR_CDPATH },
    { "COLUMNS",    VAR_COLUMNS },
    { "ENV",        VAR_ENV },
    { "HISTSIZE",   VAR_HISTSIZE },
    { "HOME",       VAR_HOME },
    { "IFS",        VAR_IFS },
    { "LINES",      VAR_LINES },
    { "PATH",       VAR_PATH },
    { "PS1",        VAR_PS1 },
    { "PS2",        VAR_PS2 },
    { "PS3",        VAR_PS3 },
    { "PS4",        VAR_PS4 },
    { "PWD",        VAR_PWD },
    { "SHELL",      VAR_SHELL },
#ifdef  GEMDOS
    { "SUFF",       VAR_SUFF },
    { "TMPDIR",     VAR_TMPDIR },
#endif  /* GEMDOS */
    { (char *)NULL, -1 }
};  /* end of reserved_names */

int var_namecmp(n1,n2)
char *n1;
char *n2;
{
    int i1,i2,rc;
    char *p1,*p2;
    register char *p,*q;
    char buffer[BUFSIZ];

    p = strchr(n1,'[');
    if (p == (char *)NULL)
        q = p;
    else q = strrchr(n1,']');
    if (p == (char *)NULL || q == (char *)NULL)
    {   /* not an indexed variable? */
        p1 = n1;
        i1 = 0;
    }
    else
    {   /* indexed variable */
        strncpy(buffer,&p[1],(int)(q-p)-1);
        buffer[(int)(q-p)-1] = '\0';
        i1 = parse_c_expression(buffer,0);
        strncpy(buffer,n1,(int)(p-n1));
        p1 = strcopy(buffer);
    }
    p = strchr(n2,'[');
    if (p == (char *)NULL)
        q = p;
    else q = strrchr(n2,']');
    if (p == (char *)NULL || q == (char *)NULL)
    {   /* not an indexed variable? */
        p2 = n2;
        i2 = 0;
    }
    else
    {   /* indexed variable */
        strncpy(buffer,&p[1],(int)(q-p)-1);
        buffer[(int)(q-p)-1] = '\0';
        i2 = parse_c_expression(buffer,0);
        strncpy(buffer,n2,(int)(p-n2));
        p2 = strcopy(buffer);
    }
    rc = strcmp(p1,p2);
    if (p1 != n1)
        free(p1);
    if (p2 != n2)
        free(p2);
    if (rc != 0)
        return rc;
    if (i1 < i2)
        return -1;
    if (i1 > i2)
        return 1;
    return 0;
}   /* end of var_namecmp */

char *var_subeval(name,side_effects)
register char *name;
int side_effects;
{
    int temp;
    register char *p,*q;
    char buffer[BUFSIZ];

    p = strchr(name,'[');
    if (p == (char *)NULL)
        return (char *)NULL;
    q = strrchr(name,']');
    if (q == (char *)NULL)
        return (char *)NULL;
    strncpy(buffer,&p[1],(int)(q-p)-1);
    buffer[(int)(q-p)-1] = '\0';
    temp = 1;
    for (q = buffer; *q; q++)
        if (!isdigit(*q))
            temp = 0;
    if (temp)
    {   /* if doesn't really need evaluation */
        return (char *)NULL;
    }
    temp = parse_c_expression(buffer,side_effects);
    strncpy(buffer,name,(int)(p-name));
    sprintf(&buffer[(int)(p-name)],"[%d]",temp);
    return strcopy(buffer);
}   /* end of var_subeval */

static void dump_var(vp,flag)
register struct variable *vp;
int flag;
{
    long temp;
    char buffer[32];
#ifndef GEMDOS
#ifndef USG
    extern long random();
#else
#endif  /* USG */
#endif  /* GEMDOS */

    io_writestring(0,vp->name);
    if (flag)
    {   /* also write out value */
        io_writestring(0,"=");
        if (vp->type & TYPE_FUNCTION)
        {   /* a "special" variable? */
            if (strcmp(vp->name,"RANDOM") == 0)
            {   /* return a random number */
#ifndef GEMDOS
#ifndef USG
                temp = random();
#else
                temp = rand();
#endif  /* USG */
#else
                temp = Random();
#endif  /* GEMDOS */
                sprintf(buffer,"%ld",temp);
                io_writestring(0,buffer);
            }
            else if (strcmp(vp->name,"SECONDS") == 0)
            {   /* how long have we been running? */
                temp = time(0L);
                sprintf(buffer,"%ld",temp-base_env.start_at);
                io_writestring(0,buffer);
            }
        }
        else io_writestring(0,vp->value);
    }
    io_writestring(0,"\n");
}   /* end of dump_var */

static void var_reserved(vp)
register struct variable *vp;
{
    register int i;
    int temp1,temp2;
    register struct hist_phrase *nh;

    for (i = 0; reserved_names[i].name != (char *)NULL; i++)
    {   /* a variable with special handling when set? */
        temp1 = strcmp(vp->name,reserved_names[i].name);
        if (temp1 > 0)
            continue;
        if (temp1 < 0)
            return;
        switch (reserved_names[i].flag)
        {   /* found a variable that needs special handling; go do it */
            case VAR_CDPATH:
                base_env.cd_path = vp->value;
                break;
            case VAR_COLUMNS:
                base_env.columns = atoi(vp->value)-1;   /* last usable column */
                break;
            case VAR_ENV:
                base_env.envfile = vp->value;
                break;
            case VAR_HISTSIZE:
                temp1 = atoi(vp->value);
                if (temp1 <= 0)
                {   /* completely kill old list */
                    if (base_env.history_size > 0)
                    {   /* need to free old stuff? */
                        for (temp2 = 0; temp2 < base_env.history_size; temp2++)
                            if (base_env.history_list[temp2].cmd != (struct phrase *)NULL)
                                phrase_free(base_env.history_list[temp2].cmd);
                        free(base_env.history_list);
                        base_env.history_list = (struct hist_phrase *)NULL;
                        base_env.history_size = 0;
                    }
#ifdef  LINED
                    History.maxSize = 1;
#endif  /* LINED */
                    return;
                }
                nh = (struct hist_phrase *)malloc(sizeof(struct hist_phrase)*temp1);
                if (nh == (struct hist_phrase *)NULL)
                {   /* out of memory? */
                    errmsg(SHERR_NOMEM,LOC("var_reserved"));
                    return;
                }
                for (temp2 = 0; temp2 < temp1; temp2++)
                {   /* make sure it starts out empty */
                    nh[temp2].cmd = (struct phrase *)NULL;
                    nh[temp2].number = -1;
                }
                for (temp2 = 0; temp2 < ((temp1<base_env.history_size)?temp1:base_env.history_size); temp2++)
                {   /* copy in old stuff */
                    nh[temp2].cmd = base_env.history_list[temp2].cmd;
                    nh[temp2].number = base_env.history_list[temp2].number;
                }
                for (; temp2 < base_env.history_size; temp2++)
                    if (base_env.history_list[temp2].cmd != (struct phrase *)NULL)
                        phrase_free(base_env.history_list[temp2].cmd);
                if (base_env.history_size > 0)
                    free(base_env.history_list);
                base_env.history_size = temp1;
#ifdef  LINED
                History.maxSize = temp1;
#endif  /* LINED */
                base_env.history_list = nh;
                        break;
            case VAR_HOME:
                base_env.homedir = vp->value;
                break;
            case VAR_IFS:
                base_env.separators = vp->value;
                break;
            case VAR_LINES:
                base_env.lines = atoi(vp->value);
                break;
            case VAR_PS1:
                base_env.prompts[0] = vp->value;
                break;
            case VAR_PS2:
                base_env.prompts[1] = vp->value;
                break;
            case VAR_PS3:
                base_env.prompts[2] = vp->value;
                break;
            case VAR_PS4:
                base_env.prompts[3] = vp->value;
                break;
            case VAR_PATH:
                base_env.exec_path = vp->value;
                break;
            case VAR_PWD:
                if (base_env.dir->current != (char *)NULL)
                    free(base_env.dir->current);
                base_env.dir->current = strcopy(vp->value);
                break;
            case VAR_SHELL:
                base_env.shellname = strcopy(vp->value);
                break;
#ifdef  GEMDOS
            case VAR_SUFF:
                base_env.exec_suff = vp->value;
                break;
            case VAR_TMPDIR:
                base_env.tmpdir = vp->value;
                break;
#endif  /* GEMDOS */
        }
        break;
    }
}   /* end of var_reserved */

void var_setarg0(arg)
char *arg;
{
    if (arg == (char *)NULL || strlen(arg) == 0)
        var_arg0 = "shell";
    else var_arg0 = arg;
    var_define0("SHELL",var_arg0,0);
}   /* end of var_setarg0 */

void var_setargs(argc,argv)
int argc;
char *argv[];
{
    register int i;

    var_argc = argc;
    var_argv = new_argv(var_argc+1);
    if (var_argv == (char **)NULL)
    {   /* enough memory */
        errmsg(SHERR_NOMEM,LOC("var_setargs"));
        var_argc = 0;
        return;
    }
    var_argv[0] = var_arg0;
    for (i = 1; i < var_argc; i++)
    {   /* now set up the args */
        var_argv[i] = strcopy(argv[i]);
    }
    var_argv[i] = (char *)NULL;
}   /* end of var_setargs */

void var_resetargs(tp)
struct token *tp;
{
    register int i;
    register struct token *tpp;

    if (var_argc > 0)
    {   /* free old args */
        for (i = 1; i < var_argc; i++)
            if (var_argv[i] != (char *)NULL)
                free(var_argv[i]);
        free(var_argv);
    }
    var_argc = 0;
    for (tpp = tp; tpp != (struct token *)NULL; tpp = tpp->next)
        var_argc++;
    var_argv = new_argv(var_argc+2);
    if (var_argv == (char **)NULL)
    {   /* enough memory */
        errmsg(SHERR_NOMEM,LOC("var_resetargs"));
        var_argc = 0;
        return;
    }
    var_argv[0] = var_arg0;
    tpp = tp;
    for (i = 0; i < var_argc; i++)
    {   /* now set up the args */
        var_argv[i+1] = strcopy(tpp->name);
        tpp = tpp->next;
    }
    var_argv[i+1] = (char *)NULL;
    var_argc++;
}   /* end of var_resetargs */

void var_shiftargs(count)
int count;
{
    register int i,j;

    if (var_argv[1] != (char *)NULL)
        free(var_argv[1]);
    for (i = 0; i < count; i++)
    {   /* shift the required number of arguments */
        for (j = 1; j < var_argc; j++)
            var_argv[j] = var_argv[j+1];
        var_argc--;
    }
}   /* end of var_shiftargs */

static void unpad(s)
register char *s;
{
    register char *p;

    while (*s == ' ')
        strcpy(s,&s[1]);
    for (p = s; *p; p++)
        /* do nothing */;
    p--;
    while (p > s && *p == ' ')
        *p-- = '\0';
}   /* end of unpad */

static void var_fixtype(vp,side_effects)
register struct variable *vp;
int side_effects;
{
    register char *p;
    register int temp;
    char buffer[16];
    char *value;

    if (vp->value == (char *)NULL)
        return;
    if (vp->type & TYPE_INTEGER)
    {   /* if need to eval value */
        temp = 1;
        for (p = vp->value; *p; p++)
            if (!isdigit(*p))
                temp = 0;
        if (!temp)
        {   /* if need to eval an expression */
            temp = parse_c_expression(vp->value,side_effects);
            sprintf(buffer,"%d",temp);
            free(vp->value);
            vp->value = strcopy(buffer);
        }
    }
    if (vp->type & TYPE_UPPERCASE)
    {   /* if should be upcased... */
        for (p = vp->value; *p; p++)
            if (islower(*p))
                *p = _toupper(*p);
    }
    if (vp->type & TYPE_LOWERCASE)
    {   /* if should be lowercased... */
        for (p = vp->value; *p; p++)
            if (isupper(*p))
                *p = _tolower(*p);
    }
    if (vp->type & TYPE_LEFTJUST)
    {   /* left justify the value */
        unpad(vp->value);
        if (vp->type & TYPE_ZEROS)
        {   /* eliminate any initial zeros */
            while (vp->value[0] == '0')
                strcpy(vp->value,&vp->value[1]);
        }
        value = new_string(vp->misc+1);
        if (value == (char *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("var_fixtype"));
            return;
        }
        sprintf(buffer,"%%-%ds",vp->misc);
        sprintf(value,buffer,vp->value);
        free(vp->value);
        vp->value = value;
    }
    if (vp->type & TYPE_RIGHTJUST)
    {   /* right justify the value */
        unpad(vp->value);
        value = new_string(vp->misc+1);
        if (value == (char *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("var_fixtype"));
            return;
        }
        if (vp->type & TYPE_ZEROS)
            sprintf(buffer,"%%0%ds",vp->misc);
        else sprintf(buffer,"%%%ds",vp->misc);
        sprintf(value,buffer,vp->value);
        free(vp->value);
        vp->value = value;
    }
    if (vp->type & TYPE_HOSTMAP)
    {   /* make into good filename */
        /* a start... */
#ifdef  GEMDOS
        for (p = vp->value; *p; p++)
        {   /* check each char... */
            if (*p == '/')
                *p = '\\';
            if (isupper(*p))
                *p = _tolower(*p);
        }
#else
        for (p = vp->value; *p; p++)
            if (*p == '\\')
                *p = '/';
#endif  /* GEMDOS */
    }
}   /* end of var_fixtype */

static void var_setvalue(svp,value)
register struct variable *svp;
register char *value;
{
    if (svp->type & TYPE_READONLY)
    {   /* can value be changed? */
        errmsg(0,LOC("var_define0"),"value of '%s' is readonly",svp->name);
        return;
    }
    if (svp->type == TYPE_DELETED)
        svp->type = 0;
    if (svp->value != (char *)NULL)
        free(svp->value);
    svp->value = strcopy(value);
    var_fixtype(svp,1);
    var_reserved(svp);
    svp->cmd = cmd_count;
    if (flag_allexport)
        svp->type |= TYPE_EXPORTED;
}   /* end of var_setvalue */

void var_define0(name,value,create)
char *name;
char *value;
int create;
{
    int rc;
    char *xname,*ptr;
    register struct variable *vp,*svp;
    struct varstack *vs;

    if (strlen(name) == 1 && (strchr("*@$!?-#",*name) != (char *)NULL || isdigit(*name)))
    {   /* can't do this directly */
        errmsg(0,LOC("var_define0"),"illegal assignment: '%s'",name);
        return;
    }

    vp = new_variable();
    if (vp == (struct variable *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("var_define0"));
        return;
    }

    xname = var_subeval(name,1);
    if (xname == (char *)NULL)
        ptr = name;
    else ptr = xname;
    vs = base_env.var;

    if (!create)
    {   /* if to use existing created slot if possible */
        for (vs = base_env.var; vs != (struct varstack *)NULL; vs = vs->next)
        {   /* for each variable table */
            for (svp = vs->table; svp != (struct variable *)NULL; )
            {   /* look for where to get the variable */
                rc = var_namecmp(ptr,svp->name);
                if (rc == 0)
                {   /* found the variable */
                    if (svp->type == TYPE_DELETED)
                        svp = (struct variable *)NULL;
                    break;
                }
                else if (rc < 0)
                {   /* go down the left side? */
                    svp = svp->left;
                }
                else
                {   /* go down right side */
                    svp = svp->right;
                }
            }
            if (svp != (struct variable *)NULL)
                break;
        }
        if (svp != (struct variable *)NULL)
        {   /* if found "correct" place for variable */
            free(vp);
            var_setvalue(svp,value);
            if (xname != (char *)NULL)
                free(xname);
            return;
        }
        for (vs = base_env.var; vs->next != (struct varstack *)NULL; vs = vs->next)
            /* do nothing */;
    }

    for (svp = vs->table; svp != (struct variable *)NULL; )
    {   /* look for where to put the variable */
        rc = var_namecmp(ptr,svp->name);
        if (rc == 0)
        {   /* replace existing value */
            free(vp);
            var_setvalue(svp,value);
            if (xname != (char *)NULL)
                free(xname);
            return;
        }
        else if (rc < 0)
        {   /* go down the left side? */
            if (svp->left == (struct variable *)NULL)
                break;
            svp = svp->left;
        }
        else
        {   /* go down right side */
            if (svp->right == (struct variable *)NULL)
                break;
            svp = svp->right;
        }
    }
    vp->name = strcopy(ptr);
    if (base_env.var->table == (struct variable *)NULL)
        base_env.var->table = vp;
    else if (rc < 0)
        svp->left = vp;
    else
        svp->right = vp;
    var_setvalue(vp,value);
    if (xname != (char *)NULL)
        free(xname);
}   /* end of var_define0 */

void var_define(word,type,create)
char *word;
int type;
int create;
{
    register char *p,*ptr;
    register struct variable *svp;
    int result,rc;
    char buffer[16];
    struct varstack *vs;
    char *xname;

    result = 0;
    for (p = word; result != 0 || *p != '='; p++)
    {   /* find end of name */
        if (*p == '[')
            result++;
        else if (*p == ']')
            result--;
    }
    /* ASSERT that *p == '=' */
    *p++ = '\0';
    xname = var_subeval(word,1);
    if (xname == (char *)NULL)
        ptr = word;
    else ptr = xname;

    svp = (struct variable *)NULL;
    for (vs = base_env.var; !create && vs != (struct varstack *)NULL; vs = vs->next)
    {   /* for each variable table */
        for (svp = vs->table; svp != (struct variable *)NULL; )
        {   /* look for where to get the variable */
            rc = var_namecmp(ptr,svp->name);
            if (rc == 0)
            {   /* found the variable */
                if (svp->type == TYPE_DELETED)
                    svp = (struct variable *)NULL;
                break;
            }
            else if (rc < 0)
            {   /* go down the left side? */
                svp = svp->left;
            }
            else
            {   /* go down right side */
                svp = svp->right;
            }
        }
        if (svp != (struct variable *)NULL)
            break;
    }
    if (svp != (struct variable *)NULL)
    {   /* known spot for variable */
        if (svp->type & TYPE_INTEGER)
        {   /* an integer type? */
            sprintf(buffer,"%d",parse_c_expression(p,1));
            var_setvalue(svp,buffer);
        }
        else
        {   /* normal strategy */
            var_setvalue(svp,p);
        }
        if (svp->type == type)
        {   /* no type change needed? */
            if (xname != (char *)NULL)
                free(xname);
            return;
        }
        var_settype(ptr,type,~0);
        if (xname != (char *)NULL)
            free(xname);
        return;
    }

    var_define0(ptr,p,create);
    var_settype(ptr,type,~0);
    *--p = '=';
    if (xname != (char *)NULL)
        free(xname);
}   /* end of var_define */

void var_settype(name,type,mask)
char *name;
int type;
int mask;
{
    register struct variable *svp;
    int rc,otype;
    char *ptr,*xname;
    struct varstack *vs;

    xname = var_subeval(name,0);
    if (xname == (char *)NULL)
        ptr = name;
    else ptr = xname;
    for (vs = base_env.var; vs != (struct varstack *)NULL; vs = vs->next)
    {   /* for each variable table */
        for (svp = vs->table; svp != (struct variable *)NULL; )
        {   /* look for where to put the variable */
            rc = var_namecmp(ptr,svp->name);
            if (rc == 0)
            {   /* found value, set type */
                otype = svp->type;
                if (type == TYPE_DELETED)
                {   /* remove this variable */
                    svp->type = type;
                    free(svp->value);
                    svp->value = (char *)NULL;
                    if (xname != (char *)NULL)
                        free(xname);
                    return;
                }
                svp->type &= ~TYPE_DELETED;
                svp->type &= mask;
                svp->type |= type;
                if (xname != (char *)NULL)
                    free(xname);
                if (svp->type != otype)
                {   /* was there any change? */
                    var_fixtype(svp,0);
                    var_reserved(svp);
                }
                return;
            }
            else if (rc < 0)
            {   /* go down the left side? */
                if (svp->left == (struct variable *)NULL)
                    break;
                svp = svp->left;
            }
            else
            {   /* go down right side */
                if (svp->right == (struct variable *)NULL)
                    break;
                svp = svp->right;
            }
        }
    }
    /* variable not found; make dummy one, set its type */
    if (type == TYPE_DELETED)
    {   /* if we really don't need to */
        if (xname != (char *)NULL)
            free(xname);
        return;
    }
    var_define0(ptr,"",0);
    var_settype(ptr,type,mask);
    if (xname != (char *)NULL)
        free(xname);
}   /* end of var_settype */

void var_setmisc(name,misc)
char *name;
int misc;
{
    register struct variable *svp;
    int rc;
    char *ptr,*xname;
    struct varstack *vs;

    xname = var_subeval(name,0);
    if (xname == (char *)NULL)
        ptr = name;
    else ptr = xname;
    for (vs = base_env.var; vs != (struct varstack *)NULL; vs = vs->next)
    {   /* for each variable table */
        for (svp = vs->table; svp != (struct variable *)NULL; )
        {   /* look for where to put the variable */
            rc = var_namecmp(ptr,svp->name);
            if (rc == 0)
            {   /* found value, set type */
                if (svp->type == TYPE_DELETED)
                {   /* really there? */
                    break;
                }
                svp->misc = misc;
                if (xname != (char *)NULL)
                    free(xname);
                var_fixtype(svp,0);
                var_reserved(svp);
                return;
            }
            else if (rc < 0)
            {   /* go down the left side? */
                if (svp->left == (struct variable *)NULL)
                    break;
                svp = svp->left;
            }
            else
            {   /* go down right side */
                if (svp->right == (struct variable *)NULL)
                    break;
                svp = svp->right;
            }
        }
    }
    /* variable not found; make dummy one, set its type */
    var_define0(ptr,"",0);
    var_setmisc(ptr,misc);
    if (xname != (char *)NULL)
        free(xname);
}   /* end of var_setmisc */

char *var_special(name)
register char *name;
{
    int rc,i;
    char *p;
    char buf[2];
    char word[BUFSIZ];

    if (strlen(name) == 2 &&
        (isdigit(name[1]) || name[1] == '*' || name[1] == '@'))
    {   /* reference to positional args? */
        p = (char *)NULL;
        rc = 0;
        if (name[1] == '0')
            p = var_arg0;
        else if (isdigit(name[1]))
        {   /* a single arg */
            if ((name[1]-'0') < var_argc)
                p = var_argv[name[1]-'0'];
            else
            {   /* supply a null string */
                p = "''";
            }
        }
        else if (name[1] == '*')
        {   /* all args, separate entities */
            rc = 0;
            for (i = 1; i < var_argc; i++)
                rc += strlen(var_argv[i])+3;
            rc++;
            p = new_string(rc);
            if (p == (char *)NULL)
            {   /* enough memory? */
                errmsg(SHERR_NOMEM,LOC("var_special"));
                return (char *)NULL;
            }
            *p = '\0';
            for (i = 1; i < var_argc; i++)
            {   /* now tack on args */
                strcat(p,"'");
                strcat(p,var_argv[i]);
                strcat(p,"'");
                if ((i+1) < var_argc)
                    strcat(p," ");
            }
        }
        else if (name[1] == '@')
        {   /* all args, one hunk */
            buf[0] = base_env.separators[0];
            buf[1] = '\0';
            rc = 0;
            for (i = 1; i < var_argc; i++)
                rc += strlen(var_argv[i])+1;
            rc += 3;
            p = new_string(rc);
            if (p == (char *)NULL)
            {   /* enough memory? */
                errmsg(SHERR_NOMEM,LOC("var_special"));
                return (char *)NULL;
            }
            *p = '\0';
            strcat(p,"'");
            for (i = 1; i < var_argc; i++)
            {   /* now tack on args */
                strcat(p,var_argv[i]);
                if ((i+1) < var_argc)
                    strcat(p,buf);
            }
            strcat(p,"'");
        }
        if (p == (char *)NULL)
            p = "";
        if (rc == 0)
            p = strcopy(p);
        return p;
    }
    if  (strlen(name) == 2 && name[1] == '?')
    {   /* status of last cmd */
        sprintf(word,"%d",cmd_lastrc);
        return strcopy(word);
    }
    if (strlen(name) == 2 && name[1] == '#')
    {   /* number of args */
        sprintf(word,"%d",var_argc);
        return strcopy(word);
    }
    if (strlen(name) == 2 && name[1] == '$')
    {   /* current pid */
#ifdef  GEMDOS
        sprintf(word,"%lx",BP);
#else
        sprintf(word,"%d",getpid());
#endif  /* GEMDOS */
        return strcopy(word);
    }
    if (strlen(name) == 2 && name[1] == '!')
    {   /* last background pid */
#ifdef  GEMDOS
        strcpy(word,"0");
#else
        sprintf(word,"%d",base_env.background_pid);
#endif  /* GEMDOS */
        return strcopy(word);
    }
    if (strlen(name) == 2 && name[1] == '-')
    {   /* status of options */
        p = word;
        *p = '\0';
        if (flag_allexport)
            *p++ = 'a';
        if (flag_noglob)
            *p++ = 'f';
        if (flag_cmdhash)
            *p++ = 'h';
        if (flag_interactive)
            *p++ = 'i';
        if (flag_keywords)
            *p++ = 'k';
        if (flag_monitor)
            *p++ = 'm';
        if (flag_noexec)
            *p++ = 'n';
        if (flag_varerr)
            *p++ = 'u';
        if (flag_echoinput)
            *p++ = 'v';
        if (flag_echoexec)
            *p++ = 'x';
        *p = '\0';
        return strcopy(word);
    }
    return (char *)NULL;
}   /* end of var_special */

char *var_normal(name)
register char *name;
{
    int rc;
    register struct variable *svp;
    struct varstack *vs;
    long temp;
    char buffer[32];
#ifndef GEMDOS
#ifndef USG
    extern long random();
#else
#endif  /* USG */
#endif  /* GEMDOS */

    for (vs = base_env.var; vs != (struct varstack *)NULL; vs = vs->next)
    {   /* for each variable table */
        for (svp = vs->table; svp != (struct variable *)NULL; )
        {   /* look for where to get the variable */
            rc = var_namecmp(&name[1],svp->name);
            if (rc == 0)
            {   /* found the variable */
                if (svp->type == TYPE_DELETED)
                    break;
                if (svp->type & TYPE_FUNCTION)
                {   /* "special" variables, like "RANDOM" and "SECONDS" */
                    if (strcmp(&name[1],"RANDOM") == 0)
                    {   /* return a random number */
#ifndef GEMDOS
#ifndef USG
                        temp = random();
#else
                        temp = rand();
#endif  /* USG */
#else
                        temp = Random();
#endif  /* GEMDOS */
                        sprintf(buffer,"%ld",temp);
                        return strcopy(buffer);
                    }
                    else if (strcmp(&name[1],"SECONDS") == 0)
                    {   /* how long have we been running? */
                        temp = time(0L);
                        sprintf(buffer,"%ld",temp-base_env.start_at);
                        return strcopy(buffer);
                    }
                }
                return strcopy(svp->value);
            }
            else if (rc < 0)
            {   /* go down the left side? */
                svp = svp->left;
            }
            else
            {   /* go down right side */
                svp = svp->right;
            }
        }
    }
    if (flag_varerr)
        errmsg(0,LOC("var_normal"),"%s: variable not found",name);
    return (char *)NULL;
}   /* end of var_normal */

static int var_0submatches(name,svp)
char *name;
register struct variable *svp;
{
    register int matches;

    matches = 0;
    if (svp->left != (struct variable *)NULL)
        matches += var_submatches(name,svp->left);
    if (wild_match(svp->name,&name[1]))
        matches++;
    if (svp->right != (struct variable *)NULL)
        matches += var_submatches(name,svp->right);
    return matches;
}   /* end of var_0submatches */

static int var_submatches(name)
char *name;
{
    struct varstack *vs;
    register int accum;

    accum = 0;
    for (vs = base_env.var; vs != (struct varstack *)NULL; vs = vs->next)
        accum += var_0submatches(name,vs->table);
    return accum;
}   /* end of var_submatches */

static int var_0subgetlen(name,svp)
char *name;
register struct variable *svp;
{
    register int length;

    length = 0;
    if (svp->left != (struct variable *)NULL)
        length += var_0subgetlen(name,svp->left);
    if (wild_match(svp->name,&name[1]))
        length += strlen(svp->value)+3;
    if (svp->right != (struct variable *)NULL)
        length += var_0subgetlen(name,svp->right);
    return length;
}   /* end of var_0subgetlen */

static int var_subgetlen(name)
char *name;
{
    struct varstack *vs;
    register int accum;                               

    accum = 0;
    for (vs = base_env.var; vs != (struct varstack *)NULL; vs = vs->next)
        accum += var_0subgetlen(name,vs->table);
    return accum;
}   /* end of var_subgetlen */

static void var_0subfilbuf(name,svp,buf,flag)
register char *name;
register struct variable *svp;
register char *buf;
int flag;
{
    if (svp->left != (struct variable *)NULL)
        var_0subfilbuf(name,svp->left,buf,flag);
    if (wild_match(svp->name,&name[1]))
    {   /* tack on the value of this name */
        if (strlen(svp->value) > 0 || flag)
        {    /* tack this one on? */
            strcat(buf,svp->value);
            strcat(buf," ");
        }
    }
    if (svp->right != (struct variable *)NULL)
        var_0subfilbuf(name,svp->right,buf,flag);
}   /* end of var_0subfilbuf */

static void var_subfilbuf(name,buf,flag)
char *name;
char *buf;
int flag;
{
    struct varstack *vs;

    for (vs = base_env.var; vs != (struct varstack *)NULL; vs = vs->next)
        var_0subfilbuf(name,vs->table,buf,flag);
}   /* end of var_subfilbuf */

char *var_getval();

char *var_expression(name)
register char *name;
{
    register char *p,*q;
    char buffer[BUFSIZ];
    int temp;

    sprintf(buffer,"#*%c[%c*%c]",ESCAPE_CHAR,ESCAPE_CHAR,ESCAPE_CHAR);
    if (wild_match(&name[1],buffer))
    {   /* number of matching subscripted vars defined */
        strcpy(buffer,name);
        sprintf(&buffer[strlen(buffer)-3],"%c[*%c]",ESCAPE_CHAR,ESCAPE_CHAR);
        temp = var_submatches(&buffer[1]);
        sprintf(buffer,"%d",temp);
        return strcopy(buffer);
    }
    if (wild_match(&name[1],"#*"))
    {   /* get length of var instead */
        p = var_getval(&name[1],1);
        strcpy(buffer,"0");
        if (p != (char *)NULL)
        {   /* if a value to look at */
            sprintf(buffer,"%d",strlen(p));
            free(p);
        }
        return strcopy(buffer);
    }
    sprintf(buffer,"*%c[%c@%c]",ESCAPE_CHAR,ESCAPE_CHAR,ESCAPE_CHAR);
    if (wild_match(&name[1],buffer))
    {   /* all values */
        strcpy(buffer,name);
        sprintf(&buffer[strlen(buffer)-3],"%c[*%c]",ESCAPE_CHAR,ESCAPE_CHAR);
        temp = var_subgetlen(buffer);
        p = new_string(temp+2);
        if (p == (char *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("var_expression"));
            return (char *)NULL;
        }
        var_subfilbuf(buffer,p,0);
        return p;
    }
    sprintf(buffer,"*%c[%c*%c]",ESCAPE_CHAR,ESCAPE_CHAR,ESCAPE_CHAR);
    if (wild_match(&name[1],buffer))
    {   /* all values */
        strcpy(buffer,name);
        sprintf(&buffer[strlen(buffer)-3],"%c[*%c]",ESCAPE_CHAR,ESCAPE_CHAR);
        temp = var_subgetlen(buffer);
        p = new_string(temp+2);
        if (p == (char *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("var_expression"));
            return (char *)NULL;
        }
        var_subfilbuf(buffer,p,1);
        return p;
    }
    sprintf(buffer,"*%c[*%c]",ESCAPE_CHAR,ESCAPE_CHAR);
    if (wild_match(&name[1],buffer))
    {   /* a subscripted variable */
        p = var_subeval(name,1);
        if (p == (char *)NULL)
            q = name;
        else q = p;
        q = var_getval(q,0);
        if (p != (char *)NULL)
            free(p);
        return q;
    }
    if (wild_match(&name[1],"*:-*"))
    {   /* alternatives, but non-null */
        p = strchr(name,':');
        /* ASSERT that p != NULL */
        while (p[1] != '-')
            p = strchr(&p[1],':');
        /* ASSERT that p != NULL */
        strncpy(buffer,name,(int)(p-name));
        buffer[(int)(p-name)] = '\0';
        q = var_getval(buffer,1);
        if (q != (char *)NULL && strlen(q) > 0)
            return q;
        if (q != (char *)NULL)
            free(q);
        return strcopy(&p[2]);
    }
    if (wild_match(&name[1],"*-*"))
    {   /* alternatives */
        p = strchr(name,'-');
        /* ASSERT that p != NULL */
        strncpy(buffer,name,(int)(p-name));
        buffer[(int)(p-name)] = '\0';
        q = var_getval(buffer,1);
        if (q != (char *)NULL)
            return q;
        return strcopy(&p[1]);
    }
    if (wild_match(&name[1],"*:=*"))
    {   /* alternatives, assignment, non-null */
        p = strchr(name,':');
        /* ASSERT that p != NULL */
        while (p[1] != '=')
            p = strchr(&p[1],':');
        /* ASSERT that p != NULL */
        strncpy(buffer,name,(int)(p-name));
        buffer[(int)(p-name)] = '\0';
        q = var_getval(buffer,1);
        if (q != (char *)NULL && strlen(q) > 0)
            return q;
        if (q != (char *)NULL)
            free(q);
        var_define0(&buffer[1],&p[2],0);
        return strcopy(&p[2]);
    }
    if (wild_match(&name[1],"*=*"))
    {   /* alternatives, assignment */
        p = strchr(name,'=');
        /* ASSERT that p != NULL */
        strncpy(buffer,name,(int)(p-name));
        buffer[(int)(p-name)] = '\0';
        q = var_getval(buffer,1);
        if (q != (char *)NULL)
            return q;
        var_define0(&buffer[1],&p[1],0);
        return strcopy(&p[1]);
    }
    sprintf(buffer,"*:%c?*",ESCAPE_CHAR);
    if (wild_match(&name[1],buffer))
    {   /* alternatives with printing, non-null */
        p = strchr(name,':');
        /* ASSERT that p != NULL */
        while (p[1] != '?')
            p = strchr(&p[1],':');
        /* ASSERT that p != NULL */
        strncpy(buffer,name,(int)(p-name));
        buffer[(int)(p-name)] = '\0';
        q = var_getval(buffer,1);
        if (q != (char *)NULL && strlen(q) != 0)
            return q;
        if (q != (char *)NULL)
            free(q);
        strcpy(buffer,&buffer[1]);
        if (p[2] != '\0')
            strcpy(buffer,&p[2]);
        else    strcat(buffer,": parameter null or not set");
        io_writestring(0,buffer);
        io_writestring(0,"\n");
        return strcopy("");
    }
    sprintf(buffer,"*%c??*",ESCAPE_CHAR);
    if (wild_match(&name[1],buffer))
    {   /* alternatives with printing */
        p = strchr(name,'?');
        /* ASSERT that p != NULL */
        strncpy(buffer,name,(int)(p-name));
        buffer[(int)(p-name)] = '\0';
        q = var_getval(buffer,1);
        if (q != (char *)NULL)
            return q;
        strcpy(buffer,&buffer[1]);
        if (p[1] != '\0')
            strcpy(buffer,&p[1]);
        else    strcat(buffer,": parameter not set");
        io_writestring(0,buffer);
        io_writestring(0,"\n");
        return strcopy("");
    }
    if (wild_match(&name[1],"*:+*"))
    {   /* substitution, non-null */
        p = strchr(name,':');
        /* ASSERT that p != NULL */
        while (p[1] != '+')
            p = strchr(&p[1],':');
        /* ASSERT that p != NULL */
        strncpy(buffer,name,(int)(p-name));
        buffer[(int)(p-name)] = '\0';
        q = var_getval(buffer,1);
        if (q != (char *)NULL && strlen(q) != 0)
        {   /* do substitution */
            free(q);
            return strcopy(&p[2]);
        }
        if (q != (char *)NULL)
            free(q);
        return strcopy("");
    }
    if (wild_match(&name[1],"*+*"))
    {   /* substitution */
        p = strchr(name,'-');
        /* ASSERT that p != NULL */
        strncpy(buffer,name,(int)(p-name));
        buffer[(int)(p-name)] = '\0';
        q = var_getval(buffer,1);
        if (q != (char *)NULL)
        {   /* do substitution */
            free(q);
            return strcopy(&p[1]);
        }
        return strcopy("");
    }
    if (wild_match(&name[1],"*##*"))
    {   /* cut from beginning */
        p = strchr(name,'#');
        while (p[1] != '#')
            p = strchr(&p[1],'#');
        /* ASSERT that p != NULL */
        strncpy(buffer,name,(int)(p-name));
        buffer[(int)(p-name)] = '\0';
        q = var_getval(buffer,1);
        if (q == (char *)NULL)
            return q;
        for (temp = strlen(q)-1; temp >= 0; temp--)
        {   /* looks for first largest portion that matches */
            strncpy(buffer,q,temp+1);
            buffer[temp+1] = '\0';
            if (wild_match(buffer,&p[2]))
            {   /* found a match? */
                strcpy(buffer,&q[temp+1]);
                free(q);
                return strcopy(buffer);
            }
        }
        return q;
    }
    if (wild_match(&name[1],"*#*"))
    {   /* cut from beginning */
        p = strchr(name,'#');
        /* ASSERT that p != NULL */
        strncpy(buffer,name,(int)(p-name));
        buffer[(int)(p-name)] = '\0';
        q = var_getval(buffer,1);
        if (q == (char *)NULL)
            return q;
        for (temp = 0; temp < strlen(q); temp++)
        {   /* looks for first portion that matches */
            strncpy(buffer,q,temp+1);
            buffer[temp+1] = '\0';
            if (wild_match(buffer,&p[1]))
            {   /* found a match? */
                strcpy(buffer,&q[temp+1]);
                free(q);
                return strcopy(buffer);
            }
        }
        return q;
    }
    if (wild_match(&name[1],"*%%*"))
    {   /* cut from beginning */
        p = strchr(name,'%');
        while (p[1] != '%')
            p = strchr(&p[1],'%');
        /* ASSERT that p != NULL */
        strncpy(buffer,name,(int)(p-name));
        buffer[(int)(p-name)] = '\0';
        q = var_getval(buffer,1);
        if (q == (char *)NULL)
            return q;
        for (temp = 0; temp < strlen(q); temp++)
        {   /* looks for first largest portion that matches */
            strcpy(buffer,&q[temp]);
            if (wild_match(buffer,&p[2]))
            {   /* found a match? */
                strncpy(buffer,q,strlen(q)-strlen(buffer));
                buffer[strlen(q)-strlen(buffer)] = '\0';
                free(q);
                return strcopy(buffer);
            }
        }
        return q;
    }
    if (wild_match(&name[1],"*%*"))
    {   /* cut from beginning */
        p = strchr(name,'%');
        /* ASSERT that p != NULL */
        strncpy(buffer,name,(int)(p-name));
        buffer[(int)(p-name)] = '\0';
        q = var_getval(buffer,1);
        if (q == (char *)NULL)
            return q;
        for (temp = strlen(q)-1; temp >= 0; temp--)
        {   /* looks for first portion that matches */
            strcpy(buffer,&q[temp]);
            if (wild_match(buffer,&p[1]))
            {   /* found a match? */
                strncpy(buffer,q,strlen(q)-strlen(buffer));
                buffer[strlen(q)-strlen(buffer)] = '\0';
                free(q);
                return strcopy(buffer);
            }
        }
        return q;
    }
    
    return (char *)NULL;
}   /* end of var_expression */

char *var_getval(name,flag)
register char *name;
int flag;
{
    register char *value;

    if (flag)
        value = var_expression(name);
    else value = (char *)NULL;
    if (value == (char *)NULL)
        value = var_special(name);
    if (value == (char *)NULL)
        value = var_normal(name);
    return value;
}   /* end of var_getval */

char *var_reference(dont_expand)
int dont_expand;
{
    int rc,count,bracketed;
    register char *p;
    char tomatch;
    char word[BUFSIZ],buf[2];

    bracketed = 0;
    buf[1] = '\0';
    p = word;
    *p++ = '$';
    rc = io_getchar(0);
    if (rc == SYM_EOF)
    {   /* end of file? */
        return (char *)NULL;
    }
    else if (rc == '{' || rc == '(')
    {   /* a "quoted" pair */
        bracketed++;
        if (rc == '{')
            tomatch = '}';
        else    tomatch = ')';
        rc = io_getchar(0);
        while (rc != SYM_EOF && rc != tomatch && rc != '\n')
        {   /* build up the variable name */
#ifdef  GEMDOS
            if (rc == '\r')
                break;
#endif  /* GEMDOS */
            *p++ = rc;
            if (p >= &word[sizeof word])
            {   /* enough space? */
                errmsg(0,LOC("var_reference"),"name too long");
                p--;
                buf[0] = rc;
                io_savestring(buf);
                break;
            }
            rc = io_getchar(0);
        }
#ifdef  GEMDOS
        if (rc == '\r' || rc == '\n')
#else
        if (rc == '\n')
#endif  /* GEMDOS */
        {   /* put eol back */
            buf[0] = '\n';
            io_savestring(buf);
        }
        if (rc == SYM_EOF)
            return (char *)NULL;
    }
    else if (isdigit(rc))
    {   /* numerical variable? */
        *p++ = rc;
    }
    else if (isalpha(rc) || rc == '_')
    {   /* named variable */
        *p++ = rc;
        while ((rc = io_getchar(0)) != SYM_EOF)
        {   /* get rest of word */
            if (rc == '[')
            {   /* a bracketed subscript */
                count = 1;
                *p++ = rc;
                if (p >= &word[sizeof word])
                {   /* enough space? */
                    errmsg(0,LOC("var_reference"),"name too long");
                    p--;
                    buf[0] = rc;
                    io_savestring(buf);
                    break;
                }
                while ((rc = io_getchar(0)) != SYM_EOF)
                {   /* rest of subscript... */
                    if (rc == '[')
                        count++;
                    *p++ = rc;
                    if (p >= &word[sizeof word])
                    {   /* enough space? */
                        errmsg(0,LOC("var_reference"),"name too long");
                        p--;
                        buf[0] = rc;
                        io_savestring(buf);
                        break;
                    }
                    if (rc == ']')
                        count--;
                    if (count == 0)
                        break;
                }
                if (rc == ']')
                    rc = io_getchar(0);
                break;
            }
            if (!isalpha(rc) && !isdigit(rc) && rc != '_')
                break;
            *p++ = rc;
            if (p >= &word[sizeof word])
            {   /* enough space? */
                errmsg(0,LOC("var_reference"),"name too long");
                p--;
                buf[0] = rc;
                io_savestring(buf);
                break;
            }
        }
        if (rc == SYM_EOF)
            return (char *)NULL;
        buf[0] = rc;
        io_savestring(buf);
    }
    else
    {   /* special variable */
        *p++ = rc;
    }
    *p = '\0';

    if (!dont_expand)
    {   /* if to expand variable */
        p = var_getval(word,1);
        if (p != (char *)NULL)
        {   /* if found */
            io_savestring(p);
            free(p);
            return (char *)NULL;
        }
    }
    rc = strlen(word);
    if (bracketed)
        rc += 2;
    p = new_string(rc+1);
    if (p == (char *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("var_reference"));
        return (char *)NULL;
    }
    if (!bracketed)
        strcpy(p,word);
    else sprintf(p,"${%s}",&word[1]);
    return p;
}   /* end of var_reference */

static void var_0dump(vp,type,flag)
register struct variable *vp;
int type;
int flag;
{
    if (vp == (struct variable *)NULL)
        return;
    if (vp->left != (struct variable *)NULL)
        var_0dump(vp->left,type,flag);
    if (vp->type != TYPE_DELETED)
    {   /* if variable really there */
        if (type == 0 || ((type & vp->type) != 0))
            dump_var(vp,flag);
    }
    if (vp->right != (struct variable *)NULL)
        var_0dump(vp->right,type,flag);
}   /* end of var_0dump */

void var_dump(type,flag)
int type;
int flag;
{
    struct varstack *vs;

    for (vs = base_env.var; vs != (struct varstack *)NULL; vs = vs->next)
        var_0dump(vs->table,type,flag);
}   /* end of var_dump */

void var_readenv()
{
    extern char **environ;
    register int i;
#ifdef  GEMDOS
    char buffer[16];
#endif  /* GEMDOS */

#ifdef  GEMDOS
    sprintf(buffer,"%lx",BP->p_parent);
    var_define0("PPID",buffer,0);
#endif  /* GEMDOS */
    if (environ != (char **)NULL)
    {   /* if an environment is present */
        for (i = 0; environ[i] != (char *)NULL; i++)
        {   /* look through environment variables */
#ifdef  MWC_ARGV
            if (strncmp(environ[i],"ARGV=",5) == 0)
                break;
#endif  /* MWC_ARGV */
            var_define(environ[i],TYPE_EXPORTED);
        }
    }
}   /* end of var_readenv */

static  int     e_length;
static  char    *e_area,*e_last;

static void e_getlength(vp)
register struct variable *vp;
{
    if (vp == (struct variable *)NULL)
        return;
    if (vp->left != (struct variable *)NULL)
        e_getlength(vp->left);
    if (vp->type != TYPE_DELETED)
    {   /* if variable really there */
        if ((vp->type & TYPE_EXPORTED) || (vp->cmd == cmd_count))
            e_length += strlen(vp->name)+strlen(vp->value)+2;
    }
    if (vp->right != (struct variable *)NULL)
        e_getlength(vp->right);
}   /* end of e_getlength */

static void e_fillarea(vp)
register struct variable *vp;
{
    if (vp == (struct variable *)NULL)
        return;
    if (vp->left != (struct variable *)NULL)
        e_fillarea(vp->left);
    if (vp->type != TYPE_DELETED)
    {   /* if variable really there */
        if ((vp->type & TYPE_EXPORTED) || (vp->cmd == cmd_count))
        {   /* put into table */
            sprintf(e_last,"%s=%s",vp->name,vp->value);
            while (*e_last)
                e_last++;
            e_last++;
        }
    }
    if (vp->right != (struct variable *)NULL)
        e_fillarea(vp->right);
}   /* end of e_fillarea */

char *var_makeenv()
{
    struct varstack *vs;

    e_length = 2;
    for (vs = base_env.var; vs != (struct varstack *)NULL; vs = vs->next)
        e_getlength(vs->table);
    e_area = new_string(e_length);
    if (e_area == (char *)NULL)
    {   /* enough memory */
        errmsg(SHERR_NOMEM,LOC("var_makeenv"));
        return (char *)NULL;
    }
    e_last = e_area;
    for (vs = base_env.var; vs != (struct varstack *)NULL; vs = vs->next)
        e_fillarea(vs->table);
    *e_last = '\0';
    return e_area;
}   /* end of var_makeenv */

void var_tablefree(vp)
register struct variable *vp;
{
    if (vp == (struct variable *)NULL)
        return;
    if (vp->left != (struct variable *)NULL)
        var_tablefree(vp->left);
    if (vp->right != (struct variable *)NULL)
        var_tablefree(vp->right);
    if (vp->value != (char *)NULL)
        free(vp->value);
    if (vp->name != (char *)NULL)
        free(vp->name);
    free(vp);
}   /* end of var_tablefree */
