/*
 * Command Input Shell
 * Dave Clemans
 * 12/88-1/89
 *
 * "spiritually" based on Bourne, Korn shells
 *
 * Utility Routines
 *
 * $Id: util.c,v 1.6 89/02/25 17:40:21 dclemans Exp $
 *
 * $Log:	util.c,v $
 * Revision 1.6  89/02/25  17:40:21  dclemans
 * miscellaneous bug fixes/speedups
 * 
 * Revision 1.5  89/02/20  20:07:09  dclemans
 * Add RCS identifiers
 * 
 */
#include <stdio.h>
#include "shell.h"

#ifdef  GEMDOS
#include <types.h>
#include <stat.h>
#else
#ifndef USG
#include <sys/param.h>
#endif  /* USG */
#include <sys/types.h>
#include <sys/stat.h>
#ifndef USG
#include <sys/file.h>
#else
#include <fcntl.h>
#define	F_OK		0	/* does file exist */
#define	X_OK		1	/* is it executable by caller */
#define	W_OK		2	/* writable by caller */
#define	R_OK		4	/* readable by caller */
#include <dirent.h>
#define MAXPATHLEN  DIRBUF
#endif  /* USG */
extern char *getwd();
#endif  /* GEMDOS */

struct to_del
{
    struct  to_del *next;
    char    file[1];            /* over-indexed; must be at end */
};
static struct to_del *deletes = (struct to_del *)NULL;

void delete_later(file)
char *file;
{
    register struct to_del *td;

    td = (struct to_del *)malloc(sizeof(*td)+strlen(file));
    if (td == (struct to_del *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("delete_later"));
        errmsg(0,LOC("delete_later"),"can't delete %s",file);
        return;
    }
    strcpy(td->file,file);
    td->next = deletes;
    deletes = td;
}   /* end of delete_later */

void do_deletes()
{
    register struct to_del *td,*otd;

    for (td = deletes; td != (struct to_del *)NULL; )
    {   /* delete files in list; free list */
        otd = td;
        td = td->next;
        unlink(otd->file);
        free(otd);
    }
    deletes = (struct to_del *)NULL;
}   /* end of do_deletes */

char *strcopy(s)
register char *s;
{
    register char *ns;

    if (s == (char *)NULL)
        return (char *)NULL;
    ns = new_string(strlen(s)+1);
    if (ns == (char *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("strcopy"));
        return (char *)NULL;
    }
    strcpy(ns,s);
    return ns;
}   /* end of strcopy */

void stripquotes(s)
register char *s;
{
    register char *p;

    for (p = s; *p != '\0'; p++)
    {   /* check for special chars */
        switch (*p)
        {   /* a special char? */
            case '\'':
                strcpy(p,p+1);
                while (*p && *p != '\'')
                    p++;
                if (*p == '\'')
                    strcpy(p,p+1);
                break;
            case '"':
                strcpy(p,p+1);
                while (*p && *p != '"')
                {   /* scan the string */
                    if (*p == ESCAPE_CHAR)
                        strcpy(p,p+1);
                    p++;
                }
                if (*p == '"')
                    strcpy(p,p+1);
                break;
            case ESCAPE_CHAR:
                strcpy(p,p+1);
                break;
            default:
                break;
        }
    }
}   /* end of stripquotes */

int isexec(path)
register char *path;
{
#ifdef  GEMDOS
    struct stat statb;

    if (stat(path,&statb) < 0)
        return 0;
    else
    {   /* check the modes */
        return 1;
    }
#else
    return (access(path,X_OK) == 0);
#endif  /* GEMDOS */
}   /* end of isexec */

int isread(path)
register char *path;
{
#ifdef  GEMDOS
    struct stat statb;

    if (stat(path,&statb) < 0)
        return 0;
    else
    {   /* check the modes */
        return 1;
    }
#else
    return (access(path,R_OK) == 0);
#endif  /* GEMDOS */
}   /* end of isread */

int isfullpath(path)
register char *path;
{
    char drive;

    if (path[0] == '.' && path[1] == DIR_SEPARATOR)
        return 1;
    if (path[0] == '.' && path[1] == '.' && path[2] == DIR_SEPARATOR)
        return 1;
    if (path[0] == DIR_SEPARATOR)
        return 1;
#ifdef	GEMDOS
    drive = path[0];
    if (islower(drive))
        drive = _toupper(drive);
    if (drive >= 'A' && drive <= 'P' && path[1] == ':')
        return 1;
#endif	/* GEMDOS */
    return 0;
}   /* end of isfullpath */

#ifdef  GEMDOS
char *xchdir(path)
register char *path;
{
    char drive,currdrive;
    register char *ldir;
    int mask,rc;
    char newpath[258];

    drive = path[0];
    if (islower(drive))
        drive = _toupper(drive);
    if (drive >= 'A' && drive <= 'P' && path[1] == ':')
    {   /* dir includes a drive spec */
        ldir = &path[2];
    }
    else
    {   /* local to current drive */
        drive = Dgetdrv() + 'A';
        ldir = path;
    }
    currdrive = Dgetdrv() + 'A';
    if (drive != currdrive)
    {   /* need to switch to another drive? */
        mask = Drvmap();
        if (!(mask && (1 << (drive-'A'))))
            return (char *)NULL;
        Dsetdrv(drive-'A');
    }
    rc = Dsetpath(ldir);
    if (rc != 0)
        return (char *)NULL;
    rc = Dgetpath(&newpath[2],drive-'A'+1);
    if (rc != 0)
        return (char *)NULL;
    newpath[0] = drive;
    newpath[1] = ':';
    return strcopy(newpath);
}   /* end of xchdir */
#else
char *xchdir(path)
register char *path;
{
    char buffer[MAXPATHLEN];

    if (chdir(path) < 0)
        return (char *)NULL;
#ifndef USG
    if (getwd(buffer) == (char *)NULL)
#else
    if (getcwd(buffer,sizeof buffer) == (char *)NULL)
#endif  /* USG */
    {   /* if couldn't find new path */
        errmsg(0,LOC("xchdir"),buffer);
        return (char *)NULL;
    }
    return strcopy(buffer);
}   /* end of xchdir */
#endif  /* GEMDOS */

int lchdir(dir,path)
char *dir;
char **path;
{
    char *ldir;
    register char *first,*last,*fullpath;

    if (strcmp(dir,"-") == 0)
    {   /* change to OLDPWD? */
        ldir = var_normal("$OLDPWD");
	if (ldir == (char *)NULL)
        {   /* something to go to */
            *path = (char *)NULL;
            return 1;
        }
        *path = xchdir(ldir);
        free(ldir);
        if (*path == (char *)NULL)
            return 1;
        io_writestring(0,*path);
        io_writestring(0,"\n");
        return 0;
    }
    if (base_env.cd_path == (char *)NULL || isfullpath(dir))
    {   /* if no path supplied */
        *path = xchdir(dir);
        if (*path == (char *)NULL)
            return 1;
#ifdef  GEMDOS
        for (ldir = *path; *ldir; ldir++)
            if (isupper(*ldir))
                *ldir = _tolower(*ldir);
#endif  /* GEMDOS */
        return 0;
    }
    for (first = last = base_env.cd_path; *last; last++)
    {   /* for each element of path */
        while (*last && *last != ',')
            last++;
        fullpath = new_string(strlen(dir)+(int)(last-first)+2);
        if (fullpath == (char *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("lchdir"));
            return 1;
        }
        if ((int)(last-first) > 0)
        {   /* if a prefix present */
            strncpy(fullpath,first,(int)(last-first));
            if (fullpath[(int)(last-first)-1] != DIR_SEPARATOR)
                fullpath[(int)(last-first)] = DIR_SEPARATOR;
            fullpath[(int)(last-first)+1] = '\0';
        }
        else    fullpath[0] = '\0';
        strcat(fullpath,dir);
        *path = xchdir(fullpath);
        free(fullpath);
        if (*path != (char *)NULL)
        {   /* if directory finally found */
#ifdef  GEMDOS
            for (ldir = *path; *ldir; ldir++)
                if (isupper(*ldir))
                    *ldir = _tolower(*ldir);
#endif  /* GEMDOS */
            io_writestring(0,*path);
            io_writestring(0,"\n");
            return 0;
        }
        first = last+1;
    }
    return 1;
}   /* end of lchdir */

int errmsg(code,file,routine,line,fmt,a0,a1,a2,a3,a4,a5,a6,a7,a8,a9)
int code;
char *file;
char *routine;
int line;
char *fmt;
{
    extern int errno;
    char buffer[BUFSIZ];
    register char *p;

    if (var_arg0 == (char *)NULL)
        var_arg0 = "";
    sprintf(buffer,"%s: %s(%s,%d): ",var_arg0,file,routine,line);
    for (p = buffer; *p; p++)
        /* do nothing */;
    switch (code)
    {   /* check for common err messages; else do normal one */
        case SHERR_NOMEM:
            strcpy(p,"out of memory");
            break;
        case SHERR_PERROR:
            /* perror(a0,a1); */
            /* fall through... */
        default:
            sprintf(p,fmt,a0,a1,a2,a3,a4,a5,a6,a7,a8,a9);
            break;
    }
    io_writestring(1,buffer);
    io_writestring(1,"\n");
    return 0;
}   /* end of errmsg */

int copy_file(old,new)
char *old;
char *new;
{
    register int ifd,ofd,rc;
    char buffer[BUFSIZ];

    ifd = open(old,0);
    if (ifd < 0)
        return -1;
    ofd = creat(new,0666);
    if (ofd < 0)
    {   /* did the file get created? */
        close(ifd);
        return -1;
    }
    while ((rc = read(ifd,buffer,sizeof buffer)) > 0)
        write(ofd,buffer,rc);
    close(ifd);
    close(ofd);
    return 0;
}   /* end of copy_file */

static struct token *copy_tokens(ftp,strip)
struct token *ftp;
int strip;
{
    register struct token *stp,*etp,*tp;

    stp = etp = (struct token *)NULL;
    while (ftp != (struct token *)NULL)
    {   /* copy the list */
        tp = new_token(strlen(ftp->name));
        if (tp == (struct token *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("copy_tokens"));
            tokens_free(stp);
            return (struct token *)NULL;
        }
        strcpy(tp->name,ftp->name);
        if (strip)
            stripquotes(tp->name);
        tp->type = ftp->type;
        if (stp == (struct token *)NULL)
            stp = etp = tp;
        else
        {   /* tack onto end */
            etp->next = tp;
            etp = tp;
        }
        ftp = ftp->next;
    }
    return stp;
}   /* end of copy_tokens */

struct phrase *copy_phrase(fpp,expand,strip,copybody)
struct phrase *fpp;
int expand;
int strip;
int copybody;
{
    register struct phrase *npp;
    struct phrase *pp,*newpp;
    register struct token *newtp;
    struct iotoken *io,*newio;
    char *ptr;
    int len;

    npp = new_phrase();
    if (npp == (struct phrase *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("copy_phrase"));
        return (struct phrase *)NULL;
    }

    if (fpp->type != (struct token *)NULL)
        len = strlen(fpp->type->name);
    else len = 0;
    newtp = new_token(len);
    if (newtp == (struct token *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("copy_phrase"));
        phrase_free(npp);
        return (struct phrase *)NULL;
    }
    if (fpp->type != (struct token *)NULL)
    {   /* if a type token to copy */
        newtp->type = fpp->type->type;
        strcpy(newtp->name,fpp->type->name);
        if (strip)
            stripquotes(newtp->name);
    }
    else
    {   /* stick in a dummy type */
        newtp->type = SYM_SEMI;
        newtp->name[0] = '\0';
    }
    npp->type = newtp;

    if (expand)
        npp->var = lex_reparse_tokens(fpp->var,strip);
    else npp->var = copy_tokens(fpp->var,strip);
    for (newtp = npp->var; newtp != (struct token *)NULL; newtp = newtp->next)
        if (newtp->next == (struct token *)NULL)
            break;
    if (newtp != (struct token *)NULL)
        npp->var_end = newtp;

    if (expand)
        npp->body = lex_reparse_tokens(fpp->body,strip);
    else npp->body = copy_tokens(fpp->body,strip);
    for (newtp = npp->body; newtp != (struct token *)NULL; newtp = newtp->next)
        if (newtp->next == (struct token *)NULL)
            break;
    if (newtp != (struct token *)NULL)
        npp->body_end = newtp;

    for (io = fpp->io; io != (struct iotoken *)NULL; io = io->next)
    {   /* for each iotoken in phrase */
        newtp = (struct token *)NULL;
        if (expand && io_pushtoken(io->file,0) == 0)
        {   /* set up for var expansion */
            newtp = lex_token(1);
            ptr = newtp->name;
        }
        else    ptr = io->file;
        newio = new_iotoken(strlen(ptr));
        if (newio == (struct iotoken *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("copy_phrase"));
            phrase_free(npp);
            if (newtp != (struct token *)NULL)
                free(newtp);
            return (struct phrase *)NULL;
        }
        if (io->tempfile != (char *)NULL)
        {   /* if temp file here, need to copy it */
            len = 16;
#ifdef  GEMDOS
            len += strlen(base_env.tmpdir);
#endif  /* GEMDOS */
            newio->tempfile = new_string(len+1);
            if (newio->tempfile == (char *)NULL)
                errmsg(SHERR_NOMEM,LOC("copy_phrase"));
            else
            {   /* copy the file */
#ifdef  GEMDOS
                sprintf(newio->tempfile,"%stemp%d.tmp",base_env.tmpdir,
                    base_env.temp_count++);
#else
                sprintf(newio->tempfile,"/tmp/sh%d.temp",base_env.temp_count++);
#endif  /* GEMDOS */
                len = copy_file(io->tempfile,newio->tempfile);
                if (len < 0)
                {   /* if copy didn't take */
                    errmsg(0,LOC("copy_phrase"),"error copying %s to %s",
                        io->tempfile,newio->tempfile);
                    free(newio->tempfile);
                    newio->tempfile = (char *)NULL;
                }
            }
        }
        newio->fd = io->fd;
        newio->type = io->type;
        strcpy(newio->file,ptr);
        if (newtp != (struct token *)NULL)
            free(newtp);
        if (npp->io == (struct iotoken *)NULL)
            npp->io = npp->io_end = newio;
        else
        {   /* tack onto end */
            npp->io_end->next = newio;
            npp->io_end = newio;
        }
    }

    for (pp = fpp->group; copybody && pp != (struct phrase *)NULL; pp = pp->next)
    {   /* for each phrase in sentence */
        newpp = copy_phrase(pp,0,0,copybody);
        if (newpp == (struct phrase *)NULL)
        {   /* enough memory? */
            /* message already printed */
            phrase_free(npp);
            return (struct phrase *)NULL;
        }
        if (npp->group == (struct phrase *)NULL)
            npp->group = npp->group_end = newpp;
        else
        {   /* tack onto end */
            npp->group_end->next = newpp;
            npp->group_end = newpp;
        }
    }

    return npp;
}   /* end of copy_phrase */

struct phrase *copy_group(fpp,expand,strip,copybody)
struct phrase *fpp;
int expand;
int strip;
int copybody;
{
    register struct phrase *ppf,*ppl,*cpp;

    ppf = ppl = (struct phrase *)NULL;
    while (fpp != (struct phrase *)NULL)
    {   /* copy list of phrases */
        cpp = copy_phrase(fpp,expand,strip,copybody);
        if (cpp == (struct phrase *)NULL)
        {   /* enough memory? */
            /* message already printed */
            if (ppf != (struct phrase *)NULL)
                sentence_free(ppf);
            return (struct phrase *)NULL;
        }
        if (ppf == (struct phrase *)NULL)
            ppf = ppl = cpp;
        else
        {   /* tack onto end */
            ppl->next = cpp;
            ppl = cpp;
        }
        fpp = fpp->next;
    }
    return ppf;
}   /* end of copy_group */

struct phrase *new_phrase()
{
    register struct phrase *pp;

    pp = (struct phrase *)malloc(sizeof(*pp));
    if (pp == (struct phrase *)NULL)
        return (struct phrase *)NULL;
    pp->type = pp->body = pp->body_end = (struct token *)NULL;
    pp->var = pp->var_end = (struct token *)NULL;
    pp->io = pp->io_end = (struct iotoken *)NULL;
    pp->group = pp->group_end = (struct phrase *)NULL;
    pp->next = (struct phrase *)NULL;

    return pp;
}   /* end of new_phrase */

struct token *new_token(len)
int len;
{
    register struct token *tp;

    tp = (struct token *)malloc(sizeof(*tp)+len);
    if (tp == (struct token *)NULL)
        return (struct token *)NULL;
    tp->next = (struct token *)NULL;
    tp->type = -1;
    tp->name[0] = '\0';

    return tp;
}   /* end of new_token */

struct iotoken *new_iotoken(len)
int len;
{
    register struct iotoken *ip;

    ip = (struct iotoken *)malloc(sizeof(*ip)+len);
    if (ip == (struct iotoken *)NULL)
        return (struct iotoken *)NULL;
    ip->fd = -1;
    ip->type = -1;
    ip->tempfile = (char *)NULL;
    ip->next = (struct iotoken *)NULL;
    ip->file[0] = '\0';

    return ip;
}   /* end of new_iotoken */

struct variable *new_variable()
{
    register struct variable *vp;

    vp = (struct variable *)malloc(sizeof(*vp));
    if (vp == (struct variable *)NULL)
        return (struct variable *)NULL;
    vp->name = vp->value = (char *)NULL;
    vp->cmd = vp->type = vp->misc = 0;
    vp->left = vp->right = (struct variable *)NULL;

    return vp;
}   /* end of new_variable */

struct aliases *new_alias()
{
    register struct aliases *ap;

    ap = (struct aliases *)malloc(sizeof(*ap));
    if (ap == (struct aliases *)NULL)
        return (struct aliases *)NULL;
    ap->name = (char *)NULL;
    ap->tp = (struct token *)NULL;
    ap->type = 0;
    ap->left = ap->right = (struct aliases *)NULL;

    return ap;
}   /* end of new_alias */

struct function *new_function()
{
    register struct function *fp;

    fp = (struct function *)malloc(sizeof(*fp));
    if (fp == (struct function *)NULL)
        return (struct function *)NULL;
    fp->name = (char *)NULL;
    fp->code = (struct phrase *)NULL;
    fp->left = fp->right = (struct function *)NULL;

    return fp;
}   /* end of new_function */

char *new_string(len)
int len;
{
    register char *cp;

    cp = (char *)malloc(len);
    if (cp == (char *)NULL)
        return (char *)NULL;
    *cp = '\0';

    return cp;
}   /* end of new_string */

struct infile *new_infile()
{
    register struct infile *ip;

    ip = (struct infile *)malloc(sizeof(*ip));
    if (ip == (struct infile *)NULL)
        return (struct infile *)NULL;
    ip->name = ip->start = ip->end = (char *)NULL;
#ifdef  GEMDOS
    ip->fp = -1;
#else
    ip->fp = -1;
#endif  /* GEMDOS */
    ip->delete = ip->nonl = ip->markeof = 0;
    ip->next = (struct infile *)NULL;

    return ip;
}   /* end of new_infile */

char **new_argv(count)
int count;
{
    register char **av;
    register int i;

    av = (char **)malloc(sizeof(char *) * (count+1));
    if (av == (char **)NULL)
        return (char **)NULL;
    for (i = 0; i <= count; i++)
        av[i] = (char *)NULL;

    return av;
}   /* end of new_argv */

struct strsave *new_strsave(s)
char *s;
{
    register char *news;
    register struct strsave *sp;

    news = strcopy(s);
    if (news == (char *)NULL)
        return (struct strsave *)NULL;
    sp = (struct strsave *)malloc(sizeof(*sp));
    if (sp == (struct strsave *)NULL)
    {   /* enough memory? */
        free(news);
        return (struct strsave *)NULL;
    }
    sp->next = (struct strsave *)NULL;
    sp->string = sp->ptr = news;

    return sp;
}   /* end of new_strsave */

void phrase_free(pp)
struct phrase *pp;
{
    register struct phrase *ppp,*opp;
    register struct iotoken *iop;
    struct iotoken *oiop;

    if (pp->type != (struct token *)NULL)
        free(pp->type);
    for (ppp = pp->group; ppp != (struct phrase *)NULL; )
    {
        opp = ppp;
        ppp = ppp->next;
        phrase_free(opp);
    }
    tokens_free(pp->body);
    tokens_free(pp->var);
    for (iop = pp->io; iop != (struct iotoken *)NULL; )
    {
        oiop = iop;
        iop = iop->next;
        if (oiop->tempfile != (char *)NULL)
        {   /* a tempfile (probably a heredoc) */
            unlink(oiop->tempfile);
            free(oiop->tempfile);
        }
        free(oiop);
    }
    free(pp);
}   /* end of phrase_free */

void sentence_free(fsp)
struct phrase *fsp;
{
/*
    struct phrase *pp,*opp;
*/
    struct phrase *osp,*sp;

    for (sp = fsp; sp != (struct phrase *)NULL; )
    {   /* delete whole paragraph */
/*
        for (pp = sp->group; pp != (struct phrase *)NULL; )
        {
            opp = pp;
            pp = pp->next;
            phrase_free(opp);
        }
*/
        osp = sp;
        sp = sp->next;
        phrase_free(osp);
    }
}   /* end of sentence_free */

void tokens_free(ftp)
struct token *ftp;
{
    register struct token *tp,*otp;

    for (tp = ftp; tp != (struct token *)NULL ; )
    {
        otp = tp;
        tp = tp->next;
        free(otp);
    }
}   /* end of tokens_free */
