/*
 * words		- routines to find words in lines for reader
 *
 * Dave Clemans, 4/84 (initial version); 1/89 (more generality)
 *
 * Entry points:
 *	_findNextWord	- find next "word" given by specified delimiters
 *	_findPrevWord	- find prev "word" given by specified delimiters
 *	_deleteNextWord	- delete next "word"
 *	_deletePrevWord	- delete prev "word"
 *	_spaceNextWord	- space to next "word"
 *	_spacePrevWord	- space to previous "word"
 *	_skipNextWord	- skip to next "word"
 *	_findNextStdWord	- find next word given by [A-Z][a-z][0-9]_
 *	_findPrevStdWord	- find prev word given by [A-Z][a-z][0-9]_
 *	_deleteNextStdWord	- delete next standard word
 *	_deletePrevStdWord	- delete prev standard word
 *	_spaceNextStdWord	- space to next word
 *	_spacePrevStdWord	- space to previous word
 *	_skipNextStdWord	- skip to next word
 *	_capitalizeStdWord	- capitalize this word
 *
 * $Id: words.c,v 1.2 89/02/20 20:20:33 dclemans Exp $
 *
 * $Log:	words.c,v $
 * Revision 1.2  89/02/20  20:20:33  dclemans
 * Add RCS identifiers
 * 
 */
#include <ctype.h>
#ifndef GEMDOS
#include <sgtty.h>
#endif  GEMDOS

#include "history.h"

/*
 * Interesting function definitions
 */
extern char *index();

/*
 * Find the next word in the global History.currentLine given by the
 * passed delimiters; return index of end of that word
 */
int _findNextWord(delimiters)
char *delimiters;
{
	register char *rc;
	register int ip;

	ip = History.currentLine->cursor+1;
	while (index(delimiters,History.currentLine->contents[ip]))
	{	/* between words */
		ip++;
		if (ip >= History.currentLine->size)
			break;
	}
	for (; ip < History.currentLine->size; ++ip)
	{	/* go through rest of line */
		rc = index(delimiters,History.currentLine->contents[ip]);
		if (rc)	/* a delimiter is found! */
			return(ip-1);
	}
	return(ip);	/* ip == end of line at this point */
};	/* end of _findNextWord */

/*
 * Find the previous word in the global History.currentLine given by the
 * passed delimiters; return index of end of that word
 */
int _findPrevWord(delimiters)
char *delimiters;
{
	register char *rc;
	register int ip;

	ip = History.currentLine->cursor-1;
	while (index(delimiters,History.currentLine->contents[ip]))
	{	/* between words */
		ip--;
		if (ip < 0)
			break;
	}
	for (; ip >= 0; --ip)
	{	/* go through rest of line */
		rc = index(delimiters,History.currentLine->contents[ip]);
		if (rc)	/* a delimiter is found! */
			return(ip+1);
	}
	return(0);	/* beginning of line */
};	/* end of _findPrevWord */

/*
 * Delete the next word in the line given by "delimiters"
 */
_deleteNextWord(delimiters)
char *delimiters;
{
	register int word;
	register int counter;

	word = _findNextWord(delimiters);
	_pushOnKillRing(History.currentLine->cursor,word);
	for (counter = 0; counter < (word-History.currentLine->cursor+1); counter++)
		_doStandardDelete('\000');
};	/* end of _deleteNextWord */

/* Delete the previous word in the line given by "delimiters"
 */
_deletePrevWord(delimiters)
char *delimiters;
{
	register int word;
	register int counter;

	word = _findPrevWord(delimiters);
	_doStandardSpace('\000');
	for (counter = 0; counter < (History.currentLine->cursor-word+1); counter++)
		_doStandardErase('\000');
};	/* end of _deletePrevWord */

/*
 * Space forward to next word in line
 */
_spaceNextWord(delimiters)
char *delimiters;
{
	register int word;

	word = _findNextWord(delimiters);
	while (History.currentLine->cursor < word)
		_doStandardSpace('\000');
};	/* end of _spaceNextWord */

/*
 * Space backward to next word in line
 */
_spacePrevWord(delimiters)
char *delimiters;
{
	register int word;

	word = _findPrevWord(delimiters);
	while (History.currentLine->cursor > word)
		_doStandardBackspace('\000');
};	/* end of _spacePrevWord */

/*
 * Skip to beginning of word after one that we are in.
 */
_skipNextWord(delimiters)
char *delimiters;
{
	register int ip;

	ip = History.currentLine->cursor;
	if (!index(delimiters,History.currentLine->contents[ip]))
	{	/* in a word; goto end */
		for (;ip < History.currentLine->size;)
		{	/* goto end of current word */
			if (index(delimiters,History.currentLine->contents[ip]))
				break;
			ip++;
		}
	}
	if (index(delimiters,History.currentLine->contents[ip]))
	{	/* between words */
		for (;ip < History.currentLine->size;)
		{	/* goto next word */
			if (!index(delimiters,History.currentLine->contents[ip]))
				break;
			ip++;
		}
	}
	while (History.currentLine->cursor < ip)
		_doStandardSpace('\000');
};	/* end of _skipNextWord */

/*
 * Find the next word in the line where a word is composed of alphanumeric's
 * and _.
 */
int _findNextStdWord()
{
	register int ip;
	register char cc;

	ip = History.currentLine->cursor+1;
	cc = History.currentLine->contents[ip];
	if (!isalpha(cc) && !isdigit(cc) && (cc != '_'))
	{	/* between words */
		for (;ip < History.currentLine->size;)
		{	/* between words; look for next one */
			cc = History.currentLine->contents[ip];
			if (isalpha(cc) || isdigit(cc) || (cc == '_'))
				break;
			ip++;
		}
	}
	for (; ip < History.currentLine->size; ip++)
	{	/* search for words */
		cc = History.currentLine->contents[ip];
		if (!isalpha(cc) && !isdigit(cc) && (cc != '_'))
			return(ip-1);
	}
	return(ip);	/* will always be end of line */
};	/* end of _findNextStdWord */

/*
 * Find the previous word in the line where a word is composed of alphanumeric's
 * and _.
 */
int _findPrevStdWord()
{
	register int ip;
	register char cc;

	ip = History.currentLine->cursor-1;
	cc = History.currentLine->contents[ip];
	if (!isalpha(cc) && !isdigit(cc) && (cc != '_'))
	{	/* between words */
		for (;ip >= 0;)
		{	/* between words; look for next one */
			cc = History.currentLine->contents[ip];
			if (isalpha(cc) || isdigit(cc) || (cc == '_'))
				break;
			ip--;
		}
	}
	for (; ip > 0; --ip)
	{	/* search for words */
		cc = History.currentLine->contents[ip];
		if (!isalpha(cc) && !isdigit(cc) && (cc != '_'))
			return(ip+1);
	}
	return(0);	/* beginning of line */
};	/* end of _findPrevStdWord */

/*
 * Delete the next word in the line
 */
_deleteNextStdWord()
{
	register int word;
	register int counter;

	word = _findNextStdWord();
	_pushOnKillRing(History.currentLine->cursor,word);
	for (counter = 0; counter < (word-History.currentLine->cursor+1); counter++)
		_doStandardDelete('\000');
};	/* end of _deleteNextStdWord */

/*
 * Delete the previous word in the line
 */
_deletePrevStdWord()
{
	register int word;
	register int counter;

	word = _findPrevStdWord();
	_doStandardSpace('\000');
	for (counter = 0; counter < (History.currentLine->cursor-word+1); counter++)
		_doStandardErase('\000');
};	/* end of _deletePrevStdWord */

/*
 * Space forward to next word in line
 */
_spaceNextStdWord()
{
	register int word;

	word = _findNextStdWord();
	while (History.currentLine->cursor < word)
		_doStandardSpace('\000');
};	/* end of _spaceNextStdWord */

/*
 * Space backward to next word in line
 */
_spacePrevStdWord()
{
	register int word;

	word = _findPrevStdWord();
	while (History.currentLine->cursor > word)
		_doStandardBackspace('\000');
};	/* end of _spacePrevStdWord */

/*
 * Skip to beginning of word after one that we are in.
 */
_skipNextStdWord()
{
	register char cc;
	register int ip;

	ip = History.currentLine->cursor;
	cc = History.currentLine->contents[ip];
	if (isalpha(cc) || isdigit(cc) || (cc == '_'))
	{	/* in a word; goto end */
		for (;ip < History.currentLine->size;)
		{	/* goto end of current word */
			if (!isalpha(cc) && !isdigit(cc) && (cc != '_'))
				break;
			ip++;
			cc = History.currentLine->contents[ip];
		}
	}
	if (!isalpha(cc) && !isdigit(cc) && (cc != '_'))
	{	/* between words */
		for (;ip < History.currentLine->size;)
		{	/* goto next word */
			if (isalpha(cc) || isdigit(cc) || (cc == '_'))
				break;
			ip++;
			cc = History.currentLine->contents[ip];
		}
	}
	while (History.currentLine->cursor < ip)
		_doStandardSpace('\000');
};	/* end of _skipNextStdWord */

/*
 * Capitalize the word that we are now in
 */
_capitalizeStdWord()
{
	register char cc;
	register int ip,size;

	ip = History.currentLine->cursor;
	cc = History.currentLine->contents[ip];
	if (!isalpha(cc) && !isdigit(cc) && (cc != '_'))
	{	/* not in word??? */
		_writechar('\007');
		return;
	}
	_spaceNextStdWord();
	_spacePrevStdWord();
	ip = _findNextStdWord();
	_pushOnKillRing(History.currentLine->cursor,ip);
	size = ip-History.currentLine->cursor+1;
	for (ip = 0; ip < size; ip++)
	{	/* go through word, capitalizing it */
		_doStandardCapitalize('\000');
		_doStandardSpace('\000');
	}
};	/* end of _capitalizeStdWord */
