/*
 * Command Input Shell
 * Dave Clemans
 * 12/88-1/89
 *
 * "spiritually" based on Bourne, Korn shells
 *
 * I/O Functions
 *
 * $Id: io.c,v 1.5 89/02/22 10:16:32 dclemans Exp $
 *
 * $Log:	io.c,v $
 * Revision 1.5  89/02/22  10:16:32  dclemans
 * Fix bugs with process waiting, pid reporting, parsing of &
 * 
 * Revision 1.4  89/02/20  20:11:49  dclemans
 * Add RCS identifiers
 * 
 */
#include <stdio.h>
#include "shell.h"

char    buffer[BUFSIZ];
char    *bufstart = buffer;
char    *bufend = buffer;
struct  infile *buffile = (struct infile *)NULL;
struct  strsave *strings = (struct strsave *)NULL;

int io_prompt(prompt_level)
int prompt_level;
{
    register char *p,*in,*out;
    char buffer[256];

    if (prompt_level > PROMPT_MAX)
    {   /* a known prompt? */
        sprintf(buffer,"? ");
    }
    else if (prompt_level < 0)
    {   /* explicit non-prompting? */
        return 0;
    }
    else
    {   /* user specified prompt */
        if (base_env.prompts[prompt_level] == (char *)NULL)
            return 0;
        base_env.prompts_issued[prompt_level]++;
        out = buffer;
        p = strcopy(base_env.prompts[prompt_level]);
        if (p == (char *)NULL)
        {   /* enough memory? */
            /* message already printed */
            return 0;
        }
        stripquotes(p);
        for (in = p; *in; in++)
        {   /* build the prompt */
            if (*in == '!')
            {   /* put in cmd number */
                sprintf(out,"%d",base_env.prompts_issued[prompt_level]);
                while (*out) out++;
            }
            else    *out++ = *in;
        }
        *out = '\0';
        free(p);
    }
    io_writestring(0,buffer);
    io_writestring(0,(char *)NULL);
    return (int)(out - buffer);
}   /* end of io_prompt */

int io_savestring(s)
char *s;
{
    register struct strsave *sp;

    if (s == (char *)NULL || strlen(s) == 0)
        return 0;
    sp = new_strsave(s);
    if (sp == (struct strsave *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("io_savestring"));
        return -1;
    }
    sp->next = strings;
    strings = sp;
    return 0;
}   /* end of io_savestring */

int io_getchar(prompt_level)
int prompt_level;
{
    register int new_char;
    register struct strsave *sp;
    int markeof,rc;
#ifdef  GEMDOS
    int i;
#endif  /* GEMDOS */
    register struct infile *ip;

    if (strings != (struct strsave *)NULL)
    {   /* scanning a "pushed back" string? */
        new_char = *strings->ptr++;
        if (*strings->ptr == '\0')
        {   /* finished with the string? */
            free(strings->string);
            sp = strings;
            strings = strings->next;
            free(sp);
        }
        return new_char;
    }
    if (buffile != (struct infile *)NULL)
    {   /* if reading inserted input */
        if (buffile->start >= buffile->end)
        {   /* if need more input */
#ifdef  GEMDOS
            rc = Fread(buffile->fp,(long)(sizeof buffile->buffer-1),buffile->buffer);
            if (rc > 0)
            {   /* got a good buffer? */
                buffile->buffer[rc] = '\0';
                for (i = 0; i < rc; i++)
                {   /* kill any double eol's */
                    if (buffile->buffer[i] == '\r' &&
                        (i+1) < rc &&
                        buffile->buffer[i+1] == '\n')
                        buffile->buffer[i] = base_env.separators[0];
                }
            }
#else
            rc = read(buffile->fp,buffile->buffer,sizeof buffile->buffer - 1);
            if (rc > 0)
                buffile->buffer[rc] = '\0';
#endif  /* GEMDOS */
            if (rc <= 0)
            {   /* at end of temp file? */
#ifdef  GEMDOS
                Fclose(buffile->fp);
#else
                close(buffile->fp);
                main_fdput(buffile->fp);
#endif  /* GEMDOS */
                if (buffile->delete)
                    delete_later(buffile->name);
                free(buffile->name);
                ip = buffile;
                buffile = buffile->next;
                markeof = ip->markeof;
                free(ip);
                if (markeof)
                    return SYM_MARKER;
                else return io_getchar(prompt_level);
            }
            buffile->buffer[sizeof buffile->buffer - 1] = '\0';
            if (flag_echoinput)
            {   /* what did we just read? */
                io_writestring(0,buffile->buffer);
            }
            for (buffile->end = buffile->buffer; *buffile->end != '\0'; buffile->end++)
                /* do nothing */;
            buffile->start = buffile->buffer;
        }
        new_char = *buffile->start++;
        if (buffile->nonl)
        {   /* no nls from this file? */
            if (new_char == '\r' || new_char == '\n')
                new_char = ' ';
        }
        return new_char;
    }
    if (bufstart >= bufend)
    {   /* need more characters? */
        if (buffile != (struct infile *)NULL)
        {   /* reading from temp file? */
        }
        else
        {   /* read from normal input */
            if (flag_interactive)
                io_prompt(prompt_level);
#ifdef  LINED
            if (flag_interactive)
            {   /* get a line */
                buffer[0] == '\0';
                rc = Reader(base_env.fd_input,buffer,sizeof buffer - 1);
                if (rc == 1 && (buffer[0] == '\r' || buffer[0] == '\n') && prompt_level <= PROMPT_MAX)
                    base_env.prompts_issued[prompt_level]--;
            }
            else
                rc = read(base_env.fd_input,buffer,sizeof buffer-1);
#else
            rc = read(base_env.fd_input,buffer,sizeof buffer-1);
            if (rc == 1 && (buffer[0] == '\r' || buffer[0] == '\n') && prompt_level <= PROMPT_MAX)
                base_env.prompts_issued[prompt_level]--;
#ifdef  GEMDOS
            if (flag_interactive && rc >= 0 && rc < (sizeof buffer -1))
            {   /* add in eol if necessary */
                if (rc == 0 && prompt_level >= 0 && prompt_level <= PROMPT_MAX)
                    base_env.prompts_issued[prompt_level]--;
                if (rc > 0 && buffer[rc-1] != '\n')
                    buffer[rc++] = '\n';
                else if (rc > 0 && buffer[rc-1] == '\032')
                    rc = 0;
                else if (rc == 0)
                    buffer[rc++] = '\n';
                io_writestring(0,"\n");
            }
#endif  /* GEMDOS */
#endif  /* LINED */
            if (rc <= 0)
                return SYM_EOF;
            buffer[rc] = '\0';
        }
        buffer[sizeof buffer - 1] = '\0';
        if (flag_echoinput)
        {   /* what did we just read? */
            io_writestring(0,buffer);
        }
        for (bufend = buffer; *bufend != '\0'; bufend++)
            /* do nothing */;
        bufstart = buffer;
    }
    new_char = *bufstart++;
    return new_char;
}   /* end of io_getchar */

void io_pushback(c)
char c;
{
    char buf[2];

    buf[0] = c;
    buf[1] = '\0';
    io_savestring(buf);
}   /* end of io_pushback */

int io_pushtoken(name,strip)
register char *name;
int strip;
{
    register char *cp;
    register int result;

    cp = new_string(strlen(name)+2);
    if (cp == (char *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("io_pushtoken"));
        return -1;
    }
    strcpy(cp,name);
    if (strip)
        stripquotes(cp);
    strcat(cp," ");
    result = io_savestring(cp);
    free(cp);
    return result;
}   /* end of io_pushtoken */

int io_pushfile(file,delete,nonl,markeof)
char *file;
int delete;
int nonl;
int markeof;
{
    register struct infile *ip;
#ifndef GEMDOS
    int tfd;
#endif  /* GEMDOS */

    ip = new_infile();
    if (ip == (struct infile *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("io_pushfile"));
        return -1;
    }
    ip->delete = delete;
    ip->nonl = nonl;
    ip->markeof = markeof;
    ip->name = strcopy(file);
    if (ip->name == (char *)NULL)
    {   /* enough memory? */
        /* message already printed */
        free(ip);
        return -1;
    }
    stripquotes(ip->name);
#ifdef  GEMDOS
    ip->fp = Fopen(ip->name,0);
    if (ip->fp < 0)
    {   /* did the file open? */
        errmsg(0,LOC("io_pushfile"),"can't open %s",file);
        free(ip);
        return -1;
    }
#else
    ip->fp = open(ip->name,0);
    if (ip->fp < 0)
    {   /* did the file open? */
        errmsg(0,LOC("io_pushfile"),"can't open %s",file);
        free(ip);
        return -1;
    }
    else
    {   /* put the fd into a safe place */
        tfd = main_fdget();
        if (dup2(ip->fp,tfd) < 0)
        {   /* everything setup? */
            errmsg(0,LOC("io_pushfile"),"can't associate %s with right fd",file);
            main_fdput(tfd);
            free(ip);
            return -1;
        }
        close(ip->fp);
        ip->fp = tfd;
    }
#endif  /* GEMDOS */
    ip->start = ip->end = ip->buffer;
    ip->next = buffile;
    buffile = ip;
    return 0;
}   /* end of io_pushfile */

void io_writestring(file,s)
int file;
char *s;
{
    static char *buffer = (char *)NULL;
    static int bufsiz = 0;
    register int fd;
    register char *in,*out;
    
    if (buffer == (char *)NULL)
    {   /* do we have any buffer space? */
        bufsiz = 128;
        buffer = new_string(bufsiz);
        if (buffer == (char *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("io_writestring"));
            return;
        }
    }

    switch (file)
    {   /* what file descriptor should we use? */
        case 0:
            fd = base_env.io->output;
            break;
        case 1:
            fd = base_env.io->errout;
            break;
        case 2:
            fd = 1;
            break;
        case 3:
            fd = 2;
            break;
        default:
            errmsg(0,LOC("io_writestring"),"illegal file specified");
            return;
    }

    for (out = buffer; *out; out++)
        /* do nothing */;
    if (s == (char *)NULL)
    {   /* just flush existing buffer */
        write(fd,buffer,(unsigned)(out - buffer));
        out = buffer;
        *out = '\0';
        return;
    }
    for (in = s; *in; in++)
    {   /* add in the string */
#ifdef  GEMDOS
        if (*in == '\n')
            *out++ = '\r';
#endif  /* GEMDOS */
        *out++ = *in;
        *out = '\0';
        if (out >= &buffer[bufsiz-1] || *in == '\n')
        {   /* dump current contents of buffer */
            write(fd,buffer,(unsigned)(out - buffer));
            out = buffer;
            *out = '\0';
        }
    }
}   /* end of io_writestring */

