/*                               -*- Mode: C -*- 
 * 
 * uSystem Version 4.3.2, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1990
 * 
 * uFile.c -- I/O routines.
 * 
 * Author           : Rick Stroobosscher
 * Created On       : Fri Feb  9 15:29:03 1990
 * Last Modified By : Hamish Macdonald
 * Last Modified On : Thu Mar  7 14:12:28 1991
 * Update Count     : 202
 */

#define __U_KERNEL__

#include <uUnix.h>
#include <uSystem.h>
#include <uFile.h>

/* definition of timeval */
#include <sys/time.h>

typedef struct uFileD {
    uSemaphore mutex;
    int file;
    uCluster cluster;
    int block;
    int flags;
} uFileD;

static char data;

inline static void uTouch( char *addr ) {

    /*
     * Used to reference shared memory so that the page tables for
     * separate address spaces will be updated properly.
     * Apparently, the I/O system does not check to see whether the
     * pages are a part of the shared memory before it raises
     * a segmentation fault.
     */

    if ( addr != NULL ) {
	data = *addr;
    } /* if */
} /* uTouch */

inline void uFileEnter( uFile file ) {
    uP( &(file->mutex) );
    file->cluster = uMigrate( uThisTask(), file->cluster );    
} /* uFileEnter */

inline void uFileLeave( uFile file ) {
    file->cluster = uMigrate( uThisTask(), file->cluster );
    uV( &(file->mutex) );
} /* uFileLeave */

inline int uFileFd( uFile file ) {
    return file->file;
} /* uFileFd */

inline static uFile uCreateFile() {
    uFile file;

    file = uMalloc( sizeof( uFileD ) );
    file->mutex = U_SEMAPHORE( 1 );
    file->cluster = uCreateCluster( 1, 0 );

    return file;
} /* uCreateFile */

inline static void uDestroyFile( uFile file ) {
    uDestroyCluster( file->cluster );
    uFree( file );
} /* uDestroyFile */

static int uSelectFile( uFile file, fd_set *set ) {

    struct timeval timeout = { 1, 0 };			/* wake up every second because of race condition between signal and select */
    int found;
    
    /* add this file descriptor to the mask of pending file descriptors */
    
    uP( &(file->cluster->mutex) );
    FD_SET( file->file, set );
    file->cluster->nfds += 1;
    uV( &(file->cluster->mutex) );
    
    /* if there is still work to be done, give up the cpu */
    
    if ( uReadyTasks() - file->cluster->nfds > -1 ) {
	
	/* give up the cpu */
	uYield();
	
	/* if the mask is still intact, that means that nobody blocked on a select */
	/* so, remove my intent, and try again. */
	uP( &(file->cluster->mutex) );
	if ( FD_ISSET( file->file, set ) ) {
	    FD_CLR( file->file, set );
	    file->cluster->nfds -= 1;
	} /* if */
	uV( &(file->cluster->mutex ) );
	
	/* tell the caller to continue polling */
	return 1;
	
    } /* if */
    
    /* prepare to block, waiting for i/o on any of */
    /* tasks who declared their intent to do i/o */
    
    uP( &(file->cluster->mutex) );
    
    found = select( FD_SETSIZE, &(file->cluster->rfds), &(file->cluster->wfds), &(file->cluster->efds), &timeout );

    FD_ZERO( &(file->cluster->rfds) );
    FD_ZERO( &(file->cluster->wfds) );
    FD_ZERO( &(file->cluster->efds) );
    file->cluster->nfds = 0;
    
    uV( &(file->cluster->mutex) );
    
    return found;
    
} /* uSelectFile */

inline static int uDoesFileBlock( uFile file ) {
    
    /*
     * Determine the type of the file,
     * so that we know whether to block on
     * access or not.
     */
    
    struct stat buf;
    
    if ( fstat( file->file, &buf ) < 0 ) {
	/* can't stat the file...  block by default */
	return U_TRUE;
    } /* if */
    	
    /*
     * examine the "type" bits
     * 
     * File access may block if file type is:
     * 	S_IFIFO : named pipe
     * 	S_IFCHR : character special file (tty/pty)
     * 	S_IFSOCK : socket
     */
    
    return
#if defined( S_IFIFO )
            (buf.st_mode & S_IFIFO) == S_IFIFO ||
#endif
            (buf.st_mode & S_IFCHR) == S_IFCHR ||
            (buf.st_mode & S_IFSOCK) == S_IFSOCK;
    
} /* uDoesFileBlock */

inline static void uSetFileAccess( uFile file ) {
    /* if the file is a blocking file, set the fcntl bits to allow non blocking access */
    if ( file->block ) {
	file->flags = fcntl( file->file, F_GETFL );
	fcntl( file->file, F_SETFL, file->flags | FNDELAY );
    } /* if */
} /* uSetFileAccess */

inline static void uClearFileAccess( uFile file ) {
    /* if the file is a blocking file, set the fcntl bits to what they originally were */
    if ( file->block ) {
	fcntl( file->file, F_SETFL, file->flags );
    } /* if */
} /* uClearFileAccess */

inline static void uOpenError( uFile file, char *path, int flags, int mode ) {
    uOpenExMsg msg;
    uException* except;
    
    msg.header.errno = errno;
    msg.path = path;
    msg.perms = NULL; /* not used for uOpen  */
    msg.flags = flags;
    msg.mode = mode;

    uFileLeave( file );
    uDestroyFile( file );
    
    switch( msg.header.errno ) {
      case EEXIST:
	msg.header.msg = "uOpen: filename exists on create.\n";
	except = &uOpenEx;
	break;
      case EPERM:
	msg.header.msg = "uOpen: pathname contains high order bit set\n";
	except = &uBadPathEx;
	break;
      case ENOTDIR:
	msg.header.msg = "uOpen: a portion of the path is not a directory\n";
	except = &uBadPathEx;
	break;
      case ENOENT:
	msg.header.msg = "uOpen: file doesn't exist or component doesn't exist or path too long\n";
	except = &uBadPathEx;
	break;
      case EFAULT:
	msg.header.msg = "uOpen: path parameter points outside process address space\n";
	except = &uBadPathEx;
	break;
      case ELOOP:
	msg.header.msg = "uOpen: symbolic link loop\n";
	except = &uBadPathEx;
	break;
      case EOPNOTSUPP:
	msg.header.msg = "uOpen: Attempt to open socket\n";
	except = &uBadPathEx;
	break;
      case EACCES:
	msg.header.msg = "uOpen: wrong permissions on file or directory\n";
	except = &uNoPermsEx;
	break;
      case EISDIR:
	msg.header.msg = "uOpen: file is a directory, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case EROFS:
	msg.header.msg = "uOpen: read only file system, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case ETXTBSY:
	msg.header.msg = "uOpen: shared text being executed, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case EBUSY:
	msg.header.msg = "uOpen: special file already opened with exlusive access\n";
	except = &uNoPermsEx;
	break;
      case EMFILE:
	msg.header.msg = "uOpen: this process has too many files open\n";
	except = &uNoFilesEx;
	break;
      case ENFILE:
	msg.header.msg = "uOpen: too many files open in the system\n";
	except = &uNoFilesEx;
	break;
      case EIO:
	msg.header.msg = "uOpen: I/O failed\n";
	except = &uOpenIOFailedEx;
	break;
      case ENOSPC:
	msg.header.msg = "uOpen: no space on filesystem\n";
	except = &uOpenNoSpaceEx;
	break;
      case ENXIO:
	msg.header.msg = "uOpen: device doesn't exist or opening comm device with no delay and no carrier\n";
	except = &uOpenEx;
	break;
      default:
	msg.header.msg = "uOpen: unknown open error\n";
	except = &uOpenEx;
	break;
    } /* switch */
    uRaise( NULL, except, &msg, sizeof(msg) );
} /* uOpenError */

uFile uOpen( char *path, int flags, int mode ) {
    
    uFile file;
    
    file = uCreateFile();
    
    uFileEnter( file );
    
    file->file = open( path, flags, mode );
    if ( file->file < 0 ) {
#ifdef __U_UNIXRC__
	uFileLeave( file );
	uDestroyFile( file );
	file = NULL;
#else
	uOpenError( file, path, flags, mode );
#endif
    } else {
	file->block = uDoesFileBlock( file );
	uSetFileAccess( file );
	uFileLeave( file );
    } /* if */
    return file;					/* return reference to file */
} /* uOpen */

inline static void uReadError( uFile file ) {
    
    uReadWriteExMsg msg;
    uException* except;
    
    msg.errno = errno;
    
    uFileLeave( file );					/* exit protocol */
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uRead: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case EFAULT:
	msg.msg = "uRead: buffer points ouside address space\n";
	except = &uReadWriteEx;
	break;
      case EBADF:
	msg.msg = "uRead: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uRead: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uRead: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    uRaise( file, except, &msg, sizeof(msg) );
} /* uReadError */

int uRead( uFile file, void *buf, int len ) {
    
    int rlen;
    
    uFileEnter( file );
    uTouch( buf + len - 1 );
    
    for ( ;; ) {
	
	/* do a non blocking read */
	rlen = read( file->file, buf, len );
	
	/* if the read was successful, break */
	if ( rlen != -1 ) break;
	
	/* if an error occurred, handle it */
	if ( errno != EINTR && errno != EWOULDBLOCK ) {
#ifdef __U_UNIXRC__
	    break;
#else
	    uReadError( file );
#endif
	} /* if */
	
	/* read failed to complete */
	rlen = uSelectFile( file, &(file->cluster->rfds) );
	if ( rlen < 0 ) {
	    if ( errno != EINTR ) {
#ifdef __U_UNIXRC__
		break;
#else
		uReadError( file );
#endif
	    } /* if */
	} /* if */
    } /* for */
    
    uFileLeave( file );

#ifndef __U_UNIXRC__
    if( rlen == 0 ) {
	uEofExMsg msg;

	msg.msg = "uRead: End of File\n";

	uRaise( file, &uEofEx, &msg, sizeof( msg ) );
    } /* if */
#endif

    return rlen;
} /* uRead */

inline static void uWriteError( uFile file ) {
    uReadWriteExMsg msg;
    uException *except;
    
    msg.errno = errno;
    
    uFileLeave( file );			/* exit protocol */
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uWrite: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uWrite: no space on file system\n";
	except = &uNoSpaceEx;
	break;
      case EFBIG:
	msg.msg = "uWrite: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uWrite: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uWrite: writing to pipe w/o reader or unconnected socket.\n";
	except = &uBadFileEx;
      case ENOMEM:
	msg.msg = "uWrite: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uWrite: unknown error\n";
	except = &uReadWriteEx;
	break;
    }
    uRaise( file, except, &msg, sizeof(msg) );
} /* uWriteError */

int uWrite( uFile file, void *buf, int len ) {
    int wlen;
    
    uFileEnter( file );    
    uTouch( buf + len - 1 );
    
    for ( ;; ) {
	
	/* do a non blocking write */
	wlen = write( file->file, buf, len );
		
	/* if the write was successful, break */
	if ( wlen != -1 ) break;
	
	/* if an error occurred, handle it */
	if ( errno != EINTR && errno != EWOULDBLOCK ) {
#ifdef __U_UNIXRC__
	    break;
#else
	    uWriteError( file );
#endif
	} /* if */
	
	/* write failed to complete */
	wlen = uSelectFile( file, &(file->cluster->wfds) );
	if ( wlen < 0 ) {
	    if ( errno != EINTR ) {
#ifdef __U_UNIXRC__
		break;
#else
		uWriteError( file );
#endif
	    } /* if */
	} /* if */
    } /* for */
    
    uFileLeave( file );
    
    return wlen;
} /* uWrite */

inline static void uLseekError( uFile file ) {
    uIOErrorExMsg msg;
    uException *except;
    
    msg.errno = errno;
    
    uFileLeave( file );
    
    switch( msg.errno ) {
      case EBADF:
	msg.msg = "uLseek: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EINVAL:
	msg.msg = "uLseek: bad parameter\n";
	except = &uIOErrorEx;
	break;
      case EPIPE:
	msg.msg = "uLseek: attempting to seek on a socket.\n";
	except = &uBadFileEx;
      default:
	msg.msg = "uLseek: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */    
    uRaise( file, except, &msg, sizeof(msg) );
} /* uLseekError */

off_t uLseek( uFile file, off_t offset, int whence ) {
    off_t pos;
    
    uFileEnter( file );
    
    pos = lseek( file->file, offset, whence );
    if ( pos == -1 ) {
#ifdef __UNIXRC__
#else
	uLseekError( file );
#endif
    } /* if */
    
    uFileLeave( file );
    
    return pos;
} /* uLseek */

inline static void uFsyncError( uFile file ) {
    uReadWriteExMsg msg;
    uException *except;
    
    msg.errno = errno;
    
    uFileLeave( file );
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFsync: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uFsync: no space on file system\n";
	except = &uNoSpaceEx;
	break;
      case EFBIG:
	msg.msg = "uFsync: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uFsync: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EINVAL:
	msg.msg = "uFsync: fsyncing a socket.\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uFsync: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFsync: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    uRaise( file, except, &msg, sizeof(msg) );
} /* uFsyncError */

int uFsync( uFile file ) {
    
    int code;
    
    uFileEnter( file );
    
    code = fsync( file->file );
    if ( code == -1 ) {
#ifdef __U_UNIXRC__
#else
	uFsyncError( file );
#endif
    } /* if */
    
    uFileLeave( file );
    
    return code;
} /* uFsync */

inline static void uCloseError( uFile file ) {
    uReadWriteExMsg msg;
    uException *except;
    
    msg.errno = errno;
    
    uFileLeave( file );
    
    switch( msg.errno ) {
      case EBADF:
	msg.msg = "uClose: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      default:
	msg.msg = "uClose: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */    
    uRaise( file, except, &msg, sizeof(msg) );
} /* uCloseError */

int uClose( uFile file ) {
    int code;
    
    uFileEnter( file );

    uClearFileAccess( file );
    
    code = close( file->file );
    if ( code == -1 ) {
#ifdef __U_UNIXRC__
#else
	uCloseError( file );
#endif
    } /* if */
    
    uFileLeave( file );
    
    uDestroyFile( file );
    
    return code;
} /* uClose */

inline static void uSocketError( uFile sock, int af, int type, int protocol ) {
    uCreateSockExMsg msg;
    uException *except;
    int len = sizeof(msg);
    
    msg.header.errno = errno;
    msg.af = af;
    msg.type = type;
    msg.protocol = protocol;
    
    uFileLeave( sock );
    uDestroyFile( sock );
    
    switch( msg.header.errno ) {
      case EACCES:
	msg.header.msg = "uSocket: Can't create socket of given type or protocol\n";
	except = &uCreateSockEx;
	break;
      case EMFILE:
	msg.header.msg = "uSocket: this process has too many files open\n";
	except = &uCreateSockEx;
	break;
      case ENFILE:
	msg.header.msg = "uSocket: too many files open in the system\n";
	except = &uCreateSockEx;
	break;
      case EPROTONOSUPPORT:
	msg.header.msg = "uSocket: bad protocol\n";
	except = &uCreateSockEx;
	break;
      case ENOBUFS:
	msg.header.msg = "uSocket: no buffer space\n";
	except = &uNoBufsEx;
	len = sizeof(uNoBufsExMsg);
	break;
      default:
	msg.header.msg = "uSocket: unknown creation error\n";
	except = &uCreateSockEx;
	break;
    } /* switch */    
    uRaise( NULL, except, &msg, len );
} /* uSocketError */

uFile uSocket( int af, int type, int protocol ) {
    uFile sock;
    
    sock = uCreateFile();
    uFileEnter( sock );
    
    sock->file = socket( af, type, protocol );		/* create socket descriptor on socket cluster */
    
    if ( sock->file == -1 ) {
#ifdef __U_UNIXRC__
	uFileLeave( sock );
	uDestroyFile( sock );
	sock = NULL;
#else
	uSocketError( sock, af, type, protocol );
#endif
    } else {
	sock->block = uDoesFileBlock( sock );
	uSetFileAccess( sock );
	uFileLeave( sock );
    } /* if */
    
    return sock;
} /* uSocket */

inline static void uBindError( uFile socket, char *name, int namelen ) {
    uBadSockAddressExMsg msg;
    uException *except;
    int len = sizeof(msg);
    
    msg.header.errno = errno;
    msg.name = name;
    msg.namelen = namelen;
    
    uFileLeave( socket );
	
    switch( msg.header.errno ) {
      case EACCES:
	msg.header.msg = "uBind: Address protected, no permission\n";
	except = &uBadSockAddressEx;
	break;
      case EFAULT:
	msg.header.msg = "uBind: addr parm not in address space\n";
	except = &uBadSockAddressEx;
	break;
      case EBADF:
	msg.header.msg = "uBind: bad file descriptor\n";
	except = &uBadFileEx;
	len = sizeof(uBadFileExMsg);
	break;
      case EINVAL:
	msg.header.msg = "uBind: this socket is already bound\n";
	except = &uBadSockAddressEx;
	break;
      case ENOTSOCK:
	msg.header.msg = "uBind: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	len = sizeof(uNotSockExMsg);
	break;
      case EADDRNOTAVAIL:
	msg.header.msg = "uBind: cannot use specified address\n";
	except = &uBadSockAddressEx;
	break;
      case EADDRINUSE:
	msg.header.msg = "uBind: specified address is already in use\n";
	except = &uBadSockAddressEx;
	break;
      default:
	msg.header.msg = "uBind: unknown socket error\n";
	except = &uBadSockAddressEx;
	break;
    } /* switch */    
    uRaise( socket, except, &msg, len );
} /* uBindError */

int uBind( uFile socket, void *name, int namelen ) {
    int code;
    
    uFileEnter( socket );
    
    code = bind( socket->file, name, namelen );
    if ( code == -1 ) {
#ifdef __U_UNIXRC__
#else
	uBindError( socket, name, namelen );
#endif
    } /* if */
    
    uFileLeave( socket );
    
    return code;
} /* uBind */

inline static void uConnectError( uFile socket, char *name, int namelen ) {
    uBadSockAddressExMsg msg;
    uException *except;
    int len = sizeof(msg);
    
    msg.header.errno = errno;
    msg.name = name;
    msg.namelen = namelen;
    
    uFileLeave( socket );
    
    switch( msg.header.errno ) {
      case EFAULT:
	msg.header.msg = "uConnect: addr parm not in address space\n";
	except = &uBadSockAddressEx;
	break;
      case EBADF:
	msg.header.msg = "uConnect: bad file descriptor\n";
	except = &uBadFileEx;
	len = sizeof(uBadFileExMsg);
	break;
      case ENOTSOCK:
	msg.header.msg = "uConnect: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	len = sizeof(uNotSockExMsg);
	break;
      case EADDRNOTAVAIL:
	msg.header.msg = "uConnect: cannot use specified address\n";
	except = &uConnFailedEx;
	break;
      case EADDRINUSE:
	msg.header.msg = "uConnect: specified address is already in use\n";
	except = &uConnFailedEx;
	break;
      case EAFNOSUPPORT:
	msg.header.msg = "uConnect: no support for this address family\n";
	except = &uConnFailedEx;
	break;
      case EISCONN:
	msg.header.msg = "uConnect: socket already connected\n";
	except = &uConnFailedEx;
	break;
      case ETIMEDOUT:
	msg.header.msg = "uConnect: connection timeout\n";
	except = &uConnFailedEx;
	break;
      case ECONNREFUSED:
	msg.header.msg = "uConnect: connection rejected\n";
	except = &uConnFailedEx;
	break;
      case ENETUNREACH:
	msg.header.msg = "uConnect: network unreachable\n";
	except = &uConnFailedEx;
	break;
      case EWOULDBLOCK:
	msg.header.msg = "uConnect: non-blocking socket, no connection yet\n";
	except = &uConnFailedEx;
	break;
      default:
	msg.header.msg = "uConnect: unknown socket error\n";
	except = &uConnFailedEx;
	break;
    } /* switch */    
    uRaise( socket, except, &msg, len );
} /* uConnectError */
    
int uConnect( uFile socket, void *name, int namelen ) {
    int code;
    
    uFileEnter( socket );
    
    code = connect( socket->file, name, namelen );
    if ( code == -1 ) {
#ifdef __U_UNIXRC__
#else
	uConnectError( socket, name, namelen );
#endif
    } /* if */
    
    uFileLeave( socket );
    
    return code;
} /* uConnect */

void uListenError( uFile socket ) {
    uSocketErrorExMsg msg;
    uException *except;
    
    msg.errno = errno;
    
    uFileLeave( socket );
    
    switch( msg.errno ) {
      case EOPNOTSUPP:
	msg.msg = "uListen: incorrect type of socket\n";
	except = &uSocketErrorEx;
	break;
      case EBADF:
	msg.msg = "uListen: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOTSOCK:
	msg.msg = "uListen: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	break;
      default:
	msg.msg = "uListen: unknown socket error\n";
	except = &uSocketErrorEx;
	break;
    } /* switch */    
    uRaise( socket, except, &msg, sizeof(msg) );
} /* uListenError */
    
int uListen( uFile socket, int log ) {
    int code;
    
    uFileEnter( socket );
    
    code = listen( socket->file, log );
    if ( code == -1 ) {
#ifdef __U_UNIXRC__
#else
	uListenError( socket );
#endif
    } /* if */
    uFileLeave( socket );
    
    return code;
} /* uListen */

inline static void uAcceptError( uFile socket, uFile connect ) {
    uSocketErrorExMsg msg;
    uException *except;
    
    msg.errno = errno;

    uFileLeave( connect );
    uDestroyFile( connect );
    uFileLeave( socket );
    
    switch( msg.errno ) {
      case EMFILE:
	msg.msg = "uAccept: Too many files open for this process\n";
	except = &uSocketErrorEx;
	break;
      case ENFILE:
	msg.msg = "uAccept: Too many file open in the system\n";
	except = &uSocketErrorEx;
	break;
      case EFAULT:
	msg.msg = "uAccept: Address parameter is not writable or not in address space\n";
	except = &uSocketErrorEx;
	break;
      case EOPNOTSUPP:
	msg.msg = "uAccept: Socket is not SOCK_STREAM, or incorrect type\n";
	except = &uSocketErrorEx;
	break;
      case EBADF:
	msg.msg = "uAccept: Bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOBUFS:
	msg.msg = "uAccept: no buffer space\n";
	except = &uNoBufsEx;
	break;
      case ENOTSOCK:
	msg.msg = "uAccept: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	break;
      default:
	msg.msg = "uAccept: unknown socket error\n";
	except = &uSocketErrorEx;
	break;
    } /* switch */    
    uRaise( socket, except, &msg, sizeof(msg) );
} /* uAcceptError */
    
uFile uAccept( uFile socket, void *addr, int *addrlen ) {
    uFile connect;
    
    uFileEnter( socket );
    connect = uCreateFile();
    uFileEnter( connect );
    
    for ( ;; ) {
	
	/* attempt a non blocking accept */
	connect->file = accept( socket->file, addr, addrlen );

	/* if accept successful, break */
	if ( connect->file != -1 ) break;

	/* if an error occurred, handle it */
	if ( errno != EINTR && errno != EWOULDBLOCK ) {
#ifdef __U_UNIXRC__
	    uFileLeave( connect );
	    uDestroyFile( connect );
	    uFileLeave( socket );
	    return NULL;
#else
	    uAcceptError( socket, connect );
#endif
	} /* if */
	
	/* accept failed to complete */
	connect->file = uSelectFile( socket, &(socket->cluster->rfds) );
	if ( connect->file < 0 ) {
	    if ( errno != EINTR ) {
#ifdef __UNIXRC__
		uFileLeave( connect );
		uDestroyFile( connect );
		uFileLeave( socket );
		return NULL;
#else
		uAcceptError( socket, connect );
#endif
	    } /* if */
	} /* if */	
    } /* for */

    connect->block = uDoesFileBlock( connect );
    uSetFileAccess( connect );
    uFileLeave( connect );
    uFileLeave( socket );
    
    return connect;
} /* uAccept */

inline static void uGetSockNameError( uFile socket ) {
    uSocketErrorExMsg msg;
    uException *except;
    
    msg.errno = errno;
    
    uFileLeave( socket );
    
    switch( msg.errno ) {
      case EFAULT:
	msg.msg = "uGetsockname: Address parameter is not writable or not in address space\n";
	except = &uSocketErrorEx;
	break;
      case EBADF:
	msg.msg = "uGetsockname: Bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOBUFS:
	msg.msg = "uGetsockname: no buffer space\n";
	except = &uNoBufsEx;
	break;
      case ENOTSOCK:
	msg.msg = "uGetsockname: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	break;
      default:
	msg.msg = "uGetsockname: unknown socket error\n";
	except = &uSocketErrorEx;
	break;
    } /* switch */    
    uRaise( socket, except, &msg, sizeof(msg) );
} /* uGetSockNameError */

int uGetsockname( uFile socket, void *name, int *namelen ) {
    int code;
    
    uFileEnter( socket );
    
    code = getsockname( socket->file, name, namelen );
    if ( code == -1 ) {
#ifdef __U_UNIXRC__
#else
	uGetSockNameError( socket );
#endif
    } /* if */
    
    uFileLeave( socket );
    
    return code;
} /* uGetsockname */

/* Local Variables: */
/* compile-command: "dmake" */
/* End: */
