/* Copyright (c) 1989  P. A. Buhr */

/*
  Unbounded buffer problem
  
  Two processes communicate through a unidirectional queue of infinite
  length. While the queue must be of finite size, it is large enough
  that in most cases it will never fill. Should it fill, an error
  message is printed and the program is terminated.
  */

#include <uSystem.h>
#include <uDelay.h>

void exit( int );
int abs( int );
long random( void );

/* shared variables for process communication */

uSemaphore full;					/* communication semaphore */
uSemaphore lock;					/* lock queue semaphore */

#define QueueSize 100

struct shrqueue {
    int front;						/* position of front of queue */
    int back;						/* position of back of queue */
    int queue[QueueSize + 1];				/* queue of integers */
}; /* shrqueue */

void qinsert( struct shrqueue *q, int elem ) {
    uP( &lock );						/* lock queue modification */
    if ( q->front == q->back ) {			/* check if queue is full */
	uPrintf( "queue full\n" );
	exit( -1 );					/* terminate all processing */
    } /* if */
    q->queue[q->back] = elem;				/* insert element in queue */
    q->back = ( q->back + 1 ) % ( QueueSize + 1 );
    uV( &lock );
} /* qinsert */

int qremove( struct shrqueue *q ) {
    int elem;
    
    uP( &lock );					/* lock queue modification */
    q->front = ( q->front + 1 ) % ( QueueSize + 1 );
    elem = q->queue[q->front];				/* remove element from queue */
    uV( &lock );
    return( elem );
} /* qremove */

void producer( struct shrqueue *q ) {
    int i;
    int N;
    
    N = 0;
    for ( i = 1; i <= 50; i += 1 ) {			/* produce a bunch of items */
	N = ( abs( random( ) ) % 100 + 1 ) + N;		/* produce a monotonic increasing positive random number */
	uPrintf( "Producer:%x, value:%d\n", uThisTask( ), N );
	qinsert( q, N );				/* insert at back of queue */
	uV( &full );					/* signal consumer */
    } /* for */
    N = -1;						/* terminate production, indicate with negative value */
    uPrintf( "Producer:%x, value:%d\n", uThisTask( ), N );
    qinsert( q, N );
    uV( &full );						/* signal consumer */
    uDie( NULL, 0 );
} /* producer */

void consumer( struct shrqueue *q ) {
    int N;
    
    /* consume until a negative element appears */
    
    do {
	uP( &full );					/* wait for producer */
	N = qremove( q );				/* remove from front of queue */
	uPrintf( "Consumer:%x, value:%d\n", uThisTask( ), N );
    } while ( N > 0 );
    uDie( NULL, 0 );
} /* consumer */

void uMain( ) {
    uTask t0, t1;
    struct shrqueue queue;

    full = U_SEMAPHORE( 0 );
    lock = U_SEMAPHORE( 1 );
    queue.front = 0;
    queue.back = 1;
    
    t0 = uEmit( producer, &queue );			/* start process */
    t1 = uEmit( consumer, &queue );			/* start process */
    
    uAbsorb( t0, NULL, 0 );				/* wait for completion of process */
    uAbsorb( t1, NULL, 0 );				/* wait for completion of process */
    uPrintf( "successful completion\n" );
} /* uMain */
