/*
  Multiple Producer and Multiple Consumer Problem with a Bounded Buffer
  
  Multiple tasks communicate through a unidirectional queue of finite length. Two semaphores are used
  to ensure that should the queue fill, the producer will wait until some free queue element appears,
  and should the queue empty, the consumer will wait until a queue element appears. Two other semaphores
  are used to make the addition and removal of an element from buffer mutually exclusive. The values
  produced go to any consumer that is available.
  */

#include <uSystem.h>
#include <uDelay.h>

extern long int random( void );

/* shared variables for process communication */

#define QueueSize 10
#define Workers 4
#define DelayRange 10
#define ElemType int

struct shrqueue {
    int front;						/* position of front of queue */
    int back;						/* position of back of queue */
    uSemaphore f_lock;					/* mutual exclusion for removal */
    uSemaphore b_lock;					/* mutual exclusion for addition */
    ElemType queue[QueueSize];				/* queue of integers */
    uSemaphore full;					/* synchronize for empty buffer */
    uSemaphore empty;					/* synchronize for full buffer */
}; /* shrqueue */

void qinsert( struct shrqueue *q, ElemType elem ) {
    int place;
    
    uP( &(q->b_lock) );					/* increment must be mutually exclusive */
    place = q->back;					/* location of back of queue */
    q->back = ( q->back + 1 ) % QueueSize;		/* increment back index */
    uV( &(q->b_lock) );
    q->queue[place] = elem;				/* insert element in queue */
} /* qinsert */

int qremove( struct shrqueue *q ) {
    ElemType elem;
    int place;
    
    uP( &(q->f_lock) );					/* increment must be mutually exclusive */
    place = q->front;					/* location of front of queue */
    q->front = ( q->front + 1 ) % QueueSize;		/* increment the front index */
    uV( &(q->f_lock) );
    elem = q->queue[place];				/* remove element from queue */
    return( elem );					/* return the element */
} /* qremove */

void Producer( struct shrqueue *q, int NoOfItems ) {
    int i, product;
    
    uPrintf("Producer:%x will produce %d items for a consumer\n", uThisTask(), NoOfItems);
    for ( i = 1; i <= NoOfItems; i += 1 ) {		/* produce a number of items */
        product = random() % 100 + 1;			/* generate random product */
        uPrintf( " Producer:%x, value:%d\n", uThisTask(), product );
        uP( &(q->empty) );				/* wait if queue is full */
        qinsert( q, product );				/* insert at back of queue */
        uV( &(q->full) );				/* signal consumer */
        uDelay( random() % DelayRange + 1 );		/* randomize the execution */
    } /* for */
    product = -1;					/* terminal value */
    uPrintf( " Producer:%x, value:%d\n", uThisTask(), product );
    uP( &(q->empty) );					/* wait if queue is full */
    qinsert( q, product );
    uV( &(q->full) );					/* signal consumer */
    uDie( NULL, 0 );
} /* Producer */

void Consumer( struct shrqueue *q ) {
    int product;
    
    /* consume until a negative element appears */
    
    do {
        uP( &(q->full) );				/* wait for producer */
        product = qremove( q );				/* remove from front of queue */
        uV( &(q->empty) );				/* signal producer about empty queue space */
        uPrintf( "Consumer :%x, value:%d\n", uThisTask(), product );
        uDelay( random() % DelayRange + 1 );		/* randomize the execution */
    } while ( product > 0 );
    uDie( NULL, 0 );
} /* Consumer */

void uMain( ) {
    int i;
    uTask p[Workers], c[Workers];
    struct shrqueue queue;

    queue.full = U_SEMAPHORE( 0 );			/* initialize bounded buffer */
    queue.empty = U_SEMAPHORE( QueueSize );
    queue.f_lock = U_SEMAPHORE( 1 );
    queue.b_lock = U_SEMAPHORE( 1 );
    queue.front = 0;
    queue.back = 0;

    for ( i = 0; i < Workers; i += 1 ) {
	p[i] = uEmit( Producer, &queue, 10 );		/* create producer */
	c[i] = uEmit( Consumer, &queue );		/* create consumer */
    } /* for */

    for ( i = 0; i < Workers; i += 1 ) {
	uAbsorb( p[i], NULL, 0 );			/* wait for completion */
	uAbsorb( c[i], NULL, 0 );			/* wait for completion */
    } /* for */

    uPrintf( "successful completion\n" );
} /* uMain */
