/* 
 * sim_ether.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.40 $
 * $Date: 1992/02/05 18:15:45 $
 */

#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <syslog.h>
#include <errno.h>
#include <signal.h>

#include <sys/types.h>
#include <netinet/in.h>
#include "x_stdio.h"
#include "xkernel.h"
#include "eth.h"
#include "eth_i.h"

/* global data of ethernet protocol */

extern int	bind( int, struct sockaddr *, int );
extern int	gethostname( char *, int );
extern int	getpid( void );
extern int	getsockopt( int, int, int, char *, int * );
extern char *	inet_ntoa( struct in_addr );
extern int	recvfrom( int, char *, int, int, struct sockaddr *, int * );
extern int	sendto( int, char *, int, int, struct sockaddr *, int );
extern int	setsockopt( int, int, int, char *, int );
extern int	socket( int, int, int );

int	traceethp;
ETHhost	ethLocalHost;

static  int	initialized = 0;
static	char 	NETNUM[16];
static	int 	ether_port;
static	int	_e;
static	int	errorCount=0;  
static	char	dummyBuf[4096];
static  char	errBuf[80];

static 	int	readether2demux( void );
static  int	push2writeether( Msg * );
static	void	init_ether( Pfi );
static	void	init_eth_blocks( void );

static Msg	emptymsg;

/* preallocated buffers */
#define MAX_ETH_BLOCKS 32
#define MAX_ETH_SIZE (EMAXPAK+sizeof(ETHhdr))

#define MAX_ERROR_COUNT 1000

#define NOREF 0			/* buffer in use flag */
#define INUSE 1

#define CLEAR_REF(_bp) (_bp)->ref = NOREF

#define RCVBUFSIZE 	50*1024		
#define EMAXPAK 	(MAX_ETH_DATA_SZ + sizeof(ETHhdr))


/* block: basic structure used to pass packets from ethernet to process */
typedef struct {
  char id;
  char ref;
  int cur_len;
  char *data;
  Msg *msg;  /* the data ptr is the msg stack */
  Semaphore sem;
  Process *procp;
} block;

/* top-level pool of buffers */
typedef struct {
  int total_blocks;
  int next_block;
  block blocks[MAX_ETH_BLOCKS];
} block_pool;

block_pool eth_block_pool;



/* changes to support full internet addressing in rom files */

void simEth2sock(ethAddr, sockAddr)
    ETHhost ethAddr;
    struct sockaddr_in *sockAddr;
{
  bzero((char *)sockAddr, sizeof (struct sockaddr_in));
  sockAddr->sin_family = AF_INET;
  /* 
   * IP address is in the first 4 bytes of the ethernet address
   * UDP port is in the 5th and 6th bytes of the ethernet address
   */
  sockAddr->sin_addr = *(struct in_addr *)&ethAddr;
  sockAddr->sin_port = (*(u_short *)((char *)&ethAddr + 4));
}


/* changed inAddr to pass-by-value to make sparc and 
   sun3 compatible - cjt 5/15 */

void sock2simEth(ethAddr, inAddr, udpPort)
     char *ethAddr;
     struct in_addr inAddr;
     int udpPort;
{
  char *cp1, *cp2;
  short tmpshrt;
  int i;

#ifdef MRED
    *(u_long *)ethAddr = *(u_long *)&inAddr;
    *(u_short *)(ethAddr + 4) = htons((u_short)udpPort);
#else

  cp1 = ethAddr;
  cp2 = (char *) &inAddr;		/* passed by value now - cjt */
  for (i=0; i<4; i++) *cp1++ = *cp2++;
  tmpshrt = htons((u_short)udpPort);
  cp2 = (char *) &tmpshrt;
  for (i=0; i<2; i++) *cp1++ = *cp2++;

#endif MRED
}


int
SetPromiscuous()
{
    return -1;
}

/*********************************************
*
* Ethernet Device
*
*********************************************/

static void
processRomFile( void )
{
    int i;

    for ( i=0; rom[i][0]; i++ ) {
	if ( ! strcmp(rom[i][0], "eth") ) {
	    if ( ! rom[i][1] || sscanf(rom[i][1], "%d", &ether_port) < 1 ) {
		xError(sprintf(errBuf, 
			       "ETH format error in line %d of the rom file",
			       i + 1));
	    }
	    break;
	}
    }
}


void
init_ether( Pfi ih )
{
  struct  sockaddr_in	addr;
  struct  hostent *h;
  struct  in_addr *in;
  int	on = 1;
  char *n, *inet_ntoa(), *strcpy();
  char	name[100];
  int	namelen=100;
  int bufSize;
  int bufSizeSize;

  xTrace0(ethp, TR_MAJOR_EVENTS, "init_ether");

  init_eth_blocks();		/* cjt */

  processRomFile();
  xTrace1(ethp, TR_MAJOR_EVENTS, "init_ether: listening on port %d", ether_port);
  if ( ether_port == 0 ) {
      Kabort("simEth: no port specified");
  }
  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr = INADDR_ANY;
  addr.sin_port = htons((unsigned short)ether_port);
  if ((_e = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    Kabort("init_ether: cannot open socket");
  }	
  setsockopt(_e, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof (on));
#ifdef SO_BROADCAST
  setsockopt(_e, SOL_SOCKET, SO_BROADCAST, (char *)&on, sizeof (on));
#endif

  /* increase receive buffer sizes from default  - menze */
  bufSize = RCVBUFSIZE;
  if (setsockopt(_e, SOL_SOCKET, SO_RCVBUF, (char *)&bufSize,
		 sizeof(bufSize))) {
      xTrace1(ethp, TR_ERRORS, "Could not set size of ethernet receive buffer to %d",
	     bufSize);
  }
  bufSizeSize = sizeof(bufSize);
  if (getsockopt(_e, SOL_SOCKET, SO_RCVBUF, (char *)&bufSize, &bufSizeSize)) {
      xTrace0(ethp, TR_ERRORS, "Could not get size of ethernet receive buffer");
  } else {
      xTrace1(ethp, TR_GROSS_EVENTS, "Receive buffer of ethernet socket: %d", bufSize);
  }
  if (getsockopt(_e, SOL_SOCKET, SO_SNDBUF, (char *)&bufSize, &bufSizeSize)) {
      xTrace0(ethp, TR_ERRORS, "Could not get size of ethernet send buffer");
  } else {
      xTrace1(ethp, TR_GROSS_EVENTS, "Send buffer of ethernet socket: %d", bufSize);
  }

  if (bind(_e, (struct sockaddr *)&addr, sizeof(addr))) {
    Kabort("init_ether: cannot bind socket");
  }
  installSignalHandler(_e, ih);
  if (fcntl(_e,F_SETFL,(FASYNC | FNDELAY)) == -1)
    Kabort("fcntl async");
  if (fcntl(_e,F_SETOWN,getpid()) == -1)
    Kabort("fcntl setown");
  gethostname(name,namelen);
  h = gethostbyname(name);
  in = (struct in_addr *) h->h_addr;
  strcpy(NETNUM, inet_ntoa(*in));
  for (n = NETNUM + strlen(NETNUM) - 1; *n != '.'; n--) {
    *n = '\0';
  }
  sock2simEth((char *)&ethLocalHost, *in, ether_port);
  xTrace1(ethp, TR_GROSS_EVENTS, "init_ether: ethernet started with address :%s:",
	 (char *)&ethLocalHost);
}


void
write_ether(buf, len)
     char	*buf;
     int	len;
{
  extern unsigned long inet_addr();
  struct  sockaddr_in	addr;
  ETHhost dest;
  
  xTrace0(ethp, TR_EVENTS, "write_ether");

  bcopy(buf, (char *)&dest, 6);
  /*
   * destination IP host and the appropriate UDP port are extracted from
   * the simulated ETH address ("dest")
   */
  simEth2sock(dest, &addr);

  xTrace3(ethp, TR_FUNCTIONAL_TRACE, "write_ether: sending %d bytes to <%d,%s>",
	 len, ntohs(addr.sin_port), inet_ntoa(addr.sin_addr));

  while (sendto(_e, buf, len, 0, (struct sockaddr *)&addr,
		sizeof(struct sockaddr)) != len) {
    xTrace0(ethp, TR_ERRORS, "write_ether: error in sendto");
    xError("sim_ether: sendto");
/*    exit(1);  */
  }
  xTrace0(ethp, TR_EVENTS, "End of write_ether");
}


int
read_ether(char *msg, int len)
{
  struct sockaddr_in	from;
  int	size, n;

  xTrace0(ethp, TR_EVENTS, "read_ether");

  size = sizeof(from);
  if ((n = recvfrom(_e, msg, len, 0, (struct sockaddr *)&from, &size)) < 0)
    return -1;

  /* was inet_ntoa(from.sin_addr.s_addr) - core dumped  cjt */
  xTrace3(ethp, TR_FUNCTIONAL_TRACE, "read_ether: receiving %d bytes from <%d,%s>",
	 n, ntohs(from.sin_port), inet_ntoa(from.sin_addr));
  return n;
}



/*
 * Parameter is useless for us.  We determine our own address rather
 * than being told about it.
 */
void
ethCtlrInit( ETHhost host )
{
    if ( ! initialized ) {
	init_ether(readether2demux);
    }
    initialized = 1;
}


void
getLocalEthHost( ETHhost *host )
{
    ethCtlrInit(*host);
    bcopy((char *)&ethLocalHost, (char *)host, sizeof(ETHhost));
}



static void
ethMsgStore( void *hdr, char *netHdr, long len, void *arg )
{
    xAssert(len == sizeof(ETHhdr));
    ((ETHhdr *)hdr)->type = htons(((ETHhdr *)hdr)->type);
    bcopy(hdr, netHdr, sizeof(ETHhdr));
}


static long
ethMsgLoad( void *hdr, char *netHdr, long len, void *arg )
{
    xAssert(len == sizeof(ETHhdr));
    bcopy(netHdr, (char *)hdr, sizeof(ETHhdr));
    ((ETHhdr *)hdr)->type = ntohs(((ETHhdr *)netHdr)->type);
    return sizeof(ETHhdr);
}



void
ethCtlrXmit( msg, dst, type )
    Msg *msg;
    ETHhost *dst;
    ETHtype type;
{
    ETHhdr	hdr;

    if (ETH_ADS_EQUAL(*dst, ethBcastHost)) {
	/*
	 * broadcast msg; goes back up as well as out
	 */
	xTrace0(ethp, TR_EVENTS, "ethCtlrXmit: bcast message");
	ethSendUp(msg, dst, type);
	/*
	 * We don't deal with broadcast addresses in this platform
	 */
	return;
    }
    hdr.src = ethLocalHost;
    hdr.dst = *dst;
    hdr.type = type;
    msgPush(msg, ethMsgStore, &hdr, sizeof(ETHhdr), NULL);
    push2writeether(msg);
}    




static void
internalDemux(block *blockp, int buflen)
{
    ETHhdr	hdr;
    Msg		msg;
    Msg		*oldmsg;
    
    xTrace0(ethp, TR_EVENTS, "in eth internal demux");
    oldmsg = blockp->msg;
    msgConstructCopy(&msg, oldmsg);
    msgTruncate(&msg, buflen);
    msgAssign(oldmsg, &emptymsg);
    msgConstructAllocate(oldmsg, MAX_ETH_SIZE, &(blockp->data));
    CLEAR_REF(blockp);
    
    if (! msgPop(&msg, ethMsgLoad, (void *)&hdr, sizeof(ETHhdr), NULL)) {
	xTrace0(ethp, TR_SOFT_ERROR, "eth_demux: incoming message too small!");
	msgDestroy(&msg);
	return;
    }
    eth_demux(msg, hdr.type, hdr.src, hdr.dst);
}




/***************************************
 *  Special function needed because we
 *  sit right on top of the device;
 *  in effect, the device driver
 ***************************************/


/*
 *   I F D E F    S P A R C
 *   **********************
 *
 */
  
#ifdef sparc

void block_handler(bp)
     block *bp;
{

  for (;;) {
    semWait(&bp->sem);		/* wait for incoming data */
    xAssert(bp->cur_len > 0 && bp->cur_len <= MAX_ETH_SIZE);
    internalDemux(bp, bp->cur_len);
  }
}


static void
init_eth_blocks()
{

  int i;
  block *bp;
  void block_handler();
  Msg *msg;

  eth_block_pool.total_blocks = MAX_ETH_BLOCKS;
  eth_block_pool.next_block = 0;

  bp = eth_block_pool.blocks;

  msgConstructEmpty(&emptymsg);

  for (i=0; i<MAX_ETH_BLOCKS; i++) {
/*
    if (!(bp->data = (char *) xMalloc(MAX_ETH_SIZE)))
      Kabort("Malloc of ethernet packets");    
*/
    msg = (Msg *)xMalloc(sizeof(Msg));
    msgConstructAllocate(msg, MAX_ETH_SIZE, &(bp->data));
    bp->msg = msg;
    bp->ref = NOREF;
    bp->id = i;
    bp->cur_len = 0;
    semInit(&bp->sem, 0);
    
    /* would be nice to get process id into struct */
    evDetach( evSchedule(block_handler, bp, 0));
    bp++;
  }
}

block *
next_block()
{
  
  block *bp;

  bp = &(eth_block_pool.blocks[eth_block_pool.next_block]);

  if (bp->ref == INUSE)
    return (block *) NULL;

  bp->ref = INUSE;
  
  eth_block_pool.next_block = 
    (eth_block_pool.next_block + 1) % eth_block_pool.total_blocks;

  return bp;
}

/* sparc version: Must pass each field of the msg structure by
   value.  Look at eth_demux to see where these are put back together.
*/

int interrupts = 0;
int checked = 0;

static int
readether2demux()
{
  int	buflen = 0;
  block *bp;			/* cjt */

  xTrace0(ethp, TR_FUNCTIONAL_TRACE, "readether2demux");

  do {
    if (!(bp = next_block())) {
      printf("ERROR: Can't get next buffer.\n");
      /* 
       * Drop this packet
       */
      if (++errorCount > MAX_ERROR_COUNT) {
	xAssert(0);
      } else {
	read_ether(dummyBuf, EMAXPAK);
	return -1;
      }
    } else if ((buflen = read_ether(bp->data, EMAXPAK)) != -1) {
      bp->cur_len = buflen;
      semSignal(&bp->sem);
    } else CLEAR_REF(bp);
  } while (buflen != -1);

  return 0;
}

#else
readether2demux()
{
  Msg	msg;
  int	buflen;
  static char buf[EMAXPAK];

  xTrace0(ethp, TR_FUNCTIONAL_TRACE, "readether2demux");

  while ((buflen = read_ether(buf, EMAXPAK)) != -1) {
    msg_make_allstack(msg, 16, buf, buflen);
    /* higher xkernel priorities mean thread can be dropped for lack
       of resources
       */
    CreateProcess4(eth_demux, STD_PRIO+1, 2 + (sizeof(Msg)+3) / 4, NULL, NULL, msg);
  }
  return 0;
}
#endif sparc


static bool
msg2Buf(char *msgPtr, long len, void *bufPtr)
{
  bcopy(msgPtr, *(char **)bufPtr, len);
  *(char **)bufPtr += len;
  return TRUE;
}


static int
push2writeether(Msg *packet)
{
  int	len;
  char	buffer[EMAXPAK];
  char	*bufPtr = buffer;
  
  xTrace0(ethp, TR_EVENTS, "push2writeether");

  if ((len = msgLen(packet)) > EMAXPAK) {
    xTrace2(ethp, TR_GROSS_EVENTS,
	    "sim ether driver: msgLen (%d) is larger than max (%d)",
	    msgLen(packet), EMAXPAK);
    return -1;
  }
#if 0
  msg_externalize(packet, buffer);
#endif
  /* Place message contents in buffer */
  msgForEach(packet, msg2Buf, &bufPtr);
  write_ether(buffer, len);
  return 0;
}


