/*
 * select.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.9 $
 * $Date: 1992/02/11 23:29:29 $
 */

#include "xkernel.h"
#include "select_i.h"


int traceselectp;


#ifdef __STDC__

static int		extractId( Part *, long * );
static void		getProtFuncs( XObj );
static void		getSessnFuncs( XObj );
static long		hdrLoad( VOID *, char *, long, VOID * );
static void		hdrStore( VOID *, char *, long, VOID * );
static void		phdr( SelHdr * );
static xkern_return_t	selectCall( XObj, Msg *, Msg * );
static int		selectControlProtl( XObj, int, char *, int );
static int		selectControlSessn( XObj, int, char *, int );
static xkern_return_t	selectDemux( XObj, XObj, Msg *, Msg * );
static xkern_return_t	selectClose( XObj );
static XObj		selectCreateSessn( XObj, XObj, ActiveKey * );
static XObj		selectOpen( XObj, XObj, XObj, Part * );
static xkern_return_t	selectOpenDisable( XObj, XObj, XObj, Part * );
static xkern_return_t 	selectOpenEnable( XObj, XObj, XObj, Part * );
static xkern_return_t	selectPop( XObj, XObj, Msg *, Msg * );

#else

static int	extractId();
static void	getProtFuncs();
static void	getSessnFuncs();
static long	hdrLoad();
static void	hdrStore();
static void	phdr();
static XObj	selectCreateSessn();

#endif __STDC__


void
select_init( self )
    XObj self;
{
    Part 	part;
    PState 	*pstate;
    
    xTrace0(selectp, TR_GROSS_EVENTS, "SELECT init");
    if ( ! xIsProtocol(xGetDown(self, 0)) ) {
	xTrace0(selectp, TR_ERRORS,
		"SELECT could not find down protocol -- not initializing");
	return;
    }
    getProtFuncs(self);
    pstate = X_NEW(PState);
    self->state = (VOID *)pstate;
    pstate->passiveMap = mapCreate(101, sizeof(PassiveKey));
    pstate->activeMap = mapCreate(101, sizeof(ActiveKey));
    partInit(&part, 1);
    partPush(part, ANY_HOST);
    if ( xOpenEnable(self, self, xGetDown(self, 0), &part) == XK_FAILURE ) {
	xTrace0(selectp,0,"select_init: could not openenable lower protocol");
    }
}


static long
hdrLoad( hdr, src, len, arg )
    VOID	*hdr, *arg;
    char 	*src;
    long  	len;
{
    xAssert(len == sizeof(SelHdr));
    bcopy( src, hdr, len );
    ((SelHdr *)hdr)->id = ntohl(((SelHdr *)hdr)->id);
    ((SelHdr *)hdr)->status = ntohl(((SelHdr *)hdr)->status);
    return len;
}


static void
hdrStore(hdr, dst, len, arg)
    VOID	*hdr, *arg;
    char 	*dst;
    long  	len;
{
    SelHdr	h;

    xAssert( len == sizeof(SelHdr) );
    h.id = htonl(((SelHdr *)hdr)->id);
    h.status = htonl(((SelHdr *)hdr)->status);
    bcopy( (char *)&h, dst, len );
}


static int
extractId( p, id )
    Part	*p;
    long	*id;
{
    long	*n;

    if ( ! p || partLen(p) < 1 ) {
	xTrace0(selectp, TR_SOFT_ERROR,
		"select extractId -- bad participant");
	return -1;
    }
    n = (long *)partPop(*p);
    if ( n == 0 || n == (long *)-1 ) {
	xTrace0(selectp, TR_SOFT_ERROR,
		"select extractId -- bad participant stack");
	return -1;
    }
    xTrace1(selectp, TR_MORE_EVENTS, "select extractID got id == %d", *n);
    *id = *n;
    return 0;
}


static XObj
selectOpen( self, hlpRcv, hlpType, p )
    XObj	self, hlpRcv, hlpType;
    Part    	*p;
{
    XObj   	s;
    PState 	*pstate = (PState *)self->state;
    ActiveKey	key;
    
    xTrace0(selectp, TR_MAJOR_EVENTS, "SELECT open");
    if ( extractId(p, &key.id) ) {
	return ERR_XOBJ;
    }
    key.lls = xOpen(self, self, xGetDown(self, 0), p);
    if ( key.lls == ERR_XOBJ ) {
	xTrace0(selectp, TR_SOFT_ERRORS, "selectOpen: could not open lls");
	return ERR_XOBJ;
    }
    s = (XObj)mapResolve(pstate->activeMap, (char *)&key);
    if ( s == ERR_XOBJ ) {
	/*
	 * session did not exist -- get an uninitialized one
	 */
	xTrace0(selectp, TR_MAJOR_EVENTS,
		"selectOpen -- no active session existed.  Creating new one.");
	s = selectCreateSessn(self, hlpRcv, &key);
    } else {
	xTrace1(selectp, TR_MAJOR_EVENTS, "Active sessn %x already exists", s);
	/*
	 * Undo the lls open
	 */
	xClose(key.lls);  
    }
    return s;
}


static XObj
selectCreateSessn( self, hlpRcv, key )
    XObj	self, hlpRcv;  		
    ActiveKey	*key;
{
    XObj 	s;
    SState	*state;
    PState	*pstate = (PState *)self->state;
    SelHdr 	*hdr;
    
    xTrace0(selectp, TR_FUNCTIONAL_TRACE, "selectCreateSessn");
    state = X_NEW(SState);
    bzero((char *)state, sizeof(SState));
    hdr = &state->hdr;
    hdr->status = SEL_OK;
    hdr->id = key->id;
    s = xCreateSessn(getSessnFuncs, hlpRcv, self, 1, &key->lls);
    s->binding = mapBind(pstate->activeMap, key, (int)s);
    if ( s->binding == ERR_BIND ) {
	xTrace0(selectp, TR_ERRORS, "selectCreateSessn -- bind failed!");
	xClose(s);
	return ERR_XOBJ;
    }
    s->state = (VOID *)state;
    return s;
}


static xkern_return_t
selectOpenEnable( self, hlpRcv, hlpType, p )
    XObj   self, hlpRcv, hlpType;
    Part    *p;
{
    PassiveKey	key;
    Enable	*e;
    PState 	*pstate = (PState *)self->state;
    
    xTrace0(selectp, TR_MAJOR_EVENTS, "SELECT open enable");
    if ( extractId(p, &key) ) {
	return XK_FAILURE;
    }
    e = (Enable *)mapResolve(pstate->passiveMap, &key);
    if ( e == ERR_ENABLE ) {
	e = X_NEW(Enable);
	e->rcnt = 1;
	e->hlpRcv = hlpRcv;
	e->hlpType = hlpType;
	xTrace1(selectp, TR_MAJOR_EVENTS, "Binding key: %d", key);
	e->binding =  mapBind(pstate->passiveMap, (char *) &key, (int)e);
	if ( e->binding == ERR_BIND ) {
	    xTrace0(selectp, TR_ERRORS, "selectEnable: could not bind hlp");
	    xFree((char *)e);
	    return XK_FAILURE;
	}
    } else {
	if ( e->hlpRcv != hlpRcv || e->hlpType != hlpType ) {
	    xTrace0(selectp, TR_SOFT_ERRORS, "selectEnable -- mismatch");
	    return XK_FAILURE;
	}
	e->rcnt++;
    }
    return XK_SUCCESS;
}


static xkern_return_t
selectOpenDisable( self, hlpRcv, hlpType, p )
    XObj   self, hlpRcv, hlpType;
    Part    *p;
{
    PassiveKey	key;
    Enable	*e;
    PState 	*pstate = (PState *)self->state;
    
    xTrace0(selectp, TR_MAJOR_EVENTS, "SELECT open disable");
    if ( extractId(p, &key) ) {
	return XK_FAILURE;
    }
    e = (Enable *)mapResolve(pstate->passiveMap, &key);
    if ( e == ERR_ENABLE || e->hlpRcv != hlpRcv || e->hlpType != hlpType ) {
	return XK_FAILURE;
    }
    if ( --e->rcnt == 0 ) {
	mapRemoveBinding(pstate->passiveMap, e->binding);
    }
    return XK_SUCCESS;
}


static xkern_return_t
selectClose(s)
    XObj   s;
{
    xTrace1(selectp, TR_EVENTS, "select_close of session %x", s);
    xAssert(xIsSession(s));
    xAssert(s->rcnt <= 0);
    xClose(xGetDown(s, 0));
    xDestroy(s);
    return XK_SUCCESS;
}


static xkern_return_t
selectCall( self, msg, rMsg )
    XObj	self;
    Msg     	*msg, *rMsg;
{
    SState	*state = (SState *)self->state;
    SelHdr 	rHdr;
    
    xTrace0(selectp, TR_EVENTS, "in select_push");
    xIfTrace(selectp, TR_DETAILED) {
	phdr(&state->hdr);
    }
    msgPush(msg, hdrStore, &state->hdr, sizeof(SelHdr), 0);
    if ( xCall(xGetDown(self, 0), msg, rMsg) == XK_FAILURE ) {
	return XK_FAILURE;
    }
    if ( msgPop(rMsg, hdrLoad, &rHdr, sizeof(SelHdr), 0) == FALSE ) {
	xTrace0(selectp, TR_ERRORS, "selectPush -- msgPop failed");
	return XK_FAILURE;
    }
    if ( rHdr.status == SEL_FAIL ) {
	xTrace0(selectp, TR_ERRORS, "selectPush -- reply indicates failure ");
	return XK_FAILURE;
    }
    return XK_SUCCESS;
}


static xkern_return_t
selectDemux( self, lls, msg, rmsg )
    XObj	self, lls;
    Msg		*msg, *rmsg;
{
    SelHdr 	hdr;
    PassiveKey 	pKey;
    ActiveKey	aKey;
    XObj   	s;
    Enable	*e;
    PState 	*pstate = (PState *)self->state;
    
    xTrace0(selectp, TR_EVENTS , "selectDemux");
    if ( msgPop(msg, hdrLoad, &hdr, sizeof(hdr), 0) == FALSE ) {
	xTrace0(selectp, TR_SOFT_ERRORS, "selectDemux -- msgPop failed!");
	return XK_FAILURE;
    }
    xIfTrace(selectp, TR_DETAILED) {
	phdr(&hdr);
    }
    aKey.id = hdr.id;
    aKey.lls = lls;
    s = (XObj) mapResolve(pstate->activeMap, (char *)&aKey);
    if ( s == ERR_XOBJ ) {
	/*
	 * no active session exists -- check for openenable
	 */
	pKey = aKey.id;
	xTrace1(selectp, TR_EVENTS, "Checking for enable using id: %d", pKey);
	e = (Enable *)mapResolve(pstate->passiveMap, (char *)&pKey);
	if ( e == ERR_ENABLE ) {
	    xTrace0(selectp, TR_EVENTS, "selectDemux: no openenable done");
	} else {
	    xTrace0(selectp, TR_MAJOR_EVENTS,
		    "select_demux creates new server session");
	    s = selectCreateSessn(self, e->hlpRcv, &aKey);
	    if ( s != ERR_XOBJ ) {
		xOpenDone(s, e->hlpType);
	    }
	}
    } else {
	xTrace0(selectp, TR_MORE_EVENTS, "selectDemux found existing sessn");
    }
    if ( s != ERR_XOBJ && xCallPop(s, lls, msg, rmsg) == XK_SUCCESS ) {
	hdr.status = SEL_OK;
    } else {
	xTrace0(selectp, TR_EVENTS, "selectDemux -- sending failure reply");
	hdr.status = SEL_FAIL;
    }
    msgPush(rmsg, hdrStore, &hdr, sizeof(hdr), 0);
    return XK_SUCCESS;
}


static xkern_return_t
selectPop( s, lls, msg, rMsg )
    XObj	s, lls;
    Msg     	*msg, *rMsg;
{
    xTrace0(selectp, TR_EVENTS, "SELECT_pop");
    return xCallDemux(s, msg, rMsg);
}


static int
selectControlProtl(self, op, buf, len)
    XObj	self;
    int 	op, len;
    char	*buf;
{
    return xControl(xGetDown(self, 0), op, buf, len);
}
  

static int
selectControlSessn(self, op, buf, len)
    XObj	self;
    int 	op, len;
    char	*buf;
{
    SState	*state = (SState *)self->state;

    switch ( op ) {
      case GETPARTICIPANTS:
	{
	    long	*idPtr;
	    int		retLen;

	    retLen = xControl(xGetDown(self, 0), op, buf, len);
	    if ( retLen < sizeof(Part) ) {
		return -1;
	    }
	    idPtr = X_NEW(long);
	    *idPtr = state->hdr.id;
	    partPush(*(Part *)buf, (VOID *)idPtr);
	    return retLen;
	}

      case GETMAXPACKET:
      case GETOPTPACKET:
	if ( xControl(xGetDown(self, 0), op, buf, len) < sizeof(int) ) {
	    return -1;
	}
	*(int *)buf -= sizeof(SelHdr);
	return (sizeof(int));
	
      default:
	return xControl(xGetDown(self, 0), op, buf, len);
    }
}
  

static void
phdr( h )
    SelHdr	*h;
{
    xTrace2(selectp, TR_ALWAYS, "SELECT header: id == %d  status == %s",
	    h->id,
	    (h->status == SEL_OK) ? "SEL_OK" :
	    (h->status == SEL_FAIL) ? "SEL_FAIL" :
	    "UNKNOWN");
}


static void
getProtFuncs( p )
    XObj 	p;
{
    p->control = selectControlProtl;
    p->open = selectOpen;
    p->openenable = selectOpenEnable;
    p->opendisable = selectOpenDisable;
    p->calldemux = selectDemux;
}


static void
getSessnFuncs( s )
    XObj	s;
{
    s->call = selectCall;
    s->callpop = selectPop;
    s->control = selectControlSessn;
    s->close = selectClose;
}

