/*
 * chan.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.25 $
 * $Date: 1992/02/05 20:13:38 $
 */

#include "xkernel.h"
#include "chan_internal.h"


/* If NO_KILLTICKET is defined, CHAN will not inform the lower protocol when it
 * can free a message.  The lower protocol will then free messages based on
 * its timeouts.
 */
/* #define NO_KILLTICKET */

/*
 * Global data
 */
int 		tracechanp=1;

#ifdef __STDC__

static void 	getProcProtl( XObj );
static void 	killticket( XObj, int * );

#else

static void 	getProcProtl();
static void 	killticket();

#endif __STDC__

/*
 * chanHdrStore - Used when calling msgPush
 */
void
chanHdrStore(hdr, dst, len, arg)
    VOID *hdr;
    char *dst;
    long int len;
    VOID *arg;
{
    CHAN_HDR h;
    
    xAssert(len == CHANHLEN);  
    h.chan = htons(((CHAN_HDR *)hdr)->chan);
    h.prot_id = htonl(((CHAN_HDR *)hdr)->prot_id);
    h.seq = htonl(((CHAN_HDR *)hdr)->seq);
    h.flags = htons(((CHAN_HDR *)hdr)->flags);
    h.boot_id = htonl(((CHAN_HDR *)hdr)->boot_id);
    h.len = htonl(((CHAN_HDR *)hdr)->len);
    bcopy((char *)(&h), dst, CHANHLEN);
}


/*
 * chanHdrLoad - Used when calling msgPop
 */
static long
chanHdrLoad(hdr, src, len, arg)
    VOID *hdr;
    char *src;
    long int len;
    VOID *arg;
{
    xAssert(len == sizeof(CHAN_HDR));  
    bcopy(src, (char *)hdr, CHANHLEN);
    ((CHAN_HDR *)hdr)->chan = ntohs(((CHAN_HDR *)hdr)->chan);
    ((CHAN_HDR *)hdr)->prot_id = ntohl(((CHAN_HDR *)hdr)->prot_id);
    ((CHAN_HDR *)hdr)->seq = ntohl(((CHAN_HDR *)hdr)->seq);
    ((CHAN_HDR *)hdr)->flags = ntohs(((CHAN_HDR *)hdr)->flags);
    ((CHAN_HDR *)hdr)->boot_id = ntohl(((CHAN_HDR *)hdr)->boot_id);
    ((CHAN_HDR *)hdr)->len = ntohl(((CHAN_HDR *)hdr)->len);
    return CHANHLEN;
}

  
static void
eventFlush(s)
    XObj s;
{
    CHAN_STATE	*state = (CHAN_STATE *)s->state;

    if ( state->event ) {
	if ( evCancel(state->event) ) {
	    /* 
	     * We successfully nuked the event, so we'll remove the
	     * reference count associated with it and free its state
	     */
	    xAssert(state->evState);
	    xFree((char *)state->evState);
	    state->evState = 0;
	    xClose(s);
	}
	state->event = 0;
    }
}


/*
 * chan_init
 */
void
chan_init(self)
    XObj self;
{
    Part	part;
    PSTATE 	*pstate;
    
    xTrace0(chanp, 1, "CHAN init");
    
    if ( ! xIsProtocol(xGetDown(self, 0)) ) {
	xTrace0(chanp, TR_ERRORS,
		"CHAN could not get get lower protl -- not initializing");
	return;
    }
    self->state = (VOID *)(pstate = X_NEW(PSTATE));
    getProcProtl(self);
    pstate->active_server_map = mapCreate(100, sizeof(ActiveID));
    pstate->active_client_map = mapCreate(100, sizeof(ActiveID));
    pstate->passive_map = mapCreate(50, sizeof(PassiveID));
    pstate->channel_number = 0;
    
    partInit(&part, 1);
    partPush(part, ANY_HOST);
    if ( xOpenEnable(self, self, xGetDown(self, 0), &part) == XK_FAILURE ) {
	xTrace0(chanp, 0, "chan_init: can't openenable transport protocol");
	xFree((char *) pstate);
    }
    xTrace0(chanp, 1, "CHAN init done");
}


/*
 * chanCloseSessn
 */
xkern_return_t
chanCloseSessn(s)
    XObj s;
{
    CHAN_STATE	*sstate;
    PSTATE 	*pstate;
    XObj	lls;
    
    xTrace0(chanp, 3, "CHAN closesessn ........................");
    xTrace1(chanp, 3, "Of session %x", s);
    xAssert( xIsSession(s) );
    xAssert( s->rcnt <= 0 );
    pstate  = (PSTATE *)s->myprotl->state;
    sstate  = (CHAN_STATE *)s->state;
    if ( s->binding ) {
	if (sstate->direction == CLIENT) {
	    mapRemoveBinding(pstate->active_client_map, s->binding);
	} else {
	    mapRemoveBinding(pstate->active_server_map, s->binding);
	}
    }
    /*
     * Free chan state
     */
    if (sstate) {
	msg_flush(sstate->saved_msg); 
	lls = xGetDown(s, 0);
	if ( lls != ERR_XOBJ ) {
	    xClose(lls);
	}
    }
    xDestroy(s);
    return XK_SUCCESS;
}


/*
 * chanDemux
 */
static xkern_return_t
chanDemux(self, lls, msg)
    XObj self;
    XObj lls;
    Msg *msg;
{
    CHAN_HDR 	hdr;
    CHAN_STATE 	*state;
    XObj   	chan_s;
    ActiveID 	ext_id;
    PassiveID 	passive_key;
    PSTATE 	*pstate;
    Map		relevant_map;
    Enable	*e;
    
    xTrace0(chanp, TR_EVENTS, "CHAN demux .............................");
    
    xAssert(xIsProtocol(self));
    pstate = (PSTATE *)self->state;
    if ( ! msgPop(msg, chanHdrLoad, (VOID *)&hdr, CHANHLEN, 0) ) {
	xError("chanDemux: msgPop returned false");
	return XK_FAILURE;
    }
    xIfTrace(chanp, 4) { 
	pChanHdr(&hdr);
    } 
    /*
     * Check for active channel
     */
    ext_id.chan		= hdr.chan;
    ext_id.prot_id 	= hdr.prot_id;
    ext_id.down_s 	= lls;
    if ((hdr.flags & REQUEST) || hdr.flags & CLNT_EXPLICIT_ACK) {
	relevant_map     = pstate->active_server_map;
	ext_id.direction = SERVER;
    } else {
	relevant_map     = pstate->active_client_map;
	ext_id.direction = CLIENT;
    }
    chan_s = (XObj) mapResolve(relevant_map, (char *) &ext_id);
    xTrace1(chanp, 4, "chanDemux: chan_s = %x", chan_s);
    /*
     * Pop to active channel
     */
    if (chan_s != ERR_XOBJ) { 
	xTrace0(chanp, 4, "chanDemux: existing channel");
	state = (CHAN_STATE *)chan_s->state;
	msgSetAttr(msg, (VOID *)&hdr);
	xPop(chan_s, lls, msg);
	return XK_SUCCESS;
    } 
    /*
     * New server channel needs to be created.  The message type should be
     * a request.  If not, a reply is coming to a session which no longer
     * exists.
     */
    if (hdr.flags & REQUEST) {
	/*
	 * Find openenable
	 */
	passive_key = ext_id.prot_id;
	e = (Enable *) mapResolve(pstate->passive_map, (char *) &passive_key);
	if (e == ERR_ENABLE) {
	    xTrace1(chanp, 4,
		    "chanDemux could not find openenable for prot %d",
		    passive_key);
	    xTrace0(chanp, 4, "chanDemux: no open enable, dropping message");
	    return XK_SUCCESS;
	}
	
	chan_s = chanSvcOpen(self, e, lls, hdr.chan, hdr.prot_id);
	if ( chan_s == ERR_XOBJ ) {
	    xTrace0(chanp, 4, "chanDemux: can't create session ");
	    return XK_SUCCESS;
	}
	/*
	 * Handle message
	 */
	msgSetAttr(msg, (VOID *)&hdr);
	xPop(chan_s, lls, msg);
	return XK_SUCCESS;
    }
    /*
     * This might be a message for a channel session which has been closed
     */ 
    xTrace0(chanp, 5, "chanDemux: dropping msg");
    
    return XK_SUCCESS;
}


int
chanCheckMsgLen(hdrLen, m)
    u_int hdrLen;
    Msg *m;
{
    u_int	dataLen;

    dataLen = msgLen(m);
    xTrace2(chanp, 4, "chan checkLen: hdr->len = %d, msg_len = %d", hdrLen,
	    dataLen);
    if (hdrLen < dataLen) {
	xTrace0(chanp, 4, "chan_pop: truncating msg");
	msgTruncate(m, hdrLen);
    } else if (hdrLen > dataLen) {
	xTrace0(chanp, 0, "chan_pop: message too short!");
	return -1;
    }
    return 0;
}


/*
 * chanControlProtl
 */
static int
chanControlProtl(self, opcode, buf, len)
    XObj self;
    int opcode;
    char *buf;
    int len;
{
    return xControl(xGetDown(self, 0), opcode, buf, len);
}
  

/*
 * chanControlSessn
 */
int
chanControlSessn(self, opcode, buf, len)
    XObj self;
    int opcode;
    char *buf;
    int len;
{
    CHAN_STATE *sstate;
    
    xTrace0(chanp, 3, "CHAN controlsessn ......................");
    xTrace1(chanp, 3, "Of session=%x", self); 
    
    sstate = (CHAN_STATE *)self->state;
    
    switch (opcode) {
	
      case GETMYPROTO:
      case GETPEERPROTO:
	checkLen(len, sizeof(long));
	*(long *)buf = sstate->hdr.prot_id;
	return sizeof(long);

      case GETMAXPACKET:
      case GETOPTPACKET:
	checkLen(len, sizeof(int));
	if ( xControl(xGetDown(self, 0), opcode, buf, len) <= 0 ) {
	    return -1;
	}
	*(int *)buf -= sizeof(CHAN_HDR);
	return sizeof(int);
	
      default:
	return xControl(xGetDown(self, 0), opcode, buf, len);
    }
}


/*
 * killticket: get rid of any dangling fragments 
 */
static void
killticket(s, t_ptr)
    XObj s;
    int *t_ptr;
{
#ifndef NO_KILLTICKET
    if (*t_ptr) {
	xAssert(xIsSession(s));
	xControl(s, FREERESOURCES, (char *)t_ptr, sizeof(int));
	*t_ptr = 0;
    }
#endif
}


/*
 * chanStatusStr - Print sequence status
 */
char *
chanStatusStr(stat)
    SEQ_STAT stat;
{
    switch (stat) {
      case old:		return "old";
      case current:	return "current";
      case new: 	return "new";
      default: 		return "UNKNOWN!!";
    }
}


/*
 * chanStateStr - Print state of server/client
 */
char *
chanStateStr(state)
    int state;
{
    switch (state) {
      case SVC_EXECUTE:	return "SVC_EXECUTE";
      case SVC_WAIT: 	return "SVC_WAIT";
      case SVC_IDLE: 	return "SVC_IDLE";
      case CLNT_FREE: 	return "CLNT_FREE";
      case CLNT_WAIT: 	return "CLNT_WAIT";
      default:		return "UNKNOWN!!";
    }
}


/*
 * pChanHdr
 */
void
pChanHdr(hdr)
    CHAN_HDR *hdr;
{
    xTrace1(chanp, TR_ALWAYS, "\t| CHAN header for channel %d:", hdr->chan);
    xTrace0(chanp, TR_ALWAYS, "\t|      msg_type: "); {
	if (hdr->flags & REQUEST) {
	    xTrace0(chanp, TR_ALWAYS, "REQUEST,");
	}
	if (hdr->flags & REPLY) {
	    xTrace0(chanp, TR_ALWAYS, "REPLY,");
	}
	if (hdr->flags & SVC_EXPLICIT_ACK) {
	    xTrace0(chanp, TR_ALWAYS, "SVC_EXPLICIT_ACK,");
	}
	if (hdr->flags & CLNT_EXPLICIT_ACK) {
	    xTrace0(chanp, TR_ALWAYS, "CLNT_EXPLICIT_ACK,");
	}
	if (hdr->flags & PROBE) {
	    xTrace0(chanp, TR_ALWAYS, "PROBE,");
	}
    }
    xTrace0(chanp, TR_ALWAYS, "\t|      flags: "); {
	if (hdr->flags & ACK_REQUESTED) {
	    xTrace0(chanp, TR_ALWAYS, "ACK_REQUESTED,");
	}
    }
    xTrace1(chanp, TR_ALWAYS, "\t|      seq: %d\n", hdr->seq);
    xTrace1(chanp, TR_ALWAYS, "\t|      prot_id: %d\n", hdr->prot_id);
    xTrace1(chanp, TR_ALWAYS, "\t|      boot_id: %d\n", hdr->boot_id);
    xTrace1(chanp, TR_ALWAYS, "\t|      len: %d\n", hdr->len);
}


/* 
 * Release timeout resources currently held.  Works for both client and
 * server
 */
void
chanFreeResources(self)
    XObj self;
{
    XObj	lls;
    CHAN_STATE	*state;

    state = (CHAN_STATE *)self->state;
    xAssert(state);
    lls = xGetDown(self, 0);
    xAssert(xIsSession(lls));
    killticket(lls, &state->ticket);
    msg_flush(state->saved_msg);
    eventFlush(self);
}


/*
 * chanReply 
 */
void
chanReply(s, hdr, flags)
    XObj s;
    CHAN_HDR *hdr;
    u_short flags;
{
    CHAN_HDR 	hdr_copy;
    Msg 		msg;
    
    xAssert(xIsSession(s));
    hdr_copy 	 = *hdr;
    hdr_copy.flags = flags;
    hdr_copy.len 	 = 0;
    xTrace0(chanp, 4, "chan_pop: Sending reply");
    xIfTrace(chanp, 4) {
	pChanHdr(&hdr_copy);
    }
    msgConstructEmpty(&msg);
    msgPush(&msg, chanHdrStore, &hdr_copy, CHANHLEN, NULL);
    xPush(s, &msg);
    msgDestroy(&msg);
}


/*
 * chanCheckSeq
 */
SEQ_STAT 
chanCheckSeq(cur_seq, new_seq)
    unsigned int cur_seq;
    unsigned int new_seq;
{
    if (cur_seq == new_seq) return(current);
    if (cur_seq > new_seq)  return(old);
    return(new);
}


long
chanGetProtNum( self, hlp )
    XObj	self, hlp;
{
    long n;

    if ( (n = relProtNum(hlp, self)) == -1 ) {
	xTrace1(chanp, TR_ERRORS,
		"chan could not get relative protocol number of %s",
		hlp->name);
    }
    return n;
}
  


/* 
 * getProcProtl
 */
static void 
getProcProtl(s)
    XObj s;
{
    xAssert(xIsProtocol(s));
    s->control 	= chanControlProtl;
    s->open   	= chanOpen;
    s->openenable = chanOpenEnable;
    s->demux 	= chanDemux;
}




