/*
 * blast_input.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.23 $
 * $Date: 1992/02/05 18:22:40 $
 */

#include "xkernel.h"
#include "blast_internal.h"

#define DONE(mask, num_frags) (mask == blastFullMask[num_frags])

#ifdef __STDC__

static void	receive_timeout( void * );
static void	free_rec_seq( MSG_STATE * );
static void	freeMstate( MSG_STATE * );
static u_short	insert_frag( BLAST_HDR *, Msg *, MSG_STATE * );
static void	reassemble( Msg *, MSG_STATE * );
static void	send_nack( MSG_STATE * );

#else

static void	receive_timeout();
static void	free_rec_seq();
static void	freeMstate();
static u_short	insert_frag();
static void	reassemble();
static void	send_nack();

#endif __STDC__


xkern_return_t
blastDemux(self, lls, dg)
    XObj self;
    XObj lls;
    Msg *dg;
{
    PSTATE 	*pstate;
    BLAST_HDR 	hdr;
    XObj   	blast_s;
    ActiveID 	active_key;
    PassiveID 	passive_key;
    Enable	*e;

#ifdef BLAST_SIM_DROPS
    static int count = 0;
    
    count++;
    xTrace1(blastp, TR_EVENTS, "blast_demux: called(%d)", count);
#else
    xTrace0(blastp, TR_EVENTS, "blast_demux");
#endif BLAST_SIM_DROPS

    xAssert(xIsProtocol(self));
    xAssert(xIsSession(lls));
    
    pstate = (PSTATE *) self->state;

    if ( msgPop(dg, blastHdrLoad, &hdr, sizeof(hdr), 0) == FALSE ) {
	xTrace0(blastp, TR_SOFT_ERRORS, "msgPop failed!");
	return XK_FAILURE;
    }
    msgSetAttr(dg, (VOID *)&hdr);
    xIfTrace(blastp, TR_MORE_EVENTS) {
	xTrace0(blastp, TR_ALWAYS, "Incoming message ");
	blast_phdr(&hdr);
    }
    /* check for active sessions */
    active_key.lls = lls;
    active_key.prot = hdr.prot_id;

    blast_s = (XObj) mapResolve(pstate->active_map, (char *) &active_key);

    xIfTrace(blastp, TR_MORE_EVENTS) {
	blastShowActiveKey(&active_key, "blast_demux");
    }
    xTrace1(blastp, TR_MORE_EVENTS,
	    "blast_demux: blast_s lookup = %x", blast_s);
    
    if (blast_s == ERR_XOBJ) { 
	/* nacks must be sent to active session */
	if (hdr.op != BLAST_SEND) {
	    xTrace1(blastp, TR_EVENTS,
		    "blast_demux: spurious msg (op %s) dropped",
		    blastOpStr(hdr.op));
	    return XK_SUCCESS;
	}
	/*
	 * We don't want two threads trying to create the same session
	 */
	semWait(&pstate->createSem);
	/*
	 * Check again for active session now that we have the lock
	 */
	blast_s = (XObj) mapResolve(pstate->active_map, (char *) &active_key);
	if (blast_s == ERR_XOBJ) {
	    /*
	     * create new session
	     */
	    passive_key = active_key.prot;
	    e = (Enable *)mapResolve(pstate->passive_map,
				     (char *)&passive_key);
	    if (e == ERR_ENABLE) {
		xTrace0(blastp, TR_EVENTS,
			"blast_demux: no open enable dropping message");
		semSignal(&pstate->createSem);
		return XK_SUCCESS;
	    }
	    xTrace0(blastp, TR_EVENTS,
		    "blast_demux found openenable, opening blast sessn");
	    blast_s = blastCreateSessn(self, e->hlpRcv, &active_key);
	    if (blast_s == ERR_XOBJ) {
		xTrace0(blastp, TR_ERRORS,
			"blast_demux: can't create session");
		semSignal(&pstate->createSem);
		return XK_SUCCESS;
	    }
	    xDuplicate(lls);
	    xOpenDone(blast_s, e->hlpType);
	}
	semSignal(&pstate->createSem);
    }
    if (msgLen(dg) > hdr.len) {
	xTrace2(blastp, TR_MORE_EVENTS,
		"Truncating message from %d to %d bytes",
		msgLen(dg), hdr.len);
	msgTruncate(dg, hdr.len);
    }
#ifdef BLAST_SIM_DROPS
    /* test timeout ability */
    {
	static int count = 0;
	if ((++count % (131 + 2 * pstate->myBlastHost.d)) == 0) {
	    xError("blast_demux: dropping random fragment");
	    return XK_SUCCESS;
	}
    }
#endif
    return xPop(blast_s, lls, dg);
}


static MSG_STATE *
newMstate(s)
    XObj s;
{
    BLAST_STATE	*state = (BLAST_STATE *)s->state;
    MSG_STATE	*mstate;
    
    if (state->rec_cache) {
	mstate = state->rec_cache;
	state->rec_cache = 0;
	mstate->mask = 0;
	mstate->old_mask = 0;
	mstate->nack_sent = 0;
	mstate->abort = FALSE;
    } else {
	xTrace0(blastp, TR_MORE_EVENTS, "blast_pop: new_state created ");
	mstate = (MSG_STATE *) xMalloc(sizeof(MSG_STATE));
	bzero((char *)mstate,sizeof(MSG_STATE));
    }
    /* 
     * Add a reference count for this message state
     */
    xDuplicate(s);
    return mstate;
}


static xkern_return_t
blastReceiverPop(s, dg, hdr)
    XObj s;
    Msg *dg;
    BLAST_HDR *hdr;
{
    BLAST_STATE	*state;
    MSG_STATE 	*mstate;
    Msg 	msg;
    int 	seq;
    
    xTrace0(blastp, TR_EVENTS, "BLAST receiverPop");
    xAssert(xIsSession(s));
    
    /*
     * If message is short just send it up
     */
    if (hdr->seq == 0) {
	xTrace0(blastp, TR_EVENTS, "blast_receiverPop: short cut");
	xTrace1(blastp, TR_DETAILED, "hdr_len = %d \n", hdr->len);
	return xDemux(s, dg);
    }
    state = (BLAST_STATE *)s->state;
    seq = hdr->seq;
    
    /* look for message state */
    mstate = (MSG_STATE *) mapResolve(state->rec_map,(char *)&seq);
    if (mstate == (MSG_STATE *)-1) {
	if (hdr->op == BLAST_RETRANSMIT) {
	    /* message must already be finished */
	    return XK_SUCCESS;
	}
	mstate = newMstate(s);
	mstate->state = state;
	mstate->hdr = *hdr;
	mstate->wait = hdr->num_frag * REC_CONST * 1000;
	mstate->binding = (Bind) mapBind(state->rec_map, (char *)&seq,
					 (int)mstate);
	if (mstate->binding == (Bind)-1) {
	    xTrace0(blastp, TR_ERRORS, "blast_pop: can't bind message state ");
	    return XK_FAILURE;
	}
	mstate->event = evSchedule(receive_timeout, mstate, mstate->wait);
    }
    /* add fragment */
    mstate->mask = insert_frag(hdr, dg, mstate);
    xTrace1(blastp, TR_MORE_EVENTS, "Inserted frag, new mask: %s",
	    blastShowMask(mstate->mask));
    if (DONE(mstate->mask, hdr->num_frag)) {
	reassemble(&msg, mstate);
	xTrace1(blastp, TR_MORE_EVENTS,
		"blast_pop: incoming msg len (no hdr): %d",
		msgLen(&msg));
	/* get rid of message */
	free_rec_seq(mstate);
	xDemux(s, &msg);
	msgDestroy(&msg);
    } else {
	/* if this is the last fragment send a nack */
#ifdef BLAST_LAST_FRAG_NACKS
	if (hdr->mask == I_TO_MASK(hdr->num_frag)) {
	    if (mstate->nack_sent == 0) {
		xTrace0(blastp, REXMIT_T, "blast_pop: last frag !");
		if (evCancel(mstate->event)) {
		    /*
		     * Event successfully cancelled
		     */
		    send_nack(mstate); 
		    mstate->event = evSchedule(receive_timeout,
					       mstate,
					       mstate->wait);
		}
	    } else {
		xTrace0(blastp, REXMIT_T, "blast_pop: last frag disabled!");
	    }
	} 
#endif BLAST_LAST_FRAG_NACKS
    }
    return XK_SUCCESS;
} 


xkern_return_t
blastPop(s, lls, dg)
    XObj s;
    XObj lls;
    Msg *dg;
{
    BLAST_HDR	*hdr;

    hdr = (BLAST_HDR *)msgGetAttr(dg);
    xAssert(hdr);
    switch (hdr->op) {
      case BLAST_NACK:
    	return blastSenderPop(s, dg, hdr);

      case BLAST_SEND:
      case BLAST_RETRANSMIT:
	return blastReceiverPop(s, dg, hdr);

      default:
	xTrace0(blastp, TR_ERRORS, "blast_pop: illegal operation ");
	return XK_SUCCESS;
    }
}


/*
 * Free the message and state associated with the incoming message referenced
 * by mstate
 */
static void
free_rec_seq(mstate)
    MSG_STATE *mstate;
{
    xTrace1(blastp, TR_EVENTS,
	    "free_rec_msg: begin seq = %d", mstate->hdr.seq);
    
    xTrace1(blastp, TR_MORE_EVENTS, "Cancelling event %s", mstate->event);
    if (evCancel(mstate->event) == FALSE) {
	/*
	 * Couldn't cancel the event before it got in the run queue.
	 * Make sure it doesn't send any retransmission requests when it
	 * does run, but let it do the freeing.
	 */
	xTrace0(blastp, TR_MORE_EVENTS,
		"Event cancel failed -- event will run");
	mstate->abort = TRUE;
    } else {
	xTrace0(blastp, TR_MORE_EVENTS, "Event cancel successful");
	freeMstate(mstate);
    }
}


static void
freeMstate(mstate)
    MSG_STATE *mstate;
{
    int		i;
    BLAST_STATE	*state;
    XObj	sessn;

    state = mstate->state;
    sessn = state->self;
    if (mstate->binding) {
	if (mapRemoveBinding(state->rec_map,mstate->binding) == -1) {
	    xTrace0(blastp, TR_ERRORS, "free_rec_seq: can't unbind!");
	    return;
	}
    } else {
	xTrace0(blastp, TR_ERRORS, "free_rec_seq: no binding?!");
    }
    xTrace1(blastp, TR_MORE_EVENTS,
	    "free_rec_msg: num_frags = %d", mstate->hdr.num_frag);
    for (i=1; i <= mstate->hdr.num_frag; i++, mstate->mask >>= 1) {
	if ( mstate->mask & 1 ) {
	    msgDestroy(&mstate->frags[i]);
	}
    }
    if (!state->rec_cache) {
	state->rec_cache = mstate;
    } else {
	xFree((char *)mstate);
    }
    /* 
     * Remove reference count for this message
     */
    xClose(sessn);
}


/* insert_frag -- Insert the fragment 'msg' into the message referenced
 * by 'mstate'.  Returns the new mask.
 */
static u_short
insert_frag(hdr, msg, mstate)
    BLAST_HDR *hdr;
    Msg *msg;
    MSG_STATE *mstate;
{
    u_short	mask;
    int 	loc;
    
    xTrace0(blastp, TR_MORE_EVENTS, "insert_frag: called");
    mask = mstate->mask;
    loc = blast_mask_to_i(hdr->mask);
    if (loc == 0) {
	xTrace0(blastp, TR_ERRORS, "insert_frag: illegal fragment");
	return mask;
    }
    xTrace1(blastp, TR_MORE_EVENTS, "insert_frag: inserting fragment %d",loc);
    /* prefer later messages */
    xTrace2(blastp, TR_MORE_EVENTS,
	    "insert_frag: hdr->len = %d msg_len = %d", hdr->len,
	    msgLen(msg));
    xAssert(hdr->len == msgLen(msg));
    if (mstate->mask & hdr->mask) {
	xTrace1(blastp, TR_MORE_EVENTS, "Freeing fragment %d", loc);
	msgAssign(&mstate->frags[loc], msg);
    } else {
	msgConstructCopy(&mstate->frags[loc], msg);
    }
    return (mask | hdr->mask);
}


/*
 * Return the message formed by the assembly of the fragments in 'mstate'
 * The fragments themselves are freed and the mstate mask is cleared to
 * indicate that there are no more valid fragments in the mstate structure.
 */
static void
reassemble(m, mstate)
    Msg *m;
    MSG_STATE *mstate;
{
    int i;

    xTrace0(blastp, TR_EVENTS, "reassemble: called");

    msgConstructCopy(m, &mstate->frags[1]);
    msgDestroy(&mstate->frags[1]);
 
    xAssert(mstate->mask == blastFullMask[mstate->hdr.num_frag]);

    for (i=2; i<=mstate->hdr.num_frag; i++) {
	xTrace1(blastp, TR_MORE_EVENTS,
		"reassemble: frag = %x", &mstate->frags[i]);
	xTrace1(blastp, TR_MORE_EVENTS, "reassemble: frag_len = %d",
		msgLen(&mstate->frags[i]));
	msgJoin(m, m, &mstate->frags[i]);
	msgDestroy(&mstate->frags[i]);
    }
    xTrace1(blastp, TR_EVENTS, "reassemble: complete msg_len = %d", msgLen(m));
    mstate->mask = 0;
}


/* receive timeout: sends nack if everything has not arrived in
 * a certain time. If nothing arrives between two nacks then 
 * msg is flushed 
 */
static void
receive_timeout(arg)
    VOID *arg;
{
    MSG_STATE	*mstate = (MSG_STATE *)arg;

    xTrace0(blastp, REXMIT_T, "blast: receive_timeout: timeout!");
    
    /*
     * Detach myself
     */
    evDetach(mstate->event);
    xIfTrace(blastp, REXMIT_T) blastShowMstate(mstate, "receive timeout");
    if ( ! mstate->abort ) {
	if (mstate->old_mask == 0) {
	    xTrace0(blastp, REXMIT_T, "receive_timeout sending first nack");
	    /* first time though */
	    mstate->old_mask = mstate->mask;
	    mstate->wait = 5*mstate->wait;
	    mstate->event =
	      evSchedule(receive_timeout, mstate, mstate->wait);
	    send_nack(mstate);
	    return;
	}
	/* check for progress */
	if (mstate->old_mask != mstate->mask) {
	    xTrace0(blastp, REXMIT_T,
		    "receive_timeout sending additional nack");
	    mstate->old_mask = mstate->mask;
	    mstate->wait = 5*mstate->wait;
	    mstate->event = evSchedule(receive_timeout, mstate, mstate->wait);
	    send_nack(mstate);
	    return;
	}
    }
    xTrace0(blastp, REXMIT_T, "receive_timeout aborting");
    /* flush message */
    freeMstate(mstate);
    return;
}


static void
send_nack(mstate)
    MSG_STATE *mstate;
{
    BLAST_STATE *state;
    BLAST_HDR	hdr;
    Msg 	m;
    
    xTrace1(blastp, REXMIT_T, "send_nack: called mask = %s",
	    blastShowMask(mstate->mask));
    state = (BLAST_STATE *) mstate->state;
    mstate->nack_sent = 1;
    
    hdr = mstate->hdr;
    hdr.op = BLAST_NACK;
    hdr.mask = mstate->mask;
    msgConstructEmpty(&m);
    msgPush(&m, blastHdrStore, &hdr, sizeof(hdr), 0);
    xIfTrace(blastp, REXMIT_T + 2) {
	blast_phdr(&hdr);
    }
    if (xPush(xGetDown(state->self, 0), &m) ==  -1) {
	xTrace0(blastp, REXMIT_T, "send_nack: can't send nack");
    }
    msgDestroy(&m);
}

