/*
 * blast.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.27 $
 * $Date: 1992/02/05 18:22:36 $
 */

/*
 * Initialization and connection establishment / teardown routines
 */

#include "xkernel.h"
#include "arp.h"
#include "blast.h"
#include "blast_internal.h"


#define checkPart(pxx, sxx, retVal) 					\
    if ( ! (pxx) || partLen(pxx) < 1 ) {				\
        xTrace1(blastp, TR_SOFT_ERROR, "%s -- bad participants", sxx); 	\
	return retVal;							\
    }


#ifdef __STDC__

static xkern_return_t	blastClose( XObj );
static void	sessnGetProc( XObj );
static void	protGetProc( XObj );

#else

static xkern_return_t	blastClose();
static void	sessnGetProc();
static void	protGetProc();

#endif __STDC__


int traceblastp=0;
unsigned int blastFullMask[MAX_FRAGS + 1];


void
blast_init(self)
    XObj self;
{
    Part 	part;
    PSTATE 	*pstate;
    int 	i;
    XObj	llp;
    
    xTrace0(blastp, TR_GROSS_EVENTS, "BLAST init");
    xAssert(xIsProtocol(self));
    
    if ((llp = xGetDown(self,0)) == ERR_XOBJ) {
	xTrace0(blastp, TR_ERRORS, "blast couldn't access lower protocol");
	return;
    }
    protGetProc(self);
    pstate = (PSTATE *)(self->state = xMalloc(sizeof(PSTATE)));
    pstate->max_seq = 0;
    pstate->active_map = mapCreate(101, sizeof(ActiveID));
    pstate->passive_map = mapCreate(101, sizeof(PassiveID));
    semInit(&pstate->outstanding_messages, OUTSTANDING_MESSAGES);
    semInit(&pstate->createSem, 1);
    pstate->max_outstanding_messages = OUTSTANDING_MESSAGES;
    for (i=1; i <= MAX_FRAGS; i++) {
	blastFullMask[i] = (1 << i) - 1;
    }
    partInit(&part, 1);
    partPush(part, ANY_HOST);
    xOpenEnable(self, self, llp, &part);
}


static long
getRelProtNum( hlp, llp, s )
    XObj	hlp, llp;
    char	*s;
{
    long	n;

    n = relProtNum(hlp, llp);
    if ( n == -1 ) {
	xTrace3(blastp, TR_SOFT_ERROR,
	       "%s: couldn't get prot num of %s relative to %s",
	       s, hlp->name, llp->name);
	return -1;
    }
    return n;
}


static XObj
blastOpen(self, hlpRcv, hlpType, p)
    XObj self, hlpRcv, hlpType;
    Part *p;
{
    XObj	s;
    XObj	lls;
    ActiveID 	active_id;
    PSTATE 	*pstate;
    
    xTrace0(blastp, TR_MAJOR_EVENTS, "BLAST open");
    checkPart(p, "blastOpen", ERR_XOBJ);
    pstate = (PSTATE *)self->state;
    semWait(&pstate->createSem);
    if ( (active_id.prot = getRelProtNum(hlpType, self, "blastOpen")) == -1 ) {
	return ERR_XOBJ;
    }
    if ( (lls = xOpen(self, self, xGetDown(self, 0), p)) == ERR_XOBJ ) {
	xTrace0(blastp, TR_MAJOR_EVENTS, "blast open: could not open lls");
	s = ERR_XOBJ;
    } else {
	xTrace0(blastp, TR_MAJOR_EVENTS, "blast_open successfully opened lls");
	active_id.lls = lls;
	xIfTrace(blastp, TR_MORE_EVENTS) {
	    blastShowActiveKey(&active_id, "blast_open");
	}
	/*
	 * is there an existing session?
	 */
	s = (XObj)mapResolve(pstate->active_map, (char *) &active_id);
	if (s == ERR_XOBJ) {
	    xTrace0(blastp, TR_MAJOR_EVENTS,
		    "blast_open creating new session");
	    s = blastCreateSessn(self, hlpRcv, &active_id);
	    if ( s == ERR_XOBJ ) {
		xClose(lls);
	    }
	} else {
	    xTrace0(blastp, TR_MAJOR_EVENTS, "blast_open: session exists");
	    xClose(lls);
	}
    }
    semSignal(&pstate->createSem);
    return s;
}    


XObj
blastCreateSessn(self, hlp, key)
    XObj self;
    XObj hlp;
    ActiveID *key;
{
    PSTATE	*pstate;
    BLAST_STATE *state;
    BLAST_HDR	*hdr;
    XObj	s;

    pstate = (PSTATE *)self->state;
    state = (BLAST_STATE *) xMalloc(sizeof(BLAST_STATE));
    bzero((char *)state, sizeof(BLAST_STATE));
    state->prot_id = key->prot;
    state->send_map = mapCreate(101, sizeof(BlastSeq));
    state->rec_map = mapCreate(101, sizeof(BlastSeq));
    /*
     * fill in header for messages that don't require fragmentation
     */
    hdr = &state->short_hdr;
    hdr->op = BLAST_SEND;
    hdr->prot_id = key->prot;
    hdr->seq = 0; /* protocol guarantees that 0 can never be a real seq */
    hdr->mask = 0;
    hdr->num_frag = 0;
    /*
     * Determine the maximum size of datagrams which this session
     * supports.  
     */
    if (xControl(key->lls, GETOPTPACKET, (char *)&state->fragmentSize,
		 sizeof(state->fragmentSize)) < 0) {
	xTrace0(blastp, TR_ERRORS,
		"Blast could not get opt packet size from lls");
	if (xControl(key->lls, GETMAXPACKET, (char *)&state->fragmentSize,
		     sizeof(state->fragmentSize)) < 0) {
	    xTrace0(blastp, TR_ERRORS,
		    "Blast could not get max packet size from lls");
	    xFree((char *)state);
	    return ERR_XOBJ;
	}
    }
    xTrace1(blastp, TR_MAJOR_EVENTS,
	    "fragment size (from llp): %d", state->fragmentSize);
    xTrace1(blastp, TR_MAJOR_EVENTS, "blast hdr len: %d", BLASTHLEN);
    state->fragmentSize -= BLASTHLEN;
    xTrace2(blastp, TR_MAJOR_EVENTS,
	    "Blast fragmenting into packets of size %d, max dgm: %d",
	    state->fragmentSize, state->fragmentSize * MAX_FRAGS);
    /*
     * create session and bind to address
     */
    s = xCreateSessn(sessnGetProc, hlp, self, 1, &key->lls);
    s->state = (VOID *)state;
    state->self = s;
    s->binding = (Bind)mapBind(pstate->active_map, (char *)key, s);
    /*
     * just to be paranoid
     */
    if (s->binding == ERR_BIND) {
	xTrace0(blastp, TR_ERRORS, "blast_open: could not bind session");
	blastClose(s);
	return ERR_XOBJ;
    }
    xTrace1(blastp, TR_MAJOR_EVENTS, "blast_open returns %x", s);
    return s;
}


static xkern_return_t
blastOpenEnable(self, hlpRcv, hlpType, p)
    XObj self, hlpRcv, hlpType;
    Part *p;
{
    PassiveID 	key;
    PSTATE 	*pstate;
    Enable	*e;
    
    xTrace0(blastp, TR_MAJOR_EVENTS, "BLAST open enable");
    checkPart(p, "blastOpenEnable", XK_FAILURE);
    pstate = (PSTATE *)self->state;
    if ( (key = getRelProtNum(hlpType, self)) == -1 ) {
	return XK_FAILURE;
    }
    xTrace1(blastp, TR_MAJOR_EVENTS,
	    "blast_openenable: ext_id.prot_id = %d", key);
    e = (Enable *) mapResolve(pstate->passive_map, (char *) &key);
    if ( e != ERR_ENABLE ) {
	if ( e->hlpRcv == hlpRcv && e->hlpType == hlpType) {
	    e->rcnt++;
	} else {
	    return XK_FAILURE;
	}
    } else {
	e = (Enable *)xMalloc(sizeof(Enable));
	e->hlpRcv = hlpRcv;
	e->hlpType = hlpType;
	e->rcnt = 1;
	e->binding = mapBind(pstate->passive_map, (char *) &key, (int) e);
	if (e->binding == ERR_BIND) {
	    xTrace0(blastp, TR_ERRORS, "Couldn't bind openenable!");
	    return XK_FAILURE;
	}
    }
    xTrace0(blastp, TR_MAJOR_EVENTS,
	    "Blast open enable returning successfully");
    return XK_SUCCESS;
}


static xkern_return_t
blastOpenDisable( self, hlpRcv, hlpType, p )
    XObj self, hlpRcv, hlpType;
    Part *p;
{
    PassiveID   key;
    PSTATE      *pstate;
    Enable	*e;
    
    xTrace0(blastp, TR_MAJOR_EVENTS, "BLAST open disable");
    checkPart(p, "blastOpenDisable", XK_FAILURE);
    pstate = (PSTATE *)self->state;
    if ( (key = getRelProtNum(hlpType, self)) == -1 ) {
	return XK_FAILURE;
    }
    e = (Enable *) mapResolve(pstate->passive_map, (char *) &key);
    if ( e == ERR_ENABLE || e->hlpRcv != hlpRcv || e->hlpType != hlpType ) {
	return XK_FAILURE;
    }
    if (--(e->rcnt) == 0) {
	mapRemoveBinding(pstate->passive_map, e->binding);
	xFree((char *)e);
    }
    return XK_SUCCESS;
}


/*
 * blastClose: blast does no caching of sessions -- when a session's
 * reference count drops to zero it is destroyed.
 */
static xkern_return_t
blastClose(s)
    XObj s;
{
    BLAST_STATE *sstate;
    PSTATE	*pstate;
    XObj	lls;
    
    xTrace1(blastp, TR_MAJOR_EVENTS, "blast_close of session %x", s);

    if (!s) return(XK_SUCCESS); 
    sstate = (BLAST_STATE *)s->state;
    pstate = (PSTATE *)s->myprotl->state;

    xAssert(s->rcnt <= 0);
    if (s->binding) {
	mapRemoveBinding(pstate->active_map,s->binding);
    }
    /* free blast state */
    if (sstate->send_map) {
	blast_mapFlush(sstate->send_map);
	mapClose(sstate->send_map);
    }
    if (sstate->rec_map) {
	blast_mapFlush(sstate->rec_map);
	mapClose(sstate->rec_map);
    }
    if ((lls = xGetDown(s, 0)) != ERR_XOBJ) {
	xClose(lls);
    }
    xDestroy(s);
    return XK_SUCCESS;
}

    	
static void
sessnGetProc(s)
    XObj s;
{
    s->push = blastPush;
    s->control = blastControlSessn;
    s->close = blastClose;
    s->pop = blastPop;
}


static void
protGetProc(p)
    XObj p;
{
    p->control = blastControlProtl;
    p->open = blastOpen;
    p->openenable = blastOpenEnable;
    p->demux = blastDemux;
    p->opendisable = blastOpenDisable;
}

