/*
 * process.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.17 $
 * $Date: 1992/02/07 17:08:16 $
 */

/*
 * Mach3
 *
 *  This file is based on process.c from the non-kernel version as
 *  of Thu Jan 23 18:08:19 1992, AND the in-kernel version from
 *  August 1991
 *
 */


#ifdef XKMACHKERNEL
#include <kern/lock.h>
#include <kern/kalloc.h>
#include <kern/syscall_subr.h>
#include <sys/varargs.h>
#else /*  XKMACHKERNEL */
#include <varargs.h>
#include <cthreads.h>
#include <sys/time.h>
#include <mach.h>
#endif XKMACHKERNEL

#include <mach/message.h>

/* xkernel.h includes xk_debug.h */
#include "xkernel.h"
#include "assert.h"
/* platform includes process.h */
#include "platform.h"

void enter_master_sledgehammer_monitor();


#define MAXARGS		6
#ifdef XKMACHKERNEL
#define MAX_XK_THREADS 16
#else
#define MAX_XK_THREADS 32
#endif XKMACHKERNEL

int total_processes = 0;
int current_xk_threads = 0;

#ifdef XKMACHKERNEL

queue_head_t			xkInfoQ;
queue_head_t			xkThreadQ;
queue_head_t			xkThreadArgQ;
decl_simple_lock_data(,	xkInfo_lock)
decl_simple_lock_data(,	xkThread_lock)
decl_simple_lock_data(,	xkThreadArg_lock)
decl_simple_lock_data(,	xkMaster_lock)

typedef struct {
	queue_head_t	q;	/* used by queue routines */
	thread_t	thread;	/* id of suspended thread */
} threadInfo;

#define XTHREAD_SELF() current_thread()
#define XTHREAD_TYPE	thread_t

#else /* XKMACHKERNEL */

mutex_t sledgehammer_concurrency_control;
#define XTHREAD_SELF()	cthread_self()
#define XTHREAD_TYPE	cthread_t
Process *Active;

#endif XKMACHKERNEL

/* the master concurrency locks */

#ifdef XKMACHKERNEL
#define	MASTER_LOCK	simple_lock( &xkMaster_lock )
#define	MASTER_UNLOCK	simple_unlock( &xkMaster_lock )
#else  /* ! XKMACHKERNEL */
#define MASTER_LOCK	mutex_lock(sledgehammer_concurrency_control)
#define MASTER_UNLOCK	mutex_unlock(sledgehammer_concurrency_control)
#endif XKMACHKERNEL

typedef struct {
#ifdef XKMACHKERNEL
	queue_head_t	q;	/* used by queue routines */
#endif XKMACHKERNEL
	int	*args;	/* pointer to argument block */
} argBlock;

/*
 *
 * threadInit
 *
 */
void threadInit()
{
  xTrace0(processcreation, TR_FULL_TRACE, "Enter event function threadInit");

#ifndef XKMACHKERNEL
  cthread_init();
  sledgehammer_concurrency_control = mutex_alloc();
#else /* XKMACHKERNEL */

  simple_lock_init( &xkMaster_lock );
  queue_init( &xkInfoQ );
  simple_lock_init( &xkInfo_lock );
  queue_init( &xkThreadQ );
  simple_lock_init( &xkThread_lock );
  queue_init( &xkThreadArgQ );
  simple_lock_init( &xkThreadArg_lock );

#endif ! XKMACHKERNEL
}

/*
 *
 * xkernel_limit_active_threads
 *
 */
static bool
xkernel_limit_active_threads(expendable, args, priority)
     int expendable;
     int *args;
     int priority;
{
#ifdef XKMACHKERNEL
  argBlock *info = (argBlock *)xMalloc(sizeof(argBlock));
#endif XKMACHKERNEL

  XTHREAD_TYPE child;

  xTrace0(processcreation, TR_FULL_TRACE, "xkernel_limit_active_threads enter");
  if (current_xk_threads > MAX_XK_THREADS && expendable ) {
    xTrace0(processcreation, TR_ERRORS, "too many active threads; dying");
    return FALSE;
  }
  if (current_xk_threads > MAX_XK_THREADS)
    xTrace1(processcreation, TR_ERRORS, "forcing essential thread beyond limit %d", current_xk_threads);
  /* we do this without a lock; could make count invalid */
  current_xk_threads++;

#ifdef XKMACHKERNEL
  /* 
    in the kernel we can't pass args to a thread, so we have to
    enqueue the arg block.  The thread will dequeue the args when
    it starts up in the master monitor
   */
  info->args = args;
  simple_lock( &xkInfo_lock );
  enqueue_tail( &xkInfoQ, &info->q );
  simple_unlock( &xkInfo_lock );
  xTrace1(processcreation, TR_EVENTS, "processcreate: create arg block at %x", args);

  child = kernel_thread( kernel_task, enter_master_sledgehammer_monitor );

#else /* XKMACHKERNEL */
  /* here we can start the thread with a pointer to arguments directly */
  child = cthread_fork(enter_master_sledgehammer_monitor, args);
  cthread_set_name(child,"xkernel thread");
  thread_priority(child,priority,FALSE);
  cthread_detach(child);
#endif XKMACHKERNEL

  xTrace1(processcreation, TR_MAJOR_EVENTS, "Created thread_id: %d", child);
  xTrace1(processswitch, TR_GROSS_EVENTS, "Created thread_id: %d", child);

  return(TRUE);
}

/*
 *
 * CreateProcess(function_ptr, priority, argcount, arg1, arg2, ...)
 *
 */
bool
CreateProcess(va_alist)
     va_dcl
{
  va_list ap;
  Pfi r;
  short prio;
  int nargs, i;
  int *arg_vec;
  int *argp;
  int expendability = 0;

  xTrace0(processcreation, TR_EVENTS, "Enter CreateProcess");

  va_start(ap);
  r = va_arg(ap, Pfi);
  
/* xkernel and Mach threads have the same priorities */
  prio = va_arg(ap, short);
  if (prio > STD_PRIO) expendability++;
  nargs = va_arg(ap, int);

  arg_vec = (int *)xMalloc((MAXARGS+2)*sizeof(int));
  argp = arg_vec;
  *argp++ = (int)r;
  *argp++ = nargs;
  for (i=0; i<nargs; i++)   *argp++ = va_arg(ap, int);

  xTrace2(processcreation, TR_MAJOR_EVENTS, "Mach CreateProcess: r %x nargs %d",
	 r,nargs);
  return(xkernel_limit_active_threads(expendability, arg_vec, prio));
}

/*
 *  CreateKernelProcess
 *
 *
 */
void
CreateKernelProcess(r, prio, arg1, arg2, arg3)
{
  CreateProcess(r, prio, 3, arg1, arg2, arg3);
}


/*
 *
 * Yield
 *
 */
void
Yield()
{
#ifdef XKMACHKERNEL

  xTrace0(processswitch, TR_FULL_TRACE, "Enter event function Yield");
  MASTER_UNLOCK;
  swtch();
  MASTER_LOCK;

#else /* XKMACHKERNEL */
  Process *active = Active;

  xTrace0(processswitch, TR_FULL_TRACE, "Enter event function Yield");
  MASTER_UNLOCK;
  cthread_yield();
  MASTER_LOCK;
  Active = active;
#endif XKMACHKERNEL
}

/*
 *
 * enter_master_sledgehammer_monitor
 *
 * Concurrency control is enforced in the xkernel by means of a monitor,
 *  in which all execution takes place. Here it is!
 * 
 * A new thread begins here.
 * Inside the kernel:
 *	Dequeue a newThreadInfo block and do what it says
 * Outside the kernel:
 *	We are passed the argBlock.
 *
 */
void
enter_master_sledgehammer_monitor
#ifdef XKMACHKERNEL
  ()
#else
  (args)
  int *args;
#endif XKMACHKERNEL
{
  Pfi user_fn;
  int nargs;
  Process *pd;
#ifdef XKMACHKERNEL
  argBlock	*info;
  int		*args;
#endif XKMACHKERNEL

  xTrace0(processcreation, TR_MAJOR_EVENTS, "enter_master_monitor: entering\n");
  MASTER_LOCK;

  xTrace0(processcreation, TR_MAJOR_EVENTS, "enter_master_monitor: locked\n");

  if ((pd = (Process *)xMalloc(sizeof(Process))) == NULL) 
    Kabort("master_sledgehammer_monitor1: no memory");

  pd->link = NULL;  pd->thread = XTHREAD_SELF();


#ifdef XKMACHKERNEL
  simple_lock( &xkInfo_lock );
  info = (argBlock *)dequeue_head(&xkInfoQ);
  simple_unlock( &xkInfo_lock );
  args = info->args;
  xTrace1(processcreation, TR_EVENTS, "processcreate: arg block at %x", args);
  xFree((char *)info);
#endif XKMACHKERNEL
  user_fn = (Pfi)*(args++);
  nargs = *(args++);

  if (nargs > MAXARGS) 
    Kabort("master_sledgehammer_monitor: cannot handle more than 6 args.");
  {
    int argv[MAXARGS];
    int i;
    for (i=0;i<nargs;i++)
      argv[i] = *(args++);
    xFree((char *)args);
#ifndef XKMACHKERNEL
    Active = pd;
#endif ! XKMACHKERNEL
    xTrace1(processcreation,TR_EVENTS,"enter_master_monitor: starting thread %d",
	    pd->thread);
    switch(nargs) {
    case 0: (user_fn)();
            break;
    case 1: (user_fn)(argv[0]);
            break;
    case 2: (user_fn)(argv[0],argv[1]);
            break;
    case 3: (user_fn)(argv[0],argv[1],argv[2]);
            break;
    case 4: (user_fn)(argv[0],argv[1],argv[2],argv[3]);
            break;
    case 5: (user_fn)(argv[0],argv[1],argv[2],argv[3],argv[4]);
            break;
    case 6: (user_fn)(argv[0],argv[1],argv[2],argv[3],argv[4],argv[5]);
            break;
    }
#ifndef XKMACHKERNEL
    Active = NULL;
#endif ! XKMACHKERNEL
  }

  xTrace1(processcreation, TR_MAJOR_EVENTS, "master_monitor: thread %d exiting", pd->thread);
  xFree((char *)pd);
  current_xk_threads--;
  MASTER_UNLOCK;
  xTrace1(processcreation, TR_MAJOR_EVENTS, "master_monitor: thread %d unlocks",
	   XTHREAD_SELF() );

#ifdef XKMACHKERNEL
  thread_terminate( current_thread() );
  thread_halt_self();
  panic( "The zombie enter_master_sledgehammer_monitor strolls" );
#endif XKMACHKERNEL
}


int delay_state;
#ifndef XKMACHKERNEL
/* where is this used ? */
struct timeval delay_time;
#endif ! XKMACHKERNEL

/*
 *
 * wake_sem
 *
 */
void
wake_sem(sp)
    Semaphore	*sp;
{
    xTrace0(processswitch, TR_FULL_TRACE, "Enter event function semSignal");
    semSignal(sp);
}

/*
 *
 * Delay
 *
 */
void
Delay(n) /* n in ms */
     int n;
{
  Semaphore s;

  xTrace0(processswitch, TR_FULL_TRACE, "Enter event function Delay");
  semInit(&s, 0);
  evDetach( evSchedule(wake_sem, (VOID *)&s , n*1000) ); /* 3rd arg is in us */
  semWait(&s);
}


/*
 *
 * semInit
 *
 */
void
semInit(s, n)
	Semaphore		*s;
	unsigned int		n;
{
        xTrace0(processswitch, TR_FULL_TRACE, "Enter event function semInit");
#ifdef XKMACHKERNEL
	Q_INIT( &s->waitQueue );
#else
	mutex_init(&s->lock);
	condition_init(&s->cond);
	s->sleepers = 0;
#endif XKMACHKERNEL
	s->count = n;
}

/*
 *
 * readP
 *
 */
void
realP(s)
	Semaphore		*s;
{
#ifdef XKMACHKERNEL
	WaitingQueue	*w;
#endif

	xTrace2(processswitch, TR_MAJOR_EVENTS, "P on %#x by 0x%x", s, XTHREAD_SELF());
	if (s->count < 0) {
		xTrace2(processswitch, TR_GROSS_EVENTS, "Blocking P on %#x by 0x%x",
							s, XTHREAD_SELF());
#ifndef	XKMACHKERNEL
		s->sleepers--;
		mutex_lock(&s->lock);	
		while (s->count <= s->sleepers) {
		  MASTER_UNLOCK;
		  condition_wait(&s->cond,&s->lock);
		  MASTER_LOCK;
		}
		s->sleepers++;
		mutex_unlock(&s->lock);
#else

		/*
		 * I own the master lock so no other xk process
		 * can change these structures while I am (running?)
		 *
		 * Add my thread_t to the list of waiters
		 * and note how many are waiting
		 */

		w = (WaitingQueue *)xMalloc( sizeof(WaitingQueue));
		assert( w != NULL );
		w->thread = current_thread();
		Q_INSERTLAST( &s->waitQueue, w );

		/* Release the master lock so that others can run */
		MASTER_UNLOCK;
		/*
		 * Suspend my thread waiting for realV to wake me up
		 * NOTE: suspend/resume uses a counter, there is
		 *  no deadlock window here.
		 */
		xTrace2(processswitch, TR_GROSS_EVENTS,
			"Suspending P on %#x by 0x%x",
			s, current_thread());

		thread_suspend( w->thread );
		thread_block((Pfv) 0);
		xTrace2(processswitch, TR_GROSS_EVENTS,
			"Resuming P on %#x by 0x%x", s, current_thread());
		/*
		 * Resumed; get the master lock back so that I can do work
		 */
		MASTER_LOCK;
#endif ! XKMACHKERNEL
		xTrace2(processswitch, TR_MAJOR_EVENTS,
			"Waking-up from P on %#x by 0x%x", s, XTHREAD_SELF());
	      }
      }

/*
 *
 * realV
 *
 */
void
realV(s)
	Semaphore		*s;
{
#ifdef XKMACHKERNEL
	WaitingQueue	*w;
#endif

	xTrace2(processswitch, TR_MAJOR_EVENTS, "V on %#x by 0x%x", s, XTHREAD_SELF());
#ifndef XKMACHKERNEL
	mutex_lock(&s->lock);
	if (s->count <= 0) {
		xTrace2(processswitch, TR_GROSS_EVENTS, "Unblocking V on %#x by 0x%x", 
							s, XTHREAD_SELF());
		condition_signal(&s->cond);
	}
	mutex_unlock(&s->lock);
#else

	/*
	 * I own the master lock so no other xk process
	 * can change these structures while I am
	 */

	/*
	 * Are there any threads waiting on this semaphore?
	 */
	if( s->count <= 0 ) {

		/*
		 * Pull the first waiter from the queue
		 */
		
		Q_REMOVEFIRST( &s->waitQueue, w );

		/*
		 * Resume that thread
		 * It will go right to sleep waiting on the master lock
		 */

		xTrace2(processswitch, TR_GROSS_EVENTS, "Resuming via V: %#x thread: 0x%x",
				  s, w->thread );

		thread_resume( w->thread );
		xFree( w );
	}
#endif ! XKMACHKERNEL
}

#ifndef XKMACHKERNEL
/*
 *
 * VAll
 *
 */
void
VAll(s)
	Semaphore		*s;
{
	xTrace2(processswitch, TR_MAJOR_EVENTS, "VAll on %#x by 0x%x", s, XTHREAD_SELF());
	s->count = 0;
	MASTER_UNLOCK;
	mutex_lock(&s->lock);
	mutex_unlock(&s->lock);
	MASTER_LOCK;
	condition_broadcast(&s->cond);
}
#endif ! XKMACHKERNEL
