/* we configure the number and hosts of the participants by setting the
 * array part_index, e.g. int part_index[] = {GARAMOND, KADMOS, FRANKLIN};
 * would have part[0] on gar, part[1] on kad, and part[2] on fra.  This
 * array needs to be lower (right before user()) because it depends on
 * a bunch of stuff.  See the declaration of it and the corresponding
 * comments below.  Also, see the options that getopt parses - we can
 * set the number of trips/test, the number of tests, and a debug
 * value that the user code uses and sets psyncs tracepsyncp with
 * ( caution - this sets the trace value for all psync users on a host,
 * not just this test).
 */

#include "upi.h"
#include "ip.h"
#include "udp.h"
#include "process.h"
#include "psync.h"


#define TRIPS_PER_TEST 2001
#define NUM_TESTS	20

#define NO  0
#define YES 1

/* these may be reset by command line options if I do getopt parsing */
;
unsigned        trips_per_test = TRIPS_PER_TEST;
unsigned        trip_num = TRIPS_PER_TEST;
unsigned        num_tests = NUM_TESTS;
unsigned        test_num = 1;	/* our current test */

#define LEGAL_OPTIONS "t:n:d:"
#define USAGE_STRING \
    "usage: bxpsync [-t trips_per_test] [-n num_tests] [-d debug_val]\n"

typedef struct {
    unsigned short  high, mid, low;
}               Enet10Address;

/* from zep/etc/ethers: or running /usr/gmt/bin/sunlist */
#define	FRANKLIN_ENET_ADDRESS 	{0x0800, 0x2001, 0xD340}
#define GARAMOND_ENET_ADDRESS	{0x0800, 0x2001, 0x90DA}
#define LUBALIN_ENET_ADDRESS    {0x0800, 0x2001, 0x51D6}
#define KADMOS_ENET_ADDRESS   	{0x0800, 0x2001, 0x5154}
#define BENGUIAT_ENET_ADDRESS   {0x0800, 0x2001, 0x51E6}
#define BEMBO_ENET_ADDRESS      {0x0800, 0x2001, 0xD982}
#define FUTURA_ENET_ADDRESS     {0x0800, 0x2001, 0x5324}
#define JENSON_ENET_ADDRESS     {0x0800, 0x2001, 0x90DA}
#define YUCCA_ENET_ADDRESS	{0x0800, 0x2001, 0x5173}
#define ORATOR_ENET_ADDRESS 	{0x0800, 0x2001, 0x4776}
#define CORONA_ENET_ADDRESS	{0x0800, 0x2001, 0x5040}
#define AURIGA_ENET_ADDRESS	{0x0800, 0x2001, 0x5040}
#define PALOVERDE_ENET_ADDRESS	{0x0800, 0x2001, 0x51d6}
#define GRANJON_ENET_ADDRESS	{0x0800, 0x2001, 0x504a}
#define GACHAM_ENET_ADDRESS	{0x0800, 0x2001, 0x5153}


/* from /etc/hosts: or running /usr/gmt/bin/sunlist */
#define FRANKLIN_IP_ADDRESS	{192, 12, 69, 36}
#define GARAMOND_IP_ADDRESS	{192, 12, 69, 19}
#define LUBALIN_IP_ADDRESS      {192, 12, 69, 45}
#define KADMOS_IP_ADDRESS       {192, 12, 69, 49}
#define BENGUIAT_IP_ADDRESS     {192, 12, 69,  8}
#define BEMBO_IP_ADDRESS        {192, 12, 69, 37}
#define FUTURA_IP_ADDRESS       {192, 12, 69, 25}
#define JENSON_IP_ADDRESS       {192, 12, 69, 32}
#define YUCCA_IP_ADDRESS	{192, 12, 69, 15}
#define ORATOR_IP_ADDRESS	{192, 12, 69, 42}
#define CORONA_IP_ADDRESS	{192, 12, 69, 47}
#define AURIGA_IP_ADDRESS	{192, 12, 69, 11}
#define PALOVERDE_IP_ADDRESS	{192, 12, 69, 16}
#define GRANJON_IP_ADDRESS	{192, 12, 69, 27}
#define GACHAM_IP_ADDRESS	{192, 12, 69, 24}

#define FRANKLIN_PSYNC_ADDRESS	{2001, FRANKLIN_IP_ADDRESS}
#define GARAMOND_PSYNC_ADDRESS	{2001, GARAMOND_IP_ADDRESS}
#define LUBALIN_PSYNC_ADDRESS   {2001, LUBALIN_IP_ADDRESS}
#define KADMOS_PSYNC_ADDRESS    {2001, KADMOS_IP_ADDRESS}
#define BENGUIAT_PSYNC_ADDRESS  {2001, BENGUIAT_IP_ADDRESS}
#define BEMBO_PSYNC_ADDRESS     {2001, BEMBO_IP_ADDRESS}
#define FUTURA_PSYNC_ADDRESS    {2001, FUTURA_IP_ADDRESS}
#define JENSON_PSYNC_ADDRESS    {2001, JENSON_IP_ADDRESS}
#define YUCCA_PSYNC_ADDRESS	{2001, YUCCA_IP_ADDRESS}
#define ORATOR_PSYNC_ADDRESS	{2001, ORATOR_IP_ADDRESS}
#define CORONA_PSYNC_ADDRESS	{2001, CORONA_IP_ADDRESS}
#define AURIGA_PSYNC_ADDRESS	{2001, AURIGA_IP_ADDRESS}
#define PALOVERDE_PSYNC_ADDRESS	{2001, PALOVERDE_IP_ADDRESS}
#define GRANJON_PSYNC_ADDRESS	{2001, GRANJON_IP_ADDRESS}
#define GACHAM_PSYNC_ADDRESS	{2001, GACHAM_IP_ADDRESS}

struct psync_host {
    char           *name;
    Enet10Address   enet_address;
    PSYNCaddr       psync_address;
};

static struct psync_host host_tab[] =
{
 {"garamond", GARAMOND_ENET_ADDRESS, GARAMOND_PSYNC_ADDRESS},
 {"franklin", FRANKLIN_ENET_ADDRESS, FRANKLIN_PSYNC_ADDRESS},
 {"benguiat", BENGUIAT_ENET_ADDRESS, BENGUIAT_PSYNC_ADDRESS},
 {"auriga", AURIGA_ENET_ADDRESS, AURIGA_PSYNC_ADDRESS},
 {"kadmos", KADMOS_ENET_ADDRESS, KADMOS_PSYNC_ADDRESS},
 {"lubalin", LUBALIN_ENET_ADDRESS, LUBALIN_PSYNC_ADDRESS},
 {"bembo", BEMBO_ENET_ADDRESS, BEMBO_PSYNC_ADDRESS},
 {"futura", FUTURA_ENET_ADDRESS, FUTURA_PSYNC_ADDRESS},
 {"jenson", JENSON_ENET_ADDRESS, JENSON_PSYNC_ADDRESS},
 {"yucca",  YUCCA_ENET_ADDRESS,  YUCCA_PSYNC_ADDRESS},
 {"orator", ORATOR_ENET_ADDRESS, ORATOR_PSYNC_ADDRESS},
 {"corona", CORONA_ENET_ADDRESS, CORONA_PSYNC_ADDRESS},
 {"gacham", GACHAM_ENET_ADDRESS, GACHAM_PSYNC_ADDRESS},
 {"paloverde", PALOVERDE_ENET_ADDRESS, PALOVERDE_PSYNC_ADDRESS},
 {"granjon", GRANJON_ENET_ADDRESS, GRANJON_PSYNC_ADDRESS}
};

#define NUM_PSYNC_HOSTS ( (sizeof(psync_host)) / (sizeof(struct host_tab)))

/* these correspond to indices in host_tab */
#define GARAMOND 0
#define FRANKLIN 1
#define BENGUIAT 2
#define AURIGA   3
#define KADMOS   4
#define LUBALIN  5
#define BEMBO    6
#define FUTURA   7
#define JENSON   8
#define YUCCA    9
#define ORATOR   10
#define CORONA   11
#define GACHAM   12
#define GRANJON  13
#define PALOVERDE 14

#define SAME_ETHERNET_ADDRESS(A,B) \
		((A).high==(B).high && (A).mid==(B).mid && (A).low==(B).low)

#define MESSAGE "xxxxxxxxyyyy3456789012345678901234567890123456"
#define MESSAGE_LENGTH 46	/* length of MESSAGE */

typedef struct {
    unsigned        sec;
    int             usec;
}               time;


time            starttime;
extern int      null();
int debug=0;  /* might  be used to set tracepsyncp, but also locally */

/* part_index is how you configure your test to run - just put the indeces
 * from host_tab into this, and you will get that many participants
 * running on those hosts.  participant 0 is the one who does the active
 * open, so host_tab[ part_index[0] ].name should be booted last.
 * Note that the number in this array must be <= MAX_PART, and there
 * must be MAX_PART demuxN funcs at the bottom (and demux[] must 
 * initialize in user to all N of them).
 */

int part_index[] = {GARAMOND, FRANKLIN};

#define NUM_PARTS ( (sizeof(part_index)) / (sizeof(int))  )


XObj           mysession;
XObj           part_protl[MAX_PART];
extern int 
demux0(), demux1(), demux2(), demux3(), demux4(), demux5(),
demux6(), demux7();	/* must be MAX_PARTS of these decls. */
int             (*demux[MAX_PART]) ();	/* will be pointed to above in
					 * user() */

#define REPLY_TO_MSG(m,part_num) \
	( ( *((unsigned *) (m + 4)) % NUM_PARTS) == part_num  ? 1  : 0 )

/* for getopt */
extern char    *optarg;
extern int      optind;

void 
user(argc, argv)
    int             argc;
    char           *argv[];

{
    int             p, r, n, x, i, z, y, c, rem,k, l;
    short           mytype = 0x98;
    int             psync_user_open_done();
    Part            whom[MAX_PART + 3];
    XObj           PSYNC;
    Enet10Address   myeaddr, my_ethernet_address();


    PSYNC = (XObj) xgetprotlbyname("psync");
    while ((c = getopt(argc, argv, LEGAL_OPTIONS)) != -1)
	switch (c) {
	case 't':
	    trips_per_test = atoi(optarg);
	    trip_num = trips_per_test;
	    break;
	case 'n':
	    num_tests = atoi(optarg);
	    break;
	case 'd':
	    debug = atoi(optarg);	    
	    xcontrolprotl(PSYNC, SETDEBUG, &debug, sizeof(int) );
	    break;
	case '?':
	case 'h':
	default:
	    printf("%s", USAGE_STRING);
	    return;
	}

    printf("\n\n\n");
    myeaddr = my_ethernet_address("My ethernet address = " /* print it */ );
    printf("\n\n\n");

    /* the number of tests must be divisible by NUM_PARTS */
    rem = trips_per_test % NUM_PARTS;
    if (rem) {
	trips_per_test += (NUM_PARTS - rem);	/* make divisible by
						 * NUM_PARTS */
	trip_num = trips_per_test;
	printf("really doing %d trips", trips_per_test);
	printf(" - must be divisible by NUM_PART(%d)\n", NUM_PARTS);
    }
    if (NUM_PARTS > MAX_PART) {
	printf("*** PANIC - NUM_PARTS(%d) > MAX_PART(%d) - fix this\n\n",
	       NUM_PARTS, MAX_PART);
	return;
    }
    /* there must be at least MAX_PART of these */
    demux[0] = demux0;
    demux[1] = demux1;
    demux[2] = demux2;
    demux[3] = demux3;
    demux[4] = demux4;
    demux[5] = demux5;
    demux[6] = demux6;
    demux[7] = demux7;

    /*
     * create a (fake) user protocol for each participant.  participant 0
     * is the one who will call the active open, so he should be called
     * last and his host booted last. 
     */

    for (i = NUM_PARTS - 1; i >= 0; i--) {
	/* only fire up participants that belong on this host */
	if (SAME_ETHERNET_ADDRESS(myeaddr,
				  host_tab[part_index[i]].enet_address)) {
	    /* create a protocol for participant i on this host */
	    printf("firing up participant %d ... ", i);
	    fill_part_list(whom, i);	/* he knows part 0 does active
					 * open */
	    part_protl[i] = (XObj)xcreateprotl(demux[i], psync_user_open_done,
					 (Pfs) null, (Pfs) null);
	    if (i > 0) {
		xopenenable(part_protl[i], PSYNC, whom);
            }
	    else {
		unsigned            buf[50];

		strcpy(buf, MESSAGE);

		mysession = (XObj)xopen(part_protl[0], PSYNC, whom);
		if ((int)mysession <= 0) {
		    printf("Not sending, other host(s) down. ");
		    printf("Boot me LAST!!!!\n\n");
		    return;

		}
		printf("\nDisconnect from ethernet now if you need to\n");
		for (x = z = 0; x < 1500000; x++)
		   y = x - (z + 2);

		/*
		 * The first word of the message is the sequence number.
		 * Currently, each participant replies to the message if
		 * the sequence number is equal to it modula NUM_PARTS.
		 * This reply policy is fixed in REPLY_TO_MSG and thus can
		 * be easily modified. 
		 */

                *buf = 1;
                trip_num--;       
        
		printf("Sending %5d (%2d/%2d)... ",
		       trips_per_test, test_num, num_tests);
		xgettime(&starttime);
		xpush(mysession, (char *)buf, MESSAGE_LENGTH);
	    }	/* if i>0 */

	    printf(" done firing up participant %d\n", i);
	}	/* if participant i is on this host */
    }  /* for all participants */

    printf("DONE firing up all participants that belong on this host\n\n");
}

fill_part_list(part, part_num)
    Part            part[];
int             part_num;	/* 0 fires up active open */

{
    int             i;

    part[0].address = (char *) 0;
    part[0].length = 0;

    if (part_num > 0) {

	part[1].address = (char *) &host_tab[part_num].psync_address;
	part[1].length = sizeof(PSYNCaddr);
	part[2].address = part[3].address = (char *) 0;
	part[2].length = part[3].length = 0;

    } else {
	/* part_num == 0 */
	part[1].address = (char *) &host_tab[0].psync_address;
	part[1].length = sizeof(PSYNCaddr);
	for (i = 0; i < NUM_PARTS; i++) {
	    part[i + 1].address = 
		(char *) &host_tab[ part_index[i] ].psync_address;
	    part[i + 1].length = sizeof(PSYNCaddr);
	}
	part[i + 1].address = (char *) 0;
	part[i + 1].length = 0;

    }

}

null()
{
    printf("***PANIC: null process executed!!!\n");
}

printtime(m, t)
    char           *m;
    time           *t;
{
    printf("%s%d:%6d\n", m, t->sec, t->usec);
}



demux0(s, m, len)
    register XObj  s;
    int             len;
    char           *m;
{

#   define MY_PART_NUM 0

    time            now, total;
    static unsigned count = 0;
    register unsigned *num_ptr = (unsigned *) (m + 4);
    char            buf[100];


    if (trip_num > 1) {
	trip_num--;

	if (REPLY_TO_MSG(m, MY_PART_NUM)) {
	    /* increment the sequence number and reply */
	    (*num_ptr)++;
            trip_num--;   /* cause I don't receive my own message */
	    xpush(s, (m + 4), MESSAGE_LENGTH);
	}
    } else {
	xgettime(&now);
	count += trips_per_test;
	subtime(&starttime, &now, &total);


	printf("%3d.%6d  (%5d total trips)\n",
	       total.sec, total.usec, count);
	if (++test_num <= num_tests) {
	    /* start another test */
	    printf("Sending %5d (%2d/%2d)... ",
		   trips_per_test, test_num, num_tests);
	    trip_num = trips_per_test - 1;
	    strcpy(buf, MESSAGE);
	    num_ptr = (unsigned *) buf;
	    *num_ptr = 1;
	    xgettime(&starttime);
	    xpush(s, buf, MESSAGE_LENGTH);
	} else
	    printf("All done with the %d tests!!\n\n", num_tests);

    }

#undef MY_PART_NUM
}



demux1(s, m, len)
    register XObj  s;
    int             len;
    char           *m;
{
    register unsigned *num_ptr = (unsigned *) (m + 4);

#   define MY_PART_NUM  1

    if (REPLY_TO_MSG(m, MY_PART_NUM)) {
	/* increment the sequence number and reply */
	(*num_ptr)++;
	xpush(s, (m + 4), MESSAGE_LENGTH);
    }
#   undef MY_PART_NUM
}


demux2(s, m, len)
    register XObj  s;
    int             len;
    char           *m;
{
    register unsigned *num_ptr = (unsigned *) (m + 4);

#   define MY_PART_NUM  2


    if (REPLY_TO_MSG(m, MY_PART_NUM)){
	/* increment the sequence number and reply */
	(*num_ptr)++;
	xpush(s, m + 4, MESSAGE_LENGTH);
    }
#   undef MY_PART_NUM
}


demux3(s, m, len)
    register XObj  s;
    int             len;
    char           *m;
{
    register unsigned *num_ptr = (unsigned *) (m + 4);

#   define MY_PART_NUM  3


    if (REPLY_TO_MSG(m, MY_PART_NUM)) {
	/* increment the sequence number and reply */
	(*num_ptr)++;
	xpush(s, (m + 4), MESSAGE_LENGTH);
    }
#   undef MY_PART_NUM
}


demux4(s, m, len)
    register XObj  s;
    int             len;
    char           *m;
{
    register unsigned *num_ptr = (unsigned *) m;

#   define MY_PART_NUM  4


    if (REPLY_TO_MSG(m, MY_PART_NUM)) {
	/* increment the sequence number and reply */
	(*num_ptr)++;
	xpush(s, m, MESSAGE_LENGTH);
    }
#   undef MY_PART_NUM
}


demux5(s, m, len)
    register XObj  s;
    int             len;
    char           *m;
{
    register unsigned *num_ptr = (unsigned *) m;

#   define MY_PART_NUM  5


    if (REPLY_TO_MSG(m, MY_PART_NUM)) {
	/* increment the sequence number and reply */
	(*num_ptr)++;
	xpush(s, m, MESSAGE_LENGTH);
    }
#   undef MY_PART_NUM
}


demux6(s, m, len)
    register XObj  s;
    int             len;
    char           *m;
{
    register unsigned *num_ptr = (unsigned *) m;

#   define MY_PART_NUM  6


    if (REPLY_TO_MSG(m, MY_PART_NUM)) {
	/* increment the sequence number and reply */
	(*num_ptr)++;
	xpush(s, m, MESSAGE_LENGTH);
    }
#   undef MY_PART_NUM
}


demux7(s, m, len)
    register XObj  s;
    int             len;
    char           *m;
{
    register unsigned *num_ptr = (unsigned *) m;

#   define MY_PART_NUM  7


    if (REPLY_TO_MSG(m, MY_PART_NUM)) {
	/* increment the sequence number and reply */
	(*num_ptr)++;
	xpush(s, m, MESSAGE_LENGTH);
    }
#   undef MY_PART_NUM
}



psync_user_open_done(s, p)
    int             s;
    Part           *p;
{
    mysession = (XObj)s;
    printf("USER: psync open complete s %x\n", s);
}


addtime(t1, t2, t3)
    time           *t1, *t2, *t3;
{
    t3->sec = t2->sec + t1->sec;
    t3->usec = t2->usec + t1->usec;
    if (t3->usec >= 1000000) {
	t3->usec -= 1000000;
	t3->sec += 1;
    }
}


subtime(t1, t2, t3)
    time           *t1, *t2, *t3;
{
    t3->sec = t2->sec - t1->sec;
    t3->usec = t2->usec - t1->usec;
    if (t3->usec < 0) {
	t3->usec += 1000000;
	t3->sec -= 1;
    }
}

multime(t1, n, t3)
    time           *t1, *t3;
    unsigned        n;	/* assume n >= 0 */
{

    t3->sec = t1->sec * n;
    t3->usec = t1->usec * n;

    while (t3->usec >= 1000000) {
	t3->usec -= 1000000;
	t3->sec++;
    }
}

divtime(t1, n, t3)
    time           *t1, *t3;
    unsigned        n;	/* assume n >= 0 */
{

    unsigned        rem;

    t3->sec = t1->sec / n;
    rem = t1->sec % n;

    t3->usec = (t1->usec + rem * 1000000) / n;

}

tmp_enet_close()
{
    printf("***PANIC: tmp_enet_close executed!!!\n\n");
}

tmp_enet_open()
{
    printf("***PANIC: tmp_enet_open executed!!!\n\n");
}

tmp_enet_demux()
{
    printf("***PANIC: tmp_enet_demux executed!!!\n\n");
}

/* my_ethernet_address does the obvious */
Enet10Address 
my_ethernet_address(print_string)
    char           *print_string;	/* NULL if we aren't supposed to
					 * print anything */
{
    XObj            ETH, p;
    Enet10Address   myeaddr;

    p = (XObj)xcreateprotl((Pfi) tmp_enet_demux, (Pfi) tmp_enet_open,
		     (Pfs) tmp_enet_close);
    ETH = (XObj)xgetprotlbyname("eth");
    if (xcontrolprotl(ETH, GETMYADDR, &myeaddr, 6) != 6) {
	printf("Cannot get my own enet addr\n");
	exit(1);	/* to somewhere */
    }
    if (print_string != NULL)
	printf("%s%x.%x.%x\n", print_string,
	       myeaddr.high, myeaddr.mid, myeaddr.low);
    xdestroyprotl(p);
    return (myeaddr);
}
