/* 
 * intelEthCmd.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

/*
 * Functions revolving around the Intel ethernet controller's 
 * Command Unit (cu), except transmission-specific stuff is in 
 * intelEthXmit.c
 *
 */

#include "xkernel.h"
#include "eth.h"
#include "intelEthType.h"
#include "intelEthRcv.h"
#include "intelEthCmd.h"
#include "intelEth.h"

Semaphore FreeCmdCntSem;
cmd_t *NextFreeCmdPtr;

boolean CuKnownNotActive;


/*
 * initCmd
 *
 * Initialize the data structures for issuing action commands.
 */

initCmd()
{
  unsigned xbdIx;		/* xmit buf desc index */
  unsigned cmdIx, nextCmdIx;	/* indices of command blocks */
  
  
  TRACE0(ie, 5, "initCmd");
  
  /*
   * Initialize the list of free command blocks. The command blocks are
   * linked in a ring, and they will always be allocated, posted, used, and
   * freed in ring order. Each command block has a linked list of xmit buffer
   * descriptors permanently associated with it, for those occasions when it
   * is instantiated as an xmit command.
   */
  
  /* make the xmit buf desc array into a bunch of linked lists */
  for (xbdIx = 0; xbdIx < NUM_XBDS; xbdIx++) {
    if ((xbdIx % XBDS_PER_CMD) == (XBDS_PER_CMD - 1)) {
      /* terminate a linked list */
      XmitBufDescArray[xbdIx].lastXbdOfCmd = TRUE;
      XmitBufDescArray[xbdIx].nextXbdPtr = NULL;
      XmitBufDescArray[xbdIx].nextXbdOffset = NULL_IE_OFFSET;
    } else {
      /* link to successor */
      XmitBufDescArray[xbdIx].lastXbdOfCmd = FALSE;
      XmitBufDescArray[xbdIx].nextXbdPtr = &XmitBufDescArray[xbdIx + 1];
      XmitBufDescArray[xbdIx].nextXbdOffset =
	IE_OFFSET_FROM_AD(&XmitBufDescArray[xbdIx + 1]);
    }
  }
  
  /* make the command block array into a ring, and associate xbd lists */
  for (cmdIx = 0; cmdIx < 2 * NUM_CMD_BLKS; cmdIx++) {
    
    nextCmdIx = (cmdIx + 1) % (2 * NUM_CMD_BLKS);	/* wrap around */
    CmdBlkArray[cmdIx].extra.nextCmdPtr = &CmdBlkArray[nextCmdIx];
    CmdBlkArray[cmdIx].prefix.nextCmdOffset =
      IE_OFFSET_FROM_AD(&CmdBlkArray[nextCmdIx]);
    CmdBlkArray[nextCmdIx].extra.prevCmdPtr = &CmdBlkArray[cmdIx];
    if (cmdIx % 2 == 0) {
      CmdBlkArray[cmdIx].extra.xbdHeadPtr =
	&XmitBufDescArray[cmdIx / 2 * XBDS_PER_CMD];
      CmdBlkArray[cmdIx].extra.firstChunk = malloc(MIN_FIRST_CHUNK);
    } else {
      CmdBlkArray[cmdIx].prefix.cmdKind = NOOP_CMD;
    }
  }
  
  NextFreeCmdPtr = &CmdBlkArray[0];	/* head of free cmd list */
  /* all cmds are free (except always keep one unallocated to avoid danger?) */
  InitSemaphore(&FreeCmdCntSem, NUM_CMD_BLKS - 1);
  SCB.cmdHeadOffset = NULL_IE_OFFSET;
  CuKnownNotActive = TRUE;
  
}				/* end initCmd */


/*
 * postCmd
 */

postCmd(cmdPtr)
     cmd_t *cmdPtr;
{
  int origPrioLevel;
  
  
  TRACE0(ie, 5, "postCmd");
  
  /* add the new command to the "posted" list (the link is already there!) */
  cmdPtr->extra.nextCmdPtr->prefix.lastPostedCmd = TRUE;
  cmdPtr->extra.prevCmdPtr->prefix.lastPostedCmd = FALSE;
  
  /* start the Command Unit if known to have left active state */
  /*
   * mask interrupts to protect critical section;
   * handleCuLeavingActiveState() plays with same stuff.
   */
  origPrioLevel = spl7();
  if (CuKnownNotActive) {
#ifndef NDEBUG
    nCUStarts++;
    IFTRACE(ie, 2) putchar('a');
#endif
    TRACE0(ie, 5, "postCmd: putting Cmd Unit in active state");
    SCB.cmdHeadOffset = IE_OFFSET_FROM_AD(cmdPtr);
    SCB.cmdUnitCtlCmd = CU_START;
    CuKnownNotActive = FALSE;
    ALERT_IE_DEV();
  } else {
#ifdef ETH_OPT_TRACE
    putchar('A');
#endif
#ifndef NDEBUG
    nCUContinues++;
    IFTRACE(ie, 2) putchar('A');
#endif
    TRACE0(ie, 5, "postCmd: Cmd Unit may be in active state");
    /* if not active, will catch on and start CU when handle intr */
  }
  splx(origPrioLevel);
  
}				/* end postCmd */


/*
 * informIeDevLocalEthAd
 */

informIeDevLocalEthAd(localEthAd)
     ethAd_t localEthAd;
{
  cmd_t *cmdPtr;
  
  TRACE3(ie, 5, "informIeDevLocalEthAd: localEthAd = %x.%x.%x",
	 localEthAd.high, localEthAd.mid, localEthAd.low);
  
  ALLOC_CMD(cmdPtr);
  
  cmdPtr->prefix.cmdKind = INDIV_AD_SETUP_CMD;
  cmdPtr->suffix.indivAd.localEthAd = localEthAd;
  
  postCmd(cmdPtr);
  
}				/* end informIeDevLocalEthAd */


/*
 * configIeDev
 */

configIeDev(promiscMode)
     int			promiscMode;
{
  cmd_t *cmdPtr;
  
  TRACE0(ie, 5, "configIeDev");
  
  ALLOC_CMD(cmdPtr);
  
  cmdPtr->prefix.cmdKind = CONFIG_CMD;
  cmdPtr->suffix.config.promiscMode = promiscMode;
  
  cmdPtr->suffix.config.bcnt = 12;
  cmdPtr->suffix.config.fifo = 12;
  cmdPtr->suffix.config.svbf = 0;
  cmdPtr->suffix.config.ext_lpb = 0;
  cmdPtr->suffix.config.int_lpb = 0;
  cmdPtr->suffix.config.srdy = 0;
  cmdPtr->suffix.config.pream_len = 2;
  cmdPtr->suffix.config.al_loc = 0;
  cmdPtr->suffix.config.addr_len = 6;
  cmdPtr->suffix.config.bof_met = 0;
  cmdPtr->suffix.config.acr = 0;
  cmdPtr->suffix.config.lin_prio = 0;
  cmdPtr->suffix.config.frame_sp = 96;
  cmdPtr->suffix.config.slot_timel = LO_BYTE_OF_2(512);
  cmdPtr->suffix.config.retry_num = 15;
  cmdPtr->suffix.config.slot_timeh = HI_BYTE_OF_2(512);
  cmdPtr->suffix.config.pad = 0;
  cmdPtr->suffix.config.bt_stf = 0;
  cmdPtr->suffix.config.crc_16 = 0;
  cmdPtr->suffix.config.ncrc = 0;
  cmdPtr->suffix.config.tono_crc = 0;
  cmdPtr->suffix.config.man = 0;
  cmdPtr->suffix.config.bc_dis = 0;
  cmdPtr->suffix.config.cdt_src = 0;
  cmdPtr->suffix.config.cdtf = 0;
  cmdPtr->suffix.config.crs_src = 0;
  cmdPtr->suffix.config.crsf = 3;
  cmdPtr->suffix.config.min_frm = 64;
  
  postCmd(cmdPtr);
  
}				/* end configIeDev */


/*
 * handleCuLeavingActiveState
 */

handleCuLeavingActiveState()
{
  /*
   *static boolean		firstTimeCuLeftActiveState = TRUE;
   */
  cmd_t *doneCmdHeadPtr;
  cmd_t *cmdPtr;
  cmd_t *lastCmdPtr;
  
#ifndef NDEBUG
  nCULeavingActives++;
  IFTRACE(ie, 2) putchar('I');
#endif
  SCB.ackCuLeftActiveState = 1;
  ALERT_IE_DEV();
  
  TRACE0(ie, 5, "handleCuLeavingActiveState");
  
  CuKnownNotActive = TRUE;
  
  /*
   * ignore the very first Cmd-Unit-left-active-state interrupt (happens when
   * start the ie dev?)
   */

  /*
   *	if( firstTimeCuLeftActiveState ){
   *		firstTimeCuLeftActiveState = FALSE;
   *		TRACE0(ie,5,"handleCuLeavingActiveState: ignoring first call");
   *		SCB.ackCuLeftActiveState = 1;
   *		SCB.cmdUnitCtlCmd = CU_NOP;
   *		ALERT_IE_DEV();
   *		return;
   *	}
   */
  
  /*
   * check whether there are any posted commands
   */

  if (SCB.cmdHeadOffset == NULL_IE_OFFSET) {
    /* interrupt corresponding to stuff that was already handled? */
    TRACE0(ie, 1,
	   "handleCu...:Cmd Unit left active state, but no cmd blk list!");
    /*		SCB.cmdUnitCtlCmd = CU_SUSPEND;	/* why not just leave IDLE? */
    return;
  }

  /*
   * There are posted commands, some of which may be done.
   */
  
  /*
   * Restart the Command Unit at the next pending command, if there is one.
   */
  
  doneCmdHeadPtr = (cmd_t *) AD_FROM_IE_OFFSET(SCB.cmdHeadOffset);
  cmdPtr = doneCmdHeadPtr;
  while (cmdPtr->prefix.cmdDone) {
    cmdPtr = cmdPtr->extra.nextCmdPtr->extra.nextCmdPtr;
  }
  
  /*
   * cmdPtr->extra.prevCmdPtr now points to the last command done, and cmdPtr
   * points to the next command to be done (which may or may not have been
   * allocated/instantiated/posted yet).
   */
  if (cmdPtr->extra.prevCmdPtr->prefix.lastPostedCmd) {
    /* last one done was also last one posted, so no cmds are pending */
#ifdef ETH_OPT_TRACE
    putchar('d');
#endif
#ifndef NDEBUG
    nCUNoCommandsPending++;
    IFTRACE(ie, 2) putchar('d');
#endif
    TRACE0(ie, 5, "handleCuLeaving...: no commands pending");
    SCB.cmdHeadOffset = NULL_IE_OFFSET;
    /*		SCB.cmdUnitCtlCmd = CU_SUSPEND;	/* why not just leave IDLE? */
  } else {
    /*
     * cmds were posted too late to be seen by the Cmd Unit, i.e. after the
     * CU had already seen a lastPostedCmd
     */
    /*    putchar('D'); */
#ifndef NDEBUG
    nCUCommandsPending++;
    IFTRACE(ie, 2) putchar('D');
#endif
    TRACE0(ie, 5, "handleCuLeaving...: there are pending commands");
#if 0
    if (cmdPtr->prefix.cmdKind != XMIT_CMD ||
	cmdPtr->prefix.cmdDone ||
	cmdPtr->prefix.lastPostedCmd) putchar('!');
#else
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
#endif
    SCB.cmdHeadOffset = IE_OFFSET_FROM_AD(cmdPtr);
    SCB.cmdUnitCtlCmd = CU_START;
    CuKnownNotActive = FALSE;
    ALERT_IE_DEV();
#ifdef ETH_OPT_TRACE
    putchar('D');
#endif
  }
  
  /*
   * Clean up the done commands.
   */
  
  /*
   * pay the price of following these links twice, to get the parallelism of
   * restarting the CU as soon as possible.
   */
  lastCmdPtr = cmdPtr;
  cmdPtr = doneCmdHeadPtr;
  while (cmdPtr != lastCmdPtr && cmdPtr->prefix.cmdDone) {
    
    if (cmdPtr->prefix.cmdKind == XMIT_CMD) {
      if (!cmdPtr->prefix.cmdSuccessful) {
	printXmitErrs(cmdPtr);
      }
      msg_free(cmdPtr->suffix.xmit.msg);
    }
    cmdPtr->prefix.cmdDone = FALSE;
    cmdPtr->extra.nextCmdPtr->prefix.cmdDone = FALSE;
    cmdPtr = cmdPtr->extra.nextCmdPtr->extra.nextCmdPtr;
    FREE_CMD;
  }
  
}				/* end handleCuLeavingActiveState */
