(* Static semantic objects for the Core - Definition v3 page 17
   This functor also provides the Core objects needed by the
   Modules elaborator *)

(* Static semantic objects - Definition v3 page 17 *)

(*
$File: Common/Environments.sml $
$Date: 1993/02/24 07:58:01 $
$Revision: 1.13 $
$Locker: birkedal $
*)

(*$Environments:
	DEC_GRAMMAR STRID IDENT CON VAR EXCON TYCON TYNAME STATOBJECT_PROP
	PRETTYPRINT SORTED_FINMAP FINMAP TIMESTAMP REPORT FLAGS LIST_HACKS
	CRASH ENVIRONMENTS
*)

functor Environments(structure DecGrammar: DEC_GRAMMAR
		     structure StrId: STRID

		     structure Ident: IDENT
		       sharing type Ident.strid = StrId.strid
			   and type Ident.longid = DecGrammar.longid
			   and type Ident.id = DecGrammar.id

		     structure Con: CON
		       sharing type Con.longid = Ident.longid
			   and type Con.strid  = StrId.strid
			   and type Con.id     = Ident.id

		     structure Var: VAR
		       sharing type Var.longid = Ident.longid

		     structure Excon: EXCON
		       sharing type Excon.id     = Ident.id
			   and type Excon.longid = Ident.longid
			   and type Excon.strid  = StrId.strid

		     structure TyCon: TYCON
		       sharing type TyCon.strid = StrId.strid

		     structure TyName: TYNAME

		     structure StatObject: STATOBJECT_PROP
		       sharing type StatObject.SyntaxTyVar = DecGrammar.tyvar
			   and type StatObject.TyName = TyName.TyName

		     structure PP: PRETTYPRINT
		       sharing type StatObject.StringTree = PP.StringTree

		     structure SortedFinMap: SORTED_FINMAP
		       sharing type SortedFinMap.StringTree = PP.StringTree

		     structure FinMap: FINMAP
		       sharing type FinMap.StringTree = PP.StringTree

		     structure Timestamp: TIMESTAMP

		     structure Report: REPORT
		       sharing type SortedFinMap.Report
				    = FinMap.Report
				    = Report.Report

		     structure Flags: FLAGS
		     structure ListHacks: LIST_HACKS
		     structure Crash: CRASH
		    ) =
  struct
    type id           = Ident.id
     and longid       = Ident.longid
     and valbind      = DecGrammar.valbind
     and Substitution = StatObject.Substitution
     and Type         = StatObject.Type
     and TypeScheme   = StatObject.TypeScheme
     and TyName       = TyName.TyName
     and con          = Con.con
     and longcon      = Con.longcon
     and longvar      = Var.longvar
     and excon        = Excon.excon
     and longexcon    = Excon.longexcon
     and tycon        = TyCon.tycon
     and longtycon    = TyCon.longtycon
     and strid        = StrId.strid
     and pat          = DecGrammar.pat
     and ty           = DecGrammar.ty

   (********
    Structure names
    ********)

    type StrName = Timestamp.stamp
    val freshStrName = Timestamp.new
    val bogus_StrName = freshStrName()

    (********
    Type name sets
    ********)

    type TyNameSet = StatObject.TyNameSet

    (********
    Structure name sets
    ********)

    type StrNameSet = StrName list

    fun singleM m = [m]

    val emptyStrNameSet = nil
    val StrNameSetUnion = ListHacks.union
    val StrNameSetFold = fn f => List.foldL(General.curry f)

    (********
    Name sets
    ********)

    datatype NameSet = NAMESET of {M: StrNameSet, T: TyNameSet}

    val emptyN = NAMESET {M = nil, T = StatObject.emptyTyNameSet}

    fun mkNameSet(M: StrNameSet, T: TyNameSet) = NAMESET {M = M, T = T}

    fun unNameSet(NAMESET {M, T}) = (M, T)

    fun mkNameSet_Str M = NAMESET {M = M, T = StatObject.emptyTyNameSet}

    fun isIn_StrName(m, NAMESET {M, ...}) = List.member m M

    fun isIn_TyName(t, NAMESET {T, ...}) = StatObject.isIn(t, T)

    fun NameSetUnion(NAMESET {M, T}, NAMESET {M = M', T = T'}) =
      NAMESET{M = ListHacks.union(M, M'), T = StatObject.TyNameSetUnion(T, T')}

    fun NameSetIntersect(NAMESET {M, T}, NAMESET {M = M', T = T'}) =
      NAMESET{M = ListHacks.intersect(M, M'), T = StatObject.TyNameSetIntersect(T, T')}

    fun NameSetMinus(NAMESET {M, T}, NAMESET {M = M', T = T'}) =
      NAMESET{M = ListHacks.minus(M, M'), T = StatObject.TyNameSetMinus(T, T')}

    fun T_of_N(NAMESET {T, ...}) = T

    fun T_in_N(T: TyNameSet) = NAMESET {M = nil, T = T}

    fun eqNameSet(NAMESET {M = M, T = T}, NAMESET {M = M', T = T'}) =
      StatObject.eqTyNameSet(T, T') andalso ListHacks.eqSet(M, M') 

    (********
    Explicit TyVars
    ********)

    type TyVarSet = StatObject.TyVar list
     and TyVar    = StatObject.TyVar

    local
      val ++ = ListHacks.union
      infix ++

      fun unguarded_opt f (Some x) = f x
	| unguarded_opt f (None  ) = []

      fun unguarded_atexp(DecGrammar.RECORDatexp(_,exprow_opt)) =
	  unguarded_opt unguarded_exprow exprow_opt
	| unguarded_atexp(DecGrammar.LETatexp(_,_,exp)) =
	  unguarded_exp exp
	| unguarded_atexp(DecGrammar.PARatexp(_,exp)) =
	  unguarded_exp exp
	| unguarded_atexp _ = []

      and unguarded_exprow(DecGrammar.EXPROW(_,_,exp,exprow_opt)) =
	  unguarded_exp exp ++ unguarded_opt unguarded_exprow exprow_opt

      and unguarded_exp(DecGrammar.ATEXPexp(_,atexp)) =
	  unguarded_atexp atexp
	| unguarded_exp(DecGrammar.APPexp(_,exp,atexp)) =
	  unguarded_exp exp ++ unguarded_atexp atexp
	| unguarded_exp(DecGrammar.TYPEDexp(_,exp,ty)) =
	  unguarded_exp exp ++ unguarded_ty ty
	| unguarded_exp(DecGrammar.HANDLEexp(_,exp,match)) =
	  unguarded_exp exp ++ unguarded_match match
	| unguarded_exp(DecGrammar.RAISEexp(_,exp)) =
	  unguarded_exp exp
	| unguarded_exp(DecGrammar.FNexp(_,match)) =
	  unguarded_match match
	| unguarded_exp(DecGrammar.UNRES_INFIXexp _) =
	  Crash.impossible "unguarded_exp(UNRES_INFIX)"

      and unguarded_match(DecGrammar.MATCH(_,mrule,match_opt)) =
	  unguarded_mrule mrule ++
	  unguarded_opt unguarded_match match_opt

      and unguarded_mrule(DecGrammar.MRULE(_,pat,exp)) =
	  unguarded_pat pat ++
	  unguarded_exp exp

      and unguarded_dec(DecGrammar.EXCEPTIONdec(_,exbind)) =
	  unguarded_exbind exbind
	| unguarded_dec(DecGrammar.LOCALdec(_,dec1,dec2)) =
	  unguarded_dec dec1 ++ unguarded_dec dec2
	| unguarded_dec(DecGrammar.SEQdec(_,dec1,dec2)) =
	  unguarded_dec dec1 ++ unguarded_dec dec2
	| unguarded_dec _ = []

      and unguarded_valbind(DecGrammar.PLAINvalbind(_,pat,exp,valbind_opt)) =
	  unguarded_pat pat ++ unguarded_exp exp ++
	  unguarded_opt unguarded_valbind valbind_opt
	| unguarded_valbind(DecGrammar.RECvalbind(_,valbind)) =
	  unguarded_valbind valbind

      and unguarded_exbind(DecGrammar.EXBIND(_,_,ty_opt,exbind_opt)) =
	  unguarded_opt unguarded_ty ty_opt ++
	  unguarded_opt unguarded_exbind exbind_opt
	| unguarded_exbind(DecGrammar.EXEQUAL(_,_,_,exbind_opt)) =
	  unguarded_opt unguarded_exbind exbind_opt

      and unguarded_atpat(DecGrammar.RECORDatpat(_,patrow_opt)) =
	  unguarded_opt unguarded_patrow patrow_opt
	| unguarded_atpat(DecGrammar.PARatpat(_,pat)) =
	  unguarded_pat pat
	| unguarded_atpat _ = []

      and unguarded_patrow(DecGrammar.DOTDOTDOT(_)) = []
	| unguarded_patrow(DecGrammar.PATROW(_,_,pat,patrow_opt)) =
	  unguarded_pat pat ++
	  unguarded_opt unguarded_patrow patrow_opt

      and unguarded_pat(DecGrammar.ATPATpat(_,atpat)) =
	  unguarded_atpat atpat
	| unguarded_pat(DecGrammar.CONSpat(_,_,atpat)) =
	  unguarded_atpat atpat
	| unguarded_pat(DecGrammar.TYPEDpat(_,pat,ty)) =
	  unguarded_pat pat ++ unguarded_ty ty
	| unguarded_pat(DecGrammar.LAYEREDpat(_,_,ty_opt,pat)) =
	  unguarded_opt unguarded_ty ty_opt ++
	  unguarded_pat pat
	| unguarded_pat(DecGrammar.UNRES_INFIXpat _) =
	  Crash.impossible "unguarded_pat(UNRES_INFIX)"

      and unguarded_ty(DecGrammar.TYVARty(_,tyvar)) =
	  [StatObject.mkExplicitTyVar tyvar]
	| unguarded_ty(DecGrammar.RECORDty(_,tyrow_opt)) =
	  unguarded_opt unguarded_tyrow tyrow_opt
	| unguarded_ty(DecGrammar.CONty(_,ty_list,_)) =
	  List.foldL
	  (fn ty => fn tyvarset => unguarded_ty ty ++ tyvarset)
	  [] ty_list
	| unguarded_ty(DecGrammar.FNty(_,ty1,ty2)) =
	  unguarded_ty ty1 ++ unguarded_ty ty2
	| unguarded_ty(DecGrammar.PARty(_,ty)) =
	  unguarded_ty ty

      and unguarded_tyrow(DecGrammar.TYROW(_,_,ty,tyrow_opt)) =
	  unguarded_ty ty ++
	  unguarded_opt unguarded_tyrow tyrow_opt
    in
      fun Scoped_TyVars
	(valbind: DecGrammar.valbind, tyvarset: TyVarSet): TyVarSet =
	 ListHacks.minus(unguarded_valbind valbind, tyvarset)
    end

    (********
    Type functions
    ********)

    type TypeFcn = StatObject.TypeFcn

    (********
    The type of items associated with identifiers in a VarEnv
    ********)

   (* MEMO: why LONGxxx? *)

    datatype VarEnvRange = LONGVAR   of TypeScheme
			 | LONGCON   of TypeScheme
			 | LONGEXCON of Type

    (* Actually, we use a slightly richer private VarEnvRange type which, for
       each constructor, remembers the fellow constructors in its type. The
       usual lookup functions return an ordinary VarEnvRange. *)

    datatype VarEnvRangePRIVATE = LONGVARpriv of TypeScheme
      				| LONGCONpriv of TypeScheme * con list
				| LONGEXCONpriv of Type

    (********
    The types of contexts and environments
    ********)

    datatype Context = CONTEXT of {T: TyNameSet, U: TyVarSet, E: Env}
	 and Env = ENV of {SE: StrEnv, TE: TyEnv, VE: VarEnv, EE: ExConEnv}
	 and StrEnv = STRENV of (strid, Str) FinMap.map
	 and TyEnv = TYENV of (tycon, TyStr) FinMap.map
	 and TyStr = TYSTR of {theta: TypeFcn, CE: ConEnv}
	 and VarEnv = VARENV of (id, VarEnvRangePRIVATE) FinMap.map
	 and ConEnv = CONENV of (con, TypeScheme) SortedFinMap.map
	 and ExConEnv = EXCONENV of (excon, Type) FinMap.map
	 and Str = STR of {m: StrName, E: Env}

    (********
    Type structures
    ********)

    fun mkTyStr(typefcn: TypeFcn, CE: ConEnv): TyStr =
      TYSTR {theta = typefcn, CE = CE}

    fun unTyStr(TYSTR {theta, CE}): TypeFcn * ConEnv =
      (theta, CE)

    fun TyStr_shares(TYSTR{theta=theta1, ...},
		     TYSTR{theta=theta2, ...}
		    ): bool =
      theta1 = theta2

    fun Theta_of(TYSTR {theta, CE}): TypeFcn =
      theta

    fun isTyName(TYSTR {theta, ...}): bool =
      case StatObject.unTyName_TypeFcn(theta) of
	Some _ => true
      | None => false

    fun CE_of_TyStr(TYSTR {theta, CE}): ConEnv =
      CE

    val bogus_TyStr = TYSTR{theta=StatObject.bogus_TypeFcn,
			    CE=CONENV SortedFinMap.empty
			   }

    fun isEmptyCE(CONENV map) = SortedFinMap.isEmpty map

    (********
    Structures
    ********)

    fun mkStr(m: StrName, E: Env): Str = STR {m = m, E = E}
    fun unStr(STR {m, E}): StrName * Env = (m, E)

    (********
    Variable environments
    ********)

    val emptyVE: VarEnv = VARENV FinMap.empty

    fun VEdom (VARENV m) = FinMap.dom m

    val bogus_VE = emptyVE

    val singleVE: id * VarEnvRangePRIVATE -> VarEnv = VARENV o FinMap.singleton

    fun VE_plus_VE(VARENV v, VARENV v'): VarEnv = 
      VARENV(FinMap.plus(v, v'))

    fun rangePRIVATE_to_range r =
      case r
	of LONGVARpriv sigma => LONGVAR sigma
	 | LONGCONpriv(sigma, _) => LONGCON sigma
	 | LONGEXCONpriv tau => LONGEXCON tau

    fun VEmap (f: VarEnvRangePRIVATE -> VarEnvRangePRIVATE) (VARENV m): VarEnv =
      VARENV(FinMap.composemap f m)

    fun VEfold (f: (VarEnvRange * 'a) -> 'a)
      	       (start: 'a)
	       (VARENV map): 'a = 
      FinMap.fold ((fn (range, a) => f(rangePRIVATE_to_range range, a)))
		  start map

    fun VEFoldPRIVATE (f: ((id * VarEnvRangePRIVATE) * 'a) -> 'a)
      		      (start: 'a) (VARENV map): 'a =
		      FinMap.Fold f start map

    fun VEFold (f: ((id * VarEnvRange) * 'a) -> 'a)
      	       (start: 'a)
	       (VARENV map): 'a = 
      FinMap.Fold (fn ((id, range), a) =>
		     f((id, rangePRIVATE_to_range range), a)
		  ) start map

    fun singleVarVE(id: Ident.id, sigma: TypeScheme): VarEnv =
      VARENV(FinMap.singleton(id, LONGVARpriv sigma))

    local
      fun f((id, LONGVARpriv sigma), VE) =
	    let
	      val VE' =
		singleVarVE(
		  id, StatObject.Close(StatObject.tyvarsTySch sigma, sigma)
		)
	    in
	      VE_plus_VE(VE, VE')
	    end

	| f _ = Crash.impossible "Environments.ClosVE"
    in
      fun ClosVE(VE: VarEnv): VarEnv =
	VEFoldPRIVATE f emptyVE VE
    end

    fun lookupVE(VE: VarEnv, id: Ident.id): VarEnvRange Option =
      let
	val VARENV m = VE
      in
	case FinMap.lookup m id
	  of Some range =>
	       (case range
		  of LONGVARpriv sigma => Some(LONGVAR sigma)
		   | LONGCONpriv(sigma, _) => Some(LONGCON sigma)
		   | LONGEXCONpriv tau => Some(LONGEXCON tau)
	       )

	   | None => None
      end

    fun lookupCons(VE: VarEnv, id: Ident.id): con list Option =
      let
	val VARENV m = VE
      in
	case FinMap.lookup m id
	  of Some range =>
	       (case range
		  of LONGVARpriv _ => None
		   | LONGCONpriv(_, cons) => Some cons
		   | LONGEXCONpriv _ => None
	       )

	   | None => None
      end

    fun onVE(S: StatObject.Substitution, VARENV m): VarEnv =
      VARENV(FinMap.composemap
	     (fn LONGVARpriv sigma => LONGVARpriv(StatObject.onScheme(S, sigma))
	       | LONGCONpriv(sigma, cons) =>
		   LONGCONpriv(StatObject.onScheme(S, sigma), cons)
	       | LONGEXCONpriv tau => LONGEXCONpriv(StatObject.on(S, tau))
	     ) m
	    )

    (********
    Constructor environments
    ********)

    val emptyCE: ConEnv = CONENV SortedFinMap.empty
    val singleCE: con * TypeScheme -> ConEnv = CONENV o SortedFinMap.singleton

    fun CE_plus_CE(CONENV m, CONENV m'): ConEnv =
      CONENV(SortedFinMap.plus Con.<(m, m'))

    fun domCE(CONENV m): con list =
      SortedFinMap.domSORTED m

    fun CEmap (f: TypeScheme -> TypeScheme) (CONENV m): ConEnv =
      CONENV(SortedFinMap.composemap f m)

    fun CEfold (f: TypeScheme * 'a -> 'a) (start: 'a) (CONENV map): 'a =
      SortedFinMap.fold f start map

    fun CEFold (f: (con * TypeScheme) * 'a -> 'a) (start: 'a) (CONENV map): 'a =
      SortedFinMap.Fold f start map

    fun isemptyCE(CONENV m): bool =
      SortedFinMap.isEmpty m

    fun equalCE(C1: ConEnv, C2: ConEnv): bool =
      C1 = C2

    (********
    Exception constructor environments
    ********)

    val emptyEE: ExConEnv = EXCONENV FinMap.empty

    fun EEdom(EXCONENV m) = FinMap.dom m
      
    fun singleEE(excon: Excon.excon, tau: Type): ExConEnv =
      EXCONENV(FinMap.singleton(excon, tau))

    fun EE_plus_EE(EXCONENV e, EXCONENV e'): ExConEnv =
      EXCONENV(FinMap.plus(e, e'))

    fun EEmap (f: Type -> Type) (EXCONENV m): ExConEnv =
      EXCONENV(FinMap.composemap f m)

    fun EEfold (f: (Type * 'a) -> 'a) (start: 'a) (EXCONENV map): 'a = 
      FinMap.fold f start map

    fun EEFold (f: ((excon * Type) * 'a) -> 'a) (start: 'a) (EXCONENV map): 'a = 
      FinMap.Fold f start map

    fun VE_of_EE(EE: ExConEnv): VarEnv =
      let
	fun f((excon, ty), m) =
	  FinMap.add(Excon.un_excon excon, LONGEXCONpriv ty, m)
      in
	VARENV(EEFold f FinMap.empty EE)
      end

    fun lookupEE(EXCONENV m, excon: Excon.excon): Type Option =
      FinMap.lookup m excon

    (********
    Type environments
    ********)

    val emptyTE: TyEnv = TYENV FinMap.empty

    val singleTE: tycon * TyStr -> TyEnv = TYENV o FinMap.singleton

    fun TE_plus_TE(TYENV t, TYENV t'): TyEnv =
      TYENV(FinMap.plus(t, t'))

    fun TEmap (f: TyStr -> TyStr) (TYENV m): TyEnv =
      TYENV(FinMap.composemap f m)

    fun TEfold (f: (TyStr * 'a) -> 'a) (start: 'a) (TYENV map): 'a = 
      FinMap.fold f start map

    fun TEFold (f: ((tycon * TyStr) * 'a) -> 'a) (start: 'a) (TYENV map): 'a = 
      FinMap.Fold f start map

    fun TEdom(TYENV map) =
      EqSet.list(FinMap.dom map)	(* MEMO: TIDY UP! *)

    fun lookupTE(TYENV m, tycon: TyCon.tycon): TyStr Option =
      FinMap.lookup m tycon

    (********
    Structure environments
    ********)

    val emptySE: StrEnv = STRENV FinMap.empty

    val singleSE: strid * Str -> StrEnv = STRENV o FinMap.singleton

    fun SE_plus_SE (STRENV s, STRENV s'): StrEnv =
      STRENV(FinMap.plus(s, s'))

    fun SEfold (f: Str * 'a -> 'a) (start: 'a) (STRENV map): 'a = 
      FinMap.fold f start map

    fun SEFold (f: ((strid * Str) * 'a) -> 'a) (start: 'a) (STRENV map): 'a = 
      FinMap.Fold f start map

    fun SEdom (STRENV map) =
      EqSet.list(FinMap.dom map)	(* MEMO: TIDY UP! *)

    fun SEmap (f: Str -> Str) (STRENV map) =
      STRENV(FinMap.composemap f map)

    fun lookupSE (STRENV map, strid: strid): Str Option =
      FinMap.lookup map strid

    (********
    Environments
    ********)

    val emptyE: Env = ENV {SE=emptySE, TE=emptyTE, VE=emptyVE, EE=emptyEE}

    val bogus_Env = emptyE

   (* and now the bogus structure: *)

    val bogus_Str = mkStr(freshStrName(), bogus_Env)

    fun VE_in_E (VE: VarEnv): Env =
      ENV{SE = emptySE, TE = emptyTE, VE = VE, EE = emptyEE}

    fun E_plus_E(ENV{SE, TE, VE, EE}, ENV{SE=SE', TE=TE', VE=VE', EE=EE'}) =
      ENV{SE=SE_plus_SE(SE, SE'), TE=TE_plus_TE(TE, TE'),
	  VE=VE_plus_VE(VE, VE'), EE=EE_plus_EE(EE, EE')
	 }

    fun SE_in_E (SE: StrEnv): Env =
      ENV {SE = SE, TE = emptyTE, VE = emptyVE, EE = emptyEE}

    fun TE_in_E (TE: TyEnv): Env =
      ENV {SE = emptySE, TE = TE, VE = emptyVE, EE = emptyEE}

    fun VE_and_TE_in_E (VE: VarEnv, TE: TyEnv): Env =
      ENV {SE = emptySE, TE = TE, VE = VE, EE = emptyEE}

    fun VE_and_EE_in_E (VE: VarEnv, EE: ExConEnv): Env =
      ENV {SE = emptySE, TE = emptyTE, VE = VE, EE = EE}

    (* Note that only VE contains free type variables *)

    fun onE (S, ENV {SE, TE, VE, EE}) =
      ENV {SE = SE, TE = TE, VE = onVE(S, VE), EE = EE}

    fun mkEnv (SE, TE, VE, EE): Env =
      ENV {SE = SE, TE = TE, VE = VE, EE = EE}

    fun unEnv (ENV {SE, TE, VE, EE}) =
      (SE, TE, VE, EE)

    fun lookupE_strid (ENV {SE, ...}, strid: strid): Str Option =
      lookupSE(SE, strid)

    fun lookupE_tycon (ENV {TE, ...}, tycon: tycon): TyStr Option =
      lookupTE(TE, tycon)

    (********
    Calculating NameSets
    ********)

    local
      fun TyNamesVE (VE: VarEnv) =
	VEfold (fn (LONGVAR sigma, T) =>
		     StatObject.TyNameSetUnion(StatObject.TyNamesTySch sigma, T)
		 | (LONGCON sigma, T) =>
		     StatObject.TyNameSetUnion(StatObject.TyNamesTySch sigma, T)
		 | (LONGEXCON tau, T) =>
		     StatObject.TyNameSetUnion(StatObject.TyNamesTy tau, T)
	       ) StatObject.emptyTyNameSet VE

      fun TyNamesCE (CE: ConEnv) =
	CEfold
	(fn (sigma, T) => StatObject.TyNameSetUnion(StatObject.TyNamesTySch sigma, T))
	StatObject.emptyTyNameSet CE

      fun TyNamesTyStr (TYSTR {theta, CE}) =
	StatObject.TyNameSetUnion(StatObject.TyNamesTypeFcn theta, TyNamesCE CE)

      fun TyNamesTE (TE: TyEnv) =
	TEfold
	(fn (tystr, T) => StatObject.TyNameSetUnion(TyNamesTyStr tystr, T)) 
	StatObject.emptyTyNameSet TE

      fun TyNamesEE (EE: ExConEnv) =
	EEfold
	(fn (ty, T) => StatObject.TyNameSetUnion(StatObject.TyNamesTy ty, T)) 
	StatObject.emptyTyNameSet EE

      fun TyNamesSE (SE: StrEnv) =
	SEfold
	(fn (str, T) => StatObject.TyNameSetUnion(TyNamesStr str, T)) 
	StatObject.emptyTyNameSet SE

      and TyNamesStr (STR {m, E}) =
	TyNamesE E

      and TyNamesE (ENV {VE, TE, EE, SE}) =
	let
	  val T1 = StatObject.TyNameSetUnion(TyNamesVE VE, TyNamesTE TE)
	  val T2 = StatObject.TyNameSetUnion(TyNamesEE EE, TyNamesSE SE)
	in
	  StatObject.TyNameSetUnion(T1, T2)
	end
    in
      fun namesE (E as ENV {SE, TE, VE, EE}): NameSet =
	NameSetUnion(namesSE SE, T_in_N (TyNamesE E))

      and namesSE (SE: StrEnv) =
	SEfold
	(fn (str, nset) => NameSetUnion(nset, namesS str))
	emptyN SE

      and namesS (STR {m, E}): NameSet =
	NameSetUnion(mkNameSet_Str [m], namesE E)
    end

    (********
    Contexts
    ********)

    fun mkC (T, E): Context =
      CONTEXT {T=T, U=nil, E=E}

    fun C_plus_VE (CONTEXT {T, U, E}, VE': VarEnv): Context =
      let
	val ENV {SE, TE, VE, EE} = E
      in
	CONTEXT
	{T = T, U = U,
	 E = ENV {SE = SE, TE = TE, VE = VE_plus_VE(VE, VE'), EE = EE}
	}
      end

    fun E_in_C E =
      CONTEXT{T = StatObject.emptyTyNameSet, U = nil, E = E}

    fun onC (S: Substitution, CONTEXT {T, U, E}) =
      CONTEXT {T = T, U = U, E = onE(S, E)}

    (********
    Function which traverses down structure environments
    ********)

    fun Traverse E [] = Some(E)
      | Traverse (ENV{SE = STRENV m, ...}) (strid :: rest) =
	case (FinMap.lookup m strid) of
	  Some(STR {m,E}) => Traverse E rest
	| None => None

    (********
    Function which looks up an identifier in the context
    ********)

    local
      fun lookupRoot f (CONTEXT{E, ...}, longid) =
	let
	  val (strid_list, id) = Ident.decompose longid
	in
	  case Traverse E strid_list
	    of None => None
	     | Some E' => let val ENV{VE, ...} = E' in f(VE, id) end
	end
    in
      val Lookup_longid = lookupRoot lookupVE

      fun lookupFellowCons longid =
	case lookupRoot lookupCons longid
	  of Some cons => cons
	   | None => Crash.impossible "Environments.lookupFellowCons"
    end

  (********
    Function which looks up a long type constructor in the context
    ********)

    fun Lookup_longtycon(CONTEXT {E, ...}, longtycon) =
      let
	val (strid_list, tycon) = TyCon.explode_LongTyCon longtycon
      in
	case (Traverse E strid_list) of
	  None => None
	| Some(E') =>
	    let
	      val ENV {TE, ...} = E'
	    in
	      lookupTE(TE, tycon)
	    end
      end

    (********
    Function which looks up a type constructor in the context
    ********)

    fun Lookup_tycon (CONTEXT {E = ENV {TE, ...}, ...}, tycon) =
      lookupTE(TE, tycon)

    (********
    Qualified structure identifiers
    ********)

    type longstrid = StrId.longstrid

    (********
    Function which looks up a structure identifier in the context
    ********)

    fun Lookup_longstrid (CONTEXT {E, ...}, longstrid) =
      let
	val (strid_list, strid) = StrId.explode_longstrid longstrid
      in
	case (Traverse E strid_list) of
	  None => None
	| Some(E') =>
	    let
	      val ENV {SE = STRENV m, ...} = E'
	    in
	      FinMap.lookup m strid
	    end
      end

    (********
    The `circled plus' operations
    ********)

    fun C_cplus_E (CONTEXT {T, U, E}, E': Env): Context =
      CONTEXT
      {T = StatObject.TyNameSetUnion(T, T_of_N (namesE E')),
       U = U, E = E_plus_E(E, E')}

    fun C_cplus_VE_and_TE (C: Context, (VE: VarEnv, TE: TyEnv)) =
      C_cplus_E(C, VE_and_TE_in_E(VE,TE))

    fun C_cplus_TE (C: Context, TE: TyEnv) =
      C_cplus_E(C, TE_in_E TE)

    (********
    The `Clos' function for constructor environments
    ********)

    fun ClosCE (CE: ConEnv): ConEnv =
      CEmap (fn sigma => StatObject.Close(StatObject.tyvarsTySch sigma, sigma)) CE

    (********
    The `Clos' function for constructor environments
    *********
    This function converts the resulting CE to a VE
    ********)

   (* CloseCE_to_VE is also responsible for including, in each VE entry,
      the entire list of constructors of this type. This info is needed
      by the compiler. *)

    fun ClosCE_to_VE(CE: ConEnv): VarEnv =
      let
	fun f((con, _), rest) = con :: rest
	val cons = CEFold f nil CE
	val cons = ListSort.sort (General.curry Con.<) cons
					(* Note: we assume duplicates have
					   been detected and faulted. *)
	val CE' = ClosCE CE

	fun f((con, sigma), VE) =
	  VE_plus_VE(singleVE(Con.con_to_id con, LONGCONpriv(sigma, cons)), VE)
      in
	CEFold f emptyVE CE'
      end

    (* Initial environment: built-in types (`->', `int', `real', `string',
			    `exn', `ref'), the constructor `ref', and the
			    value `prim'. `prim' has type (int * 'a) -> 'b.
                            Moreover the overloaded values, and therefore
			    the type `bool'.
     *)

    local
      fun mk_tystr (tyname, CE) =
	mkTyStr(StatObject.TyName_in_TypeFcn tyname, CE)

      fun te(tycon, tyname) = singleTE(tycon, mk_tystr(tyname, emptyCE))

      fun joinTE [] = emptyTE
	| joinTE (TE :: rest) = TE_plus_TE(TE, joinTE rest)

      val intTE = te(TyCon.tycon_INT, TyName.tyName_INT)
      val realTE = te(TyCon.tycon_REAL, TyName.tyName_REAL)
      val stringTE = te(TyCon.tycon_STRING, TyName.tyName_STRING)
      val exnTE = te(TyCon.tycon_EXN, TyName.tyName_EXN)

      val boolCE = CE_plus_CE(singleCE(Con.con_TRUE, 
				       StatObject.Type_in_TypeScheme 
				         StatObject.TypeBool),
			      singleCE(Con.con_FALSE,
				       StatObject.Type_in_TypeScheme 
				         StatObject.TypeBool))
      val boolTE = singleTE(TyCon.tycon_BOOL,
			    mk_tystr(TyName.tyName_BOOL, ClosCE boolCE))

      local
	val alpha = StatObject.freshTyVar{equality=false, imperative=true , overloaded=false}
	val alphaTy = StatObject.mkTypeTyVar alpha

	val refTy =
	  StatObject.mkTypeArrow(alphaTy, StatObject.mkTypeRef alphaTy)

	val beta = StatObject.freshTyVar{equality=false, imperative=true , overloaded=false}
	val betaTy = StatObject.mkTypeTyVar beta

	val refTy_to_TE =
	  StatObject.mkTypeArrow(betaTy, StatObject.mkTypeRef betaTy)

      in
	val refCE = singleCE(Con.con_REF, StatObject.Type_in_TypeScheme refTy)
	val refCE_to_TE = singleCE(Con.con_REF, StatObject.Type_in_TypeScheme refTy_to_TE)
      end

      local
	val alpha = StatObject.freshTyVar{equality=false, imperative=false, overloaded=false}
	val alphaTy = StatObject.mkTypeTyVar alpha

	val beta = StatObject.freshTyVar{equality=false, imperative=false, overloaded=false}
	val betaTy = StatObject.mkTypeTyVar beta

	val primTy = StatObject.mkTypeArrow(
		       StatObject.mkTypePair(StatObject.TypeInt, alphaTy),
		       betaTy
		     )

	val primTyScheme = StatObject.Type_in_TypeScheme primTy

	val num = StatObject.freshTyVar{equality=false, imperative=false, overloaded=true}
	val numTy = StatObject.mkTypeTyVar num

	(* Overloaded types manually closed, to ensure closing of overloaded 
	   type variable *)
	val overloadedOpArityTwoTy = StatObject.mkTypeArrow(
		       StatObject.mkTypePair(numTy, numTy),
		       numTy)
	val overloadedOpArityTwoTyScheme = 
	  StatObject.mkTypeScheme([num], overloadedOpArityTwoTy)

	val overloadedOpArityOneTy = StatObject.mkTypeArrow(
		       numTy, numTy)
	val overloadedOpArityOneTyScheme = 
	  StatObject.mkTypeScheme([num], overloadedOpArityOneTy)

        val overloadedRelOpTy = StatObject.mkTypeArrow(
		       StatObject.mkTypePair(numTy, numTy),			
		       StatObject.TypeBool)
        val overloadedRelOpTyScheme =
	  StatObject.mkTypeScheme([num], overloadedRelOpTy)

      in
	val primVE      = singleVE(Ident.id_PRIM, LONGVARpriv primTyScheme)

	val absVE       = singleVE(Ident.id_ABS, LONGVARpriv overloadedOpArityOneTyScheme)
	val negVE       = singleVE(Ident.id_NEG, LONGVARpriv overloadedOpArityOneTyScheme)
	val plusVE      = singleVE(Ident.id_PLUS, LONGVARpriv overloadedOpArityTwoTyScheme)
	val minusVE     = singleVE(Ident.id_MINUS, LONGVARpriv overloadedOpArityTwoTyScheme)
	val mulVE       = singleVE(Ident.id_MUL, LONGVARpriv overloadedOpArityTwoTyScheme)
	val lessVE      = singleVE(Ident.id_LESS, LONGVARpriv overloadedRelOpTyScheme)
	val greaterVE   = singleVE(Ident.id_GREATER, LONGVARpriv overloadedRelOpTyScheme)
	val lesseqVE    = singleVE(Ident.id_LESSEQ, LONGVARpriv overloadedRelOpTyScheme)
        val greatereqVE = singleVE(Ident.id_GREATEREQ, LONGVARpriv overloadedRelOpTyScheme)

        fun joinVE [] = emptyVE
          | joinVE (VE :: rest) = VE_plus_VE(VE, joinVE rest)
      end

      val refTE =
	singleTE(TyCon.tycon_REF, mk_tystr(TyName.tyName_REF, ClosCE refCE_to_TE))

      val initialTE = joinTE [intTE, realTE, stringTE, exnTE, refTE, boolTE]

      val initialVE = joinVE [ClosCE_to_VE refCE, ClosCE_to_VE boolCE, 
			      ClosVE primVE, absVE, negVE, plusVE, minusVE,
			      mulVE, lessVE, greaterVE, lesseqVE, greatereqVE]
    in
      val initialE =
	ENV {SE=emptySE, TE=initialTE, VE=initialVE, EE=emptyEE}
    end

    (********
    Type variable sets
    ********)

    fun C_plus_U (C: Context,tyvarset: TyVarSet): Context =
      let
	val CONTEXT {T,U,E} = C
      in
	CONTEXT {T = T, U = ListHacks.union(U,tyvarset), E = E}
      end

    fun U_of_C (C: Context): TyVarSet =
      let val CONTEXT {T,U,E} = C in U end

    fun intersect (tyvarset1: TyVarSet,tyvarset2: TyVarSet): TyVar list =
      ListHacks.intersect(tyvarset1,tyvarset2)

    (********
    Get the TyVars in a VarEnv
    ********)

    fun tyvarsVE(VARENV m: VarEnv): TyVarSet =
      FinMap.fold
      (fn (LONGVARpriv sigma, tyvarset) =>
	     ListHacks.union(tyvarset,StatObject.tyvarsTySch sigma)
	| (LONGCONpriv(sigma, _), tyvarset) =>
	     ListHacks.union(tyvarset,StatObject.tyvarsTySch sigma)
	| (LONGEXCONpriv ty, tyvarset) =>
	     ListHacks.union(tyvarset,StatObject.tyvarsTy ty))
	[] m

    (********
    Check for imperative type variables
    ********)

    structure TVResult =
      struct
	datatype T = IMP_OK
		   | FAULT of TyVar list
	datatype R = REC_OK
	           | REC_FAULT
      end

    open TVResult

    fun tvUnion(result1, result2) =
      case (result1, result2)
	of (IMP_OK, IMP_OK) => IMP_OK
	 | (IMP_OK, FAULT tvs) => FAULT tvs
	 | (FAULT tvs, IMP_OK) => FAULT tvs
	 | (FAULT tvs1, FAULT tvs2) => FAULT(ListHacks.union(tvs1, tvs2))


    fun impTyVarsE(ENV{SE, VE, ...}) =
      tvUnion(impTyVarsVE VE, impTyVarsSE SE)

    and impTyVarsVE VE =
      let
	fun check tv result =
	  tvUnion(result, case StatObject.impTyVar tv
		  	    of true => FAULT [tv]
			     | false => IMP_OK
		 )
      in
	List.foldL check IMP_OK (tyvarsVE VE)
      end

    and impTyVarsSE SE =
      SEfold
        (fn (str, result) => 
	  let
	    val (_, E) = unStr str 
	  in
	    tvUnion(result, impTyVarsE E)
	  end)
	IMP_OK
	SE

    (********
    Type realisations
    ********)

    type tyrea = StatObject.tyrea

    fun tyrea_on_CE (TyRea: tyrea) (CE: ConEnv) =
      CEmap (StatObject.tyrea_on_TypeScheme TyRea) CE

    and tyrea_on_TyStr (TyRea: tyrea) (TYSTR {theta, CE}) =
      TYSTR {theta = StatObject.tyrea_on_TypeFcn TyRea theta,
	     CE = tyrea_on_CE TyRea CE}

    and tyrea_on_TE (TyRea: tyrea) (TE: TyEnv) =
      TEmap (tyrea_on_TyStr TyRea) TE

    and tyrea_on_VE (TyRea: tyrea) (VE: VarEnv) =
      VEmap (fn LONGVARpriv sigma =>
		  LONGVARpriv(StatObject.tyrea_on_TypeScheme TyRea sigma)
	      | LONGCONpriv(sigma, cons) =>
		  LONGCONpriv(StatObject.tyrea_on_TypeScheme TyRea sigma,
			      cons
			     )
	      | LONGEXCONpriv tau =>
		  LONGEXCONpriv(StatObject.tyrea_on_Type TyRea tau)
	    ) VE

    and tyrea_on_EE (TyRea: tyrea) (EE: ExConEnv) =
      EEmap (StatObject.tyrea_on_Type TyRea) EE

    (********
    The ABS function
    ********)

    local
      fun modifyTyStr (TYSTR {theta, ...}) =
	let
	  val tyname1 =
	    case StatObject.unTyName_TypeFcn(theta) of
	      Some(tyname) => tyname
	    | None => Crash.impossible "Environments.modifyTyStr"

	  val name    = TyName.name tyname1
	  val arity   = TyName.arity tyname1
	  val tyname2 = TyName.freshTyName {name = name, arity = arity, equality = false}
	  val typefcn = StatObject.TyName_in_TypeFcn tyname2
	in
	  StatObject.mktyrea(tyname1, typefcn)
	end

      fun modifyTE (TE: TyEnv) =
	TEfold
	(fn (tystr, res) => StatObject.oo_tyrea(modifyTyStr tystr, res))
	StatObject.id_tyrea TE

      fun isEmptySE (STRENV m) = FinMap.isEmpty m

      fun tyrea_on_E (TyRea: tyrea) (ENV {SE, EE, TE, VE}) =
	if isEmptySE(SE) then
	  ENV
	  {SE = SE,
	   EE = tyrea_on_EE TyRea EE,
	   TE = tyrea_on_TE TyRea TE,
	   VE = tyrea_on_VE TyRea VE}
	else
	  Crash.impossible "Environments.ABS"

      fun strip (TE: TyEnv): TyEnv =
	TEmap (fn (TYSTR {theta, ...}) => TYSTR {theta = theta, CE = emptyCE}) TE
    in
      fun ABS (TE: TyEnv, E: Env): Env =
	let
	  val tyrea = modifyTE TE
	  val TE' = strip TE
	  val E'  = E_plus_E(TE_in_E TE', E)
	in
	  tyrea_on_E tyrea E'
	end
    end

    local
      open DecGrammar
    in
	fun dom_pat (C, pat: pat): id list =
	  let
	    fun dom_patrow(patrow: patrow): id list =
	      case patrow of
		DOTDOTDOT _ =>
		  nil
		  
	      | PATROW(_, _, pat, None) =>
		  dom_pat' pat
		  
	      | PATROW(_, _, pat, Some patrow') =>
		  dom_pat' pat @ dom_patrow patrow'
		  
	    and dom_atpat(atpat: atpat): id list =
	      case atpat of
		WILDCARDatpat _ => nil
	      | SCONatpat(_, _) => nil
	      | RECORDatpat(_, None) => nil
	      | RECORDatpat(_, Some patrow) => dom_patrow patrow
	      | PARatpat(_, pat) => dom_pat' pat
		  
	      | LONGIDatpat(i, OP_OPT(longid, _)) =>
		  (case Lookup_longid(C, longid)
		     of Some(LONGCON _) => nil
		   | Some(LONGEXCON _) => nil
		   | _ => case Ident.decompose longid
		       of (nil, id) => [id]
		     | _ => Crash.impossible "ElabDec.dom_atpat"
			 )
		     
	    and dom_pat'(pat: pat): id list =
	      case pat of
		ATPATpat(_, atpat) => dom_atpat atpat
	      | CONSpat(_, _, atpat) => dom_atpat atpat
	      | TYPEDpat(_, pat, _) => dom_pat' pat
	      | LAYEREDpat(_, OP_OPT(id, _), _, pat) => id :: dom_pat' pat
	      | UNRES_INFIXpat _ => Crash.impossible "dom_pat'(UNRES_INFIX)"
	  in
	    dom_pat' pat
	  end
    end

    (********
    The `Clos' function for valbinds
    ********)

    local
      exception BoundTwice of VarEnv (* raised if an identifier is bound
					twice in the vb --- this has been
					spotted earlier, and we choose to 
					refrain to close any of the 
					typeschemes in the VE, and just return
					VE unmodified *)

    in
    fun Clos (C: Context, vb: DecGrammar.valbind, VE: VarEnv): VarEnv =

      let
	val CONTEXT {E=ENV{SE=SE, VE=VARENV ve_map, ...}, U, ...} = C

	fun TV sigma tyvarlist =
	  ListHacks.union(StatObject.tyvarsTySch sigma, tyvarlist)

	(* Note that all tyvars in U are considered as free in C *)

	val freeTyvars = 
	  FinMap.fold
	    (fn (LONGVARpriv sigma, tyvarlist) => TV sigma tyvarlist
	      | (LONGCONpriv(sigma, _), tyvarlist) => TV sigma tyvarlist
	      | (LONGEXCONpriv tau, tyvarlist) => tyvarlist
	    ) U ve_map

	open DecGrammar 

	(* According to the Definition, p. 20 : Any variable, constructor and
	   fn expression, possibly constrained by one or more type expressions, 
	   is non-expansive; all other expressions are said to be expansive *)

	fun isExpansiveExp (exp : exp) =
        case exp of
	  ATEXPexp(_, atexp) =>  isExpansiveAtExp atexp
	| TYPEDexp(i, exp, _) => isExpansiveExp exp
	| FNexp(_, _) => false
	|  _ => true
  
	and isExpansiveAtExp (atexp : atexp) =
	case atexp of
	  IDENTatexp(_, OP_OPT(ident, withOp)) =>
	    (case Lookup_longid(C,ident) of             (* This is necessary as the *)
              Some(LONGVAR sigma) => false              (* grammar is not resolved as yet *)
	    | Some(LONGCON sigma) => false    
            | Some(LONGEXCON tau) => false
	    | None => true)
	| _ => true
	  

	(* isExpansiveId returns a function which, for every variable
	   bound in a pattern in the valbind vb, returns true if the 
	   expression corresponding to the pattern is expansive, otherwise 
	   false *)

	fun isExpansiveId (vb : valbind) =
	  let 
	    fun makemap pat exp = 
	      let 
		val b = isExpansiveExp exp
	      in
		List.foldR
		  (fn id => fn m => FinMap.add(id, b, m))
		   FinMap.empty
		   (dom_pat (C,pat))	      end  
	  in
	    case vb of
	      PLAINvalbind(_, pat, exp, None) =>
		makemap pat exp
	    | PLAINvalbind(_, pat, exp, Some vb) =>
		let val m1 = makemap pat exp
		    val m2 = isExpansiveId vb
		in
		  FinMap.mergeMap 
		    (fn (b1, b2) => raise BoundTwice VE)
		    m1 m2
		end
	    | RECvalbind(i, vb) =>
		isExpansiveId vb
	  end

	val isExpansiveId = isExpansiveId vb

        (* isVar is true iff id is a variable in domVE *)		     
	fun remake isVar id sigma =
	  let
	    val (_, ty) = StatObject.unTypeScheme sigma
	    val bv1 = ListHacks.minus(StatObject.tyvarsTy ty, freeTyvars)
	    val isExp = 
	      case (FinMap.lookup isExpansiveId id) of 
		None => false
              | Some b => if Flags.DEBUG_ELABDEC then 
			    (if b then output(std_out, (Ident.pr_id id) ^ "'s exp is expansive")
			     else output(std_out, (Ident.pr_id id) ^ "'s exp is NOT expansive");
			     b)
			  else
			    b
	    val bv2 = if isVar andalso isExp  then
	                List.all (not o StatObject.impTyVar) bv1 (* remove imperative tyvars *)
		      else 
			bv1
	  in
	    StatObject.Close(bv2, sigma)   (* Remark: We must use Close in stead of
					     mkTypeScheme to avoid generalisation
					     of overloaded type variables *)
	  end

	val VARENV m = VE
      in
	VARENV(FinMap.ComposeMap
	       (fn (id, LONGVARpriv sigma) => LONGVARpriv(remake true id sigma)
		 | (id, LONGCONpriv(sigma, cons)) => LONGCONpriv(remake false id sigma, cons)
		 | (id, LONGEXCONpriv tau) => LONGEXCONpriv tau
	       ) m
	      )
      end handle (BoundTwice VE) => VE
    end

    (********
    Function which maximises equality in a type environment
    ********)

    local
      fun checkTyStr (TYSTR {theta, CE}, TyNameSet) =
	if isemptyCE(CE) then
	  TyNameSet
	else
	  let
	    val violates_equality =
	      CEfold
	      (fn (sigma, res) => StatObject.violates_equality(TyNameSet, sigma) orelse res)
	      false CE

	    val tyname =
	      case StatObject.unTyName_TypeFcn(theta) of
		Some(tyname) => tyname
	      | None => Crash.impossible "Environments.checkTyStr"
	  in
	    if violates_equality then
	      StatObject.TyNameSetMinus(TyNameSet, StatObject.singleTyNameSet tyname)
	    else
	      TyNameSet
	  end
    in
      fun maximise_TE_equality (TyNameSet: TyNameSet, TE: TyEnv): TyNameSet =
	TEfold checkTyStr TyNameSet TE
    end

    (********
    Function which find all flexible datatype names in a type environment.
    *********
    Any TyNames which already have their equality attribute set are ignored.
    ********)

    local
      fun check (TYSTR {theta, CE}) =
	if isemptyCE(CE) then
	  StatObject.emptyTyNameSet
	else
	  let
	    val tyname =
	      case StatObject.unTyName_TypeFcn(theta) of
		Some(tyname) => tyname
	      | None => Crash.impossible "Environments.check"

	    val equality = TyName.equality tyname
	  in
	    if equality then
	      StatObject.emptyTyNameSet
	    else
	      StatObject.singleTyNameSet tyname
	  end
    in
      fun flexible_tynames (TE: TyEnv): TyNameSet =
	TEfold
	(fn (tystr, tynameset) => StatObject.TyNameSetUnion(check tystr, tynameset))
	StatObject.emptyTyNameSet TE
    end

    (********
    Function which maximises equality in a type environment.
    *********
    The VarEnv passed is also modified so that it contains
    the correct equality attributes.
    ********)

    local
      fun iterate TyNameSet TE =
	let
	  val TyNameSet' = maximise_TE_equality(TyNameSet, TE)
	in
	  if StatObject.eqTyNameSet(TyNameSet, TyNameSet') then
	    TyNameSet
	  else
	    iterate TyNameSet' TE
	end
    in
      fun maximise_equality (VE: VarEnv, TE: TyEnv): VarEnv * TyEnv =
	let
	  val TyNameSet  = flexible_tynames TE
	  val TyNameSet' = iterate TyNameSet TE
	  val tyrea = StatObject.generate_tyrea TyNameSet'
	in
	  (tyrea_on_VE tyrea VE, tyrea_on_TE tyrea TE)
	end
    end


    (********
     Syntactic type variables, and 
     function which finds the set of type variables in a ty abstract 
     syntax tree 
     ********)

    type SyntaxTyVar = StatObject.SyntaxTyVar
     and ty = DecGrammar.ty
    local
      open DecGrammar
    in
      fun syntaxtyvarsTy ty =
	case ty of 
	  TYVARty(_, tyvar) => 
	    EqSet.singleton tyvar
	| RECORDty(_, None) => 
	    EqSet.empty
	| RECORDty(_, Some tyrow) =>  
	    syntaxtyvarsTyRow tyrow
	| CONty(_, tylist, _) => 
	    List.foldL EqSet.union EqSet.empty (map syntaxtyvarsTy tylist)
	| FNty(_,ty1,ty2) => 
	    EqSet.union (syntaxtyvarsTy ty1)
	                (syntaxtyvarsTy ty2)
	| PARty(_,ty) => 
	    syntaxtyvarsTy ty

      and syntaxtyvarsTyRow (TYROW(_,_,ty,tyrowopt)) =
	EqSet.union 
	(syntaxtyvarsTy ty) 
	(case tyrowopt of
	   None => EqSet.empty
	 | Some tyrow => syntaxtyvarsTyRow tyrow)
    end

    (********
    Printing functions for longtycon's (for error reporting)
    ********)

    val pr_LongTyCon = TyCon.pr_LongTyCon

    (********
    Printing function for environments
    ********)

    type StringTree = PP.StringTree

    fun layoutSE (STRENV m) =
      let
	val l = FinMap.Fold (op ::) nil m

	fun format_strid strid =
	  implode ["structure ", StrId.pr_StrId strid, " : "]

	fun layoutPair(strid,S) =
	  PP.NODE{start=format_strid strid,
		  finish="",
		  indent=3,
		  children=[layoutStr S],
		  childsep=PP.NONE
		 }
      in
	PP.NODE{start="",
		finish="",
		indent=0, 
		children=map layoutPair l,
		childsep=PP.RIGHT " "
	       }
      end

    and layoutTE (TYENV m) =
      let
	val l = FinMap.Fold (op ::) nil m

	fun layoutPair (tycon, tystr) =
	  PP.NODE{start=TyCon.pr_TyCon tycon ^ " ",
		  finish="",
		  indent=3,
		  children=[layoutTystr tystr],
		  childsep=PP.NONE
		 }
      in
	PP.NODE{start="",
		finish="",
		indent=0,
		children=map layoutPair l,
		childsep=PP.RIGHT " "
	       }
      end

    and layoutEE (EXCONENV m) =
      let
	val l = FinMap.Fold (op ::) nil m

	fun layoutPair (excon, ty) =
	  PP.NODE{start=Excon.pr_excon excon ^ " ",
		  finish="",
		  indent=3,
		  children=[StatObject.layoutType ty],
		  childsep=PP.NONE
		 }
      in
	PP.NODE{start="",
		finish="",
		indent=0,
		children=map layoutPair l,
		childsep=PP.RIGHT " "
	       }
      end

    and layoutVE(VARENV m) =
      let
	val l = FinMap.Fold (op ::) nil m

	fun format_id id =
	  case FinMap.lookup m id
	    of Some(LONGVARpriv _) =>
		 implode ["val ", Ident.pr_id id, ":"]

	     | Some(LONGCONpriv _) =>
		 implode ["con ", Ident.pr_id id, ":"]

	     | Some(LONGEXCONpriv _) =>
		 implode ["exception ", Ident.pr_id id, ":"]

	     | None =>
		 "<failure: layoutVE.format_id>"

	fun layoutRng(LONGVARpriv sigma) =
	      StatObject.layoutTypeScheme sigma
	  | layoutRng(LONGCONpriv(sigma, _)) =
	      StatObject.layoutTypeScheme sigma
	  | layoutRng(LONGEXCONpriv tau) =
	      StatObject.layoutType tau

	fun layoutPair(id, rng)= 
	  PP.NODE{start=format_id id ^ " ",
		  finish="",
		  indent=3,
		  children=[layoutRng rng],
		  childsep = PP.NONE
		 }
      in
	PP.NODE{start="",
		finish="",
		indent=0,
		children=map layoutPair l,
		childsep=PP.RIGHT " "
	       }
      end

    and layoutStrName m = PP.LEAF(Timestamp.print m)

    and layoutStr (STR {m, E}) =
      let
	val start =
	  if Flags.DEBUG_ENVIRONMENTS then
	    "sig(" ^ Timestamp.print m ^ ") "
	  else
	    "sig "
      in
	PP.NODE{start=start,
		finish=" end",
		indent=3,
		children=[layoutEnv E],
		childsep=PP.NONE
	       }
      end

    and layoutTystr(TYSTR {theta, CE}) =
      PP.NODE{start="",
	      finish="",
	      indent=0,
	      children=[StatObject.layoutTypeFcn theta, layoutCE CE],
	      childsep=PP.RIGHT "/ "
	     }

    and layoutCE (CONENV m) =
      let
	val l = SortedFinMap.Fold (op ::) nil m

	fun layoutPair(con, typeScheme) =
	  PP.NODE{start=Con.pr_con con ^ " ",
		  finish="",
		  indent=3,
		  children=[StatObject.layoutTypeScheme typeScheme],
		  childsep=PP.NONE
		 }
      in
	PP.NODE{start="",
		finish="",
		indent=0,
		children=map layoutPair l,
		childsep=PP.RIGHT " "
	       }
      end

    and layoutEnv(ENV{SE, TE, VE, EE}) =
      PP.NODE{start="",
	      finish="",
	      indent=0, 
	      children=[layoutSE SE, layoutTE TE, layoutVE VE, layoutEE EE],
	      childsep = PP.RIGHT " "
	     }

    fun layoutNameSet(NAMESET{M, T}) =
      PP.NODE{start="{", finish = "}", indent=1,
	      children=(StatObject.layoutTyNameSet T) :: (map layoutStrName M),
	      childsep=PP.LEFT ", "
	     }

    type Report = Report.Report
    infix //
    val op // = Report.//

   (* For the TE: report TyStrs with empty ConEnv as

	type (tvs) ty

      and those with nonempty ConEnv as

	datatype (tvs) ty
	  con C1 : tyscheme
	  con C2 : tyscheme

      This is a reasonable syntax for TE's both in bindings and in specs.
      (saying `type' for `datatype' wouldn't look too good in specs.) *)

    fun reportCE(CONENV map) =
      SortedFinMap.reportMapSORTED
        (fn (con, tyScheme) =>
	   let
	     val names = StatObject.newTVNames()
	   in
	     Report.line("con " ^ Con.pr_con con ^ " : "
			 ^ StatObject.pr_TypeSchemePRETTY names tyScheme
			)
	   end
	) map

    fun reportTE{tyEnv=TYENV map, bindings} =
      FinMap.reportMapSORTED
        (TyCon.<) (fn (tycon, TYSTR{theta, CE}) =>
		     let
		       val names =
			 StatObject.newTVNames()

		       val {vars, body} =
			 StatObject.pr_TypeFcnPRETTY names theta

		       val vars' =
			 case vars of "" => "" | _ => vars ^ " "
		     in
		       if isEmptyCE CE
		       then Report.line("type " ^ vars'
					^ TyCon.pr_TyCon tycon
					^ (if bindings then " = " ^ body
					   else ""
					  )
				       )
		       else Report.line("datatype " ^ vars'
					^ TyCon.pr_TyCon tycon
				       )
			    // Report.indent(2, reportCE CE)
		     end
		  ) map

   (* Identifiers bound in the EE have type exn if non-valued or instances
      of 'a -> exn if they are constructors over 'a. *)

    fun reportEE(EXCONENV map) =
      FinMap.reportMapSORTED
        (Excon.<) (fn (excon, ty) =>
		     Report.line(
		       "exception " ^ Excon.pr_excon excon
		       ^ (if StatObject.isTypeExn ty
			  then ""
			  else " of "
			       ^ (case StatObject.unTypeArrow ty
				    of Some(domTy, _) =>
				         StatObject.pr_Type domTy

				     | None =>
					 Crash.impossible "reportEE"
				 )
			 )
		     )
		  ) map

    fun iterateSE(f, STRENV m) = FinMap.reportMapSORTED (StrId.<) f m

    fun iterateVE(f, VARENV m) =
      let
	fun f'(id, value: VarEnvRangePRIVATE) =
	  case value
	    of LONGVARpriv tyScheme =>
	         f(id, tyScheme)

	     | LONGCONpriv(tyScheme, _) =>
		 Report.null	(* We'll get the cons when we walk over
				   the TyStr's in the TE. *)

	     | LONGEXCONpriv ty =>
		 Report.null	(* We'll get the excons when we walk over
				   the EE. *)
      in
	FinMap.reportMapSORTED (Ident.<) f' m
      end
  end;

functor Test(structure DecGrammar: DEC_GRAMMAR

	     structure StrId: STRID

	     structure Ident: IDENT
	       sharing type Ident.strid = StrId.strid
		   and type Ident.id = DecGrammar.id
		   and type Ident.longid = DecGrammar.longid

	     structure Con: CON
	       sharing type Con.longid = Ident.longid
		   and type Con.strid  = StrId.strid
		   and type Con.id     = Ident.id

	     structure Var: VAR
	       sharing type Var.longid = Ident.longid

	     structure Excon: EXCON
	       sharing type Excon.id     = Ident.id
		   and type Excon.longid = Ident.longid
		   and type Excon.strid  = StrId.strid

	     structure TyCon: TYCON
	       sharing type TyCon.strid = StrId.strid

	     structure TyName: TYNAME

	     structure StatObject: STATOBJECT_PROP
	       sharing type StatObject.SyntaxTyVar = DecGrammar.tyvar
		   and type StatObject.TyName = TyName.TyName

	     structure PP: PRETTYPRINT
	       sharing type StatObject.StringTree = PP.StringTree

	     structure SortedFinMap: SORTED_FINMAP
	       sharing type SortedFinMap.StringTree = PP.StringTree

	     structure FinMap: FINMAP
	       sharing type FinMap.StringTree = PP.StringTree

	     structure Timestamp: TIMESTAMP

	     structure Report: REPORT
	       sharing type SortedFinMap.Report = FinMap.Report = Report.Report

	     structure Flags: FLAGS
	     structure ListHacks: LIST_HACKS
	     structure Crash: CRASH
	    ): sig end =
  struct
    local
      structure Unconstrained =
	Environments(structure DecGrammar = DecGrammar
		     structure Ident = Ident
		     structure Con = Con
		     structure Var = Var
		     structure Excon = Excon
		     structure TyCon = TyCon
		     structure StrId = StrId
		     structure StatObject = StatObject
		     structure TyName = TyName
		     structure PP = PP
		     structure SortedFinMap = SortedFinMap
		     structure FinMap = FinMap
		     structure Timestamp = Timestamp
		     structure Report = Report
		     structure Flags = Flags
		     structure ListHacks = ListHacks
		     structure Crash = Crash
		    )
    in
      structure Environments: ENVIRONMENTS = Unconstrained
      structure EnvironmentsProp: ENVIRONMENTS_PROP = Unconstrained
    end
  end;
