/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 * vc_call.c
 *
 * $Header: vc_call.c[1.7] Fri Apr 24 17:53:45 1992 axel@cs.tu-berlin.de accessed $
 */

/*LINTLIBRARY*/
#ifndef lint
static char *ATFSid = "$Header: vc_call.c[1.7] Fri Apr 24 17:53:45 1992 axel@cs.tu-berlin.de accessed $";
#ifdef CFFLGS
static char *ConfFlg = CFFLGS;
	/* should be defined from within Makefile */
#endif
#endif

#include <stdio.h>
#include <signal.h>
#include <atfs.h>
#include "atfsapp.h"

extern char *FindProgram();

/**/
int call (status, this_program, argvec)
     union wait *status;
     char *this_program;
     char **argvec;
{
  int pid, offered_pid;
  extern char **environ;

  if (!rindex (this_program, '/')) {
    if ((this_program = FindProgram (this_program)) == NULL)
      return 0;
  }
  
  if (!FileExecutable (this_program))
    return 0;
  
  pid = fork ();

  if (pid == -1) {
    logerr ("Can't fork");
    return 0;
  }

  if (!pid) {
    /* child */
    /* close open files ? */
    execve (this_program, argvec, environ);
    logerr ("Can't exec");
    _exit (1);
    return 1;			/* to make lint happy */
  }
  else {
    /* parent */
    /* here we must redefine actual interrupt handler. */
    while ( ((offered_pid = wait (status)) != pid) && (offered_pid != -1))
      ;				/* nothing to do */

    
    if (offered_pid == -1) {
      return 0;
    }
    /* here we must reconstruct old sighandl */
    if (wretcode(status))
      return 0;
    else
      return 1;
  }
/*NOTREACHED*/
}

/**/
int call_editor (editor, file, contents, newcontents)
     char *editor, *file, *contents;
/*     unsigned long *newcontents; */
     char **newcontents;
{
  /*
   * Calls editor "editor" with file "file" and returns its
   * contents after the editor session in "newcontents".
   * Return value is the length of the new text.
   *
   * On failure, 0 is returned to indicate the error. newcontents
   * is *not* updated and points to nowhere. On error the file
   * will be removed.
   *
   * If "contents" points to a valid text, this text is put (not appended)
   * into the temporary file before editor starts. Text must be
   * NULL terminated, otherwise strange things will happen...
   */
  
  FILE *fid;
  char *new;
  char cmd[1024];
  int length;
  union wait status;
  struct stat sbuf;
  char buf[100];
  char *argvec[10];

  *newcontents = NULL;		/* don't get confused on error. */

  Register (file, TYPEF);
  if ((fid = fopen (file, "w")) == NULL) { /* create a file */
    (void)sprintf (buf,"Can't open temporary file %s.", file);
    UnRegister (file, TYPEF);
    logerr (buf);
    return -1;
  }

  if (contents && *contents) {
    length = strlen (contents);
    
    if (fwrite (contents, sizeof (char), length, fid) != length) {
      (void)sprintf (buf, "lost bytes while writing to %s.", file);
      logerr (buf);
      (void)fclose (fid);
      RemoveFile (file);
      UnRegister (file, TYPEF);
      return -1;
    }
  }
  
  (void)fclose (fid);
  
  (void)sprintf (cmd, "%s %s", editor, file);
  (void)sprintf (buf, "starting up %s...", cmd);
  logmsg (buf);

  argvec[0] = editor;
  argvec[1] = file;
  argvec[2] = (char *) NULL;

  /*  if (retcode = system (cmd)) { */
  /*VARARGS2*/
  if (! call (&status, editor, argvec)) {
    (void)sprintf (buf, "Editor %s exited abnormally.", editor);
    logerr (buf);
    RemoveFile (file);
    UnRegister (file, TYPEF);
    return -1;
  }

  if (stat (file, &sbuf) == -1) {
    (void)sprintf (buf,"Can't stat temporary file %s.", file);
    logerr (buf);
    return -1;
  }
    
  if ((fid = fopen (file, "r")) == NULL) {
    (void)sprintf (buf,"Can't reopen temporary file %s.", file);
    logerr (buf);
    RemoveFile (file);
    UnRegister (file, TYPEF);
    return -1;
  }
  RemoveFile (file);
  UnRegister (file, TYPEF);
  length = sbuf.st_size;
  if ((new = malloc ((unsigned)(sizeof (char) * (length + 1)))) == NULL) {
    logerr ("Can't malloc for new");
    (void)fclose (fid);
    return -1;
  }
  
  if (!fread (new, sizeof (char), (Size_t)length, fid)) {
    (void)sprintf (buf, "lost bytes while reading from %s.", file);
    logerr (buf);
    (void)fclose (fid);
    return -1;
  }
  new[length] = NULL;

  (void)fclose (fid);
  *newcontents = new;
  return length;
}

/*
 * exports:
 *	int at_callcmd(char *command_processor, char *commandstring) ;
 *	Call command_processor with commandstring. Commandstring is sent
 *	to command_processor's stdin. If Commandstring is not terminated
 *	with newline, a newline is added.
 *	For return codes see at_callcmd.h.
 *
 *	int at_callcmd_errno ;
 *	Contains additional status information if at_callcmd fails.
 */


#include "config.h"
#include <errno.h>
#include <stdio.h>

extern int errno ;		/* to obtain error information after
				 * failed syscalls */

/* external functions  */

#ifndef WEXITSTATUS
#  define WEXITSTATUS(a) (((a).w_status & 0xff00) >> 8)
#endif /* WEXITSTATUS */

/* definition of return codes in atfsapp.h */

#define DELIM_CHARS   " \t"	/* delimiter characters for breaking
				 * command_processor into argv */

int at_callcmd_errno ;	/* to export additional status information */

static int sigpipe_occurred ;	/* flag: true if SIGPIPE was caught */
static int sigusr_occurred ;	/* flag: true if SIGUSR[12] was caught */

				/* function to send commandstring to child */
static int write_to_child() ;



int at_callcmd(command_processor, commandstring)
char *command_processor ;
char *commandstring ;
{
    char   **argv ;		/* built from command_processor */
    char   *cmdproc2 ;		/* copy of command_processor for strtok(3) */
    int    numtok ;		/* number of tokens in command_processor */
    int    i ;			/* loop counter */
    int    child_pid ;		/* PID of forked process */
    Wait_t child_status ;	/* exit status of child process */
    int    pipefd[2] ;		/* file descriptors for pipe to child */
    int    write_result ;	/* result of write_to_child() */
    int    wait_result ;	/* return value of wait(2) */
    int	   commandflag ;	/* do we have a commandstring? */
    static void sigusr_handler() ; /* signal handler for SIGUSR1 and SIGUSR2 */
    static Sigret_t (*old_sigusr1hand)() ; /* old signal handler for SIGUSR1 */
    static Sigret_t (*old_sigusr2hand)() ; /*  ... and SIGUSR2 */
    

    /* do we have a command string? */
    commandflag = commandstring != NULL && *commandstring != 0 ;

    /* no signals or errors yet */
    sigpipe_occurred = 0 ;
    sigusr_occurred = 0 ;
    at_callcmd_errno = 0 ;

    /* copy command_processor for strtok(3) */
    if ((cmdproc2 = malloc(strlen(command_processor) + 1)) == NULL) {
	at_callcmd_errno = ENOMEM ;
	return NO_MORE_CORE ;
    }
    strcpy(cmdproc2, command_processor) ;


    if (command_processor == NULL || *command_processor == 0
	    || strtok(cmdproc2, DELIM_CHARS) == NULL) {
	return CMDPROC_EMPTY ;
    }

    /* allocate and set up argv for child process */
    for (numtok = 1; strtok(NULL, DELIM_CHARS); numtok++)
	;

    /* copy command_processor for strtok(3) */
    if ((cmdproc2 = malloc(strlen(command_processor) + 1)) == NULL) {
	at_callcmd_errno = ENOMEM ;
	return NO_MORE_CORE ;
    }
    strcpy(cmdproc2, command_processor) ;

    argv = (char **) calloc(numtok + 1, sizeof(char *)) ;
    if (argv == NULL) {
	at_callcmd_errno = ENOMEM ;
	return NO_MORE_CORE ;
    }
    argv[0] = strtok(cmdproc2, DELIM_CHARS) ;
    for (i = 1; i < numtok; i++) {
	argv[i] = strtok(NULL, DELIM_CHARS) ;
    }
    argv[i] = NULL ;


    if (commandflag) {
	/* set up pipe to child's stdin */
	if (pipe(pipefd) != 0) {
	    at_callcmd_errno = errno ;
	    return PIPE_FAILED ;
	}
    }
    
    /* set up signal handler for SIGUSR1 and SIGUSR2 */
    old_sigusr1hand = signal(SIGUSR1, sigusr_handler) ;
    old_sigusr2hand = signal(SIGUSR2, sigusr_handler) ;

    /* and now let things go */
    switch (child_pid = vfork()) {
      case -1:			/* aehem... */
	at_callcmd_errno = errno ;
	free((char *) argv) ;
	signal(SIGUSR1, old_sigusr1hand) ;
	signal(SIGUSR2, old_sigusr2hand) ;
	return FORK_FAILED ;
	
      case 0:			/* child process */
	if (commandflag) {
	    /* set up pipe as stdin */
	    close(pipefd[1]) ;
	    close(0) ;
	    dup(pipefd[0]) ;
	    close(pipefd[0]) ;
	}

	signal(SIGUSR1, old_sigusr1hand) ;
	signal(SIGUSR2, old_sigusr2hand) ;

	/* execute program */
	execvp(argv[0], argv) ;

	/* this failed, so notify parent */
	at_callcmd_errno = errno ; /* this only works with vfork(2) */
	kill(getppid(), errno == ENOENT ? SIGUSR1 : SIGUSR2) ;
	_exit(0) ;		/* Don't flush parent's buffers! */
	break ;
	
      default:			/* parent process */
	close(pipefd[0]) ;

	/* write commandstring to child process */
	if (commandflag) {
	    write_result = write_to_child(pipefd[1], commandstring) ;
	} else {
	    write_result = OK ;
	}
	close(pipefd[1]) ;

	/* check termination of child process */
	while ((wait_result = wait(&child_status)) != child_pid) {
	    if (wait_result == -1) {
		at_callcmd_errno = errno ;
		signal(SIGUSR1, old_sigusr1hand) ; /* reset signal handlers */
		signal(SIGUSR2, old_sigusr2hand) ;
		return WAIT_ERROR ;
	    }
	}


	signal(SIGUSR1, old_sigusr1hand) ; /* reset signal handlers */
	signal(SIGUSR2, old_sigusr2hand) ;

	/* execvp(3) may have failed:
	 * if we can use vfork(2), at_callcmd_errno contains the
	 * reason of exec's failure */
	switch (sigusr_occurred) {
	  case SIGUSR1:
	    return NO_PROGRAM ;
	  case SIGUSR2:
	    return EXEC_FAILED ;
	}

	/* write to pipe may have had problems */
	if (write_result != OK) {
	    at_callcmd_errno = write_result ;
	    return WRITE_FAILED ;
	}

	/* report child's exit status or reason of death */
	if (WIFEXITED(child_status)) {
	    return WEXITSTATUS(child_status) ;
	} else {
	    /* the only case where at_callcmd_errno can't be interpreted
	     * like errno */
	    memcpy(&at_callcmd_errno, &child_status, sizeof(int)) ;
	    return CHILD_KILLED ;
	}
    }
    /* not reached, but to make gcc happy: */
    return 0 ;
}

/* write command into fd, append newline if necessary */
static int write_to_child(fd, command)
int fd ;			/* input file descriptor of pipe to child */
char *command ;			/* command string */
{
    int	written ;		/* no. of bytes written to child */
    int	cmd_write ;		/* length of rest of commandstring to write */
    int w_stat ;		/* return value of write(2) */
    static Sigret_t (*old_sigpipehand)() ; /* old signal handler for SIGPIPE */

    /* block SIGPPIPE, errno will report it anyway */
    old_sigpipehand = signal(SIGPIPE, SIG_IGN) ;
    cmd_write = strlen(command) ;
    written = 0 ;
    /* write in pieces to avoid pipe constipation */
    while (cmd_write > 0) {
	if ((w_stat = write(fd, command + written,
			    cmd_write > BUFSIZ ? BUFSIZ : cmd_write)) == -1) {
	    at_callcmd_errno = errno ;
	    signal(SIGPIPE, old_sigpipehand) ;
	    return at_callcmd_errno ;
	}
	written += w_stat ;
	cmd_write -= w_stat ;
    }
    /* append newline if necessary */
    if (command[written - 1] != '\n') {
	if (write(fd, "\n", 1) == -1) {
	    at_callcmd_errno = errno ;
	    signal(SIGPIPE, old_sigpipehand) ;
	    return errno ;
	}
    }
    signal(SIGPIPE, old_sigpipehand) ;
    return OK ;
}

/* catch SIGUSR[12] and set flag */
static void sigusr_handler(signal)
int signal ;
{
    sigusr_occurred = signal ;
}
