/*$__copyright$ */
/*
 *
 * AtFS-test -- try to repair corrupted archive
 *
 * $Header: atfsrepair.c[1.24] Fri Apr 24 17:56:13 1992 andy@cs.tu-berlin.de accessed $
 */

#include <stdio.h>

#include "atfs.h"
#include "afarchive.h"
#include "atfsrepair.h"

#include <setjmp.h>
#include <signal.h>
#include <sys/file.h>

int        verbosityLevel = NORMAL, outputVersion = AF_ARCURVERS;
Input      attrIn, dataIn;
ConstAttrs cAttrs;
Revision   revList[MAXREVS];
Udas       udaList[MAXREVS];
Note       noteList[MAXREVS];
Data       dataList[MAXREVS];
int        curRev, curUda, curNote, curData;

char *malloc(), *verifyString(), *complainString();
extern char *optarg;
extern int optind;

jmp_buf env;

#ifdef MEMDEBUG
FILE *memprot;
#endif

main (argc, argv)
     int  argc;
     char **argv;
{
  int   cleanup(), getopt(), sigHand();
  short c, i, nfiles, len, repairCache = FALSE;
  char  fileName[MAXNAMLEN+1], *af_version();
  DIR           *dirPointer;
  struct direct *dirEntry;
  
#ifdef MEMDEBUG
  memprot = fopen ("memprot", "w");
#endif

  while ((c = getopt (argc, argv, "CV:enqv")) != EOF)
    {
      switch (c)
	{
	case 'C': /* repair derived object cache */
	  repairCache = TRUE;
	  break;
	case 'V': /* output archive file version */
	  outputVersion = atoi (optarg);
	  if ((outputVersion < 2) || (outputVersion > AF_ARCURVERS))
	    {
	      fprintf (stderr, "invalid archive file version to be produced, must be between 2 and %d\n", AF_ARCURVERS);
	      exit (1);
	    }
	  break;
	case 'e': /* edit mode  */
	  verbosityLevel = EDIT;
	  break;
	case 'n': /* don't ask mode  */
	  verbosityLevel = NOASK;
	  break;
	case 'q': /* really quiet mode  */
	  verbosityLevel = SILENT;
	  break;
	case 'v': /* print current version of this program */
	  (void) printf ("This is %s version %s.\n", argv[0], af_version());
	  exit (0);
	default:
	  usage ();
	  exit (1);
	}
    }  /* end of command line parsing */
  
  (void) signal (SIGINT, sigHand);

  if (repairCache)
    if (setjmp (env) == 0)
      (void) repair ("object_cache");

  nfiles = argc - optind;
  if (nfiles == 0) /* no arguments given */
    /* repair all archive files in AtFS subdirectory */
    {
      if (repairCache)
	exit (0);

      if ((dirPointer = opendir (af_ardir (af_uniqpath (".")))) == (DIR *)0)
	exit (0);

      while ((dirEntry = readdir (dirPointer)) != (struct direct *)0)
	{
	  len = strlen (dirEntry->d_name);
	  if (af_isarchive (dirEntry->d_name) &&
	      (dirEntry->d_name[len - 1] == AF_ARCHEXT))
	    {
	      (void) strcpy (fileName, dirEntry->d_name+strlen(AF_ATFSFILEID));
	      len = strlen (fileName) - 1;
	      fileName[len] = '\0';
#if (MAXNAMLEN < 128)
	      /* On System V machines, the name might be incomplete */
	      if (len == (MAXNAMLEN-3)) /* take a look at the archive */
		{
		  (void) strcpy (fileName, afReadName (dirEntry->d_name));
		}
#endif
	      if (setjmp (env) == 0)
		(void) repair (fileName);
	    }
	}
    }
  else
    {
      for (i = 0; i < nfiles; i++)
	{
	  if (setjmp (env) == 0)
	    (void) repair (argv[i+optind]);
	}
    }
  return (0);
} /* end of main */

usage ()
{
  fprintf (stderr, "Usage: atfsrepair [-C] [-V arch_format_vers] [-e] [-q] [-v] files\n");
}

char lckFilename[MAXPATHLEN];

errorExit ()
{
  (void) unlink (lckFilename);
  exit (1);
}

cleanup ()
{
  (void) unlink (lckFilename);
  fprintf (stderr, "abort processing of this file...\n");
  longjmp (env, 1);
}

sigHand ()
{
  (void) unlink (lckFilename);
  fprintf (stderr, "abort processing of this file...stop totally (y/n) ? ");
  if (askConfirm ("y"))
    exit (0);
  longjmp (env, 1);
}

repair (filename)
     char *filename;
{
  char *fullname = af_uniqpath (filename), *sPtr, commandLine[1024];
  char *arFilename, datFilename[MAXPATHLEN], savFilename[MAXPATHLEN];
  char arTmpFilename[MAXPATHLEN], datTmpFilename[MAXPATHLEN];
  register int i, j;
  Af_user *owner, *af_garown();
  bool error, writeOk, confirm, busyEx = FALSE;
  FILE *inFile, *tmpFile, *lckFile;
  struct stat ibuf;
  int size, arVersion=0;
  off_t newDeltaSize;
  Gid_t arGid;

  if (!strcmp (filename, "object_cache"))
    {
      if (NORMAL <= verbosityLevel)
	{
	  fprintf (stdout, "cannot repair derived object caches yet -- \nthe only way ");
	  fprintf (stdout, "to fix a derived object cache is to clear it ! (y/n) ? ");
	  if (askConfirm ("y"))
	    {
	      fprintf (stdout, "clearing derived object cache...\n", filename);
	      (void) sprintf (commandLine, "rm -f %s/%s* %s/%s*\0", AF_SUBDIR, AF_BPFILEID, AF_OLDSUBDIR, AF_BPFILEID);
	      (void) system (commandLine);
	    }
	}
      else
	{
	  (void) sprintf (commandLine, "rm -f %s/%s* %s/%s*\0", AF_SUBDIR, AF_BPFILEID, AF_OLDSUBDIR, AF_BPFILEID);
	  (void) system (commandLine);
	}
      longjmp (env, 1);
    }

  cAttrs.host = af_gethostname ();
  cAttrs.syspath = af_afpath (fullname);
  cAttrs.name = af_afname (fullname);
  cAttrs.type = af_aftype (fullname);
  
  if (SILENT < verbosityLevel)
    fprintf (stdout, "%s:\n", af_unixname ((char *)0, cAttrs.name, cAttrs.type));

  arFilename = af_garname (cAttrs.syspath, cAttrs.name, cAttrs.type);
	   
  if ((owner = af_garown (arFilename, &writeOk, &arGid)) == (Af_user *)0)
    {
      if (NORMAL <= verbosityLevel)
	fprintf (stdout, "cannot determine owner of AtFS subdirectory !!\n");
      owner = af_afuser ((Uid_t) geteuid());
    }
  cAttrs.ownerName = af_entersym (owner->af_username);
  cAttrs.ownerHost = af_enterhost (owner->af_userhost);
  cAttrs.ownerDomain = af_enterdomain (owner->af_userdomain);

  /* look for Lockfile */
  (void) strcpy (lckFilename, arFilename);
  lckFilename [strlen (lckFilename) - sizeof (char)] = AF_LCKEXT;
  if (stat (lckFilename, &ibuf) != ERROR) /* lock file present */
    {
      if (NORMAL <= verbosityLevel)
	{
	  fprintf (stdout, "Archive file is locked!\n");
	  fprintf (stdout, "   This might be a spurious lockfile,\n");
	  fprintf (stdout, "   or another application is currently working on the archive file.\n");
	  fprintf (stdout, "   Ignore the lock (y/n) ? ");
	  if (!askConfirm ("y"))
	    longjmp (env, 1);
	}
    }
  else /* create lockfile */
    {
      lckFile = fopen (lckFilename, "w");
      (void) fclose (lckFile);
    }

  /* read the two archive files (if present) */
  if (inFile = fopen (arFilename, "r"))
    {
      if ((attrIn.length = af_retfsize (arFilename)) == 0)
	{
	  fprintf (stderr, "Attributes archive file is empty!\n");
	  cleanup ();
	}
      if ((attrIn.string = malloc ((unsigned) attrIn.length)) == (char *)0)
	{
	  fprintf (stderr, "repair (malloc): not enough memory\n");
	  errorExit ();
	}
      (void) fread (attrIn.string, sizeof (char), (Size_t) attrIn.length, inFile);
      (void) fclose (inFile);
      attrIn.curPos = 0;
    }
  else
    {
      /* no archive -- try to find temporary archive file */
      (void) strcpy (arTmpFilename, arFilename);
      arTmpFilename[strlen(arTmpFilename)-sizeof(char)] = AF_ARCHTMP;
      if (inFile = fopen (arTmpFilename, "r"))
	{
	  if ((attrIn.length = af_retfsize (arTmpFilename)) == 0)
	    {
	      fprintf (stderr, "Data archive file is empty!\n");
	      cleanup ();
	    }
	  if ((attrIn.string = malloc ((unsigned) attrIn.length)) == (char *)0)
	    {
	      fprintf (stderr, "repair (malloc): not enough memory\n");
	      errorExit ();
	    }
	  (void) fread (attrIn.string, sizeof (char), (Size_t) attrIn.length, inFile);
	  (void) fclose (inFile);
	  attrIn.curPos = 0;
	}
    }
	   
  (void) strcpy (datFilename, arFilename);
  datFilename[strlen(datFilename)-sizeof(char)] = AF_DATAEXT;
  if (inFile = fopen (datFilename, "r"))
    {
      dataIn.length = af_retfsize (datFilename);
      if ((dataIn.string = malloc ((unsigned) dataIn.length)) == (char *)0)
	{
	  fprintf (stderr, "repair (malloc): not enough memory\n");
	  errorExit ();
	}
      (void) fread (dataIn.string, sizeof (char), (Size_t) dataIn.length, inFile);
      (void) fclose (inFile);
    }
  else
    {
      (void) strcpy (datTmpFilename, arFilename);
      datTmpFilename[strlen(datTmpFilename)-sizeof(char)] = AF_DATATMP;
      if (inFile = fopen (datTmpFilename, "r"))
	{
	  dataIn.length = af_retfsize (datTmpFilename);
	  if ((dataIn.string = malloc ((unsigned) dataIn.length)) == (char *)0)
	    {
	      fprintf (stderr, "repair (malloc): not enough memory\n");
	      errorExit ();
	    }
	  (void) fread (dataIn.string, sizeof (char), (Size_t) dataIn.length, inFile);
	  (void) fclose (inFile);
	}
      else
	{
	  if (attrIn.string == (char *)0)
	    {
	      if (SILENT < verbosityLevel)
		fprintf (stdout, "*** No archive files for %s ***\n", af_unixname ((char *)0, cAttrs.name, cAttrs.type));
	      longjmp (env, 1);
	    }
	}
    }

  /* test existence of busy file */
  if (access (fullname, F_OK) == 0)
    busyEx = TRUE;

  /*=========
   * Phase 1
   *=========*/
  if (SILENT < verbosityLevel)
    fprintf (stdout, "*** Phase 1 (check version independent attributes) ***\n");

  /* try to determine archive version */
  if (!strncmp (attrIn.string, AF_ARHEADER, AF_SEGSTRLEN))
    arVersion = atoi (&(attrIn.string[AF_SEGSTRLEN+1]));
  
  (void) lookup (&attrIn, AF_NAMEID, 0);

  (void) nextItem (&attrIn); /* host */
  sPtr = verifyString ("hostname", attrIn, EDIT);
  if (strcmp (sPtr, cAttrs.host))
    sPtr = complainString ("hostname", sPtr, cAttrs.host, NORMAL);
  cAttrs.host = af_entersym (sPtr);
	
  (void) nextItem (&attrIn); /* syspath */
  sPtr = verifyString ("syspath", attrIn, EDIT);
  if (strcmp (sPtr, cAttrs.syspath))
    sPtr = complainString ("syspath", sPtr, cAttrs.syspath, NORMAL);
  cAttrs.syspath = af_entersym (sPtr);
	
  (void) nextItem (&attrIn); /* name */
  sPtr = verifyString ("name", attrIn, EDIT);
  if (strcmp (sPtr, cAttrs.name))
    sPtr = complainString ("name", sPtr, cAttrs.name, NORMAL);
  cAttrs.name = af_entersym (sPtr);
	
  (void) nextItem (&attrIn); /* type */
  sPtr = verifyString ("type", attrIn, EDIT);
  if (strcmp (sPtr ? sPtr : "", cAttrs.type))
    sPtr = complainString ("type", sPtr, cAttrs.type, NORMAL);
  cAttrs.type = af_entersym (sPtr);

  (void) lookup (&attrIn, AF_OWNID, 0);

  (void) nextItem (&attrIn); /* owner's name */
  sPtr = verifyString ("owner's name", attrIn, EDIT);
  if (strcmp (sPtr, cAttrs.ownerName))
    sPtr = complainString ("owner's name", sPtr, cAttrs.ownerName, NORMAL);
  cAttrs.ownerName = af_entersym (sPtr);

  (void) nextItem (&attrIn); /* owner's host */
  sPtr = verifyString ("owner's host", attrIn, EDIT);
  if (strcmp (sPtr, cAttrs.ownerHost))
    sPtr = complainString ("owner's host", sPtr, cAttrs.ownerHost, NORMAL);
  cAttrs.ownerHost = af_entersym (sPtr);

  if (arVersion > 2)
    {
      (void) nextItem (&attrIn); /* owner's domain */
      sPtr = verifyString ("owner's domain", attrIn, EDIT);
      if (strcmp (sPtr, cAttrs.ownerDomain))
	sPtr = complainString ("owner's domain", sPtr, cAttrs.ownerDomain, NORMAL);
    }
  else
    {
      sPtr = af_getdomain();
    }
  cAttrs.ownerDomain = af_entersym (sPtr);


  /*=========
   * Phase 2
   *=========*/
  if (SILENT < verbosityLevel)
    fprintf (stdout, "*** Phase 2 (check version attributes) ***\n");

  /* initialize busy version --- not *all* attributes get initialized !!! */
  revList[0].generation = AF_BUSYVERS;
  revList[0].revision = AF_BUSYVERS;
  (void) lookup (&attrIn, AF_NAMEID, 0);
  (void) nextItem (&attrIn); (void) nextItem (&attrIn);
  (void) nextItem (&attrIn); (void) nextItem (&attrIn);
  (void) nextItem (&attrIn); /* variant (ignore) */

  (void) lookup (&attrIn, AF_LOCKID, 0);
  (void) nextItem (&attrIn);
  sPtr = verifyString ("locker's name", attrIn, EDIT);
  revList[0].lockerName = af_entersym (sPtr);
  (void) nextItem (&attrIn);
  sPtr = verifyString ("locker's host", attrIn, EDIT);
  revList[0].lockerHost = af_entersym (sPtr);
  if (arVersion > 2)
    {
      (void) nextItem (&attrIn);
      sPtr = verifyString ("locker's domain", attrIn, EDIT);
    }
  else
    {
      sPtr = af_getdomain ();
    }
  revList[0].lockerDomain = af_entersym (sPtr);

  if ((revList[0].lockerName == (char *)0) && revList[0].lockerHost)
    {
      if (SILENT < verbosityLevel)
	fprintf (stdout, "Warning: lockerID inconsistent -- lock cancelled\n");
      revList[0].lockerName = (char *)0;
      revList[0].lockerHost = (char *)0;
      revList[0].lockerDomain = (char *)0;
    }
  if (revList[0].lockerName == (char *)0)
    revList[0].lockerDomain = (char *)0;

  if (revList[0].lockerName && (revList[0].lockerHost == (char *)0))
    {
      if (SILENT < verbosityLevel) 
	fprintf (stdout,"locker's host missing - inserting author's host\n");
      revList[0].lockerHost = revList[0].authorHost;
    }
  if (revList[0].lockerName && (revList[0].lockerDomain == (char *)0))
    {
      if (SILENT < verbosityLevel)
	fprintf (stdout,"locker's domain missing - inserting author's domain\n");
      revList[0].lockerHost = revList[0].authorDomain;
    }

  (void) nextItem (&attrIn);
  revList[0].lockTime = verifyDate ("locking date", attrIn, EDIT);
  if (revList[0].lockerName && (revList[0].lockTime == AF_NOTIME))
    {
      if (SILENT < verbosityLevel)
	fprintf (stdout, "Warning: locking date inconsistent -- %s\n", "setting actual date");
      revList[0].lockTime = af_acttime();
    }

  (void) lookup (&attrIn, AF_PRDID, 0);
  (void) nextItem (&attrIn);
  revList[0].predGen =
    verifyInt ("pred(gen) of busy version", attrIn, EDIT);
  (void) nextItem (&attrIn);
  revList[0].predRev =
    verifyInt ("pred(rev) of busy version", attrIn, EDIT);

  attrIn.curPos = 0;
  curRev = 0;
  while (lookup (&attrIn, AF_REVID, 1) != -1)
    {
      curRev++;
      (void) nextItem (&attrIn);
      revList[curRev].generation = verifyInt ("gen", attrIn, EDIT);
      (void) nextItem (&attrIn);
      revList[curRev].revision = verifyInt ("rev", attrIn, EDIT);
      (void) nextItem (&attrIn);
      revList[curRev].state = verifyInt ("state", attrIn, EDIT);
      (void) nextItem (&attrIn);
      revList[curRev].mode = (u_short) verifyOct ("mode", attrIn, EDIT);
      (void) nextItem (&attrIn); /* variant (ignore) */
      
      (void) lookup (&attrIn, AF_AUTHORID, 1);
      (void) nextItem (&attrIn); /* author's name */
      sPtr = verifyString ("author's name", attrIn, EDIT);
      if (sPtr == (char *)0)
	{
	  if (SILENT < verbosityLevel)
	    fprintf (stdout,"author's name missing -- inserting owner's name\n");
	  revList[curRev].authorName = cAttrs.ownerName;
	}
      else
	revList[curRev].authorName = af_entersym (sPtr);
      
      (void) nextItem (&attrIn); /* author's host */
      sPtr = verifyString ("author's host", attrIn, EDIT);
      if (sPtr == (char *)0)
	{
	  if (SILENT < verbosityLevel)
	    fprintf (stdout,"author's host missing -- inserting owner's host\n");
	  revList[curRev].authorHost = cAttrs.ownerHost;
	}
      else
	revList[curRev].authorHost = af_entersym (sPtr);

      if (arVersion > 2)
	{
	  (void) nextItem (&attrIn); /* author's domain */
	  sPtr = verifyString ("author's domain", attrIn, EDIT);
	  if (sPtr == (char *)0)
	    {
	      if (SILENT < verbosityLevel)
		fprintf (stdout,"author's domain missing -- inserting owner's domain\n");
	      revList[curRev].authorDomain = cAttrs.ownerDomain;
	    }
	  else
	    revList[curRev].authorDomain = af_entersym (sPtr);
	}
      else
	{
	  sPtr = af_getdomain ();
	  revList[curRev].authorDomain = af_entersym (sPtr);
	}

      (void) nextItem (&attrIn); /* locker's name */
      sPtr = verifyString ("locker's name", attrIn, EDIT);
      revList[curRev].lockerName = af_entersym (sPtr);

      (void) nextItem (&attrIn); /* locker's host */
      sPtr = verifyString ("locker's host", attrIn, EDIT);
      if ((sPtr == (char *)0) && (revList[curRev].lockerName != (char *)0))
	{
	  if (SILENT < verbosityLevel)
	    fprintf (stdout,"locker's host missing - inserting author's host\n");
	  revList[curRev].lockerHost = revList[curRev].authorHost;
	}
      else
	revList[curRev].lockerHost = af_entersym (sPtr);

      if (arVersion > 2)
	{
	  (void) nextItem (&attrIn); /* locker's domain */
	  sPtr = verifyString ("locker's domain", attrIn, EDIT);
	  if ((sPtr == (char *)0) && (revList[curRev].lockerName != (char *)0))
	    {
	      if (SILENT < verbosityLevel)
		fprintf (stdout,"locker's domain missing - inserting author's domain\n");
	      revList[curRev].lockerDomain = revList[curRev].authorDomain;
	    }
	  else
	    revList[curRev].lockerDomain = af_entersym (sPtr);
	}
      else
	{
	  sPtr = af_getdomain();
	  revList[curRev].lockerDomain = af_entersym (sPtr);
	}

      (void) lookup (&attrIn, AF_DATEID, 1);
      (void) nextItem (&attrIn);
      revList[curRev].modTime = verifyDate ("mod. date", attrIn, EDIT);
      if (revList[curRev].modTime == AF_NOTIME)
	{
	  if (SILENT < verbosityLevel)
	    fprintf (stdout, "Warning: modification date missing -- %s\n", "inserting actual date");
	  revList[curRev].modTime = af_acttime();
	}
      (void) nextItem (&attrIn);
      revList[curRev].accessTime = verifyDate ("acc. date", attrIn,EDIT);
      if (revList[curRev].accessTime == AF_NOTIME)
	{
	  if (SILENT < verbosityLevel)
	    fprintf (stdout, "Warning: access date missing -- %s\n", "inserting actual date");
	  revList[curRev].accessTime = af_acttime();
	}
      (void) nextItem (&attrIn);
      revList[curRev].statChangeTime =
	verifyDate ("status change date", attrIn, EDIT);
      if (revList[curRev].statChangeTime == AF_NOTIME)
	{
	  if (SILENT < verbosityLevel)
	    fprintf (stdout, "Warning: status change date missing -- %s\n", "inserting actual date");
	  revList[curRev].statChangeTime = af_acttime();
	}
      (void) nextItem (&attrIn);
      revList[curRev].saveTime = verifyDate ("save date", attrIn, EDIT);
      if (revList[curRev].saveTime == AF_NOTIME)
	{
	  if (SILENT < verbosityLevel)
	    fprintf (stdout, "Warning: save date missing -- %s\n", "inserting actual date");
	  revList[curRev].saveTime = af_acttime();
	}
      (void) nextItem (&attrIn);
      revList[curRev].lockTime = verifyDate ("lock date", attrIn, EDIT);
      if ((revList[curRev].lockTime == AF_NOTIME) &&
	  (revList[curRev].lockerName != (char *)0))
	{
	  if (SILENT < verbosityLevel)
	    fprintf (stdout, "Warning: locking date missing -- %s\n", "inserting actual date");
	  revList[curRev].lockTime = af_acttime();
	}

      (void) lookup (&attrIn, AF_REPRID, 1);
      (void) nextItem (&attrIn);
      revList[curRev].representation = verifyInt ("repr", attrIn, EDIT);
      (void) nextItem (&attrIn);
      revList[curRev].fileSize = verifyInt ("filesize", attrIn, EDIT);
      (void) nextItem (&attrIn);
      revList[curRev].deltaSize = verifyInt ("deltasize", attrIn, EDIT);
      (void) nextItem (&attrIn);
      revList[curRev].succGen = verifyInt ("succ. gen", attrIn, EDIT);
      (void) nextItem (&attrIn);
      revList[curRev].succRev = verifyInt ("succ. rev", attrIn, EDIT);
      (void) nextItem (&attrIn);
      revList[curRev].predGen = verifyInt ("pred. gen", attrIn, EDIT);
      (void) nextItem (&attrIn);
      revList[curRev].predRev = verifyInt ("pred. rev", attrIn, EDIT);
    } /* revision loop */

  /*=========
   * Phase 3
   *=========*/
  if (SILENT < verbosityLevel)
    fprintf (stdout, "*** Phase 3 (check user defined attributes) ***\n");

  attrIn.curPos = 0;
  curUda = 0;
  while (lookup (&attrIn, AF_UDAID, 1) != -1)
    {
      (void) nextItem (&attrIn);
      udaList[curUda].generation = verifyInt ("gen", attrIn, EDIT);
      (void) nextItem (&attrIn);
      udaList[curUda].revision = verifyInt ("rev", attrIn, EDIT);
      (void) nextItem (&attrIn);
      i = 0;
      size = 0;
      while (i < MAXUDAS)
	{
	  udaList[curUda].uda[i] = &attrIn.string[attrIn.curPos];
	  udaList[curUda].size = udaList[curUda].size + 
	    (strlen (udaList[curUda].uda[i]) + sizeof (char));
	  while (attrIn.string[attrIn.curPos] != '\0')
	    attrIn.curPos++;
	  attrIn.curPos++;
	  i++;
	  if (!strcmp (&attrIn.string[attrIn.curPos], AF_UDAID))
	    break;
	  if (attrIn.string[attrIn.curPos] == '\0')
	    {
	      udaList[curUda].size++;
	      break;
	    }
	} /* uda loop */
      udaList[curUda].uda[i] = (char *)0;
      curUda++;
    } /* revision loop */

  /*=========
   * Phase 4
   *=========*/
  if (SILENT < verbosityLevel)
    fprintf (stdout, "*** Phase 4 (check notes) ***\n");

  dataIn.curPos = 0;
  curNote = 0;
  while (lookup (&dataIn, AF_NOTEID, 1) != -1)
    {
      (void) nextItem (&dataIn);
      noteList[curNote].generation = verifyInt ("gen", dataIn, EDIT);
      (void) nextItem (&dataIn);
      noteList[curNote].revision = verifyInt ("rev", dataIn, EDIT);
      (void) nextItem (&dataIn);
      (void) nextLine (&dataIn); /* skip size */
      size = 0;
      noteList[curNote].contents = &dataIn.string[dataIn.curPos];
      while (((noteList[curNote].contents)[size] != AF_DATAID[0]) &&
	     ((noteList[curNote].contents)[size] != '\0'))
	size++;
      if (size == 0)
	{
	  noteList[curNote].contents = (char *)0;
	  size = 1;
	}
      else
	(noteList[curNote].contents)[size-1] = '\0';
      noteList[curNote].size = size;
      curNote++;
    } /* revision loop */

  /*=========
   * Phase 5
   *=========*/
  if (SILENT < verbosityLevel)
    fprintf (stdout, "*** Phase 5 (check data) ***\n");

  /* try to determine archive version */
  if (!strncmp (dataIn.string, AF_DATAHEADER, AF_SEGSTRLEN))
    arVersion = atoi (&(dataIn.string[AF_SEGSTRLEN+1]));
  
  dataIn.curPos = 0;
  curData = 0;
  while (lookup (&dataIn, AF_DATAID, 1) != -1)
    {
      (void) nextItem (&dataIn);
      dataList[curData].generation = verifyInt ("gen", dataIn, EDIT);
      (void) nextItem (&dataIn);
      dataList[curData].revision = verifyInt ("rev", dataIn, EDIT);
      (void) nextItem (&dataIn);
      dataList[curData].representation = verifyInt ("repr",dataIn, EDIT);
      (void) nextItem (&dataIn);
      (void) nextLine (&dataIn); /* skip size */
      size = 0;
      if (dataIn.curPos == dataIn.length)
	goto loopexit;
      dataList[curData].contents = &dataIn.string[dataIn.curPos];
      while (itemCmp (&(dataList[curData].contents)[size],
		      AF_NOTEID, AF_IDSTRLEN) &&
	     itemCmp (&(dataList[curData].contents)[size],
		      AF_DATAID, AF_IDSTRLEN))
	{
	  size++;
	  while ((dataList[curData].contents)[size] != AF_NOTEID[0])
	    {
	      size++;
	      if ((dataIn.curPos + size) == dataIn.length)
		goto loopexit;
	    }
	}
    loopexit:
      if (size == 0)
	dataList[curData].contents = (char *)0;
      dataList[curData].size = size;

      /* convert delta */
      if ((size > 0) && (arVersion == 1) && (dataList[curData].representation == AF_DELTA))
	{
	  char *afConvertDelta();
	  
	  dataList[curData].contents = afConvertDelta (dataList[curData].contents, dataList[curData].size, &newDeltaSize);
	  dataList[curData].size = newDeltaSize;
	}

      /* check plausibility of delta */
      if ((dataList[curData].size > 0) && 
	  (dataList[curData].representation == AF_DELTA))
	{
	  if ((dataList[curData].contents[0] != '\0') &&
	      (dataList[curData].contents[0] != '\01'))
	    {
	      dataList[curData].representation = AF_CHUNK;
	      if (SILENT < verbosityLevel)
		fprintf (stderr, "found bogus delta -- fixed !\n");
	    }
	}
      curData++;
    } /* revision loop */

  /*=========
   * Phase 6
   *=========*/
  if (SILENT < verbosityLevel)
    fprintf (stdout, "*** Phase 6 (check connectivity) ***\n");

  /* test order of revisions -- not yet implemented */
  /* test existence on successors and predecessors -- not yet implemented */

  /* look for zero size files (due to known AtFS error) */
  for (i=1; i<curRev; i++)
    {
      if ((revList[i].fileSize == 0) && (revList[i].representation == AF_DELTA))
	{
	  j = i;
	  while (revList[j++].fileSize == 0);
	  revList[i].fileSize = revList[j-1].fileSize;
	}
    }

  /* calculate number of revisions and size of data */
  cAttrs.noOfRevisions = curRev;

  cAttrs.datasize = 0;
  for (i=0; i<curRev; i++)
    cAttrs.datasize = cAttrs.datasize + noteList[i].size;
  for (i=0; i<curRev; i++)
    cAttrs.datasize = cAttrs.datasize + dataList[i].size;

  /* test integrity of attr and data file */
  error = FALSE;
  for (i=0; i<=cAttrs.noOfRevisions; i++)
    {
      if ((udaList[i].generation != revList[i].generation) ||
	  (udaList[i].revision != revList[i].revision))
	{
	  error = TRUE;
	  udaList[i].generation = revList[i].generation;
	  udaList[i].revision = revList[i].revision;
	}
    }
    
  for (i=0; i<cAttrs.noOfRevisions; i++)
    {
      if ((noteList[i].generation != revList[i+1].generation) ||
	  (noteList[i].revision != revList[i+1].revision))
	{
	  error = TRUE;
	  noteList[i].generation = revList[i+1].generation;
	  noteList[i].revision = revList[i+1].revision;
	}
      if ((dataList[i].generation != revList[i+1].generation) ||
	  (dataList[i].revision != revList[i+1].revision))
	{
	  error = TRUE;
	  dataList[i].generation = revList[i+1].generation;
	  dataList[i].revision = revList[i+1].revision;
	}
    }
  if (error)
    if (SILENT < verbosityLevel)
      fprintf (stderr, "version numbering inconsistent -- fixed !\n");

  /* ensure delta type consistency */
  for (i=0; i<cAttrs.noOfRevisions; i++)
    revList[i+1].representation = dataList[i].representation;


  /*==============================
   * Write temporary archive files
   *==============================*/

  /* if all revisions have been removed */
  if (!busyEx && (cAttrs.noOfRevisions == 0))
    {
      if (SILENT < verbosityLevel)
	fprintf (stdout, "*** no revisions found ! ***\n");
      (void) unlink (arFilename);
      (void) unlink (datFilename);
      (void) unlink (lckFilename);
      longjmp (env, 1);
    }

  /* open tmpfile */
  (void) strcpy (arTmpFilename, arFilename);
  arTmpFilename[strlen(arTmpFilename)-sizeof(char)] = AF_ARCHTMP;
  if ((tmpFile = fopen (arTmpFilename, "w")) == (FILE *)0)
    {
      fprintf (stderr, "cannot open tmp file\n");
      cleanup ();
    }

  /* write header */
  fprintf (tmpFile, "%s %d %d %ld\n", AF_ARHEADER, outputVersion,
	   cAttrs.noOfRevisions + 1, cAttrs.datasize);

  /* write constant attributes */
  if (busyEx)
    {
      fprintf (tmpFile, "%s %s %s %s %s %s\n", AF_NAMEID, 
	       cAttrs.host, cAttrs.syspath, cAttrs.name, NOTMT (cAttrs.type),
	       AF_NOSTRING); /* former variant string */
    }
  else
    {
      fprintf (tmpFile, "%s %s %s %s %s %s\n", AF_NAMEID, cAttrs.host,
	       cAttrs.syspath, cAttrs.name, NOTMT (cAttrs.type), AF_NOSTRING);
    }

  /* write owner */
  if (outputVersion == 2)
    {
      fprintf (tmpFile,
	       "%s %s %s\n", AF_OWNID, cAttrs.ownerName, cAttrs.ownerHost);
    }
  else
    {
      fprintf (tmpFile,
	       "%s %s %s %s\n", AF_OWNID, cAttrs.ownerName, cAttrs.ownerHost, cAttrs.ownerDomain);
    }

  /* write predecessor of busy version */
  if (busyEx)
    {
      fprintf (tmpFile, "%s %d %d\n", AF_PRDID,
	       revList[0].predGen, revList[0].predRev);
    }
  else
    {
      fprintf (tmpFile, "%s %d %d\n", AF_PRDID, AF_NOVNUM, AF_NOVNUM);
      if (revList[0].predGen != AF_NOVNUM)
	if (SILENT < verbosityLevel)
	  fprintf (stdout, "no busy version -- cancelling predecessor information !\n");
    }

  /* write locker of busy version */
  if (outputVersion == 2)
    {
      fprintf (tmpFile, "%s %s %s %d\n", AF_LOCKID,
	       NOTMT (revList[0].lockerName), NOTMT (revList[0].lockerHost),
	       revList[0].lockTime);
    }
  else
    {
      fprintf (tmpFile, "%s %s %s %s %d\n", AF_LOCKID,
	       NOTMT (revList[0].lockerName), NOTMT (revList[0].lockerHost),
	       NOTMT (revList[0].lockerDomain), revList[0].lockTime);
    }

  /* write list of version attributes */
  for (i=1; i <= cAttrs.noOfRevisions; i++)
    {
      /* write revision ID */
      fprintf (tmpFile, "%s %d %d %d %o %s\n", AF_REVID, 
	       revList[i].generation, revList[i].revision, revList[i].state,
	       revList[i].mode, AF_NOSTRING); /* former variant string */

      /* write author */
      if (outputVersion == 2)
	{
	  fprintf (tmpFile, "\t%s %s %s %s %s\n", AF_AUTHORID,
		   revList[i].authorName, revList[i].authorHost,
		   NOTMT (revList[i].lockerName), NOTMT (revList[i].lockerHost));
	}
      else
	{
	  fprintf (tmpFile, "\t%s %s %s %s %s %s %s\n", AF_AUTHORID,
		   revList[i].authorName, revList[i].authorHost,
		   revList[i].authorDomain, NOTMT (revList[i].lockerName),
		   NOTMT (revList[i].lockerHost), NOTMT (revList[i].lockerDomain));
	}
      /* write dates */
      fprintf (tmpFile, "\t%s %ld %ld %ld %ld %ld\n", AF_DATEID, 
	       revList[i].modTime, revList[i].accessTime, 
	       revList[i].statChangeTime, revList[i].saveTime,
	       revList[i].lockTime);

      /* write kind of representation and tree connects */
      fprintf (tmpFile, "\t%s %d %ld %ld %d %d %d %d\n", AF_REPRID,
	       revList[i].representation, revList[i].fileSize,
	       (revList[i].representation == AF_DELTA) ? dataList[i-1].size :
	       revList[i].deltaSize, revList[i].succGen, revList[i].succRev,
	       revList[i].predGen, revList[i].predRev);
    }

  /* write user defined attributes */
  fprintf (tmpFile, "%s\n", AF_UDASEG);

  for (i=0; i <= cAttrs.noOfRevisions; i++)
    {
      fprintf (tmpFile, "%s %d %d\n", AF_UDAID, 
	       udaList[i].generation, udaList[i].revision);
      j=0;
      while (udaList[i].uda[j])
	fprintf (tmpFile, "%s%c", udaList[i].uda[j++], '\0');
      if (j==0) /* if no user defined attribute has been written */
	(void) putc ('\0', tmpFile);
      (void) putc ('\0', tmpFile);
      (void) putc ('\n', tmpFile);
    }
  (void) fclose (tmpFile);
  /* release attrIn */
  (void) free (attrIn.string);
  attrIn.string = (char *)0;
  attrIn.length = 0;
  attrIn.curPos = 0;
      

  /* open datatmpfile */
  (void) strcpy (datTmpFilename, arFilename);
  datTmpFilename[strlen(datTmpFilename)-sizeof(char)] = AF_DATATMP;
  if ((tmpFile = fopen (datTmpFilename, "w")) == (FILE *)0)
    {
      fprintf (stderr, "cannot open tmp file\n");
      cleanup ();
    }

  fprintf (tmpFile, "%s %d\n", AF_DATAHEADER, AF_ARCURVERS);
      
  for (i=0; i < cAttrs.noOfRevisions; i++)
    {
      fprintf (tmpFile, "%s %d %d %ld\n", AF_NOTEID,
	       noteList[i].generation, noteList[i].revision, noteList[i].size);
      (void) fwrite (noteList[i].contents, sizeof(char), (Size_t)(noteList[i].size - sizeof(char)), tmpFile);
      (void) putc ('\n', tmpFile);

      fprintf (tmpFile, "%s %d %d %d %ld\n", AF_DATAID,
	       dataList[i].generation, dataList[i].revision,
	       dataList[i].representation, dataList[i].size);

      (void) fwrite (dataList[i].contents, sizeof(char), (Size_t)dataList[i].size, tmpFile);
    }
  (void) fclose (tmpFile);
  /* release dataIn */
  (void) free (dataIn.string);
  dataIn.string = (char *)0;
  dataIn.length = 0;
  dataIn.curPos = 0;
      

  /* test, if archive files are changed -- if not, exit */
  (void) sprintf (commandLine, "cmp -s %s %s\0", arFilename, arTmpFilename);
  if (system (commandLine) == 0) /* if files are identical */
    {
      (void) sprintf (commandLine, "cmp -s %s %s\0", datFilename, datTmpFilename);
      if (system (commandLine) == 0)
	{
	  (void) unlink (arTmpFilename);
	  (void) unlink (datTmpFilename);
	  (void) unlink (lckFilename);
	  if (SILENT < verbosityLevel)
	    fprintf (stdout, "*** archive files ok ! -- unchanged ***\n");
          return (AF_OK);
        }
    }

  switch (verbosityLevel)
    {
    case SILENT:
      confirm = TRUE;
      break;
    case NOASK:
      fprintf (stdout, "*** archive files inconsistent ! -- corrected ***\n");
      confirm = TRUE;
      break;
    case NORMAL:
    case EDIT:
      fprintf (stdout, "archive files for %s inconsistent - fix ? (y/n) ",
	       af_unixname ((char *)0, cAttrs.name, cAttrs.type));
      confirm = askConfirm ("y");
    }

  if (!confirm) 
    {
      (void) unlink (arTmpFilename);
      (void) unlink (datTmpFilename);
      (void) unlink (lckFilename);
      if (SILENT < verbosityLevel)
	fprintf (stdout, "*** No archive files written ***\n");
      return (AF_OK);
    }

  if (SILENT < verbosityLevel)
    fprintf (stdout, "*** Write new archive files ***\n");

  (void) strcpy (savFilename, arFilename);
  savFilename[strlen(savFilename)-sizeof(char)] = 'a';
  (void) unlink (savFilename);
  (void) link (arFilename, savFilename);
  (void) unlink (arFilename);
  if (link (arTmpFilename, arFilename) == ERROR)
    fprintf (stderr,"cannot create new archive file -- preserving tmp file\n");

  (void) chmod (arFilename, AF_ARCHMODE);
  (void) chown (arFilename, geteuid(), arGid);
  (void) unlink (arTmpFilename);

  savFilename[strlen(savFilename)-sizeof(char)] = 'd';
  (void) unlink (savFilename);
  (void) link (datFilename, savFilename);
  (void) unlink (datFilename);
  if (link (datTmpFilename, datFilename) == ERROR)
    fprintf (stderr,"cannot create new archive file -- preserving tmp file\n");
  (void) chmod (datFilename, AF_ARCHMODE);
  (void) chown (datFilename, geteuid(), arGid);
  (void) unlink (datTmpFilename);
  (void) unlink (lckFilename);

  return (AF_OK);
}

/*===========================================================================
 * askConfirm -- ask for confirmation
 *
 *==========================================================================*/

LOCAL askConfirm (expAnswer)
     char *expAnswer;
{
  char strBuf[256], answer[10], *cp;

  (void) fflush (stdin);
  (void) printf ("[%s] ", expAnswer);
  strBuf[0] = '\0';
  answer[0] = '\0';
  (void) gets (strBuf);
  (void) sscanf (strBuf, "%s", answer);
  if (answer[0] == '\0') return TRUE; /* assumption acknowledged */
  cp = answer;
  while (*cp ? (*cp++ |= ' ') : 0); /* make sure answer is lowercase */
  return !strncmp (expAnswer, answer, strlen (expAnswer));
}

/*===========================================================================
 * lookup, netItem, itemCmp
 * Functions for random positioning in input stream
 *
 *==========================================================================*/

lookup (input, searchStr, pos)
     Input *input;
     char  *searchStr;
     int   pos;
{
  if (pos == 0)
    input->curPos = 0;
  if (input->curPos == input->length)
    return (-1);

  while (itemCmp (&(input->string[input->curPos]),
		   searchStr, strlen (searchStr)))
    {
      input->curPos++;
      while (input->string[input->curPos] != searchStr[0])
	{
	  input->curPos++;
	  if (input->curPos == input->length)
	    return (-1);
	}
    }
  return (input->curPos);
}

nextItem (input)
     Input *input;
{
  if (input->curPos == input->length)
    return (-1);
  /* search next white space */
  while ((input->string[input->curPos] != ' ') && 
	 (input->string[input->curPos] != '\t') && 
	 (input->string[input->curPos] != '\n'))
    if (++input->curPos == input->length)
      return (-1);

  /* skip white spaces */
  while ((input->string[input->curPos] == ' ') || 
	 (input->string[input->curPos] == '\t') || 
	 (input->string[input->curPos] == '\n'))
    if (++input->curPos == input->length)
      return (-1);

  return (input->curPos);
}

nextLine(input)
     Input *input;
{
  if (input->curPos == input->length)
    return (-1);
  /* search beginning of next line */
  while (input->string[input->curPos] != '\n')
    if (++input->curPos == input->length)
      return (-1);

  /* skip newline */
  if (++input->curPos == input->length)
    return (-1);

  return (input->curPos);
}

char *itemCopy (str1, str2)
     char *str1, *str2;
{
  if (!strncmp (str2, AF_NOSTRING, 2))
    {
      str1[0] = '\0';
      return (str1);
    }

  while (*str1 = *str2)
    {
      if ((*str2 ==' ') || (*str2 =='\t') || (*str2 =='\n'))
	{ *str1 = '\0'; return (str1); }
      str1++;
      str2++;
    }
  return (str1);
}

itemCmp (baseStr, searchStr, len)
     char *baseStr, *searchStr;
     int  len;
{
  register int i=0;
  while (baseStr[i] == searchStr[i])
    {
      i++;
      if (i == len)
	{
	  if ((baseStr[i] == ' ') || (baseStr[i] == '\t') ||
	      (baseStr[i] == '\0') || (baseStr[i] == '\n'))
	    return (0);
	  else
	    return (1);
	}
    }
  return (1);
}

/*=========================================================================
 * verifyString, verifyInt, verifyDate
 * complainString
 * 
 *=========================================================================*/

char *verifyString (name, input, level)
     char  *name;
     Input input;
     int   level;
{
  static char value[256];

  (void) itemCopy (value, &input.string[input.curPos]);

  if (level <= verbosityLevel)
    {
      fprintf (stdout, "assume %s to be: ->%s<-, ok ? (y/n) ", name, value);
      if (askConfirm ("y"))
	goto outhere;
      fprintf (stdout, "enter new %s (*0* for empty string): ", name);
      (void) fscanf (stdin, "%s", value);
      if (!strncmp (value, "*0*", 3))
	return ((char *)0);
    }
 outhere:
  if (value[0])
    return (value);
  else
    return ((char *)0);
}
  
char *complainString (name, string1, string2, level)
     char  *name, *string1, *string2;
     int   level;
{
  if (level <= verbosityLevel)
    {
      fprintf (stdout,
	       "Inconsistency: value for %s is ->%s<- (should be ->%s<-), %s",
	       name, string1, string2, "fix ? (y/n) ");
      if (!askConfirm ("y"))
	return (string1);
    }
  return (string2);
}
  
verifyInt (name, input, level)
     char  *name;
     Input input;
     int   level;
{
  register int  value;

  value = atoi (&input.string[input.curPos]);

  if (level <= verbosityLevel)
    {
      fprintf (stdout, "assume %s to be: ->%d<-, ok ? (y/n) ", name, value);
      if (askConfirm ("y"))
	return (value);
      fprintf (stdout, "enter new %s: ", name);
      (void) fscanf (stdin, "%d", value);
    }
  return (value);
}
  
verifyOct (name, input, level)
     char  *name;
     Input input;
     int   level;
{
  int  value;

  (void) sscanf (&input.string[input.curPos], "%o", &value);

  if (level <= verbosityLevel)
    {
      fprintf (stdout, "assume %s to be: ->%o<-, ok ? (y/n) ", name, value);
      if (askConfirm ("y"))
	return (value);
      fprintf (stdout, "enter new %s: ", name);
      (void) fscanf (stdin, "%o", value);
    }
  return (value);
}
  
verifyDate (name, input, level)
     char  *name;
     Input input;
     int   level;
{
  register int  value;

  value = atoi (&input.string[input.curPos]);

  if (level <= verbosityLevel)
    {
      fprintf (stdout, "assume %s to be: ->%d<-, ok ? (y/n) ", name, value);
      if (askConfirm ("y"))
	return (value);
      fprintf (stdout, "enter new date: ");
      (void) fscanf (stdin, "%d", value);
    }
  return (value);
}
  
