/*LINTLIBRARY*/
/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 *	Shape/AtFS
 *
 *	afversions.c - operations on revisions
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP
 *					   andy@db0tui62.BITNET)
 *
 *	$Header: afvers.c[1.15] Tue Feb 11 19:23:34 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *	af_savebinary -- save derived file
 *	af_saverev -- save busy version
 *	af_newgen -- increase gen number and reset rev number
 *	af_setbusy -- set revision busy
 *	af_svnum -- set version number
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"

#ifdef MEMDEBUG
extern FILE *memprot;
#endif

/*====================================================================
 *    af_savebinary
 *
 *====================================================================*/

EXPORT af_savebinary (busykey, savekey)
     Af_key *busykey;
     Af_key *savekey; /* out */
{
  register int oldpredgen, oldpredrev, oldnrefs, retval;
  time_t oldatime;
  Af_revlist *afInitBpList();

  if (afAccessAso (busykey, AF_ATTRS))
    SFAIL ("savebinary", "", AF_EINVKEY, ERROR);
  
  if (VATTR(busykey).af_state != AF_BUSY)
    SFAIL ("savebinary", "", AF_ENOTBUSY, ERROR);

  if ((VATTR(busykey).af_mode & S_IFMT) != S_IFREG)
    SFAIL ("savebinary", "", AF_ENOTREGULAR, ERROR);

  if (VATTR(busykey).af_nrefs > 1)
    af_wng ("savebinary", "busykey has more than one reference");

  if (af_checkperm (busykey, AF_WORLD) == ERROR)
    return (ERROR);

  /* modify some attributes for saving */
  VATTR(busykey).af_state = AF_NOSTATE;
  VATTR(busykey).af_class |= AF_DERIVED;
  oldatime = VATTR(busykey).af_atime;
  VATTR(busykey).af_atime = af_acttime ();
  oldpredgen = VATTR(busykey).af_predgen;
  VATTR(busykey).af_predgen = AF_NOVNUM;
  oldpredrev = VATTR(busykey).af_predrev;
  VATTR(busykey).af_predrev = AF_NOVNUM;
  oldnrefs = VATTR(busykey).af_nrefs;
  VATTR(busykey).af_nrefs = 1;

  retval = AF_OK;
  if ((savekey->af_ldes = afInitBpList (CATTR(busykey).af_syspath))
      != (Af_revlist *)0)
    {
      if (afRplBpEntry ((Af_key *)0, busykey, savekey) != ERROR)
	{
	  savekey->af_ldes->af_access++;
#ifdef MEMDEBUG
  fprintf (memprot, "Access: %x,%d +1 (set)\ttotal: %d\n", 
	   savekey->af_ldes, savekey->af_lpos, savekey->af_ldes->af_access);
#endif
	}
      else
	retval = ERROR;
    }
  else
    retval = ERROR;

  /* restore attributes */
  VATTR(busykey).af_state = AF_BUSY;
  VATTR(busykey).af_class &= ~AF_DERIVED;
  VATTR(busykey).af_atime = oldatime;
  VATTR(busykey).af_predgen = oldpredgen;
  VATTR(busykey).af_predrev = oldpredrev;
  VATTR(busykey).af_nrefs = oldnrefs;

  return (retval);
}


/*====================================================================
 *    af_saverev
 *
 *====================================================================*/

EXPORT af_saverev (busykey, savekey, generation)
     Af_key *busykey;
     Af_key *savekey; /* out */
     int    generation;
{
  register Af_key *lastkey, *lastgenkey;
  Af_key predkey, *af_glastkey(), *af_glastgenkey();
  register Af_user *author;
  struct stat ibuf;
  
  if (afAccessAso (busykey, AF_ATTRS))
    SFAIL ("saverev", "", AF_EINVKEY, ERROR);
  
  if (VATTR(busykey).af_state != AF_BUSY)
    SFAIL ("saverev", "", AF_ENOTBUSY, ERROR);

  if ((VATTR(busykey).af_mode & S_IFMT) != S_IFREG)
    SFAIL ("saverev", "", AF_ENOTREGULAR, ERROR);

  if (busykey->af_ldes->af_nrevs > 1) /* there are saved versions */
    {
      lastkey = af_glastkey (busykey->af_ldes);
      if (generation != AF_LASTVERS)
	{
	  if (generation < 0)
	    SFAIL ("saverev", "invalid generation number", AF_EMISC, ERROR);
	  /* if generation is current generation */
	  if (VATTR(lastkey).af_gen == generation)
	    generation = AF_LASTVERS;
	  /* if generation is unknown */
	  if ((lastgenkey = af_glastgenkey (busykey->af_ldes, generation)) == (Af_key *)0)
	    SFAIL ("saverev", "invalid generation number", AF_EMISC, ERROR);
	}

      if (generation != AF_LASTVERS)
	{
	  if (af_checkperm (lastgenkey, AF_LOCKHOLDER) == ERROR)
	    SFAIL ("saverev", "", AF_ENOTLOCKED, ERROR);
	}
      else
	{
	  if (af_checkperm (lastkey, AF_LOCKHOLDER) == ERROR)
	    SFAIL ("saverev", "", AF_ENOTLOCKED, ERROR);
	}
    }

  /* get attribute buffer (pointed to by "savekey") for new version */
  /* invalidates "lastkey" */
  if (afNewAso (busykey->af_ldes, savekey, AF_SOURCE, generation) == ERROR)
    return (ERROR);

  /* set key and attributes of new version */
  VATTR(savekey).af_name = VATTR(busykey).af_name;
  VATTR(savekey).af_type = VATTR(busykey).af_type;
  VATTR(savekey).af_lckname = VATTR(busykey).af_lckname;
  VATTR(savekey).af_lckhost = VATTR(busykey).af_lckhost;
  VATTR(savekey).af_lckdomain = VATTR(busykey).af_lckdomain;
  VATTR(savekey).af_ltime = af_acttime ();
  
  if (busykey->af_ldes->af_nrevs == 1)
    {
      /* there is only one (the busy-) revision (..nrevs == 1) */
      VATTR(savekey).af_gen = AF_INITGEN;
      VATTR(savekey).af_rev = AF_INITREV;
      VATTR(savekey).af_predgen = AF_NOVNUM;
      VATTR(savekey).af_predrev = AF_NOVNUM;
      VATTR(savekey).af_succgen = AF_NOVNUM;
      VATTR(savekey).af_succrev = AF_NOVNUM;
      VATTR(busykey).af_predgen = AF_INITGEN;
      VATTR(busykey).af_predrev = AF_INITREV;
      if (af_nodelta (busykey, savekey) != AF_OK)
	return (ERROR);
    }
  else /* get some attributes from preceding revision */
    {
      lastkey = af_glastkey (busykey->af_ldes);
      if (generation == AF_LASTVERS)
	{
	  VATTR(savekey).af_gen = VATTR(lastkey).af_gen; 
	  VATTR(savekey).af_rev = VATTR(lastkey).af_rev+1;
	}
      else
	{
	  VATTR(savekey).af_gen = VATTR(lastgenkey).af_gen;
	  VATTR(savekey).af_rev = VATTR(lastgenkey).af_rev+1;;
	}
      if (VATTR(busykey).af_predgen == AF_NOVNUM)
	{
	  VATTR(savekey).af_predgen = VATTR(lastkey).af_gen;
	  VATTR(savekey).af_predrev = VATTR(lastkey).af_rev;
	}
      else
	{
	  VATTR(savekey).af_predgen = VATTR(busykey).af_predgen;
	  VATTR(savekey).af_predrev = VATTR(busykey).af_predrev;
	}
      VATTR(savekey).af_succgen = AF_NOVNUM;
      VATTR(savekey).af_succrev = AF_NOVNUM;
      VATTR(busykey).af_predgen = VATTR(savekey).af_gen;
      VATTR(busykey).af_predrev = VATTR(savekey).af_rev;
      if (af_buildkey (busykey->af_ldes, VATTR(savekey).af_predgen, VATTR(savekey).af_predrev, &predkey) == ERROR)
	predkey.af_ldes = (Af_revlist *)0;
      if (af_dodelta (busykey, &predkey, savekey) == ERROR)
	SFAIL ("saverev", "", AF_EDELTA, ERROR);
    }

  VATTR(savekey).af_state = AF_SAVED;
  VATTR(savekey).af_class = VATTR(busykey).af_class;
  author = af_afuser ((Uid_t) geteuid());
  VATTR(savekey).af_auname = af_entersym (author->af_username);
  VATTR(savekey).af_auhost = af_enterhost (author->af_userhost);
  VATTR(savekey).af_audomain = af_enterdomain (author->af_userdomain);
  (void) stat (busykey->af_ldes->af_busyfilename, &ibuf);	
  VATTR(savekey).af_mode = ibuf.st_mode;
  VATTR(savekey).af_mtime = ibuf.st_mtime;
  VATTR(savekey).af_atime = af_acttime ();
  VATTR(savekey).af_ctime = ibuf.st_ctime;
  VATTR(savekey).af_stime = af_acttime ();
  VATTR(savekey).af_notesize = 0;
  VATTR(savekey).af_note = (char *)0;
  VATTR(savekey).af_nrefs = 1;

  (void) afInitUdas (savekey);
  (void) afCopyUdas (busykey, savekey);
  
  VATTR(savekey).af_hashname = (char *)0;
  
  /* update list descriptor */
  busykey->af_ldes->af_nrevs++;
  busykey->af_ldes->af_datasize += VATTR(savekey).af_notesize;
  
  /* save changes */
  if (afAddAso (savekey) == ERROR)
    return (ERROR);
  
  return (AF_OK);
} /* af_saverev */



/*====================================================================
 *    af_newgen
 *
 *====================================================================*/

EXPORT af_newgen (key, newkey)
     Af_key *key;
     Af_key *newkey; /* out */
{
  register Af_key *lastkey, *busykey;
  Af_key *af_gbuskey(), *af_glastkey();

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("newgen", "", AF_EINVKEY, ERROR);

  if (key->af_ldes->af_nrevs == 1)
    SFAIL ("newgen", "", AF_ENOTVERS, ERROR);

  if (VATTR(key).af_class & AF_DERIVED)
    SFAIL ("newgen", "", AF_EDERIVED, ERROR);

  if ((VATTR(key).af_mode & S_IFMT) != S_IFREG)
    SFAIL ("newgen", "", AF_ENOTREGULAR, ERROR);

  lastkey = af_glastkey (key->af_ldes);
  if (af_checkperm (lastkey, AF_LOCKHOLDER) == ERROR)
    SFAIL ("newgen", "", AF_ENOTLOCKED, ERROR);

  if (afNewAso (key->af_ldes, newkey, AF_SOURCE, AF_LASTVERS) == ERROR)
    return (ERROR);

  if ((VATTR(key).af_predgen == VATTR(lastkey).af_gen) && 
      (VATTR(key).af_predrev == VATTR(lastkey).af_rev))
    {
      VATTR(key).af_predgen++;
      VATTR(key).af_predrev = 0;
    }

  /* duplicate last revision */

  VATTR(newkey).af_gen = VATTR(lastkey).af_gen+1; 
  VATTR(newkey).af_rev = 0;
  VATTR(newkey).af_predgen = VATTR(lastkey).af_gen;
  VATTR(newkey).af_predrev = VATTR(lastkey).af_rev;

  if (af_dodelta ((Af_key *)0, lastkey, newkey) == ERROR)
    SFAIL ("newgen", "", AF_EDELTA, ERROR);

  VATTR(newkey).af_name = VATTR(lastkey).af_name;
  VATTR(newkey).af_type = VATTR(lastkey).af_type;
  VATTR(newkey).af_state = VATTR(lastkey).af_state;
  VATTR(newkey).af_class = VATTR(lastkey).af_class;
  VATTR(newkey).af_auname = VATTR(lastkey).af_auname;
  VATTR(newkey).af_auhost = VATTR(lastkey).af_auhost;
  VATTR(newkey).af_audomain = VATTR(lastkey).af_audomain;
  VATTR(newkey).af_mode = VATTR(lastkey).af_mode;
  VATTR(newkey).af_mtime = VATTR(lastkey).af_mtime;
  VATTR(newkey).af_atime = VATTR(lastkey).af_atime;
  VATTR(newkey).af_ctime = VATTR(lastkey).af_ctime;
  VATTR(newkey).af_stime = VATTR(lastkey).af_stime;
  VATTR(newkey).af_notesize = VATTR(lastkey).af_notesize;
  VATTR(newkey).af_note = VATTR(lastkey).af_note;
  VATTR(newkey).af_lckname = (char *)0;
  VATTR(newkey).af_lckhost = (char *)0;
  VATTR(newkey).af_lckdomain = (char *)0;
  VATTR(newkey).af_nrefs = 1;

  (void) afInitUdas (newkey);
  (void) afCopyUdas (lastkey, newkey);

  VATTR(newkey).af_succgen = AF_NOVNUM;
  VATTR(newkey).af_succrev = AF_NOVNUM;

  /* update list descriptor */
  key->af_ldes->af_nrevs++;
  key->af_ldes->af_datasize += VATTR(newkey).af_notesize;

  /* update predecessor of busy version */
  busykey = af_gbuskey (key->af_ldes);
  if  ((VATTR(busykey).af_predgen == VATTR(lastkey).af_gen) && 
       (VATTR(busykey).af_predrev == VATTR(lastkey).af_rev))
    {
      VATTR(busykey).af_predgen = VATTR(newkey).af_gen;
      VATTR(busykey).af_predrev = VATTR(newkey).af_rev;
    }

  if (afAddAso (newkey) == ERROR)
    return (ERROR);

  return (AF_OK);
}


/*====================================================================
 *    af_setbusy
 *
 *====================================================================*/

EXPORT af_setbusy (busykey, key, prevkey)
     Af_key *busykey, *key, *prevkey;
{
  Af_key *af_gbuskey();
  int mtKey=FALSE;

  if ((key->af_ldes == (Af_revlist *)0) && (key->af_lpos == -1))
    mtKey = TRUE;
  else {
    if (afAccessAso (key, AF_ATTRS))
      SFAIL ("setbusy", "", AF_EINVKEY, ERROR);
  }

  if (busykey != (Af_key *)0) {
    if (afAccessAso (busykey, AF_ATTRS))
      SFAIL ("setbusy", "", AF_EINVKEY, ERROR);
    if (VATTR(busykey).af_state != AF_BUSY)
      SFAIL ("setbusy", "", AF_ENOTBUSY, ERROR);
    if (VATTR(busykey).af_class & AF_DERIVED)
      SFAIL ("setbusy", "", AF_EDERIVED, ERROR);
  }
  else
    busykey = af_gbuskey (key->af_ldes);

  if (af_checkperm (busykey, AF_AUTHOR) == ERROR)
    return (ERROR);

  /* determine previous predecessor of busy version */
  if (VATTR(busykey).af_predgen != AF_NOVNUM) {
    if (af_buildkey (busykey->af_ldes, VATTR(busykey).af_predgen, VATTR(busykey).af_predrev, prevkey) == ERROR)
      FAIL ("setbusy", "corrupted predecessor information", AF_EINTERNAL, ERROR);
  }
  else {
    prevkey->af_ldes = (Af_revlist *)0;
    prevkey->af_lpos = -1;
  }

  if (mtKey || (busykey->af_ldes->af_busyfilename != key->af_ldes->af_busyfilename)) {
    VATTR(busykey).af_predgen = AF_NOVNUM;
    VATTR(busykey).af_predrev = AF_NOVNUM;
  }
  else {
    VATTR(busykey).af_predgen = VATTR(key).af_gen;
    VATTR(busykey).af_predrev = VATTR(key).af_rev;
  }

  if (afUpdateAso (busykey, AF_CHANGE) == ERROR)
    return (ERROR);

  return (AF_OK);
}


/*====================================================================
 *    af_svnum
 *
 *====================================================================*/

EXPORT af_svnum (key, gen, rev)
     Af_key *key;
     int gen, rev;
{
  register Af_key *lastkey;
  Af_key predkey, *af_glastkey();

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("svnum", "", AF_EINVKEY, ERROR);

  if ((VATTR(key).af_state == AF_BUSY) || (VATTR(key).af_state > AF_PROPOSED))
    SFAIL ("svnum", "", AF_EWRONGSTATE, ERROR);

  if (!(VATTR(key).af_class & AF_DERIVED)) /* derived files can get any vnum */
    {
      if (af_checkperm (key, AF_LOCKHOLDER) == ERROR)
	SFAIL ("svnum", "", AF_ENOTLOCKED, ERROR);

      /* only the version number of the last saved version can be modified */
      lastkey = af_glastkey (key->af_ldes);
      if (af_keycmp (key, lastkey))
	SFAIL ("svnum", "can set version number only for last saved version",
	       AF_EMISC, ERROR);

      /* if new version number is smaller than the old one */
      if (gen < VATTR(key).af_gen)
	SFAIL ("svnum", "", AF_EINVVNUM, ERROR);
      if (gen == VATTR(key).af_gen)
	{
	  if (rev < VATTR(key).af_rev)
	    SFAIL ("svnum", "", AF_EINVVNUM, ERROR);
	}
      /* read data file in order to get it updated */
      if (afAccessAso (key, AF_DATA) == ERROR)
	return (ERROR);
    }

  VATTR(key).af_gen = (short)gen;
  VATTR(key).af_rev = (short)rev;

  /* if a predecessor exists, update its successor field */
  if (af_buildkey (key->af_ldes, VATTR(key).af_predgen,
		   VATTR(key).af_predrev, &predkey) == AF_OK)
    {
      VATTR((&predkey)).af_succgen = (short)gen;
      VATTR((&predkey)).af_succrev = (short)rev;
    }

  /* the predecessor is updated implicitely by af_updtvers (key) */
  if (afUpdateAso (key, AF_CHANGE) == ERROR)
    return (ERROR);

  return (AF_OK);
}

