/*$__copyright$ */
/*
 *	Shape/AtFS
 *
 *	afstore.c -- interface to archives and binary pools
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP)
 *					  (andy@db0tui62.BITNET)
 *
 *	$Header: afstore.c[1.21] Fri Mar 13 10:26:57 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *      afAccessAso -- prepare ASO for access (read Data if necessary)
 *      afUpdateAso -- Update existing ASO
 *      afNewAso    -- get space for new ASO in revision list
 *      afAddAso    -- update Revision List (contains new ASO)
 *      afDeleteAso -- delete ASO from archive
 *
 *      afGetAso    -- get ASO by name
 *      afTestAso   -- test existence of ASO (by name)
 *
 *      afBuildFile -- write content data of ASO to temporary file
 *
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"

#include <sys/file.h>

#ifdef MEMDEBUG
extern FILE *memprot;
#endif

/*=========================================================================
 * af_arlock -- lock archive
 *
 *=========================================================================*/

EXPORT af_arlock (list)
     Af_revlist *list;
{
  struct stat   ibuf;
  char          lckfilename [MAXPATHLEN];
  register FILE *lckfd;
  
  if (stat (list->af_arfilename, &ibuf) == ERROR) /* no archive file present */
    {
      /* if there should be an archive file */
      if (list->af_lastmod != (time_t) 0)
	FAIL ("arlock", "archive file lost", AF_EINTERNAL, ERROR);
    }
  else
    {
      /* if archive has changed since last read */
      if (list->af_lastmod != ibuf.st_mtime)
	FAIL ("arlock", "", AF_EARCHANGED, ERROR);
    }

  (void) strcpy (lckfilename, list->af_arfilename);
  lckfilename [strlen (lckfilename) - sizeof (char)] = AF_LCKEXT;
  if (stat (lckfilename, &ibuf) == ERROR) /* no lock file present */
    {
      /* create lockfile */
      if ((lckfd = fopen (lckfilename, "w")) == (FILE *)0)
	SFAIL ("arlock", "fopen (lockfile)", AF_ESYSERR, ERROR);
      (void) fclose (lckfd);
      af_reglckfile (lckfilename);
      return (AF_OK);
    }

  sleep (AF_LOCKTIMEOUT);
  
  if (stat (lckfilename, &ibuf) == ERROR) /* no lock file present */
    {
      if ((lckfd = fopen (lckfilename, "w")) == (FILE *)0)
	SFAIL ("arlock", "fopen (lockfile)", AF_ESYSERR, ERROR);
      (void) fclose (lckfd);
      af_reglckfile (lckfilename);
      return (AF_OK);
    }
  else
    FAIL ("arlock", "", AF_EARLOCKED, ERROR);
}

/*=========================================================================
 * af_arunlock -- unlock archive
 *
 *=========================================================================*/

EXPORT af_arunlock (list)
     Af_revlist *list;
{
  char lckfilename [MAXPATHLEN];
  struct stat ibuf;

  (void) strcpy (lckfilename, list->af_arfilename);
  lckfilename [strlen (lckfilename) - sizeof (char)] = AF_LCKEXT;

  /* update list descriptor */
  if (stat (list->af_arfilename, &ibuf) != ERROR)
    list->af_lastmod = ibuf.st_mtime;

  if (unlink (lckfilename) == ERROR)
    FAIL ("af_arunlock", "lock file lost", AF_EINTERNAL, ERROR);
  return (AF_OK);
}

/*=========================================================================
 * afAccessAso -- prepare ASO for access (read Data if necessary)
 *
 *=========================================================================*/

EXPORT afAccessAso (key, mode)
     Af_key *key;
     int    mode;
{
  if (key == (Af_key *)0)
    return (ERROR);
  if (key->af_ldes == (Af_revlist *)0)
    return (ERROR);
  if ((key->af_lpos < 0) || (key->af_lpos >= key->af_ldes->af_listlen))
    return (ERROR);

  if (mode == 0)
    return (AF_OK);

#ifdef STATISTICS
  if (key->af_ldes->af_extent & AF_ATTRS)
    _regkeyaccess (key);
#endif

  if ((key->af_ldes->af_access) && ((key->af_ldes->af_extent & mode) == mode))
    return (AF_OK);

  return (afRefreshList (key->af_ldes, mode));

}

/*=========================================================================
 * afUpdateAso -- Update existing ASO
 *
 *=========================================================================*/

extern int afCurTransaction;

EXPORT afUpdateAso (key, mode)
     Af_key *key;
     int    mode;
{
  /* if (mode == AF_ALLVERS | AF_CHANGE) */
  /* update time of last status change (ctime) for all versions */
  /* not yet implemented (update ctime) */

  if (mode & AF_CHANGE) /* update ctime only for this version */
    VATTR(key).af_ctime = af_acttime ();

  if (key->af_ldes->af_extent & AF_BPOOL)
    return (afRplBpEntry (key, key, key));
  else
    {
      /* defer write operation, if a transaction is active */
      if (afCurTransaction)
	return (afAddToTransactionList (key->af_ldes));

      if (af_arlock (key->af_ldes) == ERROR)
	return (ERROR); 

      if (afWriteArchive (key->af_ldes) == ERROR)
	{
	  (void) af_arunlock (key->af_ldes);
	  return (ERROR);
	}
      else
	return (af_arunlock (key->af_ldes));
    }
}

/*=========================================================================
 * afNewAso -- generate new ASO
 *
 *=========================================================================*/

EXPORT afNewAso (list, key, mode, generation)
     Af_revlist *list;
     Af_key     *key;
     int        mode;
     int        generation;
{
  int freepos, insertpos, i;

  if (mode != AF_SOURCE)
    FAIL ("newAso", "invalid mode", AF_EINTERNAL, ERROR);

  key->af_ldes = list;

  /* get data from archive */
  if (afRefreshList (list, AF_DATA) == ERROR)
    return (ERROR);

  if ((freepos = af_gfreepos (list)) == ERROR)
    FAIL ("newAso", "too many new revisions", AF_EINTERNAL, ERROR);

  if (generation == AF_LASTVERS)
    key->af_lpos = freepos;
  else
    {
      /* create a hole in the list */
      if ((insertpos = af_glastgenpos (list, generation)) == ERROR)
	insertpos = freepos;
      insertpos++;
      for (i=freepos; i>insertpos; i--)
	if (list->af_list[i-1].af_nrefs > 0)
	  FAIL ("newAso", "cannot create hole in history (reference count not zero)", AF_EMISC, ERROR);
      for (i=freepos; i>insertpos; i--)
	list->af_list[i] = list->af_list[i-1];
      key->af_lpos = insertpos;
    }

  VATTR(key).af_nrefs = 1;
  list->af_access++;

#ifdef STATISTICS
  _regkeyaccess (key);
#endif

  return (AF_OK);
}


/*=========================================================================
 * afAddAso -- add ASO
 *
 *=========================================================================*/

EXPORT afAddAso (key)
     Af_key *key;
{
  if (key->af_ldes->af_extent & AF_BPOOL)
    FAIL ("addvers", "cannot apply on binary pools", AF_EINTERNAL, ERROR);
  if (af_arlock (key->af_ldes) == ERROR)
    return (ERROR); 
  if (afWriteArchive (key->af_ldes) == ERROR)
    {
      (void) af_arunlock (key->af_ldes);
      return (ERROR);
    }
  else
    return (af_arunlock (key->af_ldes));
}

/*=========================================================================
 * afDeleteAso -- delete ASO from archive
 *
 *=========================================================================*/

EXPORT afDeleteAso (key)
     Af_key *key;
{
  register char *busyname;

  /* if key points to a file in a binary pool */
  if (key->af_ldes->af_extent & AF_BPOOL)
    return (afDelBpEntry (key));

  if (af_arlock (key->af_ldes) == ERROR)
    return (ERROR);

  /* if "key" points to a busy version */
  if (VATTR(key).af_state == AF_BUSY)
    {
      /* try to remove busy file */
      busyname = af_gbusname (CATTR(key).af_syspath, VATTR(key).af_name, VATTR(key).af_type);
      if (unlink (busyname) == -1)
	{
	  (void) af_arunlock (key->af_ldes);
	  FAIL ("DeleteAso", "unlink", AF_ESYSERR, ERROR);
	}
      VATTR(key).af_predgen = AF_NOVNUM;
      VATTR(key).af_predrev = AF_NOVNUM;
      VATTR(key).af_lckname = (char *)0;
      VATTR(key).af_lckhost = (char *)0;
      VATTR(key).af_lckdomain = (char *)0;
      VATTR(key).af_ltime = AF_NOTIME;
      (void) afDropUdas (key);
    }
  else
    /* remove delta */
    {
      /* read data section of archive */
      if (afRefreshList (key->af_ldes, AF_DATA) == ERROR)
	{
	  (void) af_arunlock (key->af_ldes);
	  return (ERROR);
	}
      
      key->af_ldes->af_datasize -= VATTR(key).af_notesize;
      if (VATTR(key).af_repr == AF_DELTA)
	key->af_ldes->af_datasize -= VATTR(key).af_dsize;
      else
	key->af_ldes->af_datasize -= VATTR(key).af_fsize;

      if (af_rmdelta (key) != AF_OK)
	{
	  (void) af_arunlock (key->af_ldes);
	  return (ERROR);
	}
    }

  /* clear "valid" bit and decrease refcount for corresponding archive */
  VATTR(key).af_class &= ~AF_VALID;
  key->af_ldes->af_access -= VATTR(key).af_nrefs;
#ifdef MEMDEBUG
  fprintf (memprot, "Access: %x,%d -%d (set)\ttotal: %d\n",
	   VATTR(key).af_nrefs, key->af_ldes, key->af_lpos,
	   key->af_ldes->af_access);
#endif
  VATTR(key).af_nrefs = 0;

  key->af_ldes->af_nrevs--;

  if (afWriteArchive (key->af_ldes) == ERROR)
    {
      (void) af_arunlock (key->af_ldes);
      return (ERROR);
    }
  else
    return (af_arunlock (key->af_ldes));
}

/*=========================================================================
 * afGetAso    -- get ASO by name
 *
 *=========================================================================*/

EXPORT afGetAso (syspath, name, type, gen, rev, key)
     char   *syspath, *name, *type;
     int    gen, rev;
     Af_key *key;
{
  register Af_revlist *list;
  Af_revlist          *afInitList();
  register char       *pathsym, *namesym, *typesym;
  Af_key              *keyptr, *af_glastkey(), *af_glastgenkey();

  /* build pathname (default is current directory) */
  pathsym = af_entersym (af_uniqpath (syspath));
  namesym = af_entersym (name);
  typesym = af_entersym (type);

  if ((list = afInitList (pathsym, namesym, typesym)) == (Af_revlist *)0)
    SFAIL ("getkey", "", AF_ENOREV, ERROR);

  key->af_ldes = list;
 
  /* handle special cases */
  if ((gen < 0) || (rev < 0))
    {
      switch (rev)
	{
	case AF_BUSYVERS: if (af_buildkey (list, gen, rev, key) == ERROR)
	                    SFAIL ("getkey", "", AF_ENOREV, ERROR);
	                  break;
	case AF_LASTVERS: if (gen == AF_LASTVERS)
	                    {
			      if ((keyptr = af_glastkey (list)) == (Af_key *)0)
				SFAIL ("getkey", "", AF_ENOREV, ERROR);
			      key->af_lpos = keyptr->af_lpos;
			    }
	                  else
	                    {
			      if ((keyptr = af_glastgenkey (list, gen)) == (Af_key *)0)
				SFAIL ("getkey", "", AF_ENOREV, ERROR);
			      key->af_lpos = keyptr->af_lpos;
			    }
			  break;
	case AF_FIRSTVERS: if (list->af_nrevs == 0)
	                     SFAIL ("getkey", "", AF_ENOREV, ERROR);
	                   if (gen == AF_FIRSTVERS)
			     {
			       key->af_lpos = 0;
			       while ((VATTR(key).af_state == AF_BUSY) ||
				      (!(VATTR(key).af_class & AF_VALID)))
				 {
				   if (key->af_lpos++ == list->af_listlen-1)
				     SFAIL ("getkey", "", AF_ENOREV, ERROR);
				 }
			     }
                           else
	                     {  
			       key->af_lpos = 1;
			       while ((!(VATTR(key).af_class & AF_VALID)) ||
				      (VATTR(key).af_gen != gen))
				 {
				   if (key->af_lpos++ == list->af_listlen-1)
				     SFAIL ("getkey", "", AF_ENOREV, ERROR);
				 }
			     }
			   break;
	default: SFAIL ("getkey", "", AF_ENOREV, ERROR);
	}
    }
  else
    {
      if (af_buildkey (list, gen, rev, key) == ERROR)
	SFAIL ("getkey", "", AF_ENOREV, ERROR);
    }

  key->af_ldes->af_access++;
  VATTR(key).af_nrefs++;
#ifdef MEMDEBUG
  fprintf (memprot, "Access: %x,%d +1 (set)\ttotal: %d\n", 
	   key->af_ldes, key->af_lpos, key->af_ldes->af_access);
#endif

  return (AF_OK);
}

/*=========================================================================
 * afTestAso   -- test existence of ASO (by name)
 *
 *=========================================================================*/

EXPORT afTestAso (path, name, type, mode)
     char *path, *name, *type;
     int  mode;
{
  register Af_revlist *list;
  Af_revlist          *afInitBpList(), *afTestList();
  register int        i, maxindex;
  char     *pathsym = af_entersym (path);
  char     *namesym = af_entersym (name);
  char     *typesym = af_entersym (type);

  if (mode & AF_DERIVED) /* look in binary pool */
    {
      /* lookup in binary pool */
      
      if ((list = afInitBpList (pathsym)) == (Af_revlist *)0)
	return (ERROR);
      
      maxindex = list->af_nrevs-1;
      for (i = 0; i <= maxindex; i++)
	{
	  /* skip holes in the list */
	  if (!(list->af_list[i].af_class & AF_VALID))
	    {
	      if (++maxindex == list->af_listlen)
		FAIL ("TestAso", "bad revision count", AF_EINCONSIST, ERROR);
	      continue;
	    }
	  if ((namesym == list->af_list[i].af_name) &&
	      (typesym == list->af_list[i].af_type))
	    return (AF_OK);
	}
    }
  else /* look in directory */
    {
      if (((list = afTestList (pathsym, namesym, typesym)) != (Af_revlist *)0)
	  && list->af_nrevs > 0)
	return (AF_OK);
      if (((list = afInitList (pathsym, namesym, typesym)) != (Af_revlist *)0)
	  && list->af_nrevs > 0)
	return (AF_OK);
    }
  return (ERROR);
}

/*=========================================================================
 * afBuildFile -- write content data of ASO to temporary file
 *
 *=========================================================================*/

EXPORT afBuildFile (key, name)
     Af_key *key;
     char   *name;
{
  struct utimbuf utbuf;

  if (key->af_ldes->af_extent & AF_BPOOL)
    {
      /* get file from binary pool */
      if (af_cpfile (af_bpfilename (CATTR(key).af_syspath, VATTR(key).af_hashname), VATTR(key).af_fsize, name) == ERROR)
	FAIL ("bldfile", "cpfile", AF_ESYSERR, ERROR);
      /*** set modification and access date ***/
      utbuf.actime = VATTR(key).af_atime;
      utbuf.modtime = VATTR(key).af_mtime;
      if (utime (name, &utbuf) == ERROR)
	FAIL ("bldfile", "utimes", AF_ESYSERR, ERROR);
      (void) chmod (name, (int) VATTR(key).af_mode);
    }
  else
    {
      /* get data from archive file */
      if (afRefreshList (key->af_ldes, AF_DATA) == ERROR)
	return (ERROR);
      
      if (af_undodelta (key, name) == ERROR)
	return (ERROR);
    }
  return (AF_OK);
}
