/*LINTLIBRARY*/
/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 *	Shape/AtFS
 *
 *	aflock.c -- Version locking
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP)
 *					  (andy@db0tui62.BITNET)
 *
 *	$Header: aflock.c[1.10] Fri Jan 31 18:06:05 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *	af_lock -- lock busy version
 *	af_unlock -- unlock version
 *	af_testlock -- test if version is locked
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"

/*================================================================
 *	af_lock
 *
 *================================================================*/

EXPORT Af_user *af_lock (key, locker, mode)
     Af_key  *key;
     Af_user *locker;
     int     mode;
{
  int oldlockeruid, newlockeruid, calleruid;

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("lock", "", AF_EINVKEY, (Af_user *)0);
  
  if ((VATTR(key).af_mode & S_IFMT) != S_IFREG)
    SFAIL ("lock", "", AF_ENOTREGULAR, (Af_user *)0);

  if (VATTR(key).af_class & AF_DERIVED)
    SFAIL ("lock", "", AF_EDERIVED, (Af_user *)0);

  if (!locker->af_userhost[0])
    (void) strcpy (locker->af_userhost, af_gethostname());
  if (!locker->af_userdomain[0])
    (void) strcpy (locker->af_userdomain, af_getdomain());

  if ((newlockeruid = af_getuid (locker->af_username, locker->af_userhost, locker->af_userdomain)) == ERROR)
    SFAIL ("lock", "invalid locker", AF_EMISC, (Af_user *)0);
  calleruid = geteuid ();

  /* if caller == locker, everyone who has write access may set a lock, */
  /*                      if the object is not locked by someone else */
  /* else (caller != locker), only the owner may set a lock for someone else */

  if (calleruid == newlockeruid)
    {
      if (af_checkperm (key, AF_WORLD) == ERROR)
	return ((Af_user *)0);
    }
  else
    {
      if (af_checkperm (key, AF_OWNER) == ERROR)
	return ((Af_user *)0);
    }

  /* if version is not locked or already locked by caller */
  oldlockeruid = af_getuid (VATTR(key).af_lckname, VATTR(key).af_lckhost, VATTR(key).af_lckdomain);

  if ((oldlockeruid == (Uid_t) ERROR) || (oldlockeruid == calleruid))
    {
      VATTR(key).af_lckname = af_entersym (locker->af_username);
      VATTR(key).af_lckhost = af_enterhost (locker->af_userhost);
      VATTR(key).af_lckdomain = af_enterdomain (locker->af_userdomain);
      VATTR(key).af_ltime = af_acttime();
      if (afUpdateAso (key, AF_CHANGE) == ERROR)
	return ((Af_user *)0);
      return (locker);
    }
  else
    SFAIL ("lock", "", AF_ENOTLOCKED, (Af_user *)0);
}

/*================================================================
 *	af_unlock
 *
 *================================================================*/

EXPORT Af_user *af_unlock (key, mode)
     Af_key *key;
     int    mode;
{
  static Af_user locker;

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("unlock", "", AF_EINVKEY, (Af_user *)0);

  if (af_checkperm (key, AF_OWNER | AF_LOCKHOLDER) == ERROR)
    return ((Af_user *)0);

  if (VATTR(key).af_lckname != (char *)0)
    {
      (void) strcpy (locker.af_username, VATTR(key).af_lckname);
      (void) strcpy (locker.af_userhost, VATTR(key).af_lckhost);
      (void) strcpy (locker.af_userdomain, VATTR(key).af_lckdomain);
      VATTR(key).af_lckname = (char *)0;
      VATTR(key).af_lckhost = (char *)0;
      VATTR(key).af_lckdomain = (char *)0;
      VATTR(key).af_ltime = af_acttime();
      if (afUpdateAso (key, AF_CHANGE) == ERROR)
	return ((Af_user *)0);
    }
  else
    {
      locker.af_username[0] = '\0';
      locker.af_userhost[0] = '\0';
      locker.af_userdomain[0] = '\0';
    }
  return (&locker);
}

/*================================================================
 *	af_testlock
 *
 *================================================================*/

EXPORT Af_user *af_testlock (key, mode)
     Af_key *key;
     int    mode;
{
  static Af_user locker;

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("testlock", "", AF_EINVKEY, (Af_user *)0);

  if (VATTR(key).af_lckname != (char *)0)
    {
      (void) strcpy (locker.af_username, VATTR(key).af_lckname);
      (void) strcpy (locker.af_userhost, VATTR(key).af_lckhost);
      (void) strcpy (locker.af_userdomain, VATTR(key).af_lckdomain);
    }
  else
    {
      locker.af_username[0] = '\0';
      locker.af_userhost[0] = '\0';
      locker.af_userdomain[0] = '\0';
    }

  return (&locker);
}


