/*LINTLIBRARY*/
/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 *	Shape/AtFS
 *
 *	affiles.c -- UNIX-files in AtFS
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP)
 *					  (andy@db0tui62.BITNET)
 *
 *	$Header: affiles.c[1.17] Fri Jan 31 18:06:01 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *      af_access -- see if any version of named file exists
 *	af_crkey -- create filekey
 *	af_open -- open AtFS-file
 *	af_close -- close AtFS-file
 *      af_link -- create a link to a AtFS-file
 *	af_rm -- remove AtFS-file
 *	af_restore -- restore derived file
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"

#include <sys/file.h>

extern int af_errno;

/*================================================================
 *	af_access -- see if any version of named file exists
 *
 *================================================================*/

EXPORT af_access (path, name, type, mode)
     char *path, *name, *type;
     int  mode;
{
  register char *pathname;
  
  if (!name || name[0] == '\0')
    SFAIL ("access", "no name given", AF_EMISC, ERROR);

  pathname = af_entersym (af_uniqpath (path));

  return (afTestAso (pathname, name, type, mode));
}

/*================================================================
 *	af_crkey
 *
 *================================================================*/

EXPORT af_crkey (path, name, type, key)
     char *path;
     char *name, *type;
     Af_key *key;
{
  register char *busyname, *uniqpath;
  register FILE *busyfile;
  struct stat bibuf;

  if (!name || name[0] == '\0')
    SFAIL ("crkey", "no name given", AF_EMISC, ERROR);

  uniqpath = af_uniqpath (path);
  busyname = af_gbusname (uniqpath, name, type);
  
  if (afTestAso (uniqpath, name, type, AF_SOURCE) == ERROR)
    {
      if (af_errno == AF_ENAMETOOLONG)
	return (ERROR);
    }

  /* if file does not exist -- create it */
  if ((stat (busyname, &bibuf)) == ERROR)
    {
      if ((busyfile = fopen (busyname, "w")) == (FILE *)0)
	FAIL ("crkey", "fopen", AF_ESYSERR, ERROR);
      (void) fclose (busyfile);
      (void) stat (busyname, &bibuf);
    }
  
  /* select busy version if present */
  if (afGetAso (uniqpath, name, type, AF_BUSYVERS, AF_BUSYVERS, key) == ERROR)
    return (ERROR);

  return (AF_OK);
}

/*================================================================
 *	af_open
 *
 *================================================================*/

EXPORT FILE *af_open (key, mode)
     Af_key *key;
     char   *mode;
{
  register FILE *file;
  register char *tmpname;
  
  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("open", "", AF_EINVKEY, (FILE *)0);
  
  /* if file is present as busy version */
  if (VATTR(key).af_state == AF_BUSY)
    {
      if ((file = fopen (key->af_ldes->af_busyfilename, mode)) == (FILE *)0)
	FAIL ("open", "fopen", AF_ESYSERR, (FILE *)0);
      return (file);
    }
  
  /* saved versions can be opened only for reading */
  if (mode[0] != 'r')
    SFAIL ("open", "", AF_ESAVED, (FILE *)0);
  
  /* see if file is readable */
  if (af_checkread (key) == ERROR)
    SFAIL ("open", "", AF_EACCES, (FILE *)0);

  /* build name for temporary file */
  tmpname = af_gtmpname (CATTR(key).af_syspath, VATTR(key).af_name);
  af_regtmpfile (tmpname);

  if (afBuildFile (key, tmpname) == ERROR)
    return ((FILE *)0);

  if ((file = fopen (tmpname, mode)) == (FILE *)0)
    FAIL ("open", "fopen", AF_ESYSERR, (FILE *)0);

  (void) unlink (tmpname); /* this causes the tmp file to be removed on closing */
  af_unregtmpfile (tmpname);

  VATTR(key).af_atime = af_acttime ();
/* possibly the date of last access is *not* set properly */
/* instead of */
  (void) afUpdateAso (key, 0);
/* this should be */
/* if (afUpdateAso (key, 0) == ERROR) */
/*    return ((FILE *)0); */

  return (file);
}


/*================================================================
 *	af_close
 *
 *================================================================*/

EXPORT af_close (file)
     FILE *file;
{
  return (fclose (file));
}

/*================================================================
 *	af_link
 *
 *================================================================*/

EXPORT af_link (oldkey, newkey)
     /*ARGSUSED*/
     Af_key *oldkey, *newkey;
{
  /* not yet implemented (af_link) */
}

/*================================================================
 *	af_rm
 *
 *================================================================*/

EXPORT af_rm (key)
     Af_key *key;
{
  register Af_user *locker;

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("rm", "", AF_EINVKEY, ERROR);

  /* if object is a derived object or not locked */
  locker = af_testlock (key, 0);
  if (!((VATTR(key).af_class & AF_DERIVED) || (locker->af_username[0]=='\0')))
    {
      if (af_checkperm (key, AF_LOCKHOLDER) == ERROR)
	SFAIL ("rm", "", AF_ENOTLOCKED, ERROR);
    }

  if (VATTR(key).af_nrefs > 1)
    af_wng ("rm", "deleted object has more than one reference");

  if (afDeleteAso (key) == ERROR)
    return (ERROR);

  return (AF_OK);
}

/*================================================================
 *	af_restore
 *
 *================================================================*/

EXPORT af_restore (key, restkey)
     Af_key *key, *restkey;
{
  register char *busyname;

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("restore", "", AF_EINVKEY, ERROR);

  if (!(VATTR(key).af_class & AF_DERIVED))
    SFAIL ("restore", "", AF_ENOTDERIVED, ERROR);

  /* see if file is readable */
  if (af_checkread (key) == ERROR)
    SFAIL ("restore", "", AF_EACCES, ERROR);

  busyname = af_gbusname (CATTR(key).af_syspath,
			   VATTR(key).af_name, VATTR(key).af_type);
  if (afBuildFile (key, busyname) == ERROR)
    return (ERROR);

  /* build key for restored file */
  if (afGetAso (CATTR(key).af_syspath, VATTR(key).af_name, VATTR(key).af_type,
		AF_BUSYVERS, AF_BUSYVERS, restkey) == ERROR)
    FAIL ("restore", "cannot access restored file", AF_EINTERNAL, ERROR);

  /* restore user defined attributes from binary pool */
  (void) afDropUdas (restkey);
  (void) afInitUdas (restkey);
  (void) afCopyUdas (key, restkey);

  VATTR(key).af_atime = af_acttime ();
  /* possibly the date of last access is *not* set properly */
  /* instead of */
  (void) afUpdateAso (key, 0);
  (void) afUpdateAso (restkey, 0);
  /* this should be */
  /* if (afUpdateAso (key, 0) == ERROR) */
  /*    return ((FILE *)0); */
  
  return (AF_OK);
}


