/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*LINTLIBRARY*/
/*
 *	Shape/AtFS
 *
 *	afcattrs.c - read and write complex attributes
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP
 *					   andy@db0tui62.BITNET)
 *
 *	$Header: afcattrs.c[1.13] Fri Jan 31 18:05:54 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *	af_snote -- set note (write note text)
 *	af_rnote -- return note
 *	af_sudattr -- set or modify user defined attribute
 *	af_rudattr -- return user defined attributes
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"

#ifdef MEMDEBUG
extern FILE *memprot;
#endif

char *malloc();

/*====================================================================
 *    af_snote -- set note (write note text)
 *
 *====================================================================*/

EXPORT af_snote (key, buf)
     Af_key *key;
     char   *buf;
{
  register int  len;
  char internalbuf[1];

  internalbuf[0] = '\0';

  if (afAccessAso (key, AF_DATA))
    SFAIL ("snote", "", AF_EINVKEY, ERROR);

  if (VATTR(key).af_state == AF_BUSY)
    SFAIL ("snote", "", AF_EBUSY, ERROR);

  if (VATTR(key).af_class & AF_DERIVED)
    SFAIL ("snote", "", AF_EDERIVED, ERROR);

  if (af_checkperm (key, AF_LOCKHOLDER | AF_AUTHOR | AF_OWNER) == ERROR)
    SFAIL ("snote", "", AF_EACCES, ERROR);

  if (!buf) /* if buf is a nil pointer */
    buf = internalbuf;

  len = strlen(buf) + sizeof (char); /* length of string plus nullbyte */
  if (len > VATTR(key).af_notesize)
    {
      if ((VATTR(key).af_note = af_malloc (key->af_ldes, (unsigned) (len * sizeof (char)))) == (char *)0)
	return (ERROR);
#ifdef MEMDEBUG
      fprintf (memprot, "%x(note)-AL %d bytes\n", VATTR(key).af_note,
	       len * sizeof(char));
#endif
    }
  /* change datasize in header */
  key->af_ldes->af_datasize -= VATTR(key).af_notesize;
  key->af_ldes->af_datasize += len;

  (void) strncpy (VATTR(key).af_note, buf, len);
  VATTR(key).af_notesize = len;

  /* save changes */
  if (afUpdateAso (key, AF_CHANGE) == ERROR)
    return (ERROR);
  return (AF_OK);
}

/*====================================================================
 *    af_rnote -- get pointer to note (read only)
 *
 *====================================================================*/

EXPORT char *af_rnote (key)
     Af_key *key;
{
  register char *note;

  if (afAccessAso (key, AF_DATA))
    SFAIL ("rnote", "", AF_EINVKEY, (char *)0);

  if (VATTR(key).af_state == AF_BUSY)
    SFAIL ("rnote", "", AF_EBUSY, (char *)0);

  if (VATTR(key).af_class & AF_DERIVED)
    SFAIL ("rnote", "", AF_EDERIVED, (char *)0);

  if (VATTR(key).af_notesize != 0)
    {
      if ((note = malloc ((unsigned) VATTR(key).af_notesize)) == (char *)0)
	FAIL ("rnote", "malloc", AF_ESYSERR, (char *)0);
#ifdef MEMDEBUG
      fprintf (memprot, "%x(note)-AL %d bytes\n", note, VATTR(key).af_notesize);
#endif
      (void) strcpy (note, VATTR(key).af_note);
      /* replace newline by nullbyte */
      note[VATTR(key).af_notesize-1] = '\0';
    }
  else
    {
      if ((note = malloc ((unsigned) sizeof (char))) == (char *)0)
	FAIL ("rnote", "malloc", AF_ESYSERR, (char *)0);
#ifdef MEMDEBUG
      fprintf (memprot, "%x(note)-AL %d bytes\n", note, 1);
#endif
      note[0] = '\0';
    }

  return (note);
}

/*====================================================================
 *    af_sudattr -- set or modify user defined attribute
 *               User defined attributes are strings of the following form:
 *                 name=value
 *               For manipulating user defined attributes you have three
 *               modes:
 *               AF_ADD -- add value to attribute if name is present or
 *                         add user defined attribute otherwise
 *               AF_REMOVE -- remove attribute
 *               AF_REPLACE -- replace attribute
 *
 *               Returns AF_OK on successful execution, otherwise ERROR
 *
 *               Caution: the string "attr" must not contain '\n' !!
 *
 *====================================================================*/

EXPORT af_sudattr (key, mode, attr)
     Af_key *key;
     int    mode;
     char   *attr;
{
  register char *udaptr, *tmpuda, *valptr;
  register int  tail;

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("sudattr", "", AF_EINVKEY, ERROR);

  if ((VATTR(key).af_state == AF_FROZEN) && (mode == AF_REMOVE))
    SFAIL ("sudattr", "", AF_EWRONGSTATE, ERROR);

  /* look for delimiter character in attribute string */
  if (!attr || (index (attr, AF_UDAVALDEL) != (char *)0))
    SFAIL ("sudattr", "illegal format of attribute string", AF_EMISC, ERROR);

  if (af_checkperm (key, AF_WORLD) == ERROR)
    return (ERROR);

  /* search entry */
  udaptr = afLookupUda (key, attr); 

  switch (mode)
    {
    case AF_ADD: if (udaptr != (char *)0)
                   {
		     /* build new entry and replace old one */
		     if (!(valptr = index (attr, AF_UDANAMDEL)))
		       {
			 /* no new value -- nothing to do */
			 break;
		       }
		     valptr += sizeof (char);
		     if ((tmpuda = malloc ((unsigned) ((strlen (udaptr) + strlen (valptr) +2) * sizeof (char)))) == (char *)0)
		       FAIL ("sudattr", "malloc", AF_ESYSERR, ERROR);
#ifdef MEMDEBUG
		     fprintf (memprot, "%x(tmpuda)-AL %d bytes\n", tmpuda,
			      (strlen (udaptr) + strlen (valptr) +2)
			      * sizeof (char));
#endif

		     (void) strcpy (tmpuda, udaptr);
		     tail = strlen (tmpuda);
		     tmpuda[tail] = AF_UDAVALDEL;
		     tmpuda[tail+1] = '\0';
		     (void) strcat (tmpuda, valptr);
		     (void) afReplUda (key, tmpuda);
		     free (tmpuda);
#ifdef MEMDEBUG
		     fprintf (memprot, "%x(tmpuda)-FR\n", tmpuda);
#endif
 		   }    
                 else
		   {
		     /* add new entry */
		     if (VATTR(key).af_udanum == AF_MAXUDAS)
		       SFAIL ("sudattr", "", AF_EUDASNUM, ERROR);
		     (void) afEnterUda (key, attr);
		   }
                 break;

    case AF_REMOVE: if (udaptr == (char *)0)
                      SFAIL ("sudattr", "", AF_ENOUDA, ERROR);
		    (void) afDelUda (key, udaptr);
                    break;

    case AF_REPLACE: if (udaptr == (char *)0)
                       SFAIL ("sudattr", "", AF_ENOUDA, ERROR);
                     (void) afReplUda (key, attr);
                     break;
    default: SFAIL ("sudattr", "", AF_EMODE, ERROR);
    }

  /* save changes */
  if (afUpdateAso (key, AF_CHANGE) == ERROR)
    return (ERROR);
  return (AF_OK);
}



/*====================================================================
 *    af_rudattr -- return user defined attributes
 *
 *====================================================================*/

EXPORT char *af_rudattr (key, name)
     Af_key *key;
     char   *name;
{
  register char *udattr, *entry, *valptr;

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("rudattr", "", AF_EINVKEY, (char *)0);

  if (!name)
    SFAIL ("rudattr", "no attribute name given", AF_EMISC, (char *)0);

  if ((entry = afLookupUda (key, name)) == (char *)0)
    return (char *)0;

  if ((valptr = index (entry, AF_UDANAMDEL)) != (char *)0)
    {
      if ((udattr = malloc ((unsigned) strlen(valptr) + sizeof(char))) == (char *)0)
	FAIL ("rudattr", "malloc", AF_ESYSERR, (char *)0);
      /* replace delimiters by '\n' */
      (void) strcpy (udattr, valptr+1);
      valptr = udattr;
      while ((valptr = index (valptr, AF_UDAVALDEL)) != (char *)0)
	valptr[0] = '\n';
    }
  else
    {
      if ((udattr = malloc ((unsigned) sizeof(char))) == (char *)0)
	FAIL ("rudattr", "malloc", AF_ESYSERR, (char *)0);
      udattr[0] = '\0';
    }
  
#ifdef MEMDEBUG
  fprintf (memprot, "%x(udattr)-AL %d bytes\n", udattr,
	   strlen (udattr) + sizeof (char));
#endif
  return (udattr);
}

