/*$__copyright$ */
/*
 *	Shape/AtFS
 *
 *	afbpool.c -- read/write derived object cache control files
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP)
 *					  (andy@db0tui62.BITNET)
 *
 *	$Header: afbpool.c[1.23] Wed Apr 22 17:42:45 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *      afRBpList -- read list of files in derived object cache
 *      afDelBpEntry -- delete file entry in derived object cache
 *      afRplBpEntry -- replace file entry in derived object cache
 *      afDetBpool -- detach derived object cache
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"
#include "afarchive.h"

#include <sys/file.h>
#include <fcntl.h>

#ifdef OLDDBM
#  undef NULL
#  include <dbm.h>
#else
#  ifdef OWNDBM
#    undef NULL
#    include "afdbm.h"
#  else
#    ifdef GNUDBM
#      include "gdbm.h"
#    else
#      include <ndbm.h>
#    endif
#  endif
#endif

/*==========================================================================
 * af_dbmstring -- build dbm string from attrbuf
 *
 *==========================================================================*/

LOCAL datum af_dbmstring (key, size)
     Af_key *key;
     int    size; /* size of dbm key element */
{
  static datum data;
  static char  string[MAXDBMLEN];
  char *ptrlist[AF_MAXUDAS+1], *udaptr;
  register int i;
  int  maxstrlen;

  data.dptr = string;

  /* enter all relevant attributes */

  /* AF_NOSTRING included instead of variant string */
  (void) sprintf (string, "%s %s %s %s %d %d %s %s \0",
	   CATTR(key).af_host, CATTR(key).af_syspath, VATTR(key).af_name, 
	   NOTMT (VATTR(key).af_type), VATTR(key).af_gen, VATTR(key).af_rev, 
	   AF_NOSTRING, VATTR(key).af_auname);
  (void) sprintf (&string[strlen(string)], "%s %s %o %d %d %d %d \0",
	   VATTR(key).af_auhost, VATTR(key).af_audomain, VATTR(key).af_mode,
	   VATTR(key).af_fsize,
	   VATTR(key).af_mtime, VATTR(key).af_atime, VATTR(key).af_ctime);

  (void) strcat (string, AF_UDAID);
  data.dsize = strlen (data.dptr) + sizeof (char);

  /* add user defined attributes */ 
  (void) afListUdas (key, ptrlist);
  i=0;
  udaptr = string;
  maxstrlen = (MAXDBMLEN - DBMLOSS) - size;
  while (ptrlist[i])
    {
      if ((data.dsize += ((strlen (ptrlist[i])) + sizeof (char))) >= maxstrlen)
	{
	  data.dptr = (char *)0;
	  data.dsize = 0;
	  SFAIL ("dbmstring", "", AF_ETOOLONG, data);
	}
      udaptr = &udaptr[strlen(udaptr)+1];
      (void) strcpy (udaptr, ptrlist[i]);
      i++;
    }

  udaptr[strlen(udaptr)+1] = '\0';
  data.dsize++;

  return (data);
} /* end af_dbmstring */

/*===================================================================
 * af_dbmkey -- build attrbuf from dbm entry
 *===================================================================*/

LOCAL af_dbmkey (data, dbmkey, afkey)
     datum *data, *dbmkey;
     Af_key *afkey;
{
  register char *udaptr;
  char name[MAXNAMLEN+1], type[MAXTYPLEN], dummy[4];
  char host[MAXHOSTNAMELEN], syspath[MAXPATHLEN];
  char auhost[MAXHOSTNAMELEN], auname[MAXUSERNAMELEN], audomain[MAXDOMAIN+1];

  /* do initializations */
  VATTR(afkey).af_state = AF_NOSTATE;
  VATTR(afkey).af_class = AF_DERIVED | AF_VALID;
  VATTR(afkey).af_stime = AF_NOTIME;
  VATTR(afkey).af_lckname = (char *)0;
  VATTR(afkey).af_lckhost = (char *)0;
  VATTR(afkey).af_lckdomain = (char *)0;
  VATTR(afkey).af_ltime = AF_NOTIME;
  VATTR(afkey).af_notesize = 0;
  VATTR(afkey).af_note = (char *)0;
  VATTR(afkey).af_repr = AF_FILE;
  VATTR(afkey).af_dsize = 0;
  VATTR(afkey).af_data = (char *)0;
  VATTR(afkey).af_nrefs = 0;
  VATTR(afkey).af_succgen = AF_NOVNUM;
  VATTR(afkey).af_succrev = AF_NOVNUM;
  VATTR(afkey).af_predgen = AF_NOVNUM;
  VATTR(afkey).af_predrev = AF_NOVNUM;

  /* "dummy" instead of variant */
  (void) sscanf (data->dptr, "%s%s%s%s%ld%ld%s%s%s%s%ho%ld%ld%ld%ld",
	  host, syspath, name, type,
	  &(VATTR(afkey).af_gen), &(VATTR(afkey).af_rev), 
	  dummy, auname, auhost, audomain, &(VATTR(afkey).af_mode), 
	  &(VATTR(afkey).af_fsize), &(VATTR(afkey).af_mtime), 
	  &(VATTR(afkey).af_atime),&(VATTR(afkey).af_ctime));

  if (!strcmp (type, AF_NOSTRING))
    type[0] = '\0';

  VATTR(afkey).af_name = af_entersym (name);

  if (type[0])
    VATTR(afkey).af_type = af_entersym (type);
  else
    VATTR(afkey).af_type = (char *)0;

  VATTR(afkey).af_auname = af_entersym (auname);
  VATTR(afkey).af_auhost = af_enterhost (auhost);
  VATTR(afkey).af_audomain = af_enterdomain (audomain);
  VATTR(afkey).af_hashname = af_entersym (dbmkey->dptr);

  /* read user defined attributes */

  udaptr = rindex (data->dptr, AF_UDAID[0]);
  if (strcmp (udaptr, AF_UDAID))
    {
      /* this should not happen very often */
      udaptr = index (data->dptr, AF_UDAID[0]);
      while ((udaptr) && (!strcmp (udaptr, AF_UDAID)))
	udaptr = index (udaptr, AF_UDAID[0]);
      if (!udaptr)
	FAIL ("dbmkey", "cannot find udas", AF_EINCONSIST, ERROR);
    }

  /* build up hashlist and read user defined attributes */
  afInitUdas (afkey);
  udaptr += (AF_IDSTRLEN + sizeof (char));
  while (udaptr[0])
    {
      (void) afEnterUda (afkey, udaptr);
      udaptr = (index (udaptr, '\0') + sizeof (char));
    }
  return (AF_OK);
} /* end af_dbmkey */

/*===================================================================
 * gbpsize -- get derived object cache size and owner
 *===================================================================*/

#if (defined (OLDDBM) || defined (OWNDBM))
LOCAL Af_user *gbpsize (size)
#else
# ifdef GNUDBM
LOCAL Af_user *gbpsize (db, size)
     GDBM_FILE db;
# else
LOCAL Af_user *gbpsize (db, size)
     DBM *db;
# endif
#endif
     short *size;
{
  static Af_user owner;
  datum data, dbkey;
  
  static char *stdentryname = AF_BPSTDENTRY;

  dbkey.dptr = stdentryname;
  dbkey.dsize = strlen (AF_BPSTDENTRY);

#if (defined (OLDDBM) || defined (OWNDBM))
  data = fetch (dbkey);
#else
# ifdef GNUDBM
  data = gdbm_fetch (db, dbkey);
# else
  data = dbm_fetch (db, dbkey);
# endif
#endif
  if ((data.dptr == (char *)0) || (data.dsize == 0))
    {
      *size = 0;
      return (af_afuser ((Uid_t) geteuid()));
    }
  else
    {
      (void) sscanf (data.dptr, "%hd%s%s%s", size, owner.af_username, owner.af_userhost, owner.af_userdomain);
      if (!owner.af_userdomain[0])
	(void) strcpy (owner.af_userdomain, af_getdomain ());
      return (&owner);
    }
}

/*===================================================================
 * pbpsize -- put derived object cache size and owner
 *===================================================================*/

#if (defined (OLDDBM) || defined (OWNDBM))
LOCAL pbpsize (size, owner)
#else
# ifdef GNUDBM
LOCAL pbpsize (db, size, owner)
     GDBM_FILE db;
# else
LOCAL pbpsize (db, size, owner)
     DBM     *db;
# endif
#endif
     int     size;
     Af_user *owner;
{
  datum data, dbkey;
  static char str[32 + MAXUSERNAMELEN + MAXHOSTNAMELEN + MAXDOMAIN];
  static char *stdentryname = AF_BPSTDENTRY;

  dbkey.dptr = stdentryname;
  dbkey.dsize = strlen (AF_BPSTDENTRY);

  (void) sprintf (str, "%d %s %s %s\0", size, owner->af_username, owner->af_userhost, owner->af_userdomain);
  data.dptr = str;
  data.dsize = strlen (str) + sizeof (char);

#if (defined (OLDDBM) || defined (OWNDBM))
  store (dbkey, data);
#else
# ifdef GNUDBM
  (void) gdbm_store (db, dbkey, data, GDBM_REPLACE);
# else
  (void) dbm_store (db, dbkey, data, DBM_REPLACE);
# endif
#endif
  return (AF_OK);
}

/*================================================================
 * af_lookupbpentry -- see if bpentry is loaded yet
 *                     returns position in list
 *
 *================================================================*/

LOCAL af_lookupbpentry (list, hashname)
     Af_revlist *list;
     char *hashname;
{
  register int i=0, j=0;

  for (; i < list->af_listlen; i++)
    {
      /* if entry is not valid */
      if (!(list->af_list[i].af_class & AF_VALID))
	continue;
      if (!strcmp (list->af_list[i].af_hashname, hashname))
	return (i);
      j++;
      if (j >= list->af_nrevs)
	return (ERROR);
    }
  FAIL ("lookupbpentry",
	"cannot find derived object cache entry", AF_EINTERNAL, ERROR);
}

#if (defined (OLDDBM) || defined (OWNDBM))
static char *bpool, *negList[4];
#endif

/*==========================================================================
 *	afRBpList -- read all derived object cache entries
 *
 *==========================================================================*/

EXPORT afRBpList (list)
     Af_revlist *list;
{
  datum  dbkey, data;
#if !(defined (OLDDBM) || defined (OWNDBM))
#ifdef GNUDBM
  GDBM_FILE db;
#else
  DBM    *db;
#endif
#endif
  register int i, count=0;
  Af_key       entrykey;
  Af_user      *owner;
  mode_t       oldUmask;
  char         bpfilename[MAXPATHLEN];

  if (list->af_extent & AF_COMPLETE)
    return (AF_OK);

  oldUmask = umask (002); /* enable group writing */

#if (defined (OLDDBM) || defined (OWNDBM))
  if (list->af_arfilename != bpool)
    {
      FILE *tmpfdes;
      /* lookup in negative list */
      i = 0;
      while (negList[i])
	{
	  if (list->af_arfilename == negList[i])
	    SFAIL ("rbplist", "dbminit", AF_ENOATFSDIR, ERROR);
	  i++;
	}
      if (dbminit (list->af_arfilename) != AF_OK)
	{
	  /* try to create bpool files */
	  (void) sprintf (bpfilename, "%s.dir\0", list->af_arfilename);
	  if ((tmpfdes = fopen (bpfilename, "w")) == (FILE *)0)
	    {
	      (void) afDetBpool (list);
	      /* insert into negative list */
	      i = 0;
	      while (negList[i]) if (i++ == 3) break;
	      negList[i] = list->af_arfilename;
	      SFAIL ("rbplist", "dbminit", AF_ENOATFSDIR, ERROR);
	    }
	  (void) fclose (tmpfdes);
	  (void) sprintf (bpfilename, "%s.pag\0", list->af_arfilename);
	  tmpfdes = fopen (bpfilename, "w");
	  (void) fclose (tmpfdes);
	  /* and try again */
	  if (dbminit (list->af_arfilename) != AF_OK)
	    {
	      (void) afDetBpool (list);
	      /* insert into negative list */
	      i = 0;
	      while (negList[i]) if (i++ == 3) break;
	      negList[i] = list->af_arfilename;
	      SFAIL ("rbplist", "dbminit", AF_ENOATFSDIR, ERROR);
	    }
	}
      bpool = list->af_arfilename;
    }
#else
# ifdef GNUDBM
  if ((db = gdbm_open (list->af_arfilename, 0, GDBM_READER, 0664, 0))
      == (GDBM_FILE)0)
    {
      if ((db = gdbm_open (list->af_arfilename, 0, GDBM_NEWDB, 0664, 0))
	  == (GDBM_FILE)0)
	{
	  (void) afDetBpool (list);
	  SFAIL ("rbplist", "gdbm_open", AF_ENOATFSDIR, ERROR);
	}
    }
# else
  if ((db = dbm_open (list->af_arfilename, O_RDONLY|O_CREAT, 0664)) == (DBM *)0)
    {
      (void) afDetBpool (list);
      SFAIL ("rbplist", "dbm_open", AF_ENOATFSDIR, ERROR);
    }
# endif
#endif

  (void) sprintf (bpfilename, "%s.dir\0", list->af_arfilename);
  (void) chown (bpfilename, geteuid(), list->af_owngid);
  (void) sprintf (bpfilename, "%s.pag\0", list->af_arfilename);
  (void) chown (bpfilename, geteuid(), list->af_owngid);
  (void) umask (oldUmask);

  if (list->af_listlen == 0)
    {
      list->af_listlen = af_gmaxbpsize (list->af_arfilename);
      if ((list->af_list = (Af_vattrs *)af_malloc (list, (unsigned) (list->af_listlen * sizeof(Af_vattrs)))) == (Af_vattrs *)0)
	{
	  list->af_listlen = 0;
	  return (ERROR);
	}

      bzero ((char *)list->af_list, list->af_listlen * sizeof (Af_vattrs));

      list->af_extent |= AF_BPOOL;
      list->af_extent |= AF_UXWRITE;
      list->af_nrevs = 0;
    }

  /* if list was empty up to now */
  if (list->af_nrevs == 0)
    {
#if (defined (OLDDBM) || defined (OWNDBM))
      owner = gbpsize (&list->af_nrevs);
#else /* GNUDBM and NDBM */
      owner = gbpsize (db, &list->af_nrevs);
#endif
      list->af_cattrs.af_ownname = af_entersym (owner->af_username);
      list->af_cattrs.af_ownhost = af_enterhost (owner->af_userhost);
      list->af_cattrs.af_owndomain = af_enterdomain (owner->af_userdomain);
    }
  else
    /* if there are no references to the list, delete all entries and read */
    /* the whole derived object cache. This is (hopefully) faster than */
    /* updating */
    {
      if (list->af_access == 0)
	{
	  af_frmemlist (list);
	  list->af_listlen = af_gmaxbpsize (list->af_arfilename);
	  if ((list->af_list = (Af_vattrs *)af_malloc (list, (unsigned) (list->af_listlen * sizeof(Af_vattrs)))) == (Af_vattrs *)0)
	    {
	      list->af_listlen = 0;
	      return (ERROR);
	    }
	  bzero ((char *)list->af_list, list->af_listlen * sizeof (Af_vattrs));
      
	  list->af_extent |= AF_BPOOL;
	  list->af_extent |= AF_UXWRITE;
	  list->af_nrevs = 0;
	}
    }

  i=0;
#if (defined (OLDDBM) || defined (OWNDBM))
  for (dbkey = firstkey(); dbkey.dptr != NULL; dbkey = nextkey(dbkey))
#else
# ifdef GNUDBM
  for (dbkey = gdbm_firstkey(db); dbkey.dptr != NULL; dbkey = gdbm_nextkey(db, dbkey))
# else
  for (dbkey = dbm_firstkey(db); dbkey.dptr != NULL; dbkey = dbm_nextkey(db))
# endif
#endif
    {
      if (!strncmp (dbkey.dptr, AF_BPSTDENTRY, AF_IDSTRLEN))
	continue;
      /* if there are existing references to derived object cache preserve */
      /* all attribute buffers that are aleady read in and skip used places */
      if (i == list->af_listlen) /* if list is full */
	{
	  if (af_lookupbpentry (list, dbkey.dptr) != ERROR)
	    count++;
	  else /* delete all additional entries in bpool has shrunk */
	    {
#if (defined (OLDDBM) || defined (OWNDBM))
	      delete (dbkey);
#else
# ifdef GNUDBM
	      (void) gdbm_delete (db, dbkey);
# else
	      (void) dbm_delete (db, dbkey);
# endif
#endif
	      (void) unlink (af_bpfilename (list->af_cattrs.af_syspath, dbkey.dptr));
	    }
	  continue;
	}
      count++;
      if (list->af_access > 0)
 	{
	  while (list->af_list[i].af_class & AF_VALID)
	    i++;
	  if (af_lookupbpentry (list, dbkey.dptr) != ERROR)
	      continue;
	}
#if (defined (OLDDBM) || defined (OWNDBM))
      data = fetch (dbkey);
#else
# ifdef GNUDBM
      data = gdbm_fetch (db, dbkey);
# else
      data = dbm_fetch (db, dbkey);
# endif
#endif
      if ((data.dptr == (char *)0) || (data.dsize == 0))
	FAIL ("rbplist", "derived object cache", AF_EINCONSIST, ERROR);
      entrykey.af_ldes = list;
      entrykey.af_lpos = i;
      (void) af_dbmkey (&data, &dbkey, &entrykey);
      i++;
    }

  list->af_nrevs = count;

  /* clear "incomplete" bit */
  list->af_extent |= AF_COMPLETE;

#if !(defined (OLDDBM) || defined (OWNDBM))
  dbm_close (db);
#endif
  return (AF_OK);
}

/*==========================================================================
 *	afDelBpEntry -- delete derived object cache entry
 *
 *==========================================================================*/

EXPORT afDelBpEntry (key)
     Af_key *key;
{
  datum  dbkey;
#if !(defined (OLDDBM) || defined (OWNDBM))
#ifdef GNUDBM
  GDBM_FILE db;
#else
  DBM    *db;
#endif
#else
  int i;
#endif
  short   listsize;
  Af_user *owner;
  mode_t  oldUmask;
  char    bpfilename[MAXPATHLEN];

  oldUmask = umask (002); /* enable group writing */

#if (defined (OLDDBM) || defined (OWNDBM))
  if (key->af_ldes->af_arfilename != bpool)
    {
      FILE *tmpfdes;
      /* lookup in negative list */
      i = 0;
      while (negList[i])
	{
	  if (key->af_ldes->af_arfilename == negList[i])
	    SFAIL ("delbpentry", "dbminit", AF_ENOATFSDIR, ERROR);
	  i++;
	}
      if (dbminit (key->af_ldes->af_arfilename) != AF_OK)
	{
	  /* try to create bpool files */
	  (void) sprintf (bpfilename, "%s.dir\0", key->af_ldes->af_arfilename);
	  if ((tmpfdes = fopen (bpfilename, "w")) == (FILE *)0)
	    {
	      (void) afDetBpool (key->af_ldes);
	      /* insert into negative list */
	      i = 0;
	      while (negList[i]) if (i++ == 3) break;
	      negList[i] = key->af_ldes->af_arfilename;
	      SFAIL ("delbpentry", "dbminit", AF_ENOATFSDIR, ERROR);
	    }
	  (void) fclose (tmpfdes);
	  (void) sprintf (bpfilename, "%s.pag\0", key->af_ldes->af_arfilename);
	  tmpfdes = fopen (bpfilename, "w");
	  (void) fclose (tmpfdes);
	  /* and try again */
	  if (dbminit (key->af_ldes->af_arfilename) != AF_OK)
	    {
	      (void) afDetBpool (key->af_ldes);
	      /* insert into negative list */
	      i = 0;
	      while (negList[i]) if (i++ == 3) break;
	      negList[i] = key->af_ldes->af_arfilename;
	      SFAIL ("delbpentry", "dbminit", AF_ENOATFSDIR, ERROR);
	    }
	}
      bpool = key->af_ldes->af_arfilename;
    }
#else
# ifdef GNUDBM
  if ((db = gdbm_open (key->af_ldes->af_arfilename, 0, GDBM_WRCREAT, 0664, 0))
      == (GDBM_FILE)0)
    {
      (void) afDetBpool (key->af_ldes);
      FAIL ("delbpentry", "gdbm_open", AF_ESYSERR, ERROR);
    }
# else
  if ((db = dbm_open (key->af_ldes->af_arfilename, O_RDWR|O_CREAT, 0664))
      == (DBM *)0)
    {
      (void) afDetBpool (key->af_ldes);
      FAIL ("delbpentry", "dbm_open", AF_ESYSERR, ERROR);
    }
# endif
#endif

  (void) sprintf (bpfilename, "%s.dir\0", key->af_ldes->af_arfilename);
  (void) chown (bpfilename, geteuid(), key->af_ldes->af_owngid);
  (void) sprintf (bpfilename, "%s.pag\0", key->af_ldes->af_arfilename);
  (void) chown (bpfilename, geteuid(), key->af_ldes->af_owngid);
  (void) umask (oldUmask);

  dbkey.dptr = VATTR(key).af_hashname;
  dbkey.dsize = strlen (dbkey.dptr) + sizeof (char);
#if (defined (OLDDBM) || defined (OWNDBM))
  delete (dbkey);
#else
# ifdef GNUDBM
  (void) gdbm_delete (db, dbkey);
# else
  (void) dbm_delete (db, dbkey);
# endif
#endif
  (void) unlink (af_bpfilename (CATTR(key).af_syspath, VATTR(key).af_hashname));
#if (defined (OLDDBM) || defined (OWNDBM))
  owner = gbpsize (&listsize);
#else /* GNUDBM and NDBM */
  owner = gbpsize (db, &listsize);
#endif
  key->af_ldes->af_cattrs.af_ownname = af_entersym (owner->af_username);
  key->af_ldes->af_cattrs.af_ownhost = af_enterhost (owner->af_userhost);
  key->af_ldes->af_cattrs.af_owndomain = af_enterdomain (owner->af_userdomain);
  listsize--;
  key->af_ldes->af_nrevs--;
#if (defined (OLDDBM) || defined (OWNDBM))
  (void) pbpsize (key->af_ldes->af_nrevs, owner);
#else /* GNUDBM and NDBM */
  (void) pbpsize (db, key->af_ldes->af_nrevs, owner);
#endif

  /* clear "valid" bit in attribute buffer and free allocated memory */
  (void) afDropUdas (key);
  VATTR(key).af_class &= ~AF_VALID;

#if !(defined (OLDDBM) || defined (OWNDBM))
  dbm_close (db);
#endif
  return (AF_OK);
}

/*==========================================================================
 *	afRplBpEntry -- replace derived object cache entry
 *
 *==========================================================================*/

EXPORT afRplBpEntry (oldkey, newkey, bpkey)
     Af_key *oldkey, *newkey, *bpkey;
{
  datum  dbkey, data;
#if (defined (OLDDBM) || defined (OWNDBM))
  datum  exdata;
#else
# ifdef GNUDBM
  GDBM_FILE db;
# else
  DBM    *db;
# endif
#endif
  bool   add = FALSE, found = FALSE;
  register short freepos = -1, i;
  short  listsize;
  Af_user  *owner = (Af_user *)0;
  time_t lastacc;
  struct utimbuf utbuf;
  char   *bpname;
  mode_t  oldUmask;
  char    bpfilename[MAXPATHLEN];

  if ((oldkey != (Af_key *)0) && (oldkey->af_ldes != bpkey->af_ldes))
    FAIL ("rplbpentry", "", AF_EINVKEY, ERROR);

  oldUmask = umask (002); /* enable group writing */

#if (defined (OLDDBM) || defined (OWNDBM))
  if (bpkey->af_ldes->af_arfilename != bpool)
    {
      FILE *tmpfdes;
      /* lookup in negative list */
      i = 0;
      while (negList[i])
	{
	  if (bpkey->af_ldes->af_arfilename == negList[i])
	    SFAIL ("rplbpentry", "dbminit", AF_ENOATFSDIR, ERROR);
	  i++;
	}
      if (dbminit (bpkey->af_ldes->af_arfilename) != AF_OK)
	{
	  /* try to create bpool files */
	  (void) sprintf (bpfilename, "%s.dir\0",
			  bpkey->af_ldes->af_arfilename);
	  if ((tmpfdes = fopen (bpfilename, "w")) == (FILE *)0)
	    {
	      (void) afDetBpool (bpkey->af_ldes);
	      /* insert into negative list */
	      i = 0;
	      while (negList[i]) if (i++ == 3) break;
	      negList[i] = bpkey->af_ldes->af_arfilename;
	      SFAIL ("rplbpentry", "dbminit", AF_ENOATFSDIR, ERROR);
	    }
	  (void) fclose (tmpfdes);
	  (void) sprintf (bpfilename,"%s.pag\0",bpkey->af_ldes->af_arfilename);
	  tmpfdes = fopen (bpfilename, "w");
	  (void) fclose (tmpfdes);
	  /* and try again */
	  if (dbminit (bpkey->af_ldes->af_arfilename) != AF_OK)
	    {
	      (void) afDetBpool (bpkey->af_ldes);
	      /* insert into negative list */
	      i = 0;
	      while (negList[i]) if (i++ == 3) break;
	      negList[i] = bpkey->af_ldes->af_arfilename;
	      SFAIL ("rplbpentry", "dbminit", AF_ENOATFSDIR, ERROR);
	    }
	}
      bpool = bpkey->af_ldes->af_arfilename;
    }
#else
# ifdef GNUDBM
  if ((db = gdbm_open (bpkey->af_ldes->af_arfilename, 0, GDBM_WRCREAT, 0664, 0)) == (GDBM_FILE)0)
    {
      (void) afDetBpool (bpkey->af_ldes);
      FAIL ("rplbpentry", "gdbm_open", AF_ESYSERR, ERROR);
    }
# else
  if ((db = dbm_open (bpkey->af_ldes->af_arfilename, O_RDWR|O_CREAT, 0664)) == (DBM *)0)
    {
      (void) afDetBpool (bpkey->af_ldes);
      FAIL ("rplbpentry", "dbm_open", AF_ESYSERR, ERROR);
    }
# endif
#endif

  (void) sprintf (bpfilename, "%s.dir\0", bpkey->af_ldes->af_arfilename);
  (void) chown (bpfilename, geteuid(), bpkey->af_ldes->af_owngid);
  (void) sprintf (bpfilename, "%s.pag\0", bpkey->af_ldes->af_arfilename);
  (void) chown (bpfilename, geteuid(), bpkey->af_ldes->af_owngid);
  (void) umask (oldUmask);

  if (oldkey == (Af_key *)0)
    {
      /* if we actually have to "add" a file */
#if (defined (OLDDBM) || defined (OWNDBM))
      owner = gbpsize (&listsize);
#else /* GNUDBM and NDBM */
      owner = gbpsize (db, &listsize);
#endif
      bpkey->af_ldes->af_cattrs.af_ownname = af_entersym (owner->af_username);
      bpkey->af_ldes->af_cattrs.af_ownhost = af_enterhost(owner->af_userhost);
      bpkey->af_ldes->af_cattrs.af_owndomain = af_enterdomain(owner->af_userdomain);
      if (listsize < bpkey->af_ldes->af_listlen)
	{
	  add = TRUE;
	  /* look for free attribute buffer in bplist */
	  freepos = bpkey->af_ldes->af_listlen-1;
	  while ((bpkey->af_ldes->af_list[freepos].af_class & AF_VALID)
		 && (freepos >=0))
	    freepos--;
	  if (freepos == -1)
	    {
#if !(defined (OLDDBM) || defined (OWNDBM))
	      dbm_close (db);
#endif
	      FAIL ("rplbpentry", "no free entry in derived object cache",
		    AF_EINCONSIST, ERROR);
	    }
	}
      else
	{
	  /* remove the oldest (->af_atime) file */
	  /* read in whole list of bpfiles */
	  if (afRBpList (bpkey->af_ldes) == ERROR)
	    return (ERROR);
	      
	  lastacc = af_acttime ();
	  
	  for (i=0; i < bpkey->af_ldes->af_nrevs; i++)
	    {
	      if ((!bpkey->af_ldes->af_list[i].af_nrefs) && 
		  (bpkey->af_ldes->af_list[i].af_atime < lastacc))
		{
		  freepos = i;
		  lastacc = bpkey->af_ldes->af_list[freepos].af_atime;
		  dbkey.dptr = bpkey->af_ldes->af_list[freepos].af_hashname;
		  found = TRUE;
		}
	    }

	  if (!found)
	    FAIL ("rplbpentry", "", AF_EBPFULL, ERROR);

	  dbkey.dsize = strlen (dbkey.dptr) + sizeof (char);
#if (defined (OLDDBM) || defined (OWNDBM))
	  delete (dbkey);
#else
# ifdef GNUDBM
	  (void) gdbm_delete (db, dbkey);
# else
	  (void) dbm_delete (db, dbkey);
# endif
#endif
	  (void) unlink (af_bpfilename (bpkey->af_ldes->af_cattrs.af_syspath, dbkey.dptr));
	}
    } /* if (oldkey == (Af_key *)0) then ... */
  else
    {
      if (oldkey != newkey)
	{
	  dbkey.dptr = VATTR(oldkey).af_hashname;
	  dbkey.dsize = strlen (dbkey.dptr) + sizeof (char);
#if (defined (OLDDBM) || defined (OWNDBM))
	  delete (dbkey);
#else
# ifdef GNUDBM
	  (void) gdbm_delete (db, dbkey);
# else
	  (void) dbm_delete (db, dbkey);
# endif
#endif
	  (void) unlink (af_bpfilename (CATTR(oldkey).af_syspath, dbkey.dptr));
	}
      freepos = oldkey->af_lpos;
    }

  /* enter new entry */
  if (oldkey != newkey)
    {
      i = 1;
      do
	{
	  if (i > bpkey->af_ldes->af_listlen)
	    FAIL ("rplbpentry",
		  "cannot build unique hashname", AF_EINTERNAL, ERROR);
	  dbkey.dptr =
	    af_rbphashname (VATTR(newkey).af_name,VATTR(newkey).af_type,
			    VATTR(newkey).af_gen, VATTR(newkey).af_rev,
			    newkey->af_ldes, i);
	  dbkey.dsize = strlen (dbkey.dptr) + sizeof (char);
	  data = af_dbmstring (newkey, dbkey.dsize);
	  if ((data.dptr == (char *)0) || (data.dsize == 0))
	    return (ERROR);
#if (defined (OLDDBM) || defined (OWNDBM))
	  exdata = fetch (dbkey);
#endif
	  i++;
	}
      /* as long as key is not unique, try another one */
#if (defined (OLDDBM) || defined (OWNDBM))
      while (exdata.dptr);
      store (dbkey, data);
#else
# ifdef GNUDBM
      while (gdbm_store (db, dbkey, data, GDBM_INSERT));
# else
      while (dbm_store (db, dbkey, data, DBM_INSERT));
# endif
#endif
      
      VATTR(newkey).af_hashname = dbkey.dptr;

      /*** copy file ***/
      bpname = af_bpfilename (CATTR(newkey).af_syspath, VATTR(newkey).af_hashname);
      if (af_cpfile (newkey->af_ldes->af_busyfilename, VATTR(newkey).af_fsize, bpname) == ERROR)
	FAIL ("rplbpentry", "cpfile", AF_ESYSERR, ERROR);
      /*** set modification and access date (not necessary) ***/
      utbuf.actime = VATTR(newkey).af_atime;
      utbuf.modtime = VATTR(newkey).af_mtime;
      if (utime (bpname, &utbuf) == ERROR)
	FAIL ("rplbpentry", "utimes", AF_ESYSERR, ERROR);
      (void) chmod (bpname, (int) VATTR(newkey).af_mode);
    }
  else
    {
      dbkey.dptr = VATTR(newkey).af_hashname;
      dbkey.dsize = strlen (dbkey.dptr) + sizeof (char);
      data = af_dbmstring (newkey, dbkey.dsize);
      if ((data.dptr == (char *)0) || (data.dsize == 0))
	return (ERROR);
#if (defined (OLDDBM) || defined (OWNDBM))
      store (dbkey, data);
#else
# ifdef GNUDBM
      (void) gdbm_store (db, dbkey, data, GDBM_REPLACE);
# else
      (void) dbm_store (db, dbkey, data, DBM_REPLACE);
# endif
#endif
    }

  /* build key for new entry */
  bpkey->af_lpos = freepos;

  if (oldkey != newkey)
    {
      /* copy attribute buffers */
      VATTR(bpkey) = VATTR(newkey);
      /* copy hash table */
      (void) afInitUdas (bpkey);
      (void) afCopyUdas (newkey, bpkey);
    }

  /* if an "add" was performed -- update bplist in database */
  if (add)
    {
      listsize++;
      bpkey->af_ldes->af_nrevs++;
#if (defined (OLDDBM) || defined (OWNDBM))
      (void) pbpsize (bpkey->af_ldes->af_nrevs, owner);
#else /* GNUDBM and NDBM */
      (void) pbpsize (db, bpkey->af_ldes->af_nrevs, owner);
#endif
    }
  
#if !(defined (OLDDBM) || defined (OWNDBM))
  dbm_close (db);
#endif
  return (AF_OK);
} /* afRplBpEntry */

