/*
 *  $Header: requeue.c,v 1.9 90/05/22 09:51:11 vixie Exp $
 * 
 *
 * This program cleans and organizes a sendmail delivery queue.
 * To "clean" means to delete files that are no longer useful, such as
 * orphaned df and xf files. To "organize" means to move the entire mail
 * message (qf, xf, df, etc.) into another directory if it is older than
 * a threshhold.
 *
 * Brian Reid, from an earlier exploratory version by Richard Johnsson
 * 
 *
 * usage: requeue -f fromdir -t todir [ -d age-hours ] [ -D ] [ -n ]
 *
 *   -n says "don't actually delete stale files, just find them"
 *   -D turns on debug printout
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <ctype.h>
#include <stdio.h>
#include <syslog.h>
#include <string.h>
#include <errno.h>


#define NAMESIZE	1024		/* max filename length */
#define LOCKSTALE 	3*60*60		/* lock stale at 3 hours */
#define	FILESTALE	24*60*60	/* junk file stale at 1 day */

#define TRUE 1
#define FALSE 0

char	*olddir = NULL;	/* where files are coming from */
char	*newdir = NULL;	/* where files are going */
int	deltatime = 6*60*60;	/* how old a file must be to be moved */
int	debug = 0;
int	goodfilecount = 0;
int	badfilecount = 0;
int	lockcount = 0;
int	locksbroken = 0;
int	notcopied = 0;
int	reap = 1;
int	nowTime;

typedef struct qent {
	struct qent *nextq,*hashq;
	char *qnam;
};

struct 	qent *qq, *qptr;

#define printd if(debug)printf

#define equal(s1, s2) (strcmp((s1),(s2))==0)

fatal(fmt, s1)
	char *fmt;
{
	fprintf(stderr, fmt, s1);
	fprintf(stderr, "\n");
	exit(1);
}

usage()
{
	fprintf(stderr,"usage: requeue [ -D ] [ -n ] -f fromdir -t todir [-d hours ]\n");
	fatal("       -D means 'debug printout'; -n means 'no delete of stale files.'",NULL);
}

getinfo(f,qtime,dfname)
	FILE *f;
	int *qtime;
	char *dfname;
{
	int gotT=0, gotD=0;
	char *j;
	char buf[BUFSIZ];
	while (fgets(buf, sizeof(buf), f)) {
		if (buf[0] == 'T') {
		    *qtime = atoi(&buf[1]); gotT++;
		};
		if (buf[0] == 'D') {
		    j=strchr(buf,'\n');
		    if (j!=NULL) *j=(char) NULL;
		    strcpy(dfname,&buf[1]);
		    gotD++;
		}
		if (gotT && gotD) return 0;
	}
	return -1;
}


concat2(d, s1, s2)
	char *d, *s1, *s2;
{
	strcpy(d, s1);
	strcat(d, s2);
}


concat3(d, s1, s2, s3)
	char *d, *s1, *s2, *s3;
{
	strcpy(d, s1);
	strcat(d, s2);
	strcat(d, s3);
}


bumpid(id)
	char *id;
{
	if (id[1] < 'Z') {
		id[1]++;
		return 0;
	}
	if (id[0] < 'Z') {
		id[0]++;
		id[1] = 'A';
		return 0;
	}
	return 1;
}

/* move the queue files associated with one mail message. qname is
   the pathname of the cf file and cf is a stream open on it.

   We return TRUE if the file was left behind (and hence its presence
   affects the junk collection), and FALSE if we moved the file. 
 */

int movefiles(qname,cf,save_dname)
	char *qname;
	FILE *cf;
	char *save_dname;
{
	char lname[NAMESIZE];
	char dname[NAMESIZE];
	char xnamebuf[NAMESIZE];
	char *xname;
	char newdname[NAMESIZE];
	char newlname[NAMESIZE];
	char newqname[NAMESIZE];
	char newxname[NAMESIZE];
	char tempname[NAMESIZE];
	char scratch[1024];
	char newid[10];
	int f;
	FILE *ncf, *ocf;
	char buf[BUFSIZ];
	struct stat lockstat,dfstat,cfstat,xfstat;
	int lockTime,moveflag,queuetime;

	concat2(lname, "l", qname+1);

/* First let's lock the file that we are thinking about moving */
	if (link(qname, lname) < 0) {
	        lockcount++;
		if (stat(lname, &lockstat)) {
		    sprintf(scratch,"Can't stat %s",lname);
		    perror(scratch);
		    badfilecount++;
		    return TRUE;
		}
		lockTime = lockstat.st_mtime;
		nowTime = time((long *) 0);
		printd("%s is locked; lock is %5.2f hours old.\n",
		    qname, (nowTime-lockTime)/3600.0, lockTime);
		if (nowTime-lockTime>LOCKSTALE) {
		    printd("%s lock stale. Breaking it.\n",qname);
		    locksbroken++;
		    if (unlink(lname) < 0) {
		        sprintf(scratch,"Cannot unlink %s",lname);
			perror(scratch);
			badfilecount++;
			return TRUE;
		    }
		    if (link(qname, lname) < 0) {
		        printd("still can't get lock %s", qname);
			badfilecount++;
			return TRUE;
		    }
		    
		} else {
		    notcopied++;
		    return TRUE;
		}
	}
/* Now look inside the qf file to get the queue time and the name of the df
  file. Most of the time the df file has the same name as the qf file,
  but we can't count on it. */

	queuetime= -1; strcpy(dname,"----");
	if (getinfo(cf,&queuetime,dname)) {
	    printd("Malformed cf file %s (T=%d, D=%s)\n",qname,queuetime,dname);
	    unlink(lname);
	    fstat(cf, &cfstat);
	    fclose(cf);
	    if (cfstat.st_size == (off_t) 0) {
	        printd("%s is empty. Deleting it\n",qname);
	        unlink(qname);
	    }
	    badfilecount++;
	    return TRUE;
	}
	fclose(cf);
	printd("%s is %5.2f hours old\n",qname,(nowTime-queuetime)/3600.0);
	moveflag = (nowTime - queuetime) > deltatime;

	xname = xnamebuf;
	concat2(xname, "x", qname+1);
	strcpy(newid, qname+2);
	concat3(newqname, newdir, "/qf", newid);
	concat3(newlname, newdir, "/lf", newid);
	concat3(newxname, newdir, "/xf", newid);
	concat3(newdname, newdir, "/df", newid);
	strcpy(save_dname, dname);

	if (stat(dname, &dfstat) < 0) {
	    if (errno==ENOENT) {
	        printd("%s does not exist\n",dname);
		unlink(qname);
		unlink(lname);
		return FALSE;   /* This is sort of a lie, but since
				   the qf file is gone, we can claim
				   to have moved it. */
	    } else {
	 /* in this case, don't unlink queue file */
	        sprintf(scratch,"Cannot stat %s",dname);
		perror(scratch);
		unlink(lname);
		badfilecount++;
		return TRUE;
	    }
	}
	if (!moveflag) {
	    unlink(lname);
	    notcopied++;
	    return TRUE;
	}

	if (stat(xname, &xfstat) < 0) {
	    if (errno!=ENOENT) {
	        sprintf(scratch,"Cannot stat %s",xname);
		perror(scratch);
	    } else {printd("No %s\n",xname);}
	    xname = (char *) NULL;
	}

	if (xname == (char *) NULL)
	    printd("moving %s, %s to %s, %s\n", qname, dname,
	    				newqname, newdname);
	else
	    printd("moving %s,%s,%s to %s,%s,%s\n", qname, dname, xname,
	    				newqname, newdname, newxname);

/* get the locks */
	if (link(lname, newlname) < 0) {
		printd("movefiles: target lock exists\n");
	} else if (link(qname, newqname) < 0) {
		sprintf(scratch,"link of %s to %s failed", qname, newqname);
		perror(scratch);
	} else if (link(dname, newdname) < 0) {
		sprintf(scratch,"link of %s to %s failed", dname, newdname);
		perror(scratch);
		unlink(newqname);
	} else {
		if (xname != (char *) NULL && link(xname, newxname) < 0 && errno != ENOENT) {
		    /* gee. I don't care about this error. So what
		       if we couldn't link over the xf file */
		   printd("Could not link %s to %s\n",xname,newxname);
		}

/* good news. We were able to move just by relinking the files */

		goodfilecount++;
		goto done;
	}
	if (bumpid(newid)) {unlink(lname); return TRUE;}

/* Make an unused filename by making an "nf" file. Its only purpose
   is to be used as the source of a link that probes the destination */

	sprintf(tempname, "%s/nf%d", newdir, getpid());
	unlink(tempname);
	if ((f = creat(tempname, 0600)) < 0) {
		printd("can't creat %s\n");
		return TRUE;
	}
/* A queued entry with the same qf name exists already in the destination.
   That means we have to change the ID number before we can move it, which
   means we have to change the contents of the qf file (it lists inside
   it the name of the df file). */
   
	for(;;) {
		concat3(newqname, newdir, "/qf", newid);
		concat3(newlname, newdir, "/lf", newid);
		concat3(newdname, newdir, "/df", newid);
		if (xname != (char *) NULL)
		    concat3(newxname, newdir, "/xf", newid);

		printd("moving %s, %s to %s, %s\n",
			qname, dname, newqname, newdname);
		if (link(tempname, newlname) < 0) {
			printd("movefiles: target lock exists\n");
			if ((errno != EEXIST) || bumpid(newid)) {
				unlink(tempname); unlink(lname);
				close(f);
				return TRUE;
			}
			continue;
		}
	/* First use "link" to test if this qf name is going to work. */
		if (link(tempname, newqname) < 0) {
			sprintf(scratch,"link of %s to %s failed", qname, newqname);
			perror(scratch);
			badfilecount++;
			goto tryagain;
		}
	/* OK to just link over the df file */
		if (link(dname, newdname) < 0) {
			sprintf(scratch,"link of %s to %s failed", dname, newdname);
			perror(scratch);
			unlink(newqname);
			badfilecount++;
			goto tryagain;
		}
	
	/* OK to just link over the xf file, if it exists */
		if (xname != (char *) NULL 
		    && link(xname, newxname) < 0
		    && errno != ENOENT) {
			sprintf(scratch,"link of %s to %s failed", xname, newxname);
			perror(scratch);
		/* inability to link is not a fatal error for xf file */

		}
	
	/* we have to edit the qf file as we move it. Note that we have
	   an FD open on it, since the link(tempname, newqname) succeeded */

		ncf = fdopen(f, "w");
		ocf = fopen(qname, "r");
		while (fgets(buf, sizeof(buf), ocf)) {
			if (buf[0] == 'D')
				sprintf(buf, "Ddf%s\n", newid);
			fputs(buf, ncf);
		}
		unlink(tempname);
		goodfilecount++;
		fclose(ncf);
		break;
tryagain:
		unlink(newlname);
		if (bumpid(newid)) {
			unlink(tempname);
			close(f);
			return TRUE;
		}
	}
done:
	unlink(dname);
	unlink(qname);
	if (xname != (char *) NULL) unlink(xname);
	unlink(lname);
	unlink(newlname);
	return FALSE;
}

/*
 * This routine is called with its one argument an open directory pointer
 * (suitable for use with readdir) and the other a linked list of "qent"
 * records of all of the live "qf" files that we have found in this
 * directory. Our job is to make a second pass over the directory and 
 * unlink everything we find that is stale and that is not related to 
 * any of these qf files.
 *
 */
reapjunk(fp,qtop)
	DIR *fp;
 	struct qent *qtop;
{
	struct qent *qptr,*qnp;
	int i,nq,qnum;
	struct direct *dp;
	struct qent *dindex[100];
	struct stat ffstat;
	time_t nowTime, fileTime;
	char *fn;
	int delcount=0;

	qptr = qtop;
	nq = 0;
/* 
   loop over the stored list of cf file names, count them, and build 
   100 hash threads based on the last 2 digits of the AA number
 */

	nowTime = time((long *) 0);

	for (i=0; i<100; i++) dindex[i] = (struct qent *) NULL;
	while (qptr->nextq != (struct qent *) NULL) {
	    nq++;
	    qnum=atoi(&(qptr->qnam)[2]) % 100;
	    qptr->hashq = dindex[qnum];
	    dindex[qnum] = qptr;
	    qptr = qptr->nextq;
	}

/* We've now built a hash table for fast lookup of AA numbers. Let's loop 
   over the directory and get rid of all of the junk. 
 */
	while ((dp = readdir(fp)) != NULL) {
	    fn = dp->d_name;
	    if (fn[0] == 'q' && fn[1] == 'f') goto keepit;
	    if (stat(fn,&ffstat) < 0) goto keepit;
	    if ((ffstat.st_mode & S_IFMT) != S_IFREG) goto keepit;
	    fileTime = ffstat.st_mtime;

	    /* if file is stale, see if it is garbage */
	    if (nowTime - fileTime > FILESTALE) {
	        if (!strncmp("syslog",fn,6)) goto keepit;
		if ((fn[0]) == '.') goto keepit;
		if (islower(fn[0]) && fn[1]=='f'
		  &&isupper(fn[2]) && isupper(fn[3])) {
		    qnum=atoi(&fn[4]) % 100;
		    qnp = dindex[qnum];
		    while (qnp != (struct qent *) NULL) {
		        if (!strcmp(&fn[2],qnp->qnam)) goto keepit;
			qnp = qnp->hashq;
		    }
		}

	    /* dum, da dum dum: time to unlink the file */
	    printd("unlink(%s)\n",fn);
	    if (reap) unlink(fn);
	    delcount++;
	    }
keepit: ;	    
	}
	printd("Reap=%d; found %d stale files\n",reap,delcount);
	if (delcount) {
	    if (reap) 
		syslog(LOG_INFO,"Deleted %d stale files from %s\n",
		    delcount,olddir);
	    else
		syslog(LOG_INFO,"Found %d stale files in %s (did not delete)\n",
		    delcount,olddir);
	}
	return;
}



main(argc, argv)
	int argc;
	char *argv[];
{
	int i,testfile;
	struct direct *d;
	DIR *f;
	extern int optind;
	extern char *optarg;
	char scratch[256],test[256];
	
	/* crude "ld" hack by vixie */
	(void) gethostbyname("localhost");

	while ((i=getopt(argc, argv, "f:t:d:Dn")) != EOF) {
	    switch(i) {
	        case 'f': olddir = optarg; break;
		case 't': newdir = optarg; break;
		case 'd': deltatime = 3600*atoi(optarg); break;
		case 'D': debug++; break;
		case 'n': reap = 0; break;
		case '?': usage();
	    }
	}
	if (optind < argc) usage();

 /* check options for various common errors */
	if (!olddir)
	    fatal("-f option (from dir) missing.",NULL);
	if (!newdir)
	    fatal("-t option (to dir) missing.",NULL);
	if (newdir[0] != '/')
	    fatal("-t %s: must be absolute pathname",newdir);

	if (deltatime<=0 || deltatime > 3600*250) {
	    fatal("-d option argument should be in hours. %d hours is unreasonable.",
		deltatime/3600);
	}
	printd("olddir=\"%s\"; newdir=\"%s\"; deltatime=%d\n",
		olddir, newdir, deltatime);
	if (reap) {printd("Will delete stale files\n");}
	   else   {printd("Will locate but not delete stale files\n");}

	if (chdir(olddir) < 0) {
		sprintf(scratch,"can't chdir to %s", olddir);
		perror(scratch);
		exit(1);
	}
	f = opendir(".");
	if (f == NULL)
		fatal("can't open %s as \".\"", olddir);

  /* check to make sure destination directory exists */
	sprintf(test,"%s/.test.", newdir);
	unlink(test);
	testfile = creat(test,0777);
	if (testfile < 0) {
	    sprintf(scratch,"Can't write test file %s",test);
	    perror(scratch);
	    exit(1);
	}
	close(testfile); unlink(test);

	qq = (struct qent *) malloc(sizeof(struct qent));
	qq->nextq = (struct qent *) NULL;
	qptr = qq;

	while ((d = readdir(f)) != NULL) {
		FILE *cf;
		char dname[NAMESIZE];

		nowTime = time((long *) 0);
		if (d->d_name[0] != 'q' || d->d_name[1] != 'f')
			continue;
		cf = fopen(d->d_name, "r");
		if (cf == NULL) continue;

/* 
   The "if" will succeed if the file was not moved, and thus is still
   in the queue. 
 */
		strcpy(dname, d->d_name);
		if (movefiles(d->d_name,cf, dname)) {
		    qptr = (struct qent *) malloc(sizeof(struct qent));
		    qptr->nextq = qq;
		    qq = qptr;
		    qq->qnam = (char *) malloc(1+strlen(d->d_name));
		    strcpy(qq->qnam,&(d->d_name)[2]);

	/* if df and qf files have different names, make second entry */
		    if (strcmp(&(d->d_name)[2],&dname[2])) {
			qptr = (struct qent *) malloc(sizeof(struct qent));
			qptr->nextq = qq;
			qq = qptr;
			qq->qnam = (char *) malloc(1+strlen(dname));
			strcpy(qq->qnam,&dname[2]);
		    }
		}
	}

	sprintf(scratch,"%s: %d copied (%d remain) %d locked (%d broken) %d errors.\n",
	    olddir,goodfilecount,notcopied,lockcount,locksbroken,badfilecount);
	openlog("requeue", LOG_PID);
	printd("%s\n",scratch);
	syslog(LOG_INFO,scratch);

/* Requeing complete. Now make a second pass over the directory and reap
   unreferenced files. */

	rewinddir(f);
	reapjunk(f,qq);
	closelog();
}
