/******************************************************************************
*******************************************************************************
**
** EXR0R3 - A ring 0/ring 3 IFS skeleton
** Copyright (C) 1993 by Andre Asselin
**
** R3COMM.C - Ring 3 side of communications
**
** History:
** 6/29/93 - created
**
*******************************************************************************
******************************************************************************/

#include "r3inc.h"

#pragma argsused
int main(int argc, char **argv, char **envp) {
   INITSTRUC initdata;
   unsigned long leninitdata;
   OPDATA *OpData;
   void *Buf;
   unsigned int rc;
   char name[10];

   // Print a banner to let the user know we're running
   printf("EXR0R3 - A ring 0/ring 3 IFS skeleton\n"
          "Copyright (C) 1993 by Andre Asselin\n\n");


   // Allocate the two buffers
   rc = DosAllocMem((void * *)&OpData, sizeof(initdata.OpData),
                    PAG_READ | PAG_WRITE | PAG_COMMIT);
   if (rc != NO_ERROR) {
      printf("alloc failed - OpData: %d\n", rc);
      exit(1);
   }
   rc = DosAllocMem(&Buf, 65536, PAG_READ | PAG_WRITE | PAG_COMMIT);
   if (rc != NO_ERROR) {
      printf("alloc failed - Buf: %d\n", rc);
      exit(1);
   }

   // Call local initialization entry point
   rc = FS_INIT(argv[1], 0, 0);
   if (rc != NO_ERROR) {
      printf("FS_INIT failed: rc=%d", rc);
      exit(1);
   }

   // Set up the initialization data to be passed to the IFS
   initdata.OpData = OpData;
   initdata.Buf = Buf;
   // We want the FS router to route by IFS name, so copy the name into a
   // temporary buffer because the router could destroy the name.
   strncpy(name, FS_NAME, sizeof(name));

   // Call the IFS to initialize
   leninitdata = sizeof(initdata);
   rc = DosFSCtl(NULL, 0, NULL, &initdata, sizeof(initdata), &leninitdata,
                 FSCTL_FUNC_INIT, name, -1, FSCTL_FSDNAME);
   if (rc != NO_ERROR) {
      printf("FSCtl failed - init: 0x%x %d\n", rc, rc);
      exit(1);
   }


   // We returned from the IFS-- that means we have an operation waiting.
   // Figure out what it is and route it appropriately.
   while (1) {
      if (debug)
         printf("req=%x: ", OpData->funccode);

      switch (OpData->funccode) {
      case CPFC_ATTACH:
         OpData->rc = FSD_ATTACH(OpData->attach.flag, OpData->attach.Dev,
                                 &OpData->attach.vpfsd, &OpData->attach.cdfsd,
                                 Buf, &OpData->attach.Len);
         break;

      case CPFC_CANCELLOCKREQUEST:
         OpData->rc = FS_CANCELLOCKREQUEST(&OpData->cancellockrequest.sffsi, 
                                           &OpData->cancellockrequest.sffsd,
                                           &OpData->cancellockrequest.LockRange);
         break;

      case CPFC_CHDIR:
         OpData->rc = FS_CHDIR(OpData->chdir.flag, 
                               &OpData->chdir.cdfsi,
                               &OpData->chdir.cdfsd, 
                               OpData->chdir.Dir,
                               OpData->chdir.iCurDirEnd);
         break;

      case CPFC_CHGFILEPTR:
         OpData->rc = FS_CHGFILEPTR(&OpData->chgfileptr.sffsi, 
                                    &OpData->chgfileptr.sffsd,
                                    OpData->chgfileptr.offset, 
                                    OpData->chgfileptr.type, 
                                    OpData->chgfileptr.IOflag);
         break;

      case CPFC_CLOSE:
         OpData->rc = FS_CLOSE(OpData->close.type, 
                               OpData->close.IOflag,
                               &OpData->close.sffsi, 
                               &OpData->close.sffsd);
         break;

      case CPFC_COMMIT:
         OpData->rc = FS_COMMIT(OpData->commit.type, 
                                OpData->commit.IOflag,
                                &OpData->commit.sffsi, 
                                &OpData->commit.sffsd);
         break;

      case CPFC_COPY:
         OpData->rc = FS_COPY(OpData->copy.flag, 
                              &OpData->copy.cdfsi,
                              &OpData->copy.cdfsd, 
                              OpData->copy.Src,
                              OpData->copy.iSrcCurDirEnd, 
                              OpData->copy.Dst,
                              OpData->copy.iDstCurDirEnd, 
                              OpData->copy.nameType);
         break;

      case CPFC_DELETE:
         OpData->rc = FS_DELETE(&OpData->fsdelete.cdfsi, 
                                &OpData->fsdelete.cdfsd,
                                OpData->fsdelete.File,
                                OpData->fsdelete.iCurDirEnd);
         break;

      case CPFC_FILEATTRIBUTE:
         OpData->rc = FS_FILEATTRIBUTE(OpData->fileattribute.flag, 
                                       &OpData->fileattribute.cdfsi,
                                       &OpData->fileattribute.cdfsd, 
                                       OpData->fileattribute.Name,
                                       OpData->fileattribute.iCurDirEnd, 
                                       &OpData->fileattribute.Attr);
         break;

      case CPFC_FILEINFO:
         OpData->rc = FS_FILEINFO(OpData->fileinfo.flag, 
                                  &OpData->fileinfo.sffsi,
                                  &OpData->fileinfo.sffsd, 
                                  OpData->fileinfo.level,
                                  (char *)Buf,
                                  OpData->fileinfo.cbData,
                                  OpData->fileinfo.IOflag, 
                                  &OpData->fileinfo.oError);
         break;

      case CPFC_FILELOCKS:
         OpData->rc = FS_FILELOCKS(&OpData->filelocks.sffsi, 
                                   &OpData->filelocks.sffsd,
                                   &OpData->filelocks.UnLockRange, 
                                   &OpData->filelocks.LockRange,
                                   OpData->filelocks.timeout, 
                                   OpData->filelocks.flags);
         break;

      case CPFC_FINDCLOSE:
         OpData->rc = FS_FINDCLOSE(&OpData->findclose.fsfsi, 
                                   &OpData->findclose.fsfsd);
         break;

      case CPFC_FINDFIRST:
         OpData->rc = FS_FINDFIRST(&OpData->findfirst.cdfsi, 
                                   &OpData->findfirst.cdfsd,
                                   OpData->findfirst.Name, 
                                   OpData->findfirst.iCurDirEnd,
                                   OpData->findfirst.attr, 
                                   &OpData->findfirst.fsfsi,
                                   &OpData->findfirst.fsfsd, 
                                   (char *)Buf,
                                   OpData->findfirst.cbData, 
                                   &OpData->findfirst.cMatch,
                                   OpData->findfirst.level, 
                                   OpData->findfirst.flags,
                                   &OpData->findfirst.oError);
         break;

      case CPFC_FINDFROMNAME:
         OpData->rc = FS_FINDFROMNAME(&OpData->findfromname.fsfsi, 
                                      &OpData->findfromname.fsfsd,
                                      (char *)Buf,
                                      OpData->findfromname.cbData,
                                      &OpData->findfromname.cMatch, 
                                      OpData->findfromname.level,
                                      OpData->findfromname.position, 
                                      OpData->findfromname.Name,
                                      OpData->findfromname.flags,
                                      &OpData->findfirst.oError);
         break;

      case CPFC_FINDNEXT:
         OpData->rc = FS_FINDNEXT(&OpData->findnext.fsfsi, 
                                  &OpData->findnext.fsfsd,
                                  (char *)Buf,
                                  OpData->findnext.cbData,
                                  &OpData->findnext.cMatch, 
                                  OpData->findnext.level,
                                  OpData->findnext.flags,
                                  &OpData->findfirst.oError);
         break;

      case CPFC_FLUSHBUF:
         OpData->rc = FS_FLUSHBUF(&OpData->flushbuf.vpfsd,
                                  OpData->flushbuf.flag);
         break;

      case CPFC_FSINFO:
         OpData->rc = FS_FSINFO(OpData->fsinfo.flag, 
                                &OpData->fsinfo.vpfsd,
                                (char *)Buf,
                                OpData->fsinfo.cbData,
                                OpData->fsinfo.level);
         break;

      case CPFC_MKDIR:
         OpData->rc = FS_MKDIR(&OpData->mkdir.cdfsi, 
                               &OpData->mkdir.cdfsd,
                               OpData->mkdir.Name, 
                               OpData->mkdir.iCurDirEnd,
                               (char *)Buf,
                               OpData->mkdir.flags,
                               &OpData->mkdir.oError);
         break;

      case CPFC_MOVE:
         OpData->rc = FS_MOVE(&OpData->move.cdfsi, 
                              &OpData->move.cdfsd,
                              OpData->move.Src, 
                              OpData->move.iSrcCurDirEnd,
                              OpData->move.Dst, 
                              OpData->move.iDstCurDirEnd,
                              OpData->move.flags);
         break;

      case CPFC_NEWSIZE:
         OpData->rc = FS_NEWSIZE(&OpData->newsize.sffsi, 
                                 &OpData->newsize.sffsd,
                                 OpData->newsize.len, 
                                 OpData->newsize.IOflag);
         break;

      case CPFC_OPENCREATE:
         OpData->rc = FS_OPENCREATE(&OpData->opencreate.cdfsi, 
                                    &OpData->opencreate.cdfsd,
                                    OpData->opencreate.Name, 
                                    OpData->opencreate.iCurDirEnd,
                                    &OpData->opencreate.sffsi, 
                                    &OpData->opencreate.sffsd,
                                    OpData->opencreate.ulOpenMode,
                                    OpData->opencreate.usOpenFlag,
                                    &OpData->opencreate.usAction, 
                                    OpData->opencreate.usAttr,
                                    (char *)Buf,
                                    &OpData->opencreate.fgenflag,
                                    &OpData->opencreate.oError);
         break;

      case CPFC_PATHINFO:
         OpData->rc = FS_PATHINFO(OpData->pathinfo.flag, 
                                  &OpData->pathinfo.cdfsi,
                                  &OpData->pathinfo.cdfsd, 
                                  OpData->pathinfo.Name,
                                  OpData->pathinfo.iCurDirEnd, 
                                  OpData->pathinfo.level,
                                  (char *)Buf,
                                  OpData->pathinfo.cbData, 
                                  &OpData->pathinfo.oError);
         break;

      case CPFC_PROCESSNAME:
         OpData->rc = FS_PROCESSNAME(OpData->processname.NameBuf);
         break;

      case CPFC_READ:
         OpData->rc = FS_READ(&OpData->read.sffsi, 
                              &OpData->read.sffsd,
                              (char *)Buf,
                              &OpData->read.Len,
                              OpData->read.IOflag);
         break;

      case CPFC_RMDIR:
         OpData->rc = FS_RMDIR(&OpData->rmdir.cdfsi, 
                               &OpData->rmdir.cdfsd,
                               OpData->rmdir.Name, 
                               OpData->rmdir.iCurDirEnd);
         break;

      case CPFC_VERIFYUNCNAME:
         OpData->rc = FS_VERIFYUNCNAME(OpData->verifyuncname.flag, 
                                       OpData->verifyuncname.Name);
         break;

      case CPFC_WRITE:
         OpData->rc = FS_WRITE(&OpData->write.sffsi, 
                               &OpData->write.sffsd,
                               (char *)Buf,
                               &OpData->write.Len,
                               OpData->write.IOflag);
         break;

      // This shouldn't happen!
      default:
         printf("Unknown IFS function code: %d\n", OpData->funccode);
         OpData->rc = ERROR_NOT_SUPPORTED;
         break;
      }

      // We want the FS router to route by IFS name, so copy the name into a
      // temporary buffer because the router could destroy the name.
      strncpy(name, FS_NAME, sizeof(name));

      // Say we're done with this one, and wait for next one.
      rc = DosFSCtl(NULL, 0, NULL, NULL, 0, NULL, FSCTL_FUNC_NEXT, name,
                    -1, FSCTL_FSDNAME);
      if (rc != NO_ERROR) {
         printf("FSCtl failed - next: %d\n", rc);
         exit(1);
      }
   }
}
