/****	Routines to do ICMP echo	****/

/*	Copyright (C) 1989, 1990, 1991 Massachusetts Institute of Technology
 *		Right to copy granted under the terms described
 *		in the file Copyright accompanying this distribution.
 */

/* This is a module to be included in programs that need the ability
 * to test the up/down status of a host with ICMP echo packets.
 *
 *	For now this is going to be a relatively simple thing.
 */

#ifdef EDIT_HISTORY

Nbr Date	Author		Description
--- ----	------		-----------
 3  20-Feb-89	M. A. Patton	First release version.
 4  23-Mar-89	M. A. Patton	Cosmetics from report by Chris Tengi.
 4A 24-Jun-89	M. A. Patton	Conditionalized for pre-4.3 raw IP.
 4B  9-Nov-89	Dan Long	Tighter check on received packets to reject
				replies to old pings.
 4C 24-Jan-90	M. A. Patton	Cleaned up a bit, added debug options.
				Now returns errors at initialization.
				Compute fixed length checksum in-line.
 4D 10-Feb-90	M. A. Patton	Added more robustness against false positives
				(uniqizer) and clean up send routine.
 4E 10-Mar-90	M. A. Patton	More cleanup, removed constant stuff from the
				periodic pinger() to InitICMP().
 4F 13-Mar-90	Karl Owen	DG/UX doesn't include signal.h in sys/param.h
 4G 27-Jul-90	M. A. Patton	Made uniqizer an n_short so it wraps the same
				way as the packet seq number, otherwise after
				a lot of polls they never match!
 5   9-Aug-90	M. A. Patton	Added over-ride conditionals for Raw IP stuff.
 5A 15-Aug-90	Edwin Kremer	HP-UX 7.0 doesn't have <arpa/inet.h>
 5B  1-Jan-91	Mike Patton	Smaller input buffer (don't look at extra bits,
				so why copy out of kernel?), some cleanup and
				comments added around RAW_IP stuff.  Made the
				discard code say why when debug on.
		Havard Eidnes	clean up type clash in call to inet_ntoa.
 5C 11-Jan-91	Mike Patton	Made all debugging output go to stderr to avoid
				sequencing confusion from buffering.
 5D 25-Apr-91	Mike Patton	Fixed bug in "cheap" checksum code that only
				manifests after 40000 or so packets.  It was
				a little too cheap.

#endif
/**** Documentation ****/


#ifdef DOCUMENTATION

	!!!!	needs documentation	!!!!

#endif
/**** compiler setup page ****/



/* Include the world (or a large portion of it) */
#include <errno.h>
#include <setjmp.h>

#include <sys/param.h>	/* Also gets <signal.h> and <sys/types.h> */
#include <signal.h>	/* Except on some systems... */
#include <sys/socket.h>

#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netdb.h>

#if !defined(hp9000s300)
#include <arpa/inet.h>
#endif

#include "icmp.h"
#include "common.h"
#include "debug.h"


/* This sets up RAW_IP_HEADERS defaultly if not set by config */
#if !( defined(RAW_IP_HEADERS) || defined(RAW_IP_NOHEADERS) )
#if defined(BSD) && BSD > 42
#define RAW_IP_HEADERS
#endif
#endif


/* Size of absolute minimum ICMP packet, note that there are places in
 * the code where this is known.
 */
#define MINPACKET 8


/* Maximum size of a packet we need to receive all the bits that matter.
 * This only has to cover headers we want to examine, the kernel will
 * discard any extra portion of the packet that we wouldn't look at anyway.
 */
#ifdef RAW_IP_HEADERS
#define	MAXPACKET	(sizeof(struct ip)+sizeof(struct icmp)+MINPACKET)
#else
#define MAXPACKET	(sizeof(struct icmp)+MINPACKET)
#endif
/****	Misc. data	****/



extern	int errno;


/* Where the output packet is composed */
static u_char outpack[MINPACKET];
#define ICP ((struct icmp *)(&outpack[0]))
#define WRD ((u_short *)(&outpack[0]))



/* Where input packets are received for processing */
static u_char	packet[MAXPACKET];

/* define ICMP_BASE as a "pointer" to the ICMP stuff in the packet */
#ifdef RAW_IP_HEADERS
#define ICMP_BASE	((struct icmp *)&(packet[sizeof(struct ip)]))
#else
#define ICMP_BASE	((struct icmp *)packet)
#endif


static int ntransmitted;	/* Number of packets sent in this trial */


static n_short uniqizer = 0;	/* Used in "seq" field to distinguish current
				 * polls from old ones.  Must be same type as
				 * icmp_seq field so matching works right!
				 */


static int icmp_sock;		/* Socket file descriptor */

static struct sockaddr icmp_to;	/* Who to ping */

static int MyPID;



static jmp_buf IP_Poll_Exit;
/****	Internet requires access to ICMP to poll	****/


char *
InitICMP()
{   struct protoent *proto;
    extern int sys_nerr;
    extern char *sys_errlist[];

    MyPID = getpid() & 0xFFFF;
    ICP->icmp_type = ICMP_ECHO;
    ICP->icmp_code = 0;
    ICP->icmp_id = MyPID;	/* ID */
    if ((proto = getprotobyname("icmp")) == NULL)
	return ("icmp: unknown protocol");
    if ((icmp_sock = socket(AF_INET, SOCK_RAW, proto->p_proto)) < 0)
    {	if (errno < sys_nerr)
	    return (sys_errlist[errno]);
	else
	    return ("Unknown error from socket call.");
    }
    return (NULL);
}



/*
 *			F I N I S H
 *
 * Used as the routine to run when the last alarm runs out.
 * It punts back to the poll loop and returns FALSE.
 */
static finish()
{   longjmp(IP_Poll_Exit,TRUE);
}
/****	Main poll subroutine	****/


int
IP_Poll(addr)
long addr;
{   struct sockaddr_in from;
    struct sockaddr_in *to = (struct sockaddr_in *) &icmp_to;
    extern int pinger();

    bzero( (char *)&icmp_to, sizeof(struct sockaddr) );
    to->sin_family = AF_INET;
    to->sin_addr.s_addr = addr;

    if (setjmp(IP_Poll_Exit))
    {	signal(SIGALRM, SIG_IGN);
	return (FALSE);
    }

    /* This should really arrange somehow to flush any old outstanding packets
     *	that have backed up on the socket.  In fact it would be nice if we
     *	could arrange that the socket discard them when we're not actually
     *	waiting.  I don't really know enough about all the options to figure
     *	any good way to do this (and I'm told by others you can't do it).
     *	Later, when we convert this to asynchronous operation, it will all
     *	get better again, so don't sweat it.
     */

    ntransmitted = 0;
    uniqizer++;

    signal(SIGALRM, pinger);

    pinger();			/* start things going */

    for (;;)
    {	int len = sizeof (packet);
	int fromlen = sizeof (from);
	int cc;

	if ( (cc=recvfrom(icmp_sock, packet, len, 0, &from, &fromlen)) < 0)
	{   if( errno == EINTR )
		continue;
	    perror("recvfrom");
	    continue;
	}
	if (DBG_ICMP_P) dump_packet(cc, &from); /* packet passed as global */
	if (ICMP_BASE->icmp_type != ICMP_ECHOREPLY )
	{   if (DBG_ICMP_P) fprintf(stderr,
				    "Discarded.  Type=%d, not echo reply.\n",
				    ICMP_BASE->icmp_type);
	    continue;
	}
	if (ICMP_BASE->icmp_id != MyPID )
	{   if (DBG_ICMP_P) fprintf(stderr,
				    "Discarded.  ID=%d, not my PID.\n",
				    ICMP_BASE->icmp_id);
	    continue;
	}
	if (from.sin_addr.s_addr != addr)
	{   if (DBG_ICMP_P) fprintf(stderr,
				    "Discarded.  From %x, not party tested.\n",
				    from.sin_addr.s_addr);
	    continue;
	}
	if (ICMP_BASE->icmp_seq != uniqizer)
	{   if (DBG_ICMP_P) fprintf(stderr,
				    "Discarded.  seq=%d, expecting %d.\n",
				    ICMP_BASE->icmp_seq, uniqizer);
	    continue;
	}
	/* !!!! If RAW_IP_HEADER is set, there's more bits we could look at.
	 * !!!!	Should look into which ones might be useful to test...
	 */
	/* Only need one for positive result */
	signal(SIGALRM, SIG_IGN);
	return (TRUE);
    }
    /*NOTREACHED*/
}
/****	pinger	-Send one out	****/



/* 			P I N G E R
 * 
 * Compose and transmit an ICMP ECHO REQUEST packet.  The IP packet
 * will be added on by the kernel.  The ID field is our UNIX process ID,
 * and the sequence number is an ascending integer.  We then schedule
 * another SIGALRM for 1 second from now if need to send more, else longer
 * to give time for round-trip.
 */
static pinger()
{   /**register struct icmp *icp = (struct icmp *) outpack;**/
    int i;
    extern int finish();

    ICP->icmp_seq = uniqizer;
    /* For some reason this loses late... so check to start. */
    if (ICP->icmp_seq != uniqizer)
    {	fprintf(stderr, "Setting seq lost... u=%d (%o), s=%d (%o)\n",
		ICP->icmp_seq, ICP->icmp_seq, uniqizer, uniqizer);
    }

    /* Compute ICMP checksum here, with fixed length it's easier than call */
    i = (WRD[0] + WRD[2] + WRD[3]);
    i += i>>16;
    ICP->icmp_cksum = ~i;

    i = sendto( icmp_sock, outpack, MINPACKET, 0, &icmp_to, sizeof(struct sockaddr) );
    ntransmitted++;

    if( i < 0 || i != MINPACKET )
	longjmp(IP_Poll_Exit,TRUE);

    if (ntransmitted <= NBRTRIES)
	alarm(1);
    else
    {	signal(SIGALRM, finish);
	alarm(MAXWAIT);
    }
}
/****	Print out a received packet for debugging.	****/



/*** Helper routines ***/

/*
 * 			P R _ T Y P E
 *
 * Convert an ICMP "type" field to a printable string.
 */
char *
pr_type( t )
register int t;
{   static char *ttab[] = {
		"Echo Reply",
		"ICMP 1",
		"ICMP 2",
		"Dest Unreachable",
		"Source Quence",
		"Redirect",
		"ICMP 6",
		"ICMP 7",
		"Echo",
		"ICMP 9",
		"ICMP 10",
		"Time Exceeded",
		"Parameter Problem",
		"Timestamp",
		"Timestamp Reply",
		"Info Request",
		"Info Reply"
	};

    if( t < 0 || t > 16 )
	return("OUT-OF-RANGE");

    return(ttab[t]);
}


hex_dump(p, len)
register u_char *p ;
register int len ;
{   register int i = 0, j ;
	
    while (len > 0)
    {	fprintf(stderr, "x%2.2x: ", i );
	for (j = 0; j<4; j++)
	{   if (len > 4)
	    {	fprintf(stderr, " x%8.8x", * (long *) p) ;
		p += sizeof(long) ;
		i += sizeof(long) ;
		len -= sizeof(long) ;
	    }
	    else {
		fprintf(stderr, " x%*.*x",  2*len, 2*len, * (long *) p) ;
		len = 0 ;
		break ;
	    }
	}
	fprintf(stderr, "\n") ;
    }
}


/*	dump_packet()
 */
dump_packet(cc, from_p)
int cc;				/* Length received */
struct sockaddr_in *from_p;
{   fprintf(stderr, "\nICMP: %d bytes from %s: icmp type=%d (%s) code=%d\n",
		    cc,
		    inet_ntoa( from_p->sin_addr),
		    ICMP_BASE->icmp_type,
		    pr_type(ICMP_BASE->icmp_type),
		    ICMP_BASE->icmp_code
	    );
    hex_dump((u_char *)ICMP_BASE, cc);
}
